import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { CommunicationTemplates } from 'src/common/entities/communication-templates.entity';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { CommunicationTemplatesKeysEnum } from 'src/common/enum/communication-template-keys.enum';
import { CommunicationTypeEnum } from 'src/common/enum/communication-type.enum';

@Injectable()
export class CommunicationTemplatesRepository {
  constructor(
    @InjectRepository(CommunicationTemplates)
    private readonly repo: Repository<CommunicationTemplates>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}


  async findAll(limit: number, offset: number) {
    try {
      const data = await this.commonDataService.get(
        this.repo,
        undefined,
        {},
        limit,
        offset,
        { id: 'ASC' },
      );
      const total = await this.repo.count();
      return {
        data,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    try {
      const data = await this.commonDataService.findOneById(this.repo, id, false);
      if (!data) {
        throw new InifniNotFoundException(ERROR_CODES.COMMUNICATION_TEMPLATE_NOTFOUND, null, null, id.toString());
      }
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_FIND_BY_ID_FAILED, error);
    }
  }
  async save(template: CommunicationTemplates) {
    try {
      const savedTemplate = await this.repo.save(template);
      return savedTemplate;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_SAVE_FAILED, error);
    }
  }
  async findByTemplateKey(templateKey: CommunicationTemplatesKeysEnum) {
    try {
      const data = await this.repo.findOne({ where: { templateKey: templateKey } });
      if (!data) {
        throw new InifniNotFoundException(ERROR_CODES.COMMUNICATION_TEMPLATE_NOTFOUND, null, null, templateKey);
      }
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_FIND_BY_KEY_FAILED, error);
    }
  }

  async findByTemplateKeyAndType(templateKey: CommunicationTemplatesKeysEnum, templateType: CommunicationTypeEnum) {
    try {
      const template = await this.repo.findOne({
        where: { templateKey: templateKey, templateType: templateType },
      });
      if (!template) {
        throw new InifniNotFoundException(ERROR_CODES.COMMUNICATION_TEMPLATE_NOTFOUND, null, null, `${templateKey} with type ${templateType}`);
      }
      return template;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_FIND_BY_KEY_AND_TYPE_FAILED, error);
    }
  }
  async findAllBy(data: any, limit: number, offset: number) {
    try {
      const { templateType, templateKey, category, programKey } = data;
      const query = this.repo.createQueryBuilder('template');

      if (templateType) {
        query.andWhere('template.templateType = :templateType', { templateType });
      }
      if (templateKey) {
        query.andWhere('template.templateKey = :templateKey', { templateKey });
      }
      if (category) {
        query.andWhere('template.category = :category', { category });
      }
      if (programKey) {
        query.andWhere('template.programKey = :programKey', { programKey });
      }

      query.skip(offset).take(limit);

      const [dataResult, total] = await query.getManyAndCount();

      return {
        data: dataResult,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
          numberOfRecords: dataResult.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_GET_FAILED, error);
    }
  }
}