import { Injectable } from '@nestjs/common';
import { AwsS3Service } from './awsS3.service';
import * as excelJs from 'exceljs';
import { generateTimestampedKey } from '../utils/generateTimestampedKey.util';

export interface ExcelColumn {
  key: string;
  header: string;
  width?: number;
  type?: 'string' | 'number' | 'date' | 'boolean';
}

export interface ExcelOptions {
  sheetName?: string;
  columns?: ExcelColumn[];
  makeHeaderBold?: boolean;
  autoWidth?: boolean;
}

@Injectable()
export class ExcelService {
  constructor(private readonly awsS3Service: AwsS3Service) {}

  /**
   * Convert JSON data to Excel buffer
   * @param jsonData - JSON data to be converted
   * @param options - Excel formatting options
   * @returns Excel buffer
   */
  async jsonToExcelBuffer(
    jsonData: any[], 
    options: ExcelOptions = {}
  ): Promise<Buffer> {
    try {
      if (!jsonData || jsonData.length === 0) {
        throw new Error('No data provided for Excel generation');
      }
  
      // Create a new workbook and worksheet
      const workbook = new excelJs.Workbook();
      const worksheet = workbook.addWorksheet(options.sheetName || 'Sheet 1');
  
      // Configure columns
      if (options.columns && options.columns.length > 0) {
        // Use provided column configuration
        worksheet.columns = options.columns.map(col => ({
          header: col.header,
          key: col.key,
          width: col.width || 15,
        }));
  
        // Add rows with specified columns only
        jsonData.forEach(data => {
          const rowData: any = {};
          options.columns!.forEach(col => {
            rowData[col.key] = data[col.key] || '';
          });
          worksheet.addRow(rowData);
        });
      } else {
        // Auto-generate columns from first object
        const headers = Object.keys(jsonData[0]);
        worksheet.columns = headers.map(header => ({
          header,
          key: header,
          width: 15
        }));
  
        // Add all data rows
        jsonData.forEach(data => {
          worksheet.addRow(data);
        });
      }
  
      // Style the header row
      if (options.makeHeaderBold !== false) {
        const headerRow = worksheet.getRow(1);
        headerRow.font = { bold: true };
        headerRow.fill = {
          type: 'pattern',
          pattern: 'solid',
          fgColor: { argb: 'E1E1E1' }
        };
        headerRow.alignment = { horizontal: 'left', vertical: 'middle' };
      }
  
      // Auto-fit columns if requested
      if (options.autoWidth) {
        worksheet.columns.forEach(column => {
          let maxLength = 0;
          column.eachCell!({ includeEmpty: true }, (cell) => {
            const columnLength = cell.value ? cell.value.toString().length : 10;
            if (columnLength > maxLength) {
              maxLength = columnLength;
            }
          });
          column.width = maxLength < 10 ? 10 : maxLength;
        });
      }
  
      // Convert the workbook to a buffer
      const buffer = await workbook.xlsx.writeBuffer();
      
      return Buffer.from(buffer);
    } catch (error) {
      console.error('Excel generation error:', error);
      throw new Error('Failed to generate Excel file: ' + error.message);
    }
  }

  /**
   * Convert JSON data to Excel and upload to S3
   * @param jsonData - JSON data to be converted
   * @param filename - desired filename
   * @param contentType - content type
   * @param options - Excel formatting options
   * @returns S3 URL of the uploaded file
   */
  async jsonToExcelAndUpload(
    jsonData: any[], 
    filename: string, 
    contentType: string,
    options: ExcelOptions = {}
  ): Promise<string> {
    try {
      const buffer = await this.jsonToExcelBuffer(jsonData, options);
      const s3Key = generateTimestampedKey('exports/excel', filename);
      const result = await this.awsS3Service.uploadToS3(s3Key, buffer, contentType);
      
      // Get S3 URL from bucket and key instead of Location property
      const s3Url = await this.awsS3Service.getSignedUrl(s3Key);
      
      return s3Url;
    } catch (error) {
      throw error;
    }
  }
}
