import { Injectable } from '@nestjs/common';
import { DataSource, EntityManager, IsNull } from 'typeorm';
import {
  ProgramRegistration,
  RegistrationGroupMap,
  RegistrationPairMap,
  RegistrationPair,
  RoomAllocation,
  RoomAllocationHistory,
} from '../entities';
import { AppLoggerService } from './logger.service';
import { handleKnownErrors } from '../utils/handle-error.util';
import { ERROR_CODES } from '../constants/error-string-constants';
import { formatRoomAllocationHistoryRemarks } from '../utils/room-allocation-history.util';
import { ProgramRoomInventoryMapService } from '../../room-inventory/program-room-inventory-map.service';

@Injectable()
export class AllocationClearingService {
  constructor(
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService,
    private readonly programRoomInventoryMapService: ProgramRoomInventoryMapService,
  ) {}

  /**
   * Clear all allocations (group, pair, room) for a registration with detailed history tracking
   * @param registrationId - The registration ID to clear allocations for
   * @param userId - The user performing the operation
   * @param clearanceReason - The reason for clearing allocations
   * @param transactionManager - Optional EntityManager from an existing transaction. If provided, uses it instead of creating a new transaction.
   * @returns Object containing clearing results and room inventory updates needed
   */
  async clearRegistrationAllocations(
    registrationId: number,
    userId: number,
    clearanceReason?: string,
    transactionManager?: EntityManager,
  ): Promise<{
    isGrouped: boolean;
    isPaired: boolean;
    isRoomAllocated: boolean;
    roomInventoryUpdates: Array<{ roomInventoryId: number; deallocatedCount: number }>;
  }> {
    // If a transaction manager is provided, use it directly; otherwise, create a new transaction
    if (transactionManager) {
      return await this.executeClearance(registrationId, userId, clearanceReason, transactionManager);
    }

    return await this.dataSource.transaction(async (manager) => {
      return await this.executeClearance(registrationId, userId, clearanceReason, manager);
    });
  }

  /**
   * Internal method to execute the clearance logic
   * @private
   */
  private async executeClearance(
    registrationId: number,
    userId: number,
    clearanceReason: string | undefined,
    manager: EntityManager,
  ): Promise<{
    isGrouped: boolean;
    isPaired: boolean;
    isRoomAllocated: boolean;
    roomInventoryUpdates: Array<{ roomInventoryId: number; deallocatedCount: number }>;
  }> {
    let isGrouped = false;
    let isPaired = false;
    let isRoomAllocated = false;
    let roomInventoryUpdates: Array<{ roomInventoryId: number; deallocatedCount: number }> = [];

    try {
        const groupMapRepo = manager.getRepository(RegistrationGroupMap);
        const pairMapRepo = manager.getRepository(RegistrationPairMap);
        const pairRepo = manager.getRepository(RegistrationPair);
        const roomAllocationRepo = manager.getRepository(RoomAllocation);
        const roomAllocationHistoryRepo = manager.getRepository(RoomAllocationHistory);

        const currentDate = new Date();

        // Fetch all data in parallel - single database round trip
        const [registration, existingGroup, existingPair, existingRoomAllocations] = await Promise.all([
          manager.findOne(ProgramRegistration, {
            where: { id: registrationId },
            relations: ['allocatedProgram', 'program'],
          }),
          groupMapRepo.findOne({
            where: { registrationId, deletedAt: IsNull() },
          }),
          pairMapRepo.findOne({
            where: { registrationId, deletedAt: IsNull() },
          }),
          roomAllocationRepo.find({
            where: { registrationId, deletedAt: IsNull() },
            relations: ['programRoomInventoryMap', 'programRoomInventoryMap.room'],
          }),
        ]);

        if (!registration) {
          throw new Error(`Registration ${registrationId} not found`);
        }

        // Prepare all updates
        const updatePromises: Promise<any>[] = [];

        // Clear group mappings
        if (existingGroup) {
          updatePromises.push(
            groupMapRepo.update(existingGroup.id, {
              deletedAt: currentDate,
              updatedAt: currentDate,
              updatedById: userId,
            })
          );
          isGrouped = true;

          this.logger.log('Group mapping cleared', {
            registrationId,
            groupId: existingGroup.id,
            reason: clearanceReason,
            userId,
          });
        }

        // Clear pair mappings - find and delete both sides of the pair
        if (existingPair) {
          // Find all pair mappings for the same pair (both registrations)
          const allPairMappings = await pairMapRepo.find({
            where: { 
              registrationPairId: existingPair.registrationPairId, 
              deletedAt: IsNull() 
            },
          });

          // Delete all pair mappings
          if (allPairMappings.length > 0) {
            const pairMapIds = allPairMappings.map(pm => pm.id);
            const pairRegistrationIds = allPairMappings.map(pm => pm.registrationId);
            
            updatePromises.push(
              pairMapRepo.update(pairMapIds, {
                deletedAt: currentDate,
                updatedAt: currentDate,
                updatedById: userId,
              })
            );

            // Delete the RegistrationPair entity itself
            updatePromises.push(
              pairRepo.update(existingPair.registrationPairId, {
                deletedAt: currentDate,
                updatedAt: currentDate,
                updatedById: userId,
              })
            );

            isPaired = true;

            this.logger.log('Complete pair and pair mappings cleared', {
              registrationId,
              registrationPairId: existingPair.registrationPairId,
              pairMapIds,
              pairRegistrationIds,
              reason: clearanceReason,
              userId,
            });
          }
        }

        // Handle room allocations with detailed history tracking
        if (existingRoomAllocations.length > 0) {
          // Group allocations by room inventory to count deallocations per room
          const roomInventoryDeallocations = new Map<number, number>();
          const historyRecords: RoomAllocationHistory[] = [];
          const allocationIds: number[] = [];

          // Prepare all history records and collect allocation IDs
          for (const allocation of existingRoomAllocations) {
            const roomInventoryId = allocation.programRoomInventoryMap.id;
            const currentCount = roomInventoryDeallocations.get(roomInventoryId) || 0;
            roomInventoryDeallocations.set(roomInventoryId, currentCount + 1);

            // Create detailed history record before deletion
            const historyRemarks = formatRoomAllocationHistoryRemarks(
              'cleared',
              allocation,
              clearanceReason || 'approval status change',
              registration
            );

            const historyRecord = manager.create(RoomAllocationHistory, {
              roomAllocationId: allocation.id,
              programRoomInventoryMapId: allocation.programRoomInventoryMapId,
              registrationId: allocation.registrationId,
              bedPosition: allocation.bedPosition,
              remarks: historyRemarks,
              createdById: userId,
              updatedById: userId,
              createdAt: currentDate,
              updatedAt: currentDate,
            });

            historyRecords.push(historyRecord);
            allocationIds.push(allocation.id);

            this.logger.log('Room allocation cleared with history', {
              allocationId: allocation.id,
              registrationId,
              roomId: allocation.programRoomInventoryMap.room?.id,
              roomLabel: allocation.programRoomInventoryMap.room?.label,
              bedPosition: allocation.bedPosition,
              reason: clearanceReason,
              userId,
            });
          }

          // Bulk insert history records
          if (historyRecords.length > 0) {
            updatePromises.push(roomAllocationHistoryRepo.save(historyRecords));
          }

          // Bulk update room allocations
          if (allocationIds.length > 0) {
            updatePromises.push(
              roomAllocationRepo.update(allocationIds, {
                deletedAt: currentDate,
                updatedAt: currentDate,
                updatedById: userId,
              })
            );
          }

          // Prepare room inventory updates
          roomInventoryUpdates = Array.from(roomInventoryDeallocations.entries()).map(
            ([roomInventoryId, deallocatedCount]) => ({
              roomInventoryId,
              deallocatedCount,
            })
          );

          isRoomAllocated = true;
        }

        // Execute all updates in parallel
        if (updatePromises.length > 0) {
          await Promise.all(updatePromises);
        }

        // Update room occupancy/status for each affected room inventory within the same transaction
        if (roomInventoryUpdates.length > 0) {
          for (const update of roomInventoryUpdates) {
            await this.programRoomInventoryMapService.updateRoomOccupancyAfterDeallocation(
              update.roomInventoryId,
              update.deallocatedCount,
              manager,
            );
            this.logger.log('Room inventory occupancy/status updated after deallocation', {
              roomInventoryId: update.roomInventoryId,
              deallocatedCount: update.deallocatedCount,
            });
          }
        }

      this.logger.log('Cleared allocations for approval update with history tracking', {
        registrationId,
        isGrouped,
        isPaired,
        isRoomAllocated,
        roomInventoryUpdates: roomInventoryUpdates.length,
        clearanceReason: clearanceReason || 'approval status change',
        allocatedProgram: registration.allocatedProgram?.name,
        userId,
      });

      return {
        isGrouped,
        isPaired,
        isRoomAllocated,
        roomInventoryUpdates,
      };

    } catch (error) {
      this.logger.error('Error clearing allocations for approval update', '', {
        error,
        registrationId,
        clearanceReason,
        userId,
      });
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }
}