import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  CreateDateColumn,
  UpdateDateColumn,
  ManyToOne,
  JoinColumn,
  DeleteDateColumn,
  OneToMany,
} from 'typeorm';
import { PaymentModeEnum } from '../enum/payment-mode.enum';
import { PaymentStatusEnum } from '../enum/payment-status.enum';
import { ProgramRegistration, ProgramSession, User } from '../entities/index';
import { RazorpayPaymentStatusEnum } from '../enum/razorpay-payment-status.enum';
import { Auditable } from 'src/audit-history/decorators/auditable.decorator';
import { SkipAudit } from 'src/audit-history/decorators/skip-audit.decorator';
import { PaymentEditRequest } from './payment-edit-request.entity';

@Entity('hdb_registration_payment_detail')
@Auditable()
export class RegistrationPaymentDetail {
  @PrimaryGeneratedColumn('increment', { type: 'bigint' })
  id: number;

  @Column({ name: 'registration_id', type: 'bigint' })
  registrationId: number;

  @Column({ name: 'program_session_id', type: 'bigint' })
  programSessionId: number;

  @Column({ name: 'payment_mode', type: 'enum', enum: PaymentModeEnum })
  paymentMode: PaymentModeEnum;

  @Column({ name: 'tax_amount', type: 'decimal', precision: 10, scale: 2, default: 0.00 })
  taxAmount: number;

  @Column({ name: 'tds', type: 'decimal', precision: 10, scale: 2, default: 0.00 })
  tds: number;

  @Column({ name: 'sub_total', type: 'decimal', precision: 10, scale: 2, default: 0.00 })
  subTotal: number;

  @Column({ name: 'razorpay_id', type: 'varchar', length: 100, nullable: true })
  razorpayId: string;

  @Column({ name: 'offline_meta', type: 'jsonb', nullable: true })
  offlineMeta: Record<string, any>;

  @Column({ name: 'original_amount', type: 'decimal', precision: 10, scale: 2, default: 0.00 })
  originalAmount: number;

  @Column({ name: 'gst_amount', type: 'decimal', precision: 10, scale: 2, default: 0.00 })
  gstAmount: number;

  @Column({
    name: 'payment_status',
    type: 'enum',
    enum: PaymentStatusEnum,
    default: PaymentStatusEnum.DRAFT,
  })
  paymentStatus: PaymentStatusEnum;

  @Column({ name: 'mark_as_received_date', type: 'timestamptz', nullable: true })
  markAsReceivedDate: Date;

  @Column({ name: 'portal_online_payment_status', type: 'enum', enum: PaymentStatusEnum, nullable: true })
  portalOnlinePaymentStatus: RazorpayPaymentStatusEnum | null;

  @Column({ name: 'portal_online_payment_attempt_status', type: 'jsonb', nullable: true })
  portalOnlinePaymentAttemptStatus: Record<string, any>[] | null;

  @Column({ name: 'webhook_online_payment_status', type: 'enum', enum: PaymentStatusEnum, nullable: true })
  webhookOnlinePaymentStatus: RazorpayPaymentStatusEnum | null;

  @Column({ name: 'webhook_online_payment_attempt_status', type: 'jsonb', nullable: true })
  webhookOnlinePaymentAttemptStatus: Record<string, any>[] | null;

  @Column({ name: 'gateway_online_payment_status', type: 'enum', enum: PaymentStatusEnum, nullable: true })
  gatewayOnlinePaymentStatus: RazorpayPaymentStatusEnum | null;

  @Column({ name: 'gateway_online_payment_attempt_status', type: 'jsonb', nullable: true })
  gatewayOnlinePaymentAttemptStatus: Record<string, any>[] | null;

  @Column({ name: 'payment_date', type: 'timestamptz', nullable: true })
  paymentDate: Date;

  @Column({ name: 'audit_ref_id', type: 'bigint', nullable: true })
  auditRefId: number;

  @Column({ name: 'parent_ref_id', type: 'bigint', nullable: true })
  parentRefId: number;

  @SkipAudit()
  @CreateDateColumn({ name: 'created_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP' })
  createdAt: Date;

  @SkipAudit()
  @UpdateDateColumn({ name: 'updated_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP', onUpdate: 'CURRENT_TIMESTAMP' })
  updatedAt: Date;

  @DeleteDateColumn({ name: 'deleted_at', type: 'timestamptz', nullable: true, default: null })
  deletedAt: Date;

  @SkipAudit()
  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'created_by' })
  createdBy: User;

  @SkipAudit()
  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'updated_by' })
  updatedBy: User;

  @SkipAudit()
  @ManyToOne(() => ProgramRegistration, { nullable: false, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'registration_id' })
  registration: ProgramRegistration;

  @SkipAudit()
  @ManyToOne(() => ProgramSession, { nullable: false, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'program_session_id' })
  programSession: ProgramSession;

  @SkipAudit()
  @OneToMany(() => PaymentEditRequest, (editRequest) => editRequest.payment)
  editRequests?: PaymentEditRequest[];

  constructor(partial: Partial<RegistrationPaymentDetail>) {
    Object.assign(this, partial);
  }
}
