import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  CreateDateColumn,
  UpdateDateColumn,
  ManyToOne,
  JoinColumn,
  DeleteDateColumn,
} from 'typeorm';
import { Program } from './program.entity';
import { User } from './user.entity';
import {
  ModeOfOperationEnum,
  OnlineTypeEnum,
  FrequencyEnum,
  RegistrationLevelEnum,
} from '../enum/program.enums';
import { CommonStatus } from '../enum/common-status.enum';
import { Address } from './address.entity';
import { TdsApplicabilityEnum } from '../enum/tds-applicability.enum';

/**
 * Entity representing a program session in version 1
 * Contains all session related information including scheduling, registration, and meeting details
 */
@Entity('program_session')
export class ProgramSession {
  @PrimaryGeneratedColumn('increment')
  id: number;

  @Column({ name: 'program_id', nullable: false }) // was nullable: true
  programId: number;

  @Column({ name: 'name', type: 'varchar', length: 255, nullable: false })
  name: string;

  @Column({ name: 'description', type: 'text', nullable: true })
  description: string;

  @Column({ name: 'registration_starts_at', type: 'timestamptz', nullable: true })
  registrationStartsAt: Date;

  @Column({ name: 'registration_ends_at', type: 'timestamptz', nullable: true })
  registrationEndsAt: Date;

  @Column({ name: 'checkin_at', type: 'timestamptz', nullable: true })
  checkinAt: Date;

  @Column({ name: 'checkout_at', type: 'timestamptz', nullable: true })
  checkoutAt: Date;

  @Column({ name: 'starts_at', type: 'timestamptz', nullable: true })
  startsAt: Date;

  @Column({ name: 'ends_at', type: 'timestamptz', nullable: true })
  endsAt: Date;

  @Column({ name: 'mode_of_operation', type: 'enum', enum: ModeOfOperationEnum, nullable: false })
  modeOfOperation: ModeOfOperationEnum;

  @Column({ name: 'online_type', type: 'enum', enum: OnlineTypeEnum, default: OnlineTypeEnum.NA })
  onlineType: OnlineTypeEnum;

  @Column({ name: 'max_session_duration_days', type: 'int', default: 8 })
  maxSessionDurationDays: number;

  @Column({ name: 'has_multiple_sessions', type: 'boolean', default: true })
  hasMultipleSessions: boolean;

  @Column({ name: 'frequency', type: 'enum', enum: FrequencyEnum, default: FrequencyEnum.YEARLY })
  frequency: FrequencyEnum;

  @Column({ name: 'default_start_time', type: 'time', nullable: true })
  defaultStartTime: string;

  @Column({ name: 'default_end_time', type: 'time', nullable: true })
  defaultEndTime: string;

  @Column({ name: 'duration', type: 'varchar', length: 50, nullable: true })
  duration: string;

  @Column({ name: 'requires_residence', type: 'boolean', default: false })
  requiresResidence: boolean;

  @Column({ name: 'involves_travel', type: 'boolean', default: false })
  involvesTravel: boolean;

  @Column({ name: 'has_checkin_checkout', type: 'boolean', default: false })
  hasCheckinCheckout: boolean;

  @Column({ name: 'requires_payment', type: 'boolean', default: true })
  requiresPayment: boolean;

  @Column({ name: 'requires_attendance_all_sessions', type: 'boolean', default: true })
  requiresAttendanceAllSessions: boolean;

  @Column({ name: 'allows_minors', type: 'boolean', default: false })
  allowsMinors: boolean;

  @Column({ name: 'allows_proxy_registration', type: 'boolean', default: false })
  allowsProxyRegistration: boolean;

  @Column({ name: 'requires_approval', type: 'boolean', default: false })
  requiresApproval: boolean;

  @Column({
    name: 'registration_level',
    type: 'enum',
    enum: RegistrationLevelEnum,
    default: RegistrationLevelEnum.SESSION,
  })
  registrationLevel: RegistrationLevelEnum;

  @Column({ name: 'waitlist_applicable', type: 'boolean', default: false })
  waitlistApplicable: boolean;

  @Column({ name: 'limited_seats', type: 'boolean', default: false })
  limitedSeats: boolean;

  @Column({ name: 'is_grouped_program', type: 'boolean', default: false })
  isGroupedProgram: boolean;

  @Column({ name: 'total_seats', type: 'int', nullable: true })
  totalSeats: number;

  @Column({ name: 'waitlist_trigger_count', type: 'int', nullable: true })
  waitlistTriggerCount: number;

  @Column({ name: 'available_seats', type: 'int', nullable: true })
  availableSeats: number;

  @Column({ name: 'reserved_seats', type: 'int', default: 0 })
  reservedSeats: number;

  @Column({ name: 'filled_seats', type: 'int', default: 0 })
  filledSeats: number;

  @Column({ name: 'max_capacity', type: 'int', nullable: true })
  maxCapacity: number;

  @Column({ name: 'session_fee', type: 'decimal', precision: 10, scale: 2, nullable: true })
  sessionFee: number;

  @Column({ name: 'base_price', type: 'decimal', precision: 10, scale: 2, default: 0.0 })
  basePrice: number;

  @Column({ name: 'gst_percentage', type: 'decimal', precision: 5, scale: 2, default: 18.0 })
  gstPercentage: number;

  @Column({ name: 'cgst', type: 'decimal', precision: 5, scale: 2, default: 0 })
  cgst: number;

  @Column({ name: 'sgst', type: 'decimal', precision: 5, scale: 2, default: 0 })
  sgst: number;

  @Column({ name: 'igst', type: 'decimal', precision: 5, scale: 2, default: 0 })
  igst: number;


  @Column({ name: 'tds_percent', type: 'decimal', precision: 5, scale: 2, default: 0 })
  tdsPercent: number;

  @Column({
    name: 'tds_applicability',
    type: 'enum',
    enum: TdsApplicabilityEnum,
    nullable: true,
  })
  tdsApplicability: TdsApplicabilityEnum;

  @Column({ name: 'invoice_sender_name', type: 'varchar', length: 255, nullable: true })
  invoiceSenderName: string;

  @Column({ name: 'invoice_sender_pan', type: 'varchar', length: 50, nullable: true })
  invoiceSenderPan: string;

  @Column({ name: 'invoice_sender_cin', type: 'varchar', length: 50, nullable: true })
  invoiceSenderCin: string;

  @Column({ name: 'invoice_sender_address', type: 'text', nullable: true })
  invoiceSenderAddress: string;

  @Column({ name: 'venue_address_id', type: 'int', nullable: true })
  venueAddressId: number;

  @ManyToOne(() => Address, { nullable: true })
  @JoinColumn({ name: 'venue_address_id' })
  venueAddress: Address;

  @Column({ name:'gst_number', type: 'varchar', length: 15, nullable: true })
  gstNumber: string;

  @Column({ name: 'currency', type: 'varchar', length: 3, default: 'INR' })
  currency: string;

  @Column({ name: 'venue', type: 'text', nullable: true })
  venue: string;

  @Column({ name: 'helpline_number', type: 'text', nullable: true })
  helplineNumber: string;

  @Column({ name: 'email_sender_name', type: 'text', nullable: true })
  emailSenderName: string;

  @Column({ name: 'venue_name_in_emails', type: 'text', nullable: true })
  venueNameInEmails: string;

  @Column({ name: 'launch_date', type: 'timestamptz', nullable: true })
  launchDate: Date;

  @Column({ name: 'status', type: 'enum', enum: CommonStatus, default: CommonStatus.DRAFT })
  status: CommonStatus;

  @Column({ name: 'is_active', type: 'boolean', default: true })
  isActive: boolean;

  @Column({ name: 'code', type: 'varchar', length: 50, nullable: true })
  code: string;

  @Column({ name: 'display_order', type: 'int', default: 1 })
  displayOrder: number;

  @Column({ name: 'banner_image_url', type: 'varchar', length: 1024, nullable: true })
  bannerImageUrl: string;

  @Column({ name: 'banner_animation_url', type: 'varchar', length: 1024, nullable: true })
  bannerAnimationUrl: string;

  @Column({ name: 'meeting_link', type: 'varchar', length: 500, nullable: true })
  meetingLink: string;

  @Column({ name: 'meeting_id', type: 'varchar', length: 100, nullable: true })
  meetingId: string;

  @Column({ name: 'meeting_password', type: 'varchar', length: 100, nullable: true })
  meetingPassword: string;

  @Column({ name: 'meta', type: 'json', nullable: true, default: () => "'{}'" })
  meta: Record<string, any>;

  @Column({ name: 'is_travel_required', type: 'boolean', default: false })
  isTravelRequired: boolean;

  @Column({ name: 'logo_url', type:'varchar', length: 1024, nullable: true })
  logoUrl: string;

  @CreateDateColumn({ name: 'created_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP' })
  createdAt: Date;

  @UpdateDateColumn({ name: 'updated_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP' })
  updatedAt: Date;

  @DeleteDateColumn({ name: 'deleted_at', type: 'timestamptz', nullable: true })
  deletedAt: Date;

  @Column({ name: 'created_by', type: 'int', nullable: true })
  createdBy: number;

  @Column({ name: 'updated_by', type: 'int', nullable: true })
  updatedBy: number;

  @ManyToOne(() => Program, { nullable: true })
  @JoinColumn({ name: 'program_id' })
  program: Program;

  @ManyToOne(() => User, { nullable: true })
  @JoinColumn({ name: 'created_by' })
  creator: User;

  @ManyToOne(() => User, { nullable: true })
  @JoinColumn({ name: 'updated_by' })
  updater: User;

  @Column({ name: 'checkin_ends_at', type: 'timestamptz', nullable: true })
  checkinEndsAt: Date;

  @Column({ name: 'checkout_ends_at', type: 'timestamptz', nullable: true })
  checkoutEndsAt: Date;

  constructor(partial: Partial<ProgramSession>) {
    Object.assign(this, partial);
  }
}
