import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  CreateDateColumn,
  UpdateDateColumn,
  DeleteDateColumn,
  ManyToOne,
  JoinColumn,
} from 'typeorm';
import { Program, ProgramSession, ProgramRegistration, User } from './index';
import { Auditable } from 'src/audit-history/decorators/auditable.decorator';
import { SkipAudit } from 'src/audit-history/decorators/skip-audit.decorator';

@Entity('hdb_preference')
@Auditable()
export class Preference {
  @PrimaryGeneratedColumn('increment', { type: 'bigint' })
  id: number;

  @ManyToOne(() => Program, { nullable: false, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'program_id' })
  program: Program;

  @ManyToOne(() => ProgramSession, { nullable: true, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'session_id' })
  session: ProgramSession;

  @ManyToOne(() => ProgramSession, { nullable: true, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'preferred_session_id' })
  preferredSession?: ProgramSession;

  @ManyToOne(() => Program, { nullable: true, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'preferred_program_id' })
  preferredProgram?: Program;

  @ManyToOne(() => ProgramRegistration, { nullable: false, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'registration_id' })
  registration: ProgramRegistration;

  @Column({ name: 'priority_order', type: 'int', default: 0 })
  priorityOrder: number;

  @Column({ name: 'audit_ref_id', type: 'bigint', nullable: true })
  auditRefId: number;

  @Column({ name: 'parent_ref_id', type: 'bigint', nullable: true })
  parentRefId: number;

  @SkipAudit()
  @CreateDateColumn({ name: 'created_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP' })
  createdAt: Date;

  @SkipAudit()
  @UpdateDateColumn({ name: 'updated_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP', onUpdate: 'CURRENT_TIMESTAMP' })
  updatedAt: Date;

  @DeleteDateColumn({ name: 'deleted_at', type: 'timestamptz', nullable: true, default: null })
  deletedAt: Date;

  @SkipAudit()
  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'created_by' })
  createdBy: User;

  @SkipAudit()
  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'updated_by' })
  updatedBy: User;

  constructor(partial: Partial<Preference>) {
    Object.assign(this, partial);
  }
}