import { Injectable, Inject, forwardRef } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { AuditHistoryLog } from 'src/common/entities/audit-history-log.entity';
import { Repository, Between, FindOptionsWhere } from 'typeorm';
import { RegistrationActivityQueryDto } from '../dto/registration-activity-query.dto';
import { UserService } from 'src/user/user.service';
import {
  IntelligentActivityResponse,
  ActivityLogItem,
  ActivityLogChange,
} from '../interfaces/intelligent-activity.interface';
import { ProgramService } from 'src/program/program.service';
import { EXCLUDED_ACTIVITY_SUBCATEGORIES } from '../audit-history.constants';

@Injectable()
export class AuditHistoryService {
  constructor(
    @InjectRepository(AuditHistoryLog)
    private auditHistoryLogRepository: Repository<AuditHistoryLog>,
    @Inject(forwardRef(() => UserService))
    private userService: UserService,
    @Inject (forwardRef(() => ProgramService))
    private programService: ProgramService,
  ) {}

  async createAuditLog(log: Partial<AuditHistoryLog>): Promise<AuditHistoryLog> {
    const entity = this.auditHistoryLogRepository.create(log);
    return this.auditHistoryLogRepository.save(entity);
  }


  /**
   * Group logs by requestId
   */
  private groupLogsByRequestId(logs: AuditHistoryLog[]): Map<string, AuditHistoryLog[]> {
    const groups = new Map<string, AuditHistoryLog[]>();

    logs.forEach((log) => {
      const requestId = log.requestId || 'no-request-id';
      if (!groups.has(requestId)) {
        groups.set(requestId, []);
      }
      groups.get(requestId)!.push(log);
    });

    return groups;
  }

  /**
   * Get user information for all users involved in the logs
   */
  private async getUserMapForLogs(logs: AuditHistoryLog[]): Promise<Map<number, string>> {
    const userIds: Set<number> = new Set();

    logs.forEach((log) => {
      if (log.userId && typeof log.userId === 'string' && log.userId.trim() !== '' && 
          log.userId !== 'null' && log.userId !== 'undefined') {
        const cleanUserId = log.userId.replace(/^['"]|['"]$/g, '').trim();
        const numericUserId = parseInt(cleanUserId, 10);
        if (!isNaN(numericUserId)) {
          userIds.add(numericUserId);
        }
      }
    });

    const userMap = new Map<number, string>();
    if (userIds.size > 0) {
      try {
        const userPromises = Array.from(userIds).map(async (userId) => {
          try {
            const user = await this.userService.getUserById(userId);
            return { userId, name: user.legalFullName || user.fullName || user.firstName || 'Seeker' };
          } catch (error) {
            console.warn(`Failed to fetch user with ID ${userId}:`, error);
            return { userId, name: 'Seeker' };
          }
        });

        const users = await Promise.all(userPromises);
        users.forEach(({ userId, name }) => {
          userMap.set(userId, name);
        });
      } catch (error) {
        console.error('Error fetching user information:', error);
      }
    }

    return userMap;
  }





  /**
   * Get intelligent activity log grouped by requestId with context-aware messages
   */
  async getIntelligentActivityLog(
    registrationId: string,
    query: RegistrationActivityQueryDto,
  ): Promise<IntelligentActivityResponse> {
    const where: FindOptionsWhere<AuditHistoryLog> = { parentRefId: registrationId };

    if (query.startDate && query.endDate) {
      where.createdAt = Between(new Date(query.startDate), new Date(query.endDate));
    }

    const logs = await this.auditHistoryLogRepository.find({
      where,
      relations: ['details'],
      order: { createdAt: 'ASC' },
    });

    // Group logs by requestId
    const groupedLogs = this.groupLogsByRequestId(logs);

    // Get user information for all involved users
    const userMap = await this.getUserMapForLogs(logs);

    // Process each group to create intelligent activity items
    const activities: ActivityLogItem[] = [];
    const categoryCounts: { [key: string]: number } = {};

    for (const [requestId, groupLogs] of groupedLogs.entries()) {
      const entities = [...new Set(groupLogs.map((log) => log.entityType))];

      if (entities.includes('ProgramRegistration') && entities.includes('Preference')) {
        const registrationLogs = groupLogs.filter(
          (log) => log.entityType !== 'Preference',
        );
        const preferenceLogs = groupLogs.filter(
          (log) => log.entityType === 'Preference',
        );

        const registrationActivity = await this.createIntelligentActivityItem(
          requestId,
          registrationLogs,
          userMap,
        );
        const preferenceActivity = await this.createIntelligentActivityItem(
          requestId,
          preferenceLogs,
          userMap,
        );

        if (registrationActivity) {
          activities.push(registrationActivity);
          categoryCounts[registrationActivity.category] =
            (categoryCounts[registrationActivity.category] || 0) + 1;
        }

        if (preferenceActivity) {
          activities.push(preferenceActivity);
          categoryCounts[preferenceActivity.category] =
            (categoryCounts[preferenceActivity.category] || 0) + 1;
        }
      } else {
        const activityItem = await this.createIntelligentActivityItem(
          requestId,
          groupLogs,
          userMap,
        );
        if (activityItem) {
          activities.push(activityItem);
          categoryCounts[activityItem.category] =
            (categoryCounts[activityItem.category] || 0) + 1;
        }
      }
    }

    // Sort activities by timestamp (newest first)
    // Filter out activities with excluded subcategories
    const filteredActivities = activities.filter(
      activity => !EXCLUDED_ACTIVITY_SUBCATEGORIES.includes(activity.subCategory || '')
    );
    
    filteredActivities.sort((a, b) => new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime());

    return {
      data: filteredActivities,
      totalActivities: filteredActivities.length,
      dateRange: {
        startDate: query.startDate,
        endDate: query.endDate,
      }
      // ,
      // summary: {
      //   categories: categoryCounts,
      //   totalApiCalls: groupedLogs.size,
      // },
    };
  }

  /**
   * Create intelligent activity item based on entity combinations and actions
   */
  private async createIntelligentActivityItem(
    requestId: string,
    logs: AuditHistoryLog[],
    userMap: Map<number, string>
  ): Promise<ActivityLogItem | null> {
    if (logs.length === 0) return null;

    // Extract entity types and actions
    const entities = [...new Set(logs.map(log => log.entityType))];
    const actions = [...new Set(logs.map(log => log.action))];
    
    // Get user information
    const firstLog = logs[0];
    let performedBy: number | null = null;
    let performedByName: string | null = null;

    if (firstLog.userId && typeof firstLog.userId === 'string' && firstLog.userId.trim() !== '' && 
        firstLog.userId !== 'null' && firstLog.userId !== 'undefined') {
      const cleanUserId = firstLog.userId.replace(/^['"]|['"]$/g, '').trim();
      const numericUserId = parseInt(cleanUserId, 10);
      if (!isNaN(numericUserId)) {
        performedBy = numericUserId;
        performedByName = userMap.get(numericUserId) || 'Seeker';
      }
    }

    // Analyze entity combinations and determine message
    const { message, category, subCategory } = await this.analyzeEntityCombination(
      entities,
      actions,
      logs,
    );

    // Extract changes
    const changes = this.extractIntelligentChanges(logs);

    return {
      requestId: requestId || 'unknown',
      timestamp: firstLog.createdAt.toISOString(),
      message,
      category,
      subCategory,
      performedBy,
      performedByName,
      // changes,
      metadata: {
        entities,
        actions,
        apiEndpoint: firstLog.apiEndpoint || undefined,
        httpMethod: firstLog.httpMethod || undefined,
      },
    };
  }

  /**
   * Analyze entity combination to determine appropriate message and category
   */
  private async analyzeEntityCombination(
    entities: string[],
    actions: string[],
    logs: AuditHistoryLog[]
  ): Promise<{ message: string; category: string; subCategory?: string; }> {
    
    // Check for swap-related activities
    if (entities.includes('ProgramRegistrationSwap')) {
      return this.analyzeSwapActivity(logs);
    }

    // Check for rating and recommendation activities
    if (entities.includes('ProgramRegistrationRmRating') || entities.includes('ProgramRegistrationRecommendations')) {
      return this.analyzeRatingRecommendationActivity(entities, logs);
    }

    // Check for payment activities
    if (entities.includes('RegistrationPaymentDetail') || entities.includes('RegistrationInvoiceDetail')) {
      return this.analyzePaymentActivity(entities, logs);
    }

    if (entities.includes('SeekerProgramExperience')) {
      return this.analyzeSeekerProgramExperienceActivity(logs);
    }

    // Check for travel activities
    if (entities.includes('RegistrationTravelPlan') || entities.includes('RegistrationTravelInfo')) {
      return this.analyzeTravelActivity(entities, logs);
    }

    // Check for goodies activities
    if (entities.includes('ProgramRegistrationGoodies')) {
      return this.analyzeGoodiesActivity(logs);
    }

    // Check for registration and preference activities
    if (entities.includes('ProgramRegistration') && entities.includes('Preference')) {
      return await this.analyzeRegistrationActivity(entities, actions, logs);
    }

    // Check for preference only activities
    if (entities.includes('Preference')) {
      return await this.analyzePreferenceActivity(logs, entities, actions);
    }

    // Check for approval activities
    if (entities.includes('RegistrationApproval')) {
      return await this.analyzeApprovalActivity(logs);
    }

    // Check for registration-related activities
    if (entities.includes('ProgramRegistration')) {
      return this.analyzeRegistrationActivity(entities, actions, logs);
    }

    // Check for cancellation
    if (this.isCancellationActivity(logs)) {
      return {
        message: 'Registration cancelled',
        category: 'Cancellation',
      };
    }

    // If entities length is 1 and entity is RegistrationQuestionAnswer
    if (entities.length === 1 && entities[0] === 'RegistrationQuestionAnswer') {
      return {
        message: 'Registration details updated',
        category: 'Registration',
        subCategory: 'Question Answered',
      };
    }

    // If entities length is 1 and entity is CommunicationTrack
    if (entities.length === 1 && entities[0] === 'CommunicationTrack') {
      return this.analyzeCommunicationTrackActivity(logs);
    }

    // Default fallback
    return {
      message: `Updated ${entities.join(', ')}`,
      category: 'Other',
    };
  }

  /**
   * Analyze swap-related activities
   */
  private async analyzeSwapActivity(logs: AuditHistoryLog[]): Promise<{ message: string; category: string; subCategory?: string; }> {
    const swapLogs = logs.filter(log => log.entityType === 'ProgramRegistrationSwap');
    
    for (const log of swapLogs) {
      if (log.action === 'INSERT') {
        // Check swap type from details
        const swapTypeDetail = log.details?.find(d => d.fieldName === 'type');
        if (swapTypeDetail?.newValue === 'wants_swap') {
          return {
            message: 'Swap request created - wants swap',
            category: 'Swap',
            subCategory: 'wants_swap',
          };
        } else if (swapTypeDetail?.newValue === 'can_shift') {
          return {
            message: 'Swap request created - can shift',
            category: 'Swap',
            subCategory: 'can_shift',
          };
        }
        return {
          message: 'Swap request created',
          category: 'Swap',
        };
      } else if (log.action === 'UPDATE') {
        const statusDetail = log.details?.find(d => d.fieldName === 'status');
        const swapAllocatedDetail = log.details?.find(d => d.fieldName === 'allocatedProgramId');
        if (statusDetail?.newValue === 'on_hold') {
          return {
            message: 'Registration moved to swap demand',
            category: 'Swap',
            subCategory: 'on_hold',
          };
        } else if (statusDetail?.newValue === 'rejected') {
          return {
            message: 'Swap request on hold',
            category: 'Swap',
            subCategory: 'rejected',
          };
        } else if (statusDetail?.newValue === 'closed') {
          return {
            message: 'Swap request cancelled',
            category: 'Swap',
            subCategory: 'cancelled',
          };
        } else if (statusDetail?.newValue === 'accepted') {
          // get the new allocated program id details get program name from program service refer preference activity log
          const programDetails = await this.programService.findByIds([swapAllocatedDetail?.newValue]);
          return {
            message: programDetails[0]?.name ? `Swap request accepted - allocated to ${programDetails[0]?.name}` : 'Swap request accepted',
            category: 'Swap',
            subCategory: 'accepted',
          };
        }
        return {
          message: 'Swap request updated',
          category: 'Swap',
        };
      }
    }

    return {
      message: 'Swap activity',
      category: 'Swap',
    };
  }

  /**
   * Analyze rating and recommendation activities
   */
  private analyzeRatingRecommendationActivity(
    entities: string[],
    logs: AuditHistoryLog[]
  ): { message: string; category: string; subCategory?: string } {
    const hasRating = entities.includes('ProgramRegistrationRmRating');
    const hasRecommendation = entities.includes('ProgramRegistrationRecommendations');

    if (hasRating && hasRecommendation) {
      const ratingAction = logs.find(log => log.entityType === 'ProgramRegistrationRmRating')?.action;
      const recommendationAction = logs.find(log => log.entityType === 'ProgramRegistrationRecommendations')?.action;
      
      if (ratingAction === 'INSERT' && recommendationAction === 'INSERT') {
        return {
          message: 'RM rating and recommendation added',
          category: 'RM Review',
          subCategory: 'created',
        };
      } else {
        return {
          message: 'RM rating and recommendation updated',
          category: 'RM Review',
          subCategory: 'updated',
        };
      }
    } else if (hasRating) {
      const ratingLog = logs.find(log => log.entityType === 'ProgramRegistrationRmRating');
      return {
        message: ratingLog?.action === 'INSERT' ? 'RM rating added' : 'RM rating updated',
        category: 'RM Review',
        subCategory: ratingLog?.action === 'INSERT' ? 'rating_created' : 'rating_updated',
      };
    } else if (hasRecommendation) {
      const recommendationLog = logs.find(log => log.entityType === 'ProgramRegistrationRecommendations');
      return {
        message: recommendationLog?.action === 'INSERT' ? 'RM recommendation added' : 'RM recommendation updated',
        category: 'RM Review',
        subCategory: recommendationLog?.action === 'INSERT' ? 'recommendation_created' : 'recommendation_updated',
      };
    }

    return {
      message: 'RM review activity',
      category: 'RM Review',
    };
  }

  /**
   * Analyze payment activities
   */
  private analyzePaymentActivity(
    entities: string[],
    logs: AuditHistoryLog[]
  ): { message: string; category: string; subCategory?: string } {
    const hasPayment = entities.includes('RegistrationPaymentDetail');
    const hasInvoice = entities.includes('RegistrationInvoiceDetail');

    if (hasPayment) {
      const paymentLog = logs.find(log => log.entityType === 'RegistrationPaymentDetail');
      const paymentStatusDetail = paymentLog?.details?.find(d => d.fieldName === 'paymentStatus');
      
      if (paymentStatusDetail?.newValue?.includes('online')) {
        let message = 'Online payment';
        if (hasInvoice) {
          const invoiceLog = logs.find(log => log.entityType === 'RegistrationInvoiceDetail');
          const invoiceStatusDetail = invoiceLog?.details?.find(d => d.fieldName === 'invoiceStatus');
          // if invoice_status is completed, then it is invoice generated or else it is invoice pending
          if (paymentStatusDetail.newValue.includes('pending')) {
            message += ' - Pending';
          } else if (paymentStatusDetail.newValue.includes('completed')) {
            message += ' - Completed';
          }
          if (invoiceStatusDetail?.newValue?.includes('pending')) {
            message += ' - Invoice pending';
          } else if (invoiceStatusDetail?.newValue?.includes('completed')) {
            message += ' - Invoice completed';
          }
        }
        return {
          message,
          category: 'Payment',
          subCategory: 'online',
        };
      } else if (paymentStatusDetail?.newValue?.includes('offline')) {

        const invoiceLog = logs.find(log => log.entityType === 'RegistrationInvoiceDetail');
        const invoiceStatusDetail = invoiceLog?.details?.find(d => d.fieldName === 'invoiceStatus');

        // check whether completed or pending
        let message = 'Offline payment';
        if (paymentStatusDetail.newValue.includes('pending')) {
          message += ' - Pending';
        } else if (paymentStatusDetail.newValue.includes('completed')) {
          message += ' - Completed';
        }
        if (hasInvoice) {
          if (invoiceStatusDetail?.newValue?.includes('pending')) {
            message += ' - Invoice pending';
          } else if (invoiceStatusDetail?.newValue?.includes('completed')) {
            message += ' - Invoice completed';
          }
        }

        return {
          message,
          category: 'Payment',
          subCategory: 'offline',
        };
      }
    }

    if (hasInvoice) {
      const invoiceLog = logs.find(log => log.entityType === 'RegistrationInvoiceDetail');
      const invoiceStatusDetail = invoiceLog?.details?.find(d => d.fieldName === 'invoiceStatus');
      const invoiceIssueDateDetail = invoiceLog?.details?.find(d => d.fieldName === 'invoiceIssuedDate');
      if (invoiceStatusDetail?.newValue?.includes('completed')) {
        return {
          message: 'Invoice generated',
          category: 'Payment',
          subCategory: 'invoice_created',
        };
      } else if (invoiceStatusDetail?.newValue?.includes('pending')) {
        return {
          message: 'Invoice pending',
          category: 'Payment',
          subCategory: 'invoice_pending',
        };
      } else if (invoiceIssueDateDetail?.newValue) {
        return {
          message: 'Invoice resent successfully',
          category: 'Payment',
          subCategory: 'invoice_resent',
        };
      }
      return {
        message: 'Invoice Drafted',
        category: 'Payment',
        subCategory: 'invoice',
      };
    }

    return {
      message: 'Payment Drafted',
      category: 'Payment',
    };
  }

  private analyzeSeekerProgramExperienceActivity(
    logs: AuditHistoryLog[]
  ): { message: string; category: string; subCategory?: string } {
    const experienceLog = logs.find(log => log.entityType === 'SeekerProgramExperience');
    const isViewedTrue = experienceLog?.details?.some(d => d.fieldName === 'isViewed' && d.newValue === true);
    if (experienceLog?.action === 'INSERT') {

      return {
        message: 'Seeker program experience added',
        category: 'Seeker Program Experience',
        subCategory: 'created',
      };
    } else if (experienceLog?.action === 'UPDATE') {
      
      if (isViewedTrue) {
        return {
          message: 'Seeker program experience viewed',
          category: 'Seeker Program Experience',
          subCategory: 'seeker_program_experience_viewed',
        };
      }
      return {
        message: 'Seeker program experience updated',
        category: 'Seeker Program Experience',
        subCategory: 'updated',
      };
    } else if (experienceLog?.action === 'DELETE') {
      return {
        message: 'Seeker program experience removed',
        category: 'Seeker Program Experience',
        subCategory: 'deleted',
      };
    }

    return {
      message: 'Seeker program experience activity',
      category: 'Seeker Program Experience',
    };
  }

  /**
   * Analyze travel activities
   */
  private analyzeTravelActivity(
    entities: string[],
    logs: AuditHistoryLog[]
  ): { message: string; category: string; subCategory?: string } {
    const hasTravelPlan = entities.includes('RegistrationTravelPlan');
    const hasTravelInfo = entities.includes('RegistrationTravelInfo');

    if (hasTravelPlan && hasTravelInfo) {
      const planAction = logs.find(log => log.entityType === 'RegistrationTravelPlan')?.action;
      const infoAction = logs.find(log => log.entityType === 'RegistrationTravelInfo')?.action;
      
      if (planAction === 'INSERT' || infoAction === 'INSERT') {
        return {
          message: 'Travel details added',
          category: 'Travel',
          subCategory: 'created',
        };
      } else {
        return {
          message: 'Travel details updated',
          category: 'Travel',
          subCategory: 'updated',
        };
      }
    } else if (hasTravelPlan) {
      const planLog = logs.find(log => log.entityType === 'RegistrationTravelPlan');
      return {
        message: planLog?.action === 'INSERT' ? 'Travel plan added' : 'Travel plan updated',
        category: 'Travel',
        subCategory: planLog?.action === 'INSERT' ? 'plan_created' : 'plan_updated',
      };
    } else if (hasTravelInfo) {
      const infoLog = logs.find(log => log.entityType === 'RegistrationTravelInfo');
      return {
        message: infoLog?.action === 'INSERT' ? 'Travel info added' : 'Travel info updated',
        category: 'Travel',
        subCategory: infoLog?.action === 'INSERT' ? 'info_created' : 'info_updated',
      };
    }

    return {
      message: 'Travel activity',
      category: 'Travel',
    };
  }

  /**
   * Analyze goodies activities
   */
  private analyzeGoodiesActivity(logs: AuditHistoryLog[]): { message: string; category: string; subCategory?: string } {
    const goodiesLog = logs.find(log => log.entityType === 'ProgramRegistrationGoodies');
    
    if (goodiesLog?.action === 'INSERT') {

      return {
        message: 'Goodies preferences added',
        category: 'Goodies',
        subCategory: 'created',
      };
    } else if (goodiesLog?.action === 'UPDATE') {
      
      return {
        message: 'Goodies Data updated',
        category: 'Goodies',
        subCategory: 'updated',
      };
    } else if (goodiesLog?.action === 'DELETE') {
      return {
        message: 'Goodies Data removed',
        category: 'Goodies',
        subCategory: 'deleted',
      };
    }

    return {
      message: 'Goodies activity',
      category: 'Goodies',
    };
  }

  /**
   * Analyze registration activities
   */
  private async analyzeRegistrationActivity(
    entities: string[],
    actions: string[],
    logs: AuditHistoryLog[]
  ): Promise<{ message: string; category: string; subCategory?: string; }> {
    const hasRegistration = entities.includes('ProgramRegistration');
    const hasPreference = entities.includes('Preference');

    if (hasRegistration) {
      const registrationLog = logs.find(log => log.entityType === 'ProgramRegistration');
      
      if (registrationLog?.action === 'INSERT') {
        const statusDetail = registrationLog.details?.find(d => d.fieldName === 'registrationStatus');
        if (statusDetail?.newValue === 'save_as_draft') {
          return {
            message: 'Registration saved as draft',
            category: 'Registration',
            subCategory: 'draft',
          };
        }
        return {
          message: 'Registration created',
          category: 'Registration',
          subCategory: 'created',
        };
      } else if (registrationLog?.action === 'UPDATE') {
        if (hasPreference) {
          // Check if preferences have significant changes
          const preferenceLogs = logs.filter(log => log.entityType === 'Preference');
          if (preferenceLogs.length > 0) {
            return {
              message: 'Registration and preferences updated',
              category: 'Registration',
              subCategory: 'updated_with_preferences',
            };
          } else {
            return {
              message: 'Registration details updated',
              category: 'Registration',
              subCategory: 'updated',
            };
          }
        }
        
        // Check if it's status change or basic details update
        const statusDetail = registrationLog.details?.find(d => d.fieldName === 'registrationStatus');
        if (statusDetail && (statusDetail.newValue === 'pending' || statusDetail.newValue === 'pending_approval')) {
          return {
            message: 'Registration submitted for approval',
            category: 'Registration',
            subCategory: 'submitted',
          };
        }
        
        return {
          message: 'Registration details updated',
          category: 'Registration',
          subCategory: 'updated',
        };
      }
    }

    if (hasPreference) {
      return await this.analyzePreferenceActivity(logs, entities, actions);
    }

    return {
      message: 'Registration activity',
      category: 'Registration',
    };
  }

  /**
   * Analyze approval activities
   */
  private analyzeApprovalActivity(logs: AuditHistoryLog[]): { message: string; category: string; subCategory?: string } {
    const approvalLog = logs.find(log => log.entityType === 'RegistrationApproval');
    const registrationLog = logs.find(log => log.entityType === 'ProgramRegistration');
    const statusDetail = approvalLog?.details?.find(d => d.fieldName === 'approvalStatus');
    // if registrationLog old value is save_as_draft, then it is pending then give message as registration is updated from save_as_draft to pending
    if (registrationLog) {
      const registrationStatusDetail = registrationLog.details?.find(d => d.fieldName === 'registrationStatus');
      if (registrationStatusDetail?.oldValue === 'save_as_draft' && registrationStatusDetail.newValue === 'pending') {
        return {
          message: 'Registration updated from draft to save - pending for approval',
          category: 'Approval',
          subCategory: 'draft_to_pending',
        };
      }
    }
    if (statusDetail?.newValue === 'approved') {
      return {
        message: 'Registration blessed',
        category: 'Approval',
        subCategory: 'blessed',
      };
    } else if (statusDetail?.newValue === 'rejected') {
      return {
        message: 'Registration on hold',
        category: 'Approval',
        subCategory: 'rejected',
      };
    } else if (statusDetail?.newValue === 'on_hold') {
      return {
        message: 'Registration moved to swap demand',
        category: 'Approval',
        subCategory: 'on_hold',
      };
    } else if (statusDetail?.newValue === 'cancelled') {
      return {
        message: 'Registration cancelled',
        category: 'Approval',
        subCategory: 'cancelled',
      };
    } else if (statusDetail?.newValue === 'pending') {
      return {
        message: 'Registration saved & approval pending',
        category: 'Approval',
        subCategory: 'pending',
      };
    }

    return {
      message: 'Approval status updated',
      category: 'Approval',
    };
  }

  /**
   * Check if this is a cancellation activity
   */
  private isCancellationActivity(logs: AuditHistoryLog[]): boolean {
    return logs.some(log => {
      if (log.entityType === 'ProgramRegistration') {
        const cancellationDetail = log.details?.find(d => 
          d.fieldName === 'cancellationReason' || 
          d.fieldName === 'cancellationComments'
        );
        return cancellationDetail !== undefined;
      }
      if (log.entityType === 'RegistrationApproval') {
        const statusDetail = log.details?.find(d => 
          d.fieldName === 'approvalStatus' && 
          d.newValue === 'cancelled'
        );
        return statusDetail !== undefined;
      }
      return false;
    });
  }

  /**
   * Analyze communication track activities
   */
  private analyzeCommunicationTrackActivity(logs: AuditHistoryLog[]): { message: string; category: string; subCategory?: string } {
    const communicationLog = logs.find(log => log.entityType === 'CommunicationTrack');
    
    if (!communicationLog) {
      return {
        message: 'Communication activity',
        category: 'Communication',
        subCategory: 'unknown',
      };
    }

    const communicationType = communicationLog?.details?.find(d => d.fieldName === 'typ')?.newValue;
    const communicationStatus = communicationLog?.details?.find(d => d.fieldName === 'stats')?.newValue;
    const action = communicationLog.action;

    // When both type and status are present, create meaningful messages
    if (communicationType && communicationStatus) {
      let message = '';
      let subCategory = '';

      if (communicationType === 'email') {
        if (communicationStatus === 'success') {
          message = 'Email sent successfully';
          subCategory = 'email_success';
        } else if (communicationStatus === 'failed') {
          message = 'Email delivery failed';
          subCategory = 'email_failed';
        } else {
          message = `Email ${communicationStatus}`;
          subCategory = `email_${communicationStatus}`;
        }
      } else if (communicationType === 'whatsapp') {
        if (communicationStatus === 'success') {
          message = 'WhatsApp message sent successfully';
          subCategory = 'whatsapp_success';
        } else if (communicationStatus === 'failed') {
          message = 'WhatsApp message delivery failed';
          subCategory = 'whatsapp_failed';
        } else {
          message = `WhatsApp ${communicationStatus}`;
          subCategory = `whatsapp_${communicationStatus}`;
        }
      } else {
        // For other communication types
        if (communicationStatus === 'success') {
          message = `${communicationType} sent successfully`;
          subCategory = `${communicationType}_success`;
        } else if (communicationStatus === 'failed') {
          message = `${communicationType} delivery failed`;
          subCategory = `${communicationType}_failed`;
        } else {
          message = `${communicationType} ${communicationStatus}`;
          subCategory = `${communicationType}_${communicationStatus}`;
        }
      }

      return {
        message,
        category: 'Communication',
        subCategory,
      };
    }

    // Fallback when only one or neither is present
    if (communicationType) {
      return {
        message: `${communicationType} communication ${action === 'INSERT' ? 'initiated' : 'updated'}`,
        category: 'Communication',
        subCategory: `${communicationType}_${action === 'INSERT' ? 'initiated' : 'updated'}`,
      };
    }

    if (communicationStatus) {
      return {
        message: `Communication ${communicationStatus}`,
        category: 'Communication',
        subCategory: communicationStatus,
      };
    }

    return {
      message: 'Communication activity',
      category: 'Communication',
      subCategory: 'tracked',
    };
  }

  /**
   * Extract changes with intelligent value formatting
   */
  private extractIntelligentChanges(logs: AuditHistoryLog[]): ActivityLogChange[] {
    const changes: ActivityLogChange[] = [];

    logs.forEach(log => {
      log.details?.forEach(detail => {
        const change: ActivityLogChange = {
          field: detail.fieldName,
          oldValue: detail.oldValue,
          newValue: detail.newValue,
          displayOldValue: this.formatDisplayValue(detail.oldValue),
          displayNewValue: this.formatDisplayValue(detail.newValue),
        };
        changes.push(change);
      });
    });

    return changes;
  }

  /**
   * Format values for display (convert yes/no to true/false, format timestamps, etc.)
   */
  private formatDisplayValue(value: any): string {
    if (value === null || value === undefined) {
      return 'Not set';
    }

    // Convert yes/no to true/false
    if (typeof value === 'string') {
      const lowerValue = value.toLowerCase();
      if (lowerValue === 'yes') return 'true';
      if (lowerValue === 'no') return 'false';
    }

    // Format timestamps
    if (typeof value === 'string' && value.match(/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}/)) {
      try {
        return new Date(value).toLocaleString();
      } catch (e) {
        return value;
      }
    }

    // Convert boolean to string
    if (typeof value === 'boolean') {
      return value.toString();
    }

    return value.toString();
  }

  /**
   * Analyze preference activities with detailed change categorization
   */
  private async analyzePreferenceActivity(logs: AuditHistoryLog[], entities: string[], actions: string[]): Promise<{ message: string; category: string; subCategory?: string; }> {
    const preferenceLogs = logs.filter(log => log.entityType === 'Preference');
    
    // Early return if no preference logs
    if (preferenceLogs.length === 0) {
      return {
        message: 'Preference activity',
        category: 'Preferences',
      };
    }

    const registrationQuestionAnswerLogs = logs.filter(log => log.entityType === 'RegistrationQuestionAnswer');
    const preferenceUpdated = registrationQuestionAnswerLogs.length > 0;
    
    // Check if preferences have meaningful changes
    const hasPreferenceChanged = this.checkPreferenceAnswerChanges(registrationQuestionAnswerLogs);

    // Count action types efficiently
    const actionCounts = this.countPreferenceActions(preferenceLogs);
    const { insertCount, updateCount, deleteCount } = actionCounts;

    // Extract and sort preference data by priority
    const sortedProgramNames = await this.extractSortedProgramNames(preferenceLogs);

    // Generate intelligent messages based on the type of changes
    return this.generatePreferenceMessage({
      preferenceUpdated,
      hasPreferenceChanged,
      actionCounts,
      sortedProgramNames
    });
  }

  /**
   * Check if registration question answer logs have meaningful preference changes
   */
  private checkPreferenceAnswerChanges(registrationQuestionAnswerLogs: AuditHistoryLog[]): boolean {
    return registrationQuestionAnswerLogs.some(log => {
      return log.details?.some(detail => {
        if (detail.fieldName !== 'answerValue') return false;
        
        try {
          const decodedNewValue = decodeURIComponent(detail.newValue ?? '');
          const decodedOldValue = decodeURIComponent(detail.oldValue ?? '');
          
          const isNewValueEmpty = !decodedNewValue || decodedNewValue === '[]';
          const isOldValueEmpty = !decodedOldValue || decodedOldValue === '[]';
          
          // Return true if there's a meaningful change (not both empty)
          return !(isNewValueEmpty && isOldValueEmpty);
        } catch (error) {
          console.warn('Error decoding preference values:', error);
          return false;
        }
      });
    });
  }

  /**
   * Count different types of preference actions
   */
  private countPreferenceActions(preferenceLogs: AuditHistoryLog[]): {
    insertCount: number;
    updateCount: number;
    deleteCount: number;
  } {
    return preferenceLogs.reduce((counts, log) => {
      switch (log.action) {
        case 'INSERT':
          counts.insertCount++;
          break;
        case 'UPDATE':
          counts.updateCount++;
          break;
        case 'DELETE':
          counts.deleteCount++;
          break;
      }
      return counts;
    }, { insertCount: 0, updateCount: 0, deleteCount: 0 });
  }

  /**
   * Extract and sort program names by priority order
   */
  private async extractSortedProgramNames(preferenceLogs: AuditHistoryLog[]): Promise<string> {
    try {
      // Extract preference data with priority and program info
      const preferenceData = this.extractPreferenceData(preferenceLogs);
      
      if (preferenceData.length === 0) {
        return '';
      }

      // Sort by priority order
      const sortedData = preferenceData.sort((a, b) => {
        const priorityA = a.priorityOrder ?? 999;
        const priorityB = b.priorityOrder ?? 999;
        return priorityA - priorityB;
      });

      // Extract program IDs
      const programIds = sortedData
        .map(item => item.preferredProgram?.id)
        .filter(id => id != null);

      if (programIds.length === 0) {
        return '';
      }

      // Fetch program details
      const programDetails = await this.programService.findByIds(programIds);
      const programMap = new Map(programDetails.map(program => [program.id, program]));

      // Get program names in priority order
      const programNames = programIds
        .map(id => programMap.get(id)?.name)
        .filter(name => name != null);

      return programNames.join(', ');
    } catch (error) {
      console.error('Error extracting sorted program names:', error);
      return '';
    }
  }

  /**
   * Extract preference data from logs
   */
  private extractPreferenceData(preferenceLogs: AuditHistoryLog[]): Array<{
    auditLogId: string;
    priorityOrder?: number;
    preferredProgram?: any;
  }> {
    const preferenceDetails = preferenceLogs.flatMap(log => log.details || []);
    const programDetails = preferenceDetails.filter(detail => detail.fieldName === 'preferredProgram');
    const programAuditIds = programDetails
      .map(detail => detail.auditLogId)
      .filter(id => id != null);

    const relevantLogs = preferenceLogs.filter(log =>
      programAuditIds.includes(Number(log.id))
    );

    return relevantLogs.map(log => {
      const logData = {
        auditLogId: log.id,
        priorityOrder: undefined as number | undefined,
        preferredProgram: undefined as any
      };

      log.details?.forEach(detail => {
        if (detail.fieldName === 'priorityOrder') {
          logData.priorityOrder = detail.newValue;
        } else if (detail.fieldName === 'preferredProgram') {
          logData.preferredProgram = detail.newValue;
        }
      });

      return logData;
    });
  }

  /**
   * Generate preference message based on analysis results
   */
  private generatePreferenceMessage({
    preferenceUpdated,
    hasPreferenceChanged,
    actionCounts,
    sortedProgramNames
  }: {
    preferenceUpdated: boolean;
    hasPreferenceChanged: boolean;
    actionCounts: { insertCount: number; updateCount: number; deleteCount: number };
    sortedProgramNames: string;
  }): { message: string; category: string; subCategory?: string } {
    const { insertCount, updateCount, deleteCount } = actionCounts;
    const programNamesText = sortedProgramNames || 'preferences';

    // Check for preference update with meaningful changes
    if (preferenceUpdated && insertCount > 0 && hasPreferenceChanged) {
      return {
        message: `Preference updated ${programNamesText}`,
        category: 'Preferences',
        subCategory: 'preference_updated',
      };
    }

    // Pure insert (new preferences)
    if (insertCount > 0 && updateCount === 0 && deleteCount === 0) {
      return {
        message: `Preference added ${programNamesText}`,
        category: 'Preferences',
        subCategory: 'added',
      };
    }

    // Pure update (existing preferences modified)
    if (updateCount > 0 && insertCount === 0 && deleteCount === 0) {
      return {
        message: sortedProgramNames ? `Preference updated ${programNamesText}` : 'Preference updated to Any HDB/MSD',
        category: 'Preferences',
        subCategory: 'updated',
      };
    }

    // Mixed operations or other cases
    return {
      message: `Preference updated ${programNamesText}`,
      category: 'Preferences',
      subCategory: 'updated',
    };
  }
}
