import { CallHandler, ExecutionContext, Injectable, NestInterceptor } from '@nestjs/common';
import { Observable } from 'rxjs';
import { tap } from 'rxjs/operators';
import { Reflector } from '@nestjs/core';
import { AuditHistoryService } from '../services/audit-history.service';
import { USER_TRACE_KEY } from '../decorators/user-trace.decorator';
import { DataSource } from 'typeorm';
import { AuditHistoryLogType } from '../audit-history.constants';

@Injectable()
export class UserTraceInterceptor implements NestInterceptor {
  constructor(
    private auditHistoryService: AuditHistoryService,
    private reflector: Reflector,
    private dataSource: DataSource,
  ) {}

  intercept(context: ExecutionContext, next: CallHandler): Observable<any> {
    const action = this.reflector.get<string>(USER_TRACE_KEY, context.getHandler());

    if (!action) {
      return next.handle();
    }

    const request = context.switchToHttp().getRequest();
    return next.handle().pipe(
      tap((response) => {
        const process = async () => {
          let userId: any;
          if (request.path === '/login') {
            if (response && response.statusCode === 200) {
              const user = await this.dataSource.getRepository('users').findOne({
                where: [{ emailId: request.body.userName }, { loginName: request.body.userName }],
                select: ['userId'],
              });
              userId = user?.userId;
            } else {
              return;
            }
          } else {
            userId = request?.headers?.userid;
          }
          try {
            // Generate a requestId if one doesn't exist
            const requestId = request.id || request.headers['x-request-id'] || `req-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
            
            await this.auditHistoryService.createAuditLog({
              entityType: 'User',
              entityName: 'User',
              entityId: userId,
              action,
              userId: userId,
              ipAddress: request.ip || request.connection?.remoteAddress,
              userAgent: request.headers['user-agent'],
              requestId: requestId,
              type: AuditHistoryLogType.TRACE,
            });
          } catch (err) {
            console.error('Failed to save user trace', err);
          }
        };
        void process();
      }),
    );
  }
}
