# 📘 Backend Registration System – CodeX Integration Guide

This document provides **coding standards**, **file references**, and **project context** for CodeX to understand, modify, or enhance the backend registration system. It includes how modules are structured, enforced conventions, and external references.

---

## 🧱 Coding Standards & File References

### 1. 🔐 Type Safety & DTOs
- All code **must be in TypeScript** to enforce strict typing.
  - **Reference:** `program-question.repository.ts`, `program.entity.ts`
- Use **DTOs for request and response validation**.
  - **Reference:** `update-program-question.dto.ts`
- Maintain **strong typing** across:
  - Entities
  - DTOs
  - Service methods

---

### 2. 🚨 Error Handling
- Use **centralized custom exceptions** for reusable, meaningful error responses.
  - **Reference:**  
    - `src/common/exceptions/inifni-not-found.exception.ts`  
    - `src/common/exceptions/inifni-bad-request.exception.ts`
- Store error messages and codes in shared constants.
  - **Reference:** `common/constants/error-string-constants.ts`
- Use `try/catch` blocks in repositories and services.
  - **Reference:** `program-question.repository.ts`
- Add new error codes to:
  - `common/i18n/error-messages.ts` under `ERROR_MESSAGES`

---

### 3. 🧩 Repository & Service Patterns
- Enforce **Separation of Concerns**:
  - **Repositories** → database logic
  - **Services** → business logic
  - **Reference:** `program-question.repository.ts`, `program-question.service.ts`
- Use `@Injectable()` from **NestJS**.
- Apply `try/catch` consistently in controllers/services/repositories.
- Prefer `CommonDataService` methods in repositories where available.

---

### 4. ✅ Validation
- Use `class-validator` decorators in DTOs.
  - **Reference:** `update-program-question.dto.ts`
- Validate all inputs **before DB operations**.
- For entities, use enums from `common/enum`; create if not available.

---

### 5. 🗃️ Soft Deletion & Auditing
- Implement soft deletion using `deletedAt`.
  - **Reference:** `program-question.entity.ts`
- Include auditing fields in all entities:
  - `createdBy`, `updatedBy`, `createdAt`, `updatedAt`

---

### 6. 🧹 Naming & Formatting
- **camelCase** → variables
- **PascalCase** → class names
- Enforce linting/formatting via:
  - `.eslintrc.js`
  - `.prettierrc`
- Use:
  - **Single quotes**
  - **Trailing commas**
- Store reusable messages in:
  - `common/constants/string-constants.ts`

---

### 7. 🧪 Testing
- Use **Jest** and **Supertest** for:
  - Unit tests
  - Integration tests
- **Reference:** `program-question.service.spec.ts`

---

### 8. 📚 Documentation
- Annotate APIs using **Swagger decorators**.
  - **Reference:** `program-question.controller.ts`
- Include:
  - HTTP status codes
  - Request bodies
  - Route/query parameters
  - Authentication
  - Role-based access
- Add meaningful **code comments** across layers.

---

### 9. 🛢️ Database & ORM
- Use **TypeORM** for entity/relationship management.
  - **Reference:** `program-question.entity.ts`, `program.entity.ts`
- Use enums for status and fixed values.
- Apply:
  - Soft deletes
  - Unique constraints

---

### 10. 🌐 API Design
- Support **pagination**, **filtering**, and **sorting** for all GET endpoints.
- Implement API **versioning** and maintain **clear route structures**.

---

### 11. 🔒 Security
- Validate inputs to prevent:
  - SQL injection
  - XSS attacks
- Store secrets in **environment variables**.
- Use guards like:
  - `FirebaseAuthGuard`
  - `RolesGuard`
- Avoid exposing internal error details in responses.

---

### 12. 📝 Logging
- Use a **central logging service**.
  - **Reference:** `logger.service.ts`
- Log with:
  - `debug`, `info`, `error` levels
  - Contextual metadata

---

## 📦 Project Context: Folder `projectContext`

### 1. 📄 `programData.csv`
- Contains basic **program type definitions**.
- Reference for form/program generation logic.

### 2. 📄 `registrationData.csv`
- Sample data expected from **user registration forms**.
- Use as input/output example for validation and DTO mapping.

---

### 🔄 AI Integration (Inferred Usage)
- AI modules assist in:
  - Generating/suggesting form fields
  - Creating program/session skeletons based on context
- Ensure output from AI modules complies with the **coding standards above**.

---

> Ensure **every new feature** or **change in logic**:
> - Matches the patterns shown in the reference files.
> - Passes validation and typing requirements.
> - Includes meaningful documentation and test coverage.

---
