import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { EntityManager, In, IsNull, Repository } from 'typeorm';
import {
  ProgramRegistration,
  User,
  UserParticipationSummary,
  UserRegistrationMap,
} from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { userRegistrationMapConstMessages } from 'src/common/constants/strings-constants';

@Injectable()
export class UserRegistrationMapRepository {
  constructor(
    @InjectRepository(UserRegistrationMap)
    private readonly mapRepository: Repository<UserRegistrationMap>,
    @InjectRepository(User)
    private readonly userRepository: Repository<User>,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepository: Repository<ProgramRegistration>,
    @InjectRepository(UserParticipationSummary)
    private readonly summaryRepository: Repository<UserParticipationSummary>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  private getMapRepository(manager?: EntityManager): Repository<UserRegistrationMap> {
    return manager ? manager.getRepository(UserRegistrationMap) : this.mapRepository;
  }

  private getUserRepository(manager?: EntityManager): Repository<User> {
    return manager ? manager.getRepository(User) : this.userRepository;
  }

  private getRegistrationRepository(manager?: EntityManager): Repository<ProgramRegistration> {
    return manager ? manager.getRepository(ProgramRegistration) : this.registrationRepository;
  }

  private getSummaryRepository(manager?: EntityManager): Repository<UserParticipationSummary> {
    return manager ? manager.getRepository(UserParticipationSummary) : this.summaryRepository;
  }

  async findUserById(userId: number, manager?: EntityManager): Promise<User | null> {
    this.logger.log(userRegistrationMapConstMessages.VALIDATING_USER(userId));
    return this.commonDataService.findOneById(this.getUserRepository(manager), userId, false);
  }

  async findRegistrationsByIds(
    registrationIds: number[],
    manager?: EntityManager,
  ): Promise<ProgramRegistration[]> {
    this.logger.log(userRegistrationMapConstMessages.VALIDATING_REGISTRATIONS(registrationIds));
    try {
      return await this.getRegistrationRepository(manager).find({
        where: { id: In(registrationIds) },
        relations: [
          'program',
          'programSession',
          'programSession.program',
          'allocatedProgram',
          'allocatedSession',
        ],
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_GET_FAILED, error);
    }
  }

  async findExistingSummary(
    userId: number,
    programId: number,
    subProgramId: number | null,
    sessionId: number | null,
    manager?: EntityManager,
  ): Promise<UserParticipationSummary | null> {
    try {
      const where: any = {
        userId,
        programId,
        deletedAt: IsNull(),
      };

      if (subProgramId !== null && subProgramId !== undefined) {
        where.subProgramId = subProgramId;
      } else {
        where.subProgramId = IsNull();
      }

      if (sessionId !== null && sessionId !== undefined) {
        where.sessionId = sessionId;
      } else {
        where.sessionId = IsNull();
      }

      return await this.getSummaryRepository(manager).findOne({ where });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_GET_FAILED, error);
    }
  }

  createSummary(partial: Partial<UserParticipationSummary>, manager?: EntityManager) {
    return this.getSummaryRepository(manager).create(partial);
  }

  async saveSummary(
    summary: UserParticipationSummary,
    manager?: EntityManager,
  ): Promise<UserParticipationSummary> {
    try {
      return await this.getSummaryRepository(manager).save(summary);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_SUMMARY_FAILED, error);
    }
  }

  createMapping(partial: Partial<UserRegistrationMap>, manager?: EntityManager) {
    return this.getMapRepository(manager).create(partial);
  }

  async saveMapping(
    mapping: UserRegistrationMap,
    manager?: EntityManager,
  ): Promise<UserRegistrationMap> {
    try {
      return await this.getMapRepository(manager).save(mapping);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_SAVE_FAILED, error);
    }
  }

  async saveMappings(
    mappings: UserRegistrationMap[],
    manager?: EntityManager,
  ): Promise<UserRegistrationMap[]> {
    try {
      return await this.getMapRepository(manager).save(mappings);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_SAVE_FAILED, error);
    }
  }

  async findMappingByUserAndRegistration(
    userId: number,
    registrationId: number,
    includeDeleted = false,
    manager?: EntityManager,
  ): Promise<UserRegistrationMap | null> {
    try {
      return await this.getMapRepository(manager).findOne({
        where: {
          user: { id: userId },
          registration: { id: registrationId },
          ...(includeDeleted ? {} : { deletedAt: IsNull() }),
        },
        relations: ['user', 'registration', 'summary'],
        withDeleted: includeDeleted,
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_GET_FAILED, error);
    }
  }

  async findActiveMappingsForUser(
    userId: number,
    manager?: EntityManager,
  ): Promise<UserRegistrationMap[]> {
    this.logger.log(userRegistrationMapConstMessages.FETCHING_EXISTING_MAPPINGS(userId));
    try {
      return await this.getMapRepository(manager).find({
        where: {
          user: { id: userId },
          deletedAt: IsNull(),
        },
        relations: ['user', 'registration','registration.program'],
        order: { id: 'ASC' },
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_GET_FAILED, error);
    }
  }

  async findMappingsForRegistrations(
    userId: number,
    registrationIds: number[],
    includeDeleted = false,
    manager?: EntityManager,
  ): Promise<UserRegistrationMap[]> {
    try {
      return await this.getMapRepository(manager).find({
        where: {
          user: { id: userId },
          registration: { id: In(registrationIds) },
          ...(includeDeleted ? {} : { deletedAt: IsNull() }),
        },
        relations: ['user', 'registration', 'summary'],
        withDeleted: includeDeleted,
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_GET_FAILED, error);
    }
  }

  async restoreMapping(
    mapping: UserRegistrationMap,
    manager?: EntityManager,
  ): Promise<UserRegistrationMap> {
    try {
      return await this.getMapRepository(manager).recover(mapping);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_RESTORE_FAILED, error);
    }
  }

  async softDeleteMappings(
    userId: number,
    registrationIds: number[],
    manager?: EntityManager,
  ): Promise<void> {
    this.logger.log(userRegistrationMapConstMessages.REMOVING_MAPPINGS(userId, registrationIds));
    try {
      const queryBuilder = (manager ?? this.mapRepository.manager)
        .createQueryBuilder()
        .softDelete()
        .from(UserRegistrationMap)
        .where('user_id = :userId', { userId });

      if (registrationIds.length) {
        queryBuilder.andWhere('registration_id IN (:...registrationIds)', { registrationIds });
      }

      await queryBuilder.execute();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_DELETE_FAILED, error);
    }
  }

  async softDeleteMappingsNotIn(
    userId: number,
    registrationIdsToKeep: number[],
    manager?: EntityManager,
  ): Promise<void> {
    this.logger.log(userRegistrationMapConstMessages.REMOVING_MAPPINGS(userId));
    try {
      const queryBuilder = (manager ?? this.mapRepository.manager)
        .createQueryBuilder()
        .softDelete()
        .from(UserRegistrationMap)
        .where('user_id = :userId', { userId });

      if (registrationIdsToKeep.length) {
        queryBuilder.andWhere('registration_id NOT IN (:...registrationIdsToKeep)', {
          registrationIdsToKeep,
        });
      }

      await queryBuilder.execute();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_REGISTRATION_MAP_DELETE_FAILED, error);
    }
  }

  /**
   * Soft deletes all user registration mappings for a specific user
   * @param {number} userId - The ID of the user whose mappings should be soft deleted
   * @param {EntityManager} manager - The transaction manager
   * @returns {Promise<void>} A promise that resolves when all mappings are soft deleted
   */
  async softDeleteAllUserMappings(userId: number, manager?: EntityManager): Promise<void> {
    const repo = manager ? manager.getRepository(UserRegistrationMap) : this.mapRepository;
    await repo.update(
      { user: { id: userId }, deletedAt: IsNull() },
      { deletedAt: new Date() }
    );
  }
}
