import {
  Body,
  Controller,
  HttpCode,
  HttpStatus,
  Param,
  ParseIntPipe,
  Post,
  Put,
  Res,
  UseGuards,
  ValidationPipe,
} from '@nestjs/common';
import {
  ApiBearerAuth,
  ApiBody,
  ApiOperation,
  ApiParam,
  ApiResponse,
  ApiSecurity,
  ApiTags,
} from '@nestjs/swagger';
import { Response } from 'express';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import {
  SWAGGER_API_RESPONSE,
  userProgramExperienceConstMessages,
} from 'src/common/constants/strings-constants';
import { CreateUserProgramExperienceDto } from './dto/create-user-program-experience.dto';
import { UpdateUserProgramExperienceDto } from './dto/update-user-program-experience.dto';
import { UserProgramExperienceService } from './user-program-experience.service';

@ApiTags('user-program-experience')
@Controller('user-program-experience')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('rm','relational_manager','shoba', 'rm_support')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class UserProgramExperienceController {
  constructor(
    private readonly service: UserProgramExperienceService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: userProgramExperienceConstMessages.CREATE_OPERATION })
  @ApiBody({ type: CreateUserProgramExperienceDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({
    status: HttpStatus.INTERNAL_SERVER_ERROR,
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR,
  })
  async create(
    @Body(new ValidationPipe({ transform: true, whitelist: true }))
    dto: CreateUserProgramExperienceDto,
    @Res() res: Response,
  ) {
    this.logger.log(userProgramExperienceConstMessages.CREATE_REQUEST_RECEIVED(dto.userId), dto);
    try {
      const data = await this.service.create(dto);
      await this.responseService.success(
        res,
        userProgramExperienceConstMessages.EXPERIENCES_CREATED,
        // data,
        {},
        HttpStatus.CREATED,
      );
    } catch (error) {
      await handleControllerError(res, error);
    }
  }

  @Put(':userId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: userProgramExperienceConstMessages.UPDATE_OPERATION })
  @ApiParam({ name: 'userId', type: Number, description: 'Seeker user identifier' })
  @ApiBody({ type: UpdateUserProgramExperienceDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.UPDATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({
    status: HttpStatus.INTERNAL_SERVER_ERROR,
    description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR,
  })
  async update(
    @Param('userId', ParseIntPipe) userId: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true }))
    dto: UpdateUserProgramExperienceDto,
    @Res() res: Response,
  ) {
    this.logger.log(userProgramExperienceConstMessages.UPDATE_REQUEST_RECEIVED(userId), dto);
    try {
      const data = await this.service.update(userId, dto);
      await this.responseService.success(
        res,
        userProgramExperienceConstMessages.EXPERIENCES_UPDATED,
        data,
      );
    } catch (error) {
      await handleControllerError(res, error);
    }
  }
}
