import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  ParseBoolPipe,
  ParseIntPipe,
  Post,
  Put,
  Query,
  Req,
  Res,
  UseGuards,
  ValidationPipe,
} from '@nestjs/common';
import {
  ApiBearerAuth,
  ApiBody,
  ApiOperation,
  ApiParam,
  ApiQuery,
  ApiResponse,
  ApiSecurity,
  ApiTags,
} from '@nestjs/swagger';
import { Response } from 'express';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import {
  seekerExperienceRecordMessages,
  SWAGGER_API_RESPONSE,
} from 'src/common/constants/strings-constants';
import { SeekerProgramExperienceService } from './seeker-program-experience.service';
import { CreateSeekerProgramExperienceDto } from './dto/create-seeker-program-experience.dto';
import { UpdateSeekerProgramExperienceDto } from './dto/update-seeker-program-experience.dto';
import { DeleteSeekerProgramExperienceDto } from './dto/delete-seeker-program-experience.dto';
import { FilterSeekerProgramExperienceDto } from './dto/filter-seeker-program-experience.dto';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';


@ApiTags('seeker-program-experience')
@Controller('seeker-program-experience')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin', 'rm', 'relational_manager', 'shoba', 'rm_support', 'viewer', 'mahatria', 'operational_manger', 'finance_manager')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class SeekerProgramExperienceController {
  constructor(
    private readonly service: SeekerProgramExperienceService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: seekerExperienceRecordMessages.CREATE_OPERATION })
  @ApiBody({ type: CreateSeekerProgramExperienceDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  async create(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: CreateSeekerProgramExperienceDto,
    @Res() res: Response,
    @Req() req: any,
  ) {
    const seeker = req.user;
    this.logger.log(seekerExperienceRecordMessages.CREATE_REQUEST(dto.registrationId));
    try {
      const data = await this.service.create(dto, seeker);
      await this.responseService.success(
        res,
        seekerExperienceRecordMessages.CREATED,
        data,
        HttpStatus.CREATED,
      );
    } catch (error) {
      await handleControllerError(res, error);
    }
  }

  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: seekerExperienceRecordMessages.UPDATE_OPERATION })
  @ApiParam({ name: 'id', type: Number })
  @ApiBody({ type: UpdateSeekerProgramExperienceDto })
  async update(
    @Param('id', ParseIntPipe) id: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: UpdateSeekerProgramExperienceDto,
    @Res() res: Response,
    @Req() req: any,
  ) {
    const seeker = req.user;
    this.logger.log(seekerExperienceRecordMessages.UPDATE_REQUEST(id));
    try {
      const data = await this.service.update(id, dto, seeker);
      await this.responseService.success(res, seekerExperienceRecordMessages.UPDATED, data);
    } catch (error) {
      await handleControllerError(res, error);
    }
  }

  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: seekerExperienceRecordMessages.GET_OPERATION })
  @ApiParam({ name: 'id', type: Number })
  @ApiQuery({ name: 'includeDeleted', required: false, type: Boolean })
  async getById(
    @Param('id', ParseIntPipe) id: number,
    @Query('includeDeleted', new ParseBoolPipe({ optional: true })) includeDeleted: boolean,
    @Res() res: Response,
  ) {
    this.logger.log(seekerExperienceRecordMessages.FETCH_REQUEST(id));
    try {
      const data = await this.service.findById(id, includeDeleted);
      await this.responseService.success(res, seekerExperienceRecordMessages.FETCHED, data);
    } catch (error) {
      await handleControllerError(res, error);
    }
  }

  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: seekerExperienceRecordMessages.LIST_OPERATION })
  @ApiQuery({ name: 'approvalStatus', required: false, enum: ApprovalStatusEnum, description: 'Filter by registration approval status' })
  async getAll(
    @Query(new ValidationPipe({ transform: true, whitelist: true })) query: FilterSeekerProgramExperienceDto,
    @Res() res: Response,
  ) {
    this.logger.log(seekerExperienceRecordMessages.LISTING(query.limit, query.offset, query.search));
    try {
      const data = await this.service.findAll(query);
      await this.responseService.success(res, seekerExperienceRecordMessages.LISTED, data);
    } catch (error) {
      await handleControllerError(res, error);
    }
  }

  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: seekerExperienceRecordMessages.DELETE_OPERATION })
  @ApiParam({ name: 'id', type: Number })
  @ApiBody({ type: DeleteSeekerProgramExperienceDto })
  async delete(
    @Param('id', ParseIntPipe) id: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: DeleteSeekerProgramExperienceDto,
    @Res() res: Response,
    @Req() req: any,
  ) {
    this.logger.log(seekerExperienceRecordMessages.DELETE_REQUEST(id));
    const seeker = req.user;
    try {
      const data = await this.service.softDelete(id, dto, seeker);
      await this.responseService.success(res, seekerExperienceRecordMessages.DELETED, data);
    } catch (error) {
      await handleControllerError(res, error);
    }
  }
  @Get('registration/:registrationId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get seeker program experience by registrationId' })
  @ApiParam({ name: 'registrationId', type: Number })
  @ApiQuery({ name: 'includeDeleted', required: false, type: Boolean })
  async getByRegistrationId(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    @Query('includeDeleted', new ParseBoolPipe({ optional: true })) includeDeleted: boolean,
    @Res() res: Response,
  ) {
    this.logger.log(`Fetch seeker program experience by registrationId: ${registrationId}`);
    try {
      const data = await this.service.findByRegistrationId(registrationId, includeDeleted);
      if (!data) {
        return await this.responseService.success(
          res,
          `No seeker program experience found for the given registrationId: ${registrationId}`,
          {},
        );
      }
      await this.responseService.success(res, 'Fetched seeker program experience by registrationId', data);
    } catch (error) {
      await handleControllerError(res, error);
    }
  }

  @Delete('registration/:registrationId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Delete seeker program experience by registrationId' })
  @ApiParam({ name: 'registrationId', type: Number })
  @ApiBody({ type: DeleteSeekerProgramExperienceDto })
  async deleteByRegistrationId(
    @Param('registrationId', ParseIntPipe) registrationId: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: DeleteSeekerProgramExperienceDto,
    @Res() res: Response,
    @Req() req: any,
  ) {
    this.logger.log(`Delete seeker program experience by registrationId: ${registrationId}`);
    const seeker = req.user;
    try {
      await this.service.deleteByRegistrationId(registrationId, seeker, dto);
      await this.responseService.success(
        res,
        `Deleted seeker program experience by registrationId: ${registrationId}`,
        { registrationId },
      );
    } catch (error) {
      await handleControllerError(res, error);
    }
  }
}