import { Injectable } from '@nestjs/common';
import { RoomInventoryRepository } from './room-inventory.repository';
import { ProgramRoomInventoryMapRepository } from './program-room-inventory-map.repository';
import { RoomStatus } from '../common/enums/room-status.enum';
import { RoomStatusKpiDto } from './dto/room-inventory-response.dto';
import { ProgramRoomInventoryMap } from '../common/entities/program-room-inventory-map.entity';
import { AppLoggerService } from '../common/services/logger.service';
import { handleKnownErrors } from '../common/utils/handle-error.util';
import { ERROR_CODES } from '../common/constants/error-string-constants';
import { UserRepository } from '../user/user.repository';
import { ROLE_KEYS, roomInventoryMessages } from '../common/constants/strings-constants';
import { LookupDataRepository } from 'src/lookup-data/lookup-data.repository';
import { FilterConfig } from '../common/interfaces/interface';
import {
  ROOM_INVENTORY_AGE_FILTER_OPTIONS,
  ROOM_INVENTORY_GENDER_FILTER_OPTIONS,
  ROOM_INVENTORY_HDB_FILTER_OPTIONS,
  ROOM_INVENTORY_ROOMMATE_PREFERENCE_OPTIONS,
  ROOM_INVENTORY_PAIRED_STATUS_OPTIONS,
  ROOM_INVENTORY_DEPARTURE_DATE_FILTER_OPTIONS,
  ROOM_INVENTORY_RETURN_TERMINAL_OPTIONS,
  ROOM_INVENTORY_FILTER_CONFIG_BASE,
  ROOM_STATUS_CONFIG,
  ROOM_INVENTORY_CONTROLLER_FILTERS,
  ROOM_INVENTORY_REPORT_ACCESS_MATRIX,
  ROOM_INVENTORY_REPORT_DEFINITIONS,
} from '../common/constants/constants';
import { ProgramRegistrationRepository } from 'src/program-registration/program-registration.repository';
import InifniBadRequestException from '../common/exceptions/infini-badrequest-exception';
import { UpdateRoomInventoryDto } from './dto/update-room-inventory.dto';
import { InifniNotFoundException } from '../common/exceptions/infini-notfound-exception';
import { ExcelService } from '../common/services/excel.service';
import { formatDateTimeIST, calculateAge } from '../common/utils/common.util';

/**
 * Service for managing room inventory operations
 * Provides business logic for room inventory listing, KPI calculations, and data aggregation
 */
@Injectable()
export class RoomInventoryService {
  constructor(
    private readonly roomInventoryRepository: RoomInventoryRepository,
    private readonly programRoomInventoryMapRepository: ProgramRoomInventoryMapRepository,
    private readonly userRepository: UserRepository,
    private readonly logger: AppLoggerService,
    private readonly lookupDataRepository: LookupDataRepository,
    private readonly programRegistrationRepository: ProgramRegistrationRepository,
    private readonly excelService: ExcelService,
  ) {}

  /**
   * Retrieves all room inventory items with complete relations and calculates KPIs
   * @param limit - Number of records per page or 'all' to get all records
   * @param offset - Offset for pagination or 'all' to skip pagination
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param searchText - Optional search text for room number or label
   * @param filter - Optional advanced filter object with additional criteria
   * @returns Promise with raw room inventory data, KPIs, and filter options
   */
  async getAllRoomInventoryWithKpis(
    limit: number | 'all' = 10,
    offset: number | 'all' = 0,
    programId?: number,
    subProgramId?: number,
    searchText?: string,
    filter?: any,
    userRoles?: string[]
  ): Promise<{
    data: ProgramRoomInventoryMap[];
    kpis: RoomStatusKpiDto[];
    filters: Array<{
      key: string;
      label: string;
      type: string;
      sortable: boolean;
      filterable: boolean;
      order: number;
      options: Array<{ label: string; value: string }>;
    }>;
    reports: Array<{
      code: string;
      label: string;
      description: string;
    }>;
    total: number;
    limit: number;
    offset: number;
  }> {
    this.logger.log(roomInventoryMessages.ROOM_INVENTORY_WITH_KPIS_REQUEST_RECEIVED, {
      limit,
      offset,
      programId,
      subProgramId,
      searchText,
      filter
    });

    try {
      // Get room inventory data with pagination from repository
      // Using default parameters (undefined) to load all relations for UI display
      const { data, totalRecords } = await this.roomInventoryRepository.findRoomInventoryWithPagination(
        limit,
        offset,
        programId,
        subProgramId,
        searchText,
        filter,
        undefined, // selectFields - use defaults
        undefined  // includeRelations - load all relations for UI
      );

      // Calculate KPIs for all data (not just paginated results)
      const kpis = await this.calculateRoomInventoryKpis(programId, subProgramId, searchText);

      // Extract filter IDs from filter object
      const venueIds = Array.isArray(filter?.venues) ? filter.venues : [];
      const blockIds = Array.isArray(filter?.block) ? filter.block : [];
      
      // Clean and validate IDs - convert strings to numbers and remove duplicates
      const cleanedVenueIds: number[] = venueIds.length > 0 
        ? ([...new Set(venueIds.map((id: any) => typeof id === 'string' ? parseInt(id) : id).filter((id: any) => !isNaN(id)))] as number[])
        : [];
      
      const cleanedBlockIds: number[] = blockIds.length > 0 
        ? ([...new Set(blockIds.map((id: any) => typeof id === 'string' ? parseInt(id) : id).filter((id: any) => !isNaN(id)))] as number[])
        : [];
      
      // Get filter dropdown options for property, block, and floor with cascading filters
      const [propertyOptions, blockOptions, floorOptions] = await Promise.all([
        this.roomInventoryRepository.getDistinctVenues(programId, subProgramId),
        this.roomInventoryRepository.getDistinctBlocks(programId, subProgramId, cleanedVenueIds),
        this.roomInventoryRepository.getDistinctFloors(programId, subProgramId, cleanedVenueIds, cleanedBlockIds),
      ]);

      // Add default "All" option at the top
      propertyOptions.unshift({ label: roomInventoryMessages.LABEL_ALL, value: [...new Set(propertyOptions.map(option => option.value))].join(',') });
      blockOptions.unshift({ label: roomInventoryMessages.LABEL_ALL, value: [...new Set(blockOptions.map(option => option.value))].join(',') });
      floorOptions.unshift({ label: roomInventoryMessages.LABEL_ALL, value: [...new Set(floorOptions.map(option => option.value))].join(',') });
      // Build filters in the detailed format
      const filters: FilterConfig[] = ROOM_INVENTORY_CONTROLLER_FILTERS.map(baseFilter => {
        const filterConfig: FilterConfig = { 
          ...baseFilter,
          options: []
        };
        
        // Assign appropriate options based on filter key
        switch (filterConfig.key) {
          case roomInventoryMessages.FILTER_KEY_VENUES:
            filterConfig.options = propertyOptions;
            break;
          case roomInventoryMessages.FILTER_KEY_BLOCK:
            filterConfig.options = blockOptions;
            break;
          case roomInventoryMessages.FILTER_KEY_FLOOR:
            filterConfig.options = floorOptions;
            break;
          default:
            filterConfig.options = [];
            this.logger.warn(roomInventoryMessages.UNKNOWN_FILTER_KEY_IN_CONTROLLER, { key: filterConfig.key });
        }
        
        return filterConfig;
      });

      // Get available reports based on user roles
      const reports = userRoles ? this.getAvailableReports(userRoles) : [];

      this.logger.log(roomInventoryMessages.ROOM_INVENTORY_DATA_PROCESSED, {
        totalRecords,
        currentPageRecords: data.length,
        kpisSummary: {
          statusBreakdownCount: kpis.length
        },
        filters: {
          propertyCount: propertyOptions.length,
          blockCount: blockOptions.length
        },
        reportsCount: reports.length
      });

      return {
        data,
        kpis,
        filters,
        reports,
        total: totalRecords,
        limit: limit === roomInventoryMessages.PAGINATION_ALL ? totalRecords : (limit as number),
        offset: offset === roomInventoryMessages.PAGINATION_ALL ? 0 : (offset as number)
      };

    } catch (error) {
      this.logger.error(roomInventoryMessages.ERROR_IN_ROOM_INVENTORY_RETRIEVAL, error.stack, {
        limit,
        offset,
        programId,
        subProgramId,
        searchText,
        filter
      });
      handleKnownErrors(ERROR_CODES.ROOM_INVENTORY_GET_FAILED, error);
    }
  }

  /**
   * Calculates KPIs for room inventory - returns array of status and count directly from repository
   * Maps all RoomStatus enum values with their counts from database
   * @param programId - Optional program ID filter for KPI calculation
   * @param subProgramId - Optional sub program ID filter for KPI calculation
   * @param searchText - Optional search text filter for KPI calculation
   * @returns Promise<RoomStatusKpiDto[]> Array of all room statuses with their counts
   */
  private async calculateRoomInventoryKpis(
    programId?: number,
    subProgramId?: number,
    searchText?: string
  ): Promise<RoomStatusKpiDto[]> {
    this.logger.debug(roomInventoryMessages.CALCULATING_ROOM_INVENTORY_KPIS, {
      programId,
      subProgramId,
      searchText
    });

    try {
      // Get status breakdown from database (only returns statuses that exist)
      const statusBreakdown = await this.roomInventoryRepository.calculateStatusBreakdown(
        programId,
        subProgramId,
      );

      // Create a map of key to count from database results
      const statusCountMap = new Map<string, number>();
      let totalCount = 0;

      statusBreakdown.forEach((item) => {
        statusCountMap.set(item.status, Number(item.count));
        totalCount += Number(item.count);
      });

      // Define all possible statuses from RoomStatus enum with their labels
      const allStatuses: { status: string[]; label: string; key: string }[] = ROOM_STATUS_CONFIG.map(
        (statusInfo) => ({
          status: [...statusInfo.status],
          label: statusInfo.label,
          key: statusInfo.key,
        }),
      );

      // Get allocation status breakdown from program registration repository
      const allocationStatusBreakdown =
        await this.programRegistrationRepository.getRegistrationCountByAllocationStatus(
          programId,
          subProgramId,
        );
      statusCountMap.set(roomInventoryMessages.STATUS_KEY_ALLOCATED_SEEKERS, allocationStatusBreakdown.allocated || 0);  
      
      // Map all statuses with their counts (0 if not in database) and include canClick property
      const kpis: RoomStatusKpiDto[] = allStatuses.map((statusInfo) => ({
        key: statusInfo.key,
        status: [...statusInfo.status],
        label: statusInfo.label,
        count:
          statusInfo.key === roomInventoryMessages.STATUS_KEY_ALL
            ? totalCount
            : statusCountMap.get(statusInfo.key) || 0,
      }));

      this.logger.debug(roomInventoryMessages.ROOM_INVENTORY_KPIS_CALCULATED, {
        kpisCount: kpis.length,
        totalCount: totalCount,
        programId,
        subProgramId,
      });

      return kpis;
    } catch (error) {
      this.logger.error(roomInventoryMessages.ERROR_CALCULATING_KPIS, error.stack, {
        programId,
        subProgramId,
        searchText
      });
      handleKnownErrors(ERROR_CODES.ROOM_INVENTORY_KPI_CALCULATION_FAILED, error);
    }
  }

  /**
   * Get comprehensive filter configuration for room inventory
   * Provides all available filter options with sortability and filterability settings
   * Following the same pattern as registration service filters
   * 
   * @param filter - Optional filter object containing programId, subProgramId, venues, block, floor arrays
   * @returns Promise<FilterConfig[]> Complete filter configuration with all available options
   */
  async getFilterConfiguration(
    filter?: any
  ): Promise<FilterConfig[]> {
    this.logger.log(roomInventoryMessages.ROOM_INVENTORY_FILTER_CONFIG_REQUEST_RECEIVED, {
      filter
    });

    try {
      // Extract filter properties - handle data processing in service
      const programId = filter?.programId;
      const subProgramId = filter?.subProgramId;
      const venueIds = Array.isArray(filter?.venues) ? filter.venues : [];
      const blockIds = Array.isArray(filter?.block) ? filter.block : [];
      const floorIds = Array.isArray(filter?.floor) ? filter.floor : [];

      // Clean and validate IDs - convert strings to numbers and remove duplicates
      const cleanedVenueIds: number[] = venueIds.length > 0 
        ? ([...new Set(venueIds.map((id: any) => typeof id === 'string' ? parseInt(id) : id).filter((id: any) => !isNaN(id)))] as number[])
        : [];
      
      const cleanedBlockIds: number[] = blockIds.length > 0 
        ? ([...new Set(blockIds.map((id: any) => typeof id === 'string' ? parseInt(id) : id).filter((id: any) => !isNaN(id)))] as number[])
        : [];
      
      const cleanedFloorIds: number[] = floorIds.length > 0 
        ? ([...new Set(floorIds.map((id: any) => typeof id === 'string' ? parseInt(id) : id).filter((id: any) => !isNaN(id)))] as number[])
        : [];

      this.logger.debug(roomInventoryMessages.FILTER_IDS_CLEANED, {
        originalVenueIds: venueIds,
        cleanedVenueIds,
        originalBlockIds: blockIds,
        cleanedBlockIds,
        originalFloorIds: floorIds,
        cleanedFloorIds
      });

      // Get dynamic data for filter options from repository
      const [
        availableVenues,
        availableBlocks,
        availableFloors,
        availableRoomTypes,
        availableBedTypes,
        occupancyRanges
      ] = await Promise.all([
        this.roomInventoryRepository.getDistinctVenues(programId, subProgramId),
        this.roomInventoryRepository.getDistinctBlocks(programId, subProgramId, cleanedVenueIds),
        this.roomInventoryRepository.getDistinctFloors(programId, subProgramId, cleanedVenueIds, cleanedBlockIds),
        this.roomInventoryRepository.getDistinctRoomTypes(programId, subProgramId, cleanedVenueIds, cleanedBlockIds, cleanedFloorIds),
        this.roomInventoryRepository.getDistinctBedTypes(programId, subProgramId, cleanedVenueIds, cleanedBlockIds, cleanedFloorIds),
        this.roomInventoryRepository.getOccupancyRanges(programId, subProgramId, cleanedVenueIds, cleanedBlockIds, cleanedFloorIds)
      ]);

      // Get cities from lookup data using repository
      const citiesResponse = await this.lookupDataRepository.findActiveByCategory(roomInventoryMessages.LOOKUP_CATEGORY_CITY_NAME);
      const cities = citiesResponse.data;

      const rmUsers = await this.userRepository.getUsersByRoleKey(ROLE_KEYS.RELATIONAL_MANAGER);

      // Transform lookup data into filter options
      let availableLocations = cities
        .map((city) => ({
          value: city.label?.trim() || '',
          label: city.label?.trim() || '',
        }))
        .filter((location) => location.value !== '' && location.label !== '')
        .sort((a, b) => a.label.localeCompare(b.label));

      // Check if 'Other' city exists, if not, add it at the end
      const otherCity = availableLocations.find(
        (location) => location.label.toLowerCase() === roomInventoryMessages.LABEL_OTHER.toLowerCase(),
      );
      if (otherCity) {
        availableLocations = availableLocations.filter(
          (location) => location.label.toLowerCase() !== roomInventoryMessages.LABEL_OTHER.toLowerCase(),
        );
        availableLocations.push(otherCity);
      } else {
        availableLocations.push({ label: roomInventoryMessages.LABEL_OTHER, value: roomInventoryMessages.LABEL_OTHER });
      }
  
      // Transform RM users to filter options format in service layer
      const availableRMs = rmUsers
        .map(user => ({
          value: user.id,
          label: user.orgUsrName || user.fullName || `${user.firstName || ''} ${user.lastName || ''}`.trim() || roomInventoryMessages.LABEL_UNNAMED_RM
        }))
        .filter(rm => rm.label !== roomInventoryMessages.LABEL_UNNAMED_RM)
        .sort((a, b) => a.label.localeCompare(b.label));

      // Create filter configuration using constants with FilterConfig interface
      const filters: FilterConfig[] = ROOM_INVENTORY_FILTER_CONFIG_BASE.map(baseConfig => {
        const config: FilterConfig = { 
          ...baseConfig,
          options: []
        };
        
        // Assign appropriate options based on filter key
        switch (config.key) {
          case roomInventoryMessages.FILTER_KEY_AGE:
            config.options = [...ROOM_INVENTORY_AGE_FILTER_OPTIONS];
            break;
          case roomInventoryMessages.FILTER_KEY_GENDER:
            config.options = [...ROOM_INVENTORY_GENDER_FILTER_OPTIONS];
            break;
          case roomInventoryMessages.FILTER_KEY_DEPARTURE_DATETIME:
            config.options = [...ROOM_INVENTORY_DEPARTURE_DATE_FILTER_OPTIONS];
            break;
          case roomInventoryMessages.FILTER_KEY_NO_OF_HDBS:
            config.options = [...ROOM_INVENTORY_HDB_FILTER_OPTIONS];
            break;
          case roomInventoryMessages.FILTER_KEY_PREFERRED_ROOMMATE:
            config.options = [...ROOM_INVENTORY_ROOMMATE_PREFERENCE_OPTIONS];
            break;
          case roomInventoryMessages.FILTER_KEY_RM_CONTACT:
            config.options = availableRMs;
            break;
          case roomInventoryMessages.FILTER_KEY_CITY:
            config.options = availableLocations;
            break;
          case roomInventoryMessages.FILTER_KEY_PAIRED_STATUS:
            config.options = [...ROOM_INVENTORY_PAIRED_STATUS_OPTIONS];
            break;
          case roomInventoryMessages.FILTER_KEY_OCCUPANCY:
            config.options = [
              { label: roomInventoryMessages.FILTER_LABEL_OCCUPANCY, min: 0, max: 10, eq: 15 }
            ];
            config.placeholder = roomInventoryMessages.FILTER_PLACEHOLDER_OCCUPANCY;
            break;
          case roomInventoryMessages.FILTER_KEY_FLOOR:
            config.options = availableFloors;
            break;
          case roomInventoryMessages.FILTER_KEY_RETURN_TRAVEL_TERMINAL:
            config.options = [...ROOM_INVENTORY_RETURN_TERMINAL_OPTIONS];
            break;
          default:
            config.options = [];
            this.logger.warn(roomInventoryMessages.UNKNOWN_FILTER_KEY_IN_CONFIG, { key: config.key });
        }
        
        return config;
      });

      // Sort filters by order
      const sortedFilters = filters.sort((a, b) => a.order - b.order);

      this.logger.log(roomInventoryMessages.ROOM_INVENTORY_FILTER_CONFIG_GENERATED, {
        totalFilters: sortedFilters.length,
        appliedFilters: {
          programId,
          subProgramId,
          venueIds: cleanedVenueIds,
          blockIds: cleanedBlockIds,
          floorIds: cleanedFloorIds
        },
        dynamicOptionsLoaded: {
          venues: availableVenues.length,
          blocks: availableBlocks.length,
          floors: availableFloors.length,
          roomTypes: availableRoomTypes.length,
          bedTypes: availableBedTypes.length,
          occupancyRange: occupancyRanges.length > 0 ? `${occupancyRanges[0].minOccupancy}-${occupancyRanges[0].maxOccupancy}` : 'N/A'
        }
      });

      return sortedFilters;

    } catch (error) {
      this.logger.error(roomInventoryMessages.ERROR_GENERATING_FILTER_CONFIG, error.stack, {
        filter
      });
      handleKnownErrors(ERROR_CODES.ROOM_INVENTORY_FILTER_CONFIG_FAILED, error);
    }
  }

  /**
   * Update room inventory details with conditional status management
   * @param id - Room inventory identifier
   * @param updateRoomInventoryDto - Payload containing fields to update
   * @param updatedById - User performing the update
   * @returns Promise<ProgramRoomInventoryMap>
   */
  async updateRoomInventory(
    id: number,
    updateRoomInventoryDto: UpdateRoomInventoryDto,
    updatedById?: number,
  ): Promise<ProgramRoomInventoryMap> {
    this.logger.log(roomInventoryMessages.ROOM_INVENTORY_UPDATE_REQUEST_RECEIVED, {
      id,
      updateRoomInventoryDto,
      updatedById
    });

    try {
      // Fetch room inventory with allocations to check for existing allocations
      const roomInventory = await this.programRoomInventoryMapRepository.findByIdWithRelations(id);

      if (!roomInventory) {
        throw new InifniNotFoundException(
          ERROR_CODES.ROOM_INVENTORY_NOT_FOUND,
          null,
          null,
          id.toString(),
        );
      }

      // Check if trying to set room to reserved status when allocations exist
      const isSettingToReserved = updateRoomInventoryDto.isReserved === true;
      const hasActiveAllocations = roomInventory.roomAllocations && roomInventory.roomAllocations.length > 0;
      
      if (isSettingToReserved && hasActiveAllocations) {
        this.logger.error(
          roomInventoryMessages.ROOM_CANNOT_SET_RESERVED_WITH_ALLOCATIONS,
          undefined,
          {
            roomInventoryId: id,
            activeAllocations: roomInventory.roomAllocations.length,
            allocationIds: roomInventory.roomAllocations.map(allocation => allocation.id)
          }
        );
        
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_INVENTORY_RESERVED_WITH_ALLOCATIONS,
          null,
          null,
          roomInventoryMessages.ROOM_HAS_ACTIVE_ALLOCATIONS(roomInventory.roomAllocations.length),
        );
      }

      const totalCapacity = roomInventory.room?.occupancy || 0;
      const newRemainingOccupancy = updateRoomInventoryDto.remainingOccupancy ?? roomInventory.remainingOccupancy;

      if (newRemainingOccupancy < 0) {
        throw new InifniBadRequestException(
          ERROR_CODES.ROOM_INVENTORY_VALIDATION_FAILED,
          null,
          null,
          roomInventory?.id.toString()
        );
      }

      if (totalCapacity && newRemainingOccupancy > totalCapacity) {
        throw new InifniBadRequestException(
          ERROR_CODES.REMAINING_OCCUPANCY_EXCEEDS_CAPACITY,
          null,
          null,
          roomInventory?.id.toString()
        );
      }

      const updatedIsReserved = updateRoomInventoryDto.isReserved ?? roomInventory.isReserved;
      const resolvedRoomStatus = this.resolveRoomStatus(
        updatedIsReserved,
        newRemainingOccupancy,
        totalCapacity,
        updateRoomInventoryDto.roomStatus,
        roomInventory.roomStatus
      );

      const updatedInventory = await this.roomInventoryRepository.save(
        new ProgramRoomInventoryMap({
          ...roomInventory,
          ...updateRoomInventoryDto,
          remainingOccupancy: newRemainingOccupancy,
          isReserved: updatedIsReserved,
          reservedFor: updatedIsReserved
            ? updateRoomInventoryDto.reservedFor ?? roomInventory.reservedFor
            : undefined,
          roomStatus: resolvedRoomStatus,
          occupiedStartsAt: updateRoomInventoryDto.occupiedStartsAt ?? roomInventory.occupiedStartsAt,
          occupiedEndsAt: updateRoomInventoryDto.occupiedEndsAt ?? roomInventory.occupiedEndsAt,
          updatedById: updatedById ?? roomInventory.updatedById ?? roomInventory.createdById,
        }),
      );

      this.logger.log(roomInventoryMessages.ROOM_INVENTORY_UPDATED_SUCCESSFULLY, {
        id,
        newRemainingOccupancy,
        resolvedRoomStatus,
        updatedIsReserved,
      });

      return updatedInventory;
    } catch (error) {
      this.logger.error(roomInventoryMessages.ERROR_UPDATING_ROOM_INVENTORY, error.stack, {
        id,
        updateRoomInventoryDto
      });
      handleKnownErrors(ERROR_CODES.ROOM_INVENTORY_UPDATE_FAILED, error);
    }
  }

  /**
   * Determine final room status based on reservation flag and occupancy values
   * Validates occupancy constraints for each status and maintains existing status if invalid
   * @param isReserved - Indicates if the room is reserved
   * @param remainingOccupancy - Remaining occupancy after update
   * @param totalCapacity - Total room capacity
   * @param requestedStatus - Optional status provided by client
   * @param currentStatus - Current status of the room (for fallback)
   * @returns RoomStatus
   */
  private resolveRoomStatus(
    isReserved: boolean,
    remainingOccupancy: number,
    totalCapacity: number,
    requestedStatus?: RoomStatus,
    currentStatus?: RoomStatus,
  ): RoomStatus {
    // Reserved rooms always have RESERVED status
    if (isReserved) {
      return RoomStatus.RESERVED;
    }

    // If a specific status is requested (and not RESERVED)
    if (requestedStatus && requestedStatus !== RoomStatus.RESERVED) {
      // Validate occupancy constraints for the requested status
      switch (requestedStatus) {
        case RoomStatus.AVAILABLE:
          // Available: remaining occupancy should equal total capacity
          if (totalCapacity > 0 && remainingOccupancy === totalCapacity) {
            return RoomStatus.AVAILABLE;
          }
          break;
          
        case RoomStatus.PARTIALLY_ALLOTTED:
          // Partially allotted: remaining occupancy must be less than total capacity and greater than 0
          if (totalCapacity > 0 && remainingOccupancy > 0 && remainingOccupancy < totalCapacity) {
            return RoomStatus.PARTIALLY_ALLOTTED;
          }
          break;
          
        case RoomStatus.ALLOTTED:
          // Allotted: remaining occupancy should be zero
          if (remainingOccupancy === 0) {
            return RoomStatus.ALLOTTED;
          }
          break;
      }
      
      // If requested status doesn't match occupancy constraints, return current status or auto-determine
      if (currentStatus) {
        return currentStatus;
      }
    }

    // Auto-determine status based on occupancy when no valid requested status
    if (totalCapacity > 0 && remainingOccupancy === totalCapacity) {
      return RoomStatus.AVAILABLE;
    }

    if (remainingOccupancy === 0) {
      return RoomStatus.ALLOTTED;
    }

    if (totalCapacity > 0 && remainingOccupancy > 0 && remainingOccupancy < totalCapacity) {
      return RoomStatus.PARTIALLY_ALLOTTED;
    }

    // Fallback to current status if available, otherwise AVAILABLE
    return currentStatus || RoomStatus.AVAILABLE;
  }

  /**
   * Get available reports for user based on their roles
   * @param userRoles - Array of user role names
   * @returns Array of report configurations accessible to the user
   */
  getAvailableReports(userRoles: string[]): Array<{
    code: string;
    label: string;
    description: string;
  }> {
    const availableReports: Array<{
      code: string;
      label: string;
      description: string;
    }> = [];

    // Iterate through all report definitions
    for (const [reportCode, reportDef] of Object.entries(ROOM_INVENTORY_REPORT_DEFINITIONS)) {
      const allowedRoles = ROOM_INVENTORY_REPORT_ACCESS_MATRIX[reportCode];
      
      // Check if user has permission to access this report
      if (allowedRoles && userRoles.some(role => allowedRoles.includes(role))) {
        availableReports.push({
          code: reportCode,
          label: reportDef.label,
          description: reportDef.description,
        });
      }
    }

    return availableReports;
  }

  /**
   * Validate report access for API endpoint
   * @param userRoles - Array of user roles
   * @param reportCode - Report code to validate
   * @returns Promise<boolean> indicating access permission
   */
  async validateReportAccess(userRoles: string[], reportCode: string): Promise<boolean> {
    const allowedRoles = ROOM_INVENTORY_REPORT_ACCESS_MATRIX[reportCode];
    if (!allowedRoles) return false;
    
    return userRoles.some(role => allowedRoles.includes(role));
  }

  /**
   * Generate report based on report code and filters
   * @param reportCode - Code of the report to generate
   * @param options - Report generation options
   * @returns Promise<string> containing the S3 URL
   */
  async generateReport(reportCode: string, options: {
    programId?: number;
    subProgramId?: number;
    search?: string;
    filters?: any;
    userRoles: string[];
  }): Promise<string> {
    // Validate report definition exists
    const reportDef = ROOM_INVENTORY_REPORT_DEFINITIONS[reportCode];
    if (!reportDef) {
      throw new InifniBadRequestException(
        ERROR_CODES.INVALID_REPORT_CODE,
        null,
        null,
        roomInventoryMessages.INVALID_REPORT_CODE_MESSAGE(reportCode)
      );
    }

    try {
      // Get data based on report type
      let data: any[] = [];
      
      switch (reportCode) {
        case roomInventoryMessages.REPORT_CODE_ROOM_INVENTORY_ALLOCATION:
          data = await this.getRoomInventoryAllocationReportData(options);
          break;
        default:
          throw new InifniBadRequestException(
            ERROR_CODES.INVALID_REPORT_CODE,
            null,
            null,
            roomInventoryMessages.UNSUPPORTED_REPORT_CODE(reportCode)
          );
      }

      // Check if data is empty
      if (!data || data.length === 0) {
        throw new InifniBadRequestException(
          ERROR_CODES.NO_DATA_FOUND_FOR_REPORT,
          null,
          null,
          roomInventoryMessages.NO_DATA_FOR_REPORT(reportDef.label)
        );
      }

      // Generate Excel file and upload to S3
      const filename = `${reportCode}_${new Date().getTime()}`;
      const excelOptions = {
        sheetName: reportDef.label,
        makeHeaderBold: true,
        autoWidth: true
      };
      
      const s3Url = await this.excelService.jsonToExcelAndUpload(
        data,
        filename,
        roomInventoryMessages.MIME_TYPE_EXCEL,
        excelOptions
      );
      
      return s3Url;
    } catch (error) {
      this.logger.error(roomInventoryMessages.ERROR_GENERATING_REPORT(reportCode), error);
      throw error;
    }
  }

  /**
   * Get data for Room Inventory Allocation report
   * Creates a row for each allocation in each room
   */
  private async getRoomInventoryAllocationReportData(options: any): Promise<any[]> {
    // Fetch data with specific relations needed for the report
    // Only load the minimal relations required: room hierarchy (room, floor, block, venue)
    // and allocation data (allocations, registration, rmUser) for performance
    const { data } = await this.roomInventoryRepository.findRoomInventoryWithPagination(
      roomInventoryMessages.PAGINATION_ALL as any,
      roomInventoryMessages.PAGINATION_ALL as any,
      options.programId,
      options.subProgramId,
      '',
      options.filters || {},
      undefined,
      {
        room: true,
        floor: true,
        block: true,
        venue: true,
        address: false,
        allocations: true,
        registration: true,
        rmUser: true,
        travelPlan: false,
        regPair: false,
        allocatedProgram: false
      }
    );

    const reportRows: any[] = [];
    let serialNumber = 1;

    // Iterate through each room inventory
    data.forEach(inventory => {
      const room = inventory.room;
      const floor = room?.floor;
      const block = floor?.block;
      const venue = block?.venue;
      const allocations = inventory.roomAllocations || [];

      // If room has allocations, create a row for each allocation
      if (allocations.length > 0) {
        allocations.forEach(allocation => {
          const registration = allocation.registration;
          reportRows.push({
            'S.No.': serialNumber++,
            [roomInventoryMessages.COLUMN_VENUE]: venue?.label || '',
            [roomInventoryMessages.COLUMN_BLOCK]: block?.label || '',
            [roomInventoryMessages.COLUMN_FLOOR]: floor?.label || '',
            [roomInventoryMessages.COLUMN_ROOM_NUMBER]: room?.roomNumber || '',
            [roomInventoryMessages.COLUMN_IS_RESERVED]: inventory.isReserved ? roomInventoryMessages.LABEL_YES : roomInventoryMessages.LABEL_NO,
            [roomInventoryMessages.COLUMN_RESERVED_FOR]: inventory.reservedFor || '',
            [roomInventoryMessages.COLUMN_HDB_NAME]: registration?.fullName || '',
            [roomInventoryMessages.COLUMN_GENDER]: registration?.gender || '',
            [roomInventoryMessages.COLUMN_MOBILE]: registration?.mobileNumber || '',
            [roomInventoryMessages.COLUMN_RM_CONTACT]: allocation?.registration?.rmContactUser?.fullName || '',
            '# HDBs': registration?.noOfHDBs || '',
            [roomInventoryMessages.COLUMN_CITY]: registration?.city || '',
          });
        });
      } else {
        // If no allocations, still show the room with empty seeker details
        reportRows.push({
          'S.No.': serialNumber++,
          [roomInventoryMessages.COLUMN_VENUE]: venue?.label || '',
          [roomInventoryMessages.COLUMN_BLOCK]: block?.label || '',
          [roomInventoryMessages.COLUMN_FLOOR]: floor?.label || '',
          [roomInventoryMessages.COLUMN_ROOM_NUMBER]: room?.roomNumber || '',
          [roomInventoryMessages.COLUMN_IS_RESERVED]: inventory.isReserved ? roomInventoryMessages.LABEL_YES : roomInventoryMessages.LABEL_NO,
          [roomInventoryMessages.COLUMN_RESERVED_FOR]: inventory.reservedFor || '',
          [roomInventoryMessages.COLUMN_HDB_NAME]: '',
          [roomInventoryMessages.COLUMN_GENDER]: '',
          [roomInventoryMessages.COLUMN_MOBILE]: '',
          [roomInventoryMessages.COLUMN_RM_CONTACT]: '',
          '# HDBs': '',
          [roomInventoryMessages.COLUMN_CITY]: '',
        });
      }
    });

    return reportRows;
  }

}