import { Injectable, Logger } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository, SelectQueryBuilder } from 'typeorm';
import { ProgramRoomInventoryMap } from '../common/entities/program-room-inventory-map.entity';
import { GenderEnum } from 'src/common/enum/gender.enum';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';

/**
 * Repository for managing program room inventory map database operations
 * Provides data access methods for room inventory validation and management
 */
@Injectable()
export class RoomInventoryRepository {
  private readonly logger = new Logger(RoomInventoryRepository.name);

  constructor(
    @InjectRepository(ProgramRoomInventoryMap)
    private readonly repository: Repository<ProgramRoomInventoryMap>,
  ) {}

  /**
   * Find room inventory mapping by ID and program ID
   * @param id - Room inventory map ID
   * @param programId - Program ID
   * @returns Promise<ProgramRoomInventoryMap | null>
   */
  async findByIdAndProgramId(id: number, programId: number): Promise<ProgramRoomInventoryMap | null> {
    return this.repository.findOne({
      where: {
        id,
        programId,
      },
      relations: ['room', 'program', 'subProgram'],
    });
  }

  /**
   * Find room inventory mapping by ID and sub-program ID
   * @param id - Room inventory map ID
   * @param subProgramId - Sub-program ID
   * @returns Promise<ProgramRoomInventoryMap | null>
   */
  async findByIdAndSubProgramId(id: number, subProgramId: number): Promise<ProgramRoomInventoryMap | null> {
    return this.repository.findOne({
      where: {
        id,
        subProgramId,
      },
      relations: ['room', 'program', 'subProgram'],
    });
  }

  /**
   * Find room inventory mapping by ID
   * @param id - Room inventory map ID
   * @returns Promise<ProgramRoomInventoryMap | null>
   */
  async findById(id: number): Promise<ProgramRoomInventoryMap | null> {
    return this.repository.findOne({
      where: { id },
      relations: ['room', 'program', 'subProgram'],
    });
  }

  /**
   * Persist room inventory changes
   * @param roomInventory - Room inventory entity to be saved
   * @returns Promise<ProgramRoomInventoryMap>
   */
  async save(roomInventory: ProgramRoomInventoryMap): Promise<ProgramRoomInventoryMap> {
    this.logger.debug('Saving room inventory changes', { id: roomInventory.id });

    return this.repository.save(roomInventory);
  }

  /**
   * Update remaining occupancy for a room inventory
   * @param id - Room inventory map ID
   * @param remainingOccupancy - New remaining occupancy value
   * @returns Promise<void>
   */
  async updateRemainingOccupancy(id: number, remainingOccupancy: number): Promise<void> {
    await this.repository.update(id, { remainingOccupancy });
  }

  /**
   * Builds base query builder with all necessary joins and filters
   * Handles array-based filters from UI for venues, blocks, floors, roomType, bedType, gender, and age
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param searchText - Optional search text for room number or label
   * @param filter - Optional advanced filter object with additional criteria
   * @param selectFields - Optional array of field paths to select (e.g., ['inventory.id', 'room.roomNumber'])
   * @param includeRelations - Optional flags to control which relations to join and load
   * @returns SelectQueryBuilder<ProgramRoomInventoryMap>
   */
  
  private buildBaseQueryBuilder(
    programId?: number,
    subProgramId?: number,
    searchText?: string,
    filter?: any,
    selectFields?: string[],
    includeRelations?: {
      room?: boolean;
      floor?: boolean;
      block?: boolean;
      venue?: boolean;
      address?: boolean;
      allocations?: boolean;
      registration?: boolean;
      rmUser?: boolean;
      travelPlan?: boolean;
      regPair?: boolean;
      allocatedProgram?: boolean;
    }
  ): SelectQueryBuilder<ProgramRoomInventoryMap> {
    // Default to loading all relations if not specified
    const relations = includeRelations || {
      room: true,
      floor: true,
      block: true,
      venue: true,
      address: true,
      allocations: true,
      registration: true,
      rmUser: true,
      travelPlan: true,
      regPair: true,
      allocatedProgram: true,
    };

    // Start building query with base inventory selection
    let queryBuilder = this.repository.createQueryBuilder('inventory');

    // Apply custom select fields if provided, otherwise use default inventory fields
    if (selectFields && selectFields.length > 0) {
      queryBuilder = queryBuilder.select(selectFields);
    } else {
      queryBuilder = queryBuilder.select([
        'inventory.id',
        'inventory.roomId',
        'inventory.programId',
        'inventory.subProgramId',
        'inventory.remainingOccupancy',
        'inventory.isReserved',
        'inventory.reservedFor',
        'inventory.roomStatus',
        'inventory.remarks',
        'inventory.deletedAt',
        'inventory.roomCategory'
      ]);
    }

    // Conditionally join and select room relation
    if (relations.room) {
      queryBuilder = queryBuilder
        .leftJoin('inventory.room', 'room')
        .addSelect([
          'room.label',
          'room.description',
          'room.roomNumber',
          'room.occupancy',
          'room.roomType',
          'room.bedType',
          'room.remarks',
          'room.deletedAt',
        ]);
    }

    // Conditionally join and select floor relation (requires room)
    if (relations.floor && relations.room) {
      queryBuilder = queryBuilder
        .leftJoin('room.floor', 'floor')
        .addSelect([
          'floor.id',
          'floor.label',
          'floor.description',
          'floor.deletedAt',
        ]);
    }

    // Conditionally join and select block relation (requires floor)
    if (relations.block && relations.floor && relations.room) {
      queryBuilder = queryBuilder
        .leftJoin('floor.block', 'block')
        .addSelect([
          'block.id',
          'block.label',
          'block.description',
          'block.deletedAt',
        ]);
    }

    // Conditionally join and select venue relation (requires block)
    if (relations.venue && relations.block && relations.floor && relations.room) {
      queryBuilder = queryBuilder
        .leftJoin('block.venue', 'venue')
        .addSelect([
          'venue.id',
          'venue.label',
          'venue.description',
          'venue.addressId',
          'venue.type',
          'venue.deletedAt',
        ]);
    }

    // Conditionally join and select address relation (requires venue)
    if (relations.address && relations.venue && relations.block && relations.floor && relations.room) {
      queryBuilder = queryBuilder
        .leftJoin('venue.address', 'address')
        .addSelect([
          'address.id',
          'address.addr1',
          'address.addr2',
          'address.city',
          'address.state',
          'address.pincode',
          'address.country',
        ]);
    }

    // Conditionally join and select allocations relation
    if (relations.allocations) {
      queryBuilder = queryBuilder
        .leftJoin('inventory.roomAllocations', 'allocation')
        .addSelect([
          'allocation.id',
          'allocation.remarks',
          'allocation.bedPosition',
        ]);
    }

    // Conditionally join and select registration relation (requires allocations)
    if (relations.registration && relations.allocations) {
      queryBuilder = queryBuilder
        .leftJoin('allocation.registration', 'registration', 
          'registration.id = allocation.registration_id AND EXISTS (SELECT 1 FROM hdb_registration_approval regApproval WHERE regApproval.registration_id = registration.id AND regApproval.approval_status = :approvedStatus)'
        )
        .setParameter('approvedStatus', ApprovalStatusEnum.APPROVED)
        .addSelect([
          'registration.id',
          'registration.registrationSeqNumber',
          'registration.fullName',
          'registration.gender',
          'registration.mobileNumber',
          'registration.emailAddress',
          'registration.registrationStatus',
          'registration.registrationDate',
          'registration.preferredRoomMate',
          'registration.city',
          'registration.otherCityName',
          'registration.countryName',
          'registration.dob',
          'registration.profileUrl',
          'registration.noOfHDBs',
        ]);
    }

    // Conditionally join and select rmUser relation (requires registration)
    if (relations.rmUser && relations.registration && relations.allocations) {
      queryBuilder = queryBuilder
        .leftJoin('registration.rmContactUser', 'rmUser')
        .addSelect(['rmUser.fullName']);
    }

    // Conditionally join and select travelPlan relation (requires registration)
    if (relations.travelPlan && relations.registration && relations.allocations) {
      queryBuilder = queryBuilder
        .leftJoin('registration.travelPlans', 'travelPlan')
        .addSelect([
          'travelPlan.departureDatetime',
          'travelPlan.returnTerminal',
        ]);
    }

    // Conditionally join and select regPair relation (requires registration)
    if (relations.regPair && relations.registration && relations.allocations) {
      queryBuilder = queryBuilder
        .leftJoin('registration.registrationPairMaps', 'regPairMap')
        .addSelect(['regPairMap.id'])
        .leftJoin('regPairMap.registrationPair', 'regPair')
        .addSelect([
          'regPair.id',
          'regPair.pairCode',
          'regPair.seqNumber',
        ]);
    }

    // Conditionally join and select allocatedProgram relation (requires registration)
    if (relations.allocatedProgram && relations.registration && relations.allocations) {
      queryBuilder = queryBuilder
        .leftJoin('registration.allocatedProgram', 'allocatedProgram')
        .addSelect(['allocatedProgram.code']);
    }

    // Apply base WHERE conditions - only check deletedAt for joined tables
    queryBuilder = queryBuilder.where('inventory.deletedAt IS NULL');
    
    if (relations.room) {
      queryBuilder = queryBuilder
        .andWhere('room.deletedAt IS NULL')
        .andWhere('room.roomStatus = :activeStatus', { activeStatus: 'active' });
    }
    
    if (relations.floor && relations.room) {
      queryBuilder = queryBuilder.andWhere('floor.deletedAt IS NULL');
    }
    
    if (relations.block && relations.floor && relations.room) {
      queryBuilder = queryBuilder.andWhere('block.deletedAt IS NULL');
    }
    
    if (relations.venue && relations.block && relations.floor && relations.room) {
      queryBuilder = queryBuilder.andWhere('venue.deletedAt IS NULL');
    }
  
    // Apply basic filters
    if (programId) {
      queryBuilder.andWhere('inventory.programId = :programId', { programId });
    }
  
    if (subProgramId) {
      queryBuilder.andWhere('inventory.subProgramId = :subProgramId', { subProgramId });
    }
  
    // Handle search text - include complete room data when person name matches
    if (searchText) {
      const searchPattern = `%${searchText}%`;
      queryBuilder.andWhere(
        new Brackets(qb => {
          qb.where('room.roomNumber ILIKE :searchText', { searchText: searchPattern })
            .orWhere('floor.label ILIKE :searchText', { searchText: searchPattern })
            .orWhere(
              subQuery => {
                return `inventory.id IN (${subQuery
                  .subQuery()
                  .select('DISTINCT inv.id')
                  .from('program_room_inventory_map', 'inv')
                  .innerJoin('room_allocation', 'alloc', 'alloc.program_room_inventory_map_id = inv.id')
                  .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                  .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                  .where('reg.full_name ILIKE :searchText')
                  .andWhere('inv.deleted_at IS NULL')
                  .andWhere('alloc.deleted_at IS NULL')
                  .andWhere('reg.deleted_at IS NULL')
                  .andWhere('regApp.approval_status = :approvedStatus')
                  .getQuery()})`;
              },
              { searchText: searchPattern, approvedStatus: ApprovalStatusEnum.APPROVED }
            );
        })
      );
    }
  
    // Apply advanced filter options
    if (filter) {
      // Filter by departure terminal
      if (
        filter.returnTravelTerminal &&
        typeof filter.returnTravelTerminal === 'string' &&
        filter.returnTravelTerminal.trim() !== ''
      ) {
        queryBuilder.andWhere(
          new Brackets(qb => {
            qb.where('travelPlan.returnTerminal = :returnTerminal', {
              returnTerminal: filter.returnTravelTerminal,
            })
              .orWhere(
                subQuery => {
                  return `inventory.id IN (${subQuery
                    .subQuery()
                    .select('DISTINCT alloc.program_room_inventory_map_id')
                    .from('room_allocation', 'alloc')
                    .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                    .innerJoin('hdb_registration_travel_plan', 'travel', 'travel.registration_id = reg.id')
                    .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                    .where('travel.return_terminal = :returnTerminal')
                    .andWhere('alloc.deleted_at IS NULL')
                    .andWhere('reg.deleted_at IS NULL')
                    .andWhere('regApp.approval_status = :approvedStatus')
                    .getQuery()})`;
                },
                { returnTerminal: filter.returnTravelTerminal, approvedStatus: ApprovalStatusEnum.APPROVED }  
              );
          })
        );
      }
  
      // Filter by room status from advanced filter
      if (filter.roomStatus) {
        if (
          Array.isArray(filter.roomStatus) &&
          filter.roomStatus.length > 0 &&
          !filter.roomStatus.includes('ALL')
        ) {
          queryBuilder.andWhere('inventory.roomStatus IN (:...roomStatuses)', {
            roomStatuses: filter.roomStatus,
          });
        } else if (
          typeof filter.roomStatus === 'string' &&
          filter.roomStatus !== 'ALL' &&
          filter.roomStatus.trim() !== ''
        ) {
          queryBuilder.andWhere('inventory.roomStatus = :roomStatus', {
            roomStatus: filter.roomStatus,
          });
        }
      }
  
      // Filter by reservation status
      if (filter.isReserved !== undefined) {
        queryBuilder.andWhere('inventory.isReserved = :isReserved', {
          isReserved: filter.isReserved,
        });
      }
  
      // Filter by venues (array of venue IDs)
      if (filter.venues && Array.isArray(filter.venues) && filter.venues.length > 0) {
        const venueIds = filter.venues
          .map((id) => (typeof id === 'string' ? parseInt(id, 10) : id))
          .filter((id) => !isNaN(id));
        const uniqueVenueIds = [...new Set(venueIds)];
  
        if (uniqueVenueIds.length > 0) {
          queryBuilder.andWhere('venue.id IN (:...venueIds)', { venueIds: uniqueVenueIds });
        }
      }
  
      // Filter by blocks (array of block IDs)
      if (filter.block && Array.isArray(filter.block) && filter.block.length > 0) {
        const blockIds = filter.block
          .map((id) => (typeof id === 'string' ? parseInt(id, 10) : id))
          .filter((id) => !isNaN(id));
        const uniqueBlockIds = [...new Set(blockIds)];
  
        if (uniqueBlockIds.length > 0) {
          queryBuilder.andWhere('block.id IN (:...blockIds)', { blockIds: uniqueBlockIds });
        }
      }
  
      // Filter by floors (array of floor IDs)
      if (filter.floor && Array.isArray(filter.floor) && filter.floor.length > 0) {
        const floorIds = filter.floor
          .map((id) => (typeof id === 'string' ? parseInt(id, 10) : id))
          .filter((id) => !isNaN(id));
        const uniqueFloorIds = [...new Set(floorIds)];
  
        if (uniqueFloorIds.length > 0) {
          queryBuilder.andWhere('floor.id IN (:...floorIds)', { floorIds: uniqueFloorIds });
        }
      }
  
      // Filter by occupancy (exact match or range)
      if (filter.occupancy && Array.isArray(filter.occupancy)) {
        queryBuilder.andWhere(
          new Brackets(qb => {
            filter.occupancy.forEach((occupancyFilter, index) => {
              if (occupancyFilter.min !== undefined && occupancyFilter.max !== undefined) {
                qb.orWhere(
                  `room.occupancy BETWEEN :minOccupancy${index} AND :maxOccupancy${index}`,
                  {
                    [`minOccupancy${index}`]: occupancyFilter.min,
                    [`maxOccupancy${index}`]: occupancyFilter.max,
                  }
                );
              } else if (occupancyFilter.min !== undefined) {
                qb.orWhere(`room.occupancy >= :minOccupancy${index}`, {
                  [`minOccupancy${index}`]: occupancyFilter.min,
                });
              } else if (occupancyFilter.max !== undefined) {
                qb.orWhere(`room.occupancy <= :maxOccupancy${index}`, {
                  [`maxOccupancy${index}`]: occupancyFilter.max,
                });
              } else if (occupancyFilter.minExcluded !== undefined) {
                qb.orWhere(`room.occupancy > :minExcludedOccupancy${index}`, {
                  [`minExcludedOccupancy${index}`]: occupancyFilter.minExcluded,
                });
              } else if (occupancyFilter.maxExcluded !== undefined) {
                qb.orWhere(`room.occupancy < :maxExcludedOccupancy${index}`, {
                  [`maxExcludedOccupancy${index}`]: occupancyFilter.maxExcluded,
                });
              } else if (occupancyFilter.eq !== undefined) {
                qb.orWhere(`room.occupancy = :eqOccupancy${index}`, {
                  [`eqOccupancy${index}`]: occupancyFilter.eq,
                });
              }
            });
          })
        );
      }
  
      // Filter by room type (array of room types)
      if (filter.roomType && Array.isArray(filter.roomType) && filter.roomType.length > 0) {
        queryBuilder.andWhere('LOWER(room.roomType) IN (:...roomTypes)', {
          roomTypes: filter.roomType.map((type) => type.toLowerCase()),
        });
      } else if (filter.roomType && typeof filter.roomType === 'string') {
        queryBuilder.andWhere('LOWER(room.roomType) = LOWER(:roomType)', {
          roomType: filter.roomType,
        });
      }
  
      // Filter by bed type (array of bed types)
      if (filter.bedType && Array.isArray(filter.bedType) && filter.bedType.length > 0) {
        queryBuilder.andWhere('LOWER(room.bedType) IN (:...bedTypes)', {
          bedTypes: filter.bedType.map((type) => type.toLowerCase()),
        });
      } else if (filter.bedType && typeof filter.bedType === 'string') {
        queryBuilder.andWhere('LOWER(room.bedType) = LOWER(:bedType)', {
          bedType: filter.bedType,
        });
      }
  
      // Filter by gender (array of gender values)
      if (filter.gender && Array.isArray(filter.gender) && filter.gender.length > 0) {
        const genderValues = filter.gender
          .map((g) => {
            if (g.toLowerCase().trim() === 'male') {
              return GenderEnum.MALE;
            } else if (g.toLowerCase().trim() === 'female') {
              return GenderEnum.FEMALE;
            }
            return null;
          })
          .filter((g) => g !== null);

        if (genderValues.length > 0) {
          queryBuilder.andWhere(
            subQuery => {
              return `inventory.id IN (${subQuery
                .subQuery()
                .select('DISTINCT alloc.program_room_inventory_map_id')
                .from('room_allocation', 'alloc')
                .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                .where('reg.gender IN (:...genderValues)')
                .andWhere('alloc.deleted_at IS NULL')
                .andWhere('reg.deleted_at IS NULL')
                .andWhere('regApp.approval_status = :approvedStatus')
                .getQuery()})`;
            },
            { genderValues, approvedStatus: ApprovalStatusEnum.APPROVED }
          );
        }
      }
  
      // Filter by age range (array of age range values)
      if (filter.age && Array.isArray(filter.age) && filter.age.length > 0) {
        queryBuilder.andWhere(
          new Brackets(qb => {
            filter.age.forEach((ageFilter, index) => {
              if (ageFilter?.min && ageFilter?.max) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .where(`DATE_PART('year', AGE(reg.dob)) BETWEEN :minAge${index} AND :maxAge${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`minAge${index}`]: ageFilter.min,
                    [`maxAge${index}`]: ageFilter.max,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (ageFilter?.min) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .where(`DATE_PART('year', AGE(reg.dob)) >= :minAge${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`minAge${index}`]: ageFilter.min,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (ageFilter?.max) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .where(`DATE_PART('year', AGE(reg.dob)) <= :maxAge${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`maxAge${index}`]: ageFilter.max,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (ageFilter?.minExcluded !== undefined) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .where(`DATE_PART('year', AGE(reg.dob)) > :minExcludedAge${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`minExcludedAge${index}`]: ageFilter.minExcluded,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (ageFilter?.maxExcluded !== undefined) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .where(`DATE_PART('year', AGE(reg.dob)) < :maxExcludedAge${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`maxExcludedAge${index}`]: ageFilter.maxExcluded,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (ageFilter?.eq !== undefined) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .where(`DATE_PART('year', AGE(reg.dob)) = :eqAge${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`eqAge${index}`]: ageFilter.eq,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              }
            });
          })
        );
      }
  
      // Filter by preferred roommate (array of values: 'yes' or 'no')
      if (
        filter.preferredRoommate &&
        Array.isArray(filter.preferredRoommate) &&
        filter.preferredRoommate.length > 0
      ) {
        const hasYes = filter.preferredRoommate.some((val) => val.toLowerCase() === 'yes');
        const hasNo = filter.preferredRoommate.some((val) => val.toLowerCase() === 'no');

        if (hasYes && hasNo) {
          // Both selected - no filter needed (show all)
        } else if (hasYes) {
          // Only 'yes' selected - show rooms where registrations have preferred roommate
          queryBuilder.andWhere(
            subQuery => {
              return `inventory.id IN (${subQuery
                .subQuery()
                .select('DISTINCT alloc.program_room_inventory_map_id')
                .from('room_allocation', 'alloc')
                .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                .where('reg.preferred_room_mate IS NOT NULL')
                .andWhere('reg.preferred_room_mate != :empty')
                .andWhere('alloc.deleted_at IS NULL')
                .andWhere('reg.deleted_at IS NULL')
                .andWhere('regApp.approval_status = :approvedStatus')
                .getQuery()})`;
            },
            { empty: '', approvedStatus: ApprovalStatusEnum.APPROVED }
          );
        } else if (hasNo) {
          // Only 'no' selected - show rooms where registrations have no preferred roommate
          queryBuilder.andWhere(
            subQuery => {
              return `inventory.id IN (${subQuery
                .subQuery()
                .select('DISTINCT alloc.program_room_inventory_map_id')
                .from('room_allocation', 'alloc')
                .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                .where('(reg.preferred_room_mate IS NULL OR reg.preferred_room_mate = :empty)')
                .andWhere('alloc.deleted_at IS NULL')
                .andWhere('reg.deleted_at IS NULL')
                .andWhere('regApp.approval_status = :approvedStatus')
                .getQuery()})`;
            },
            { empty: '', approvedStatus: ApprovalStatusEnum.APPROVED }
          );
        }
      }
  
      // Filter by RM (Relationship Manager) contact (array of RM user IDs)
      if (filter.rmContact && Array.isArray(filter.rmContact) && filter.rmContact.length > 0) {
        const rmContactIds = filter.rmContact
          .map((id) => (typeof id === 'string' ? parseInt(id, 10) : id))
          .filter((id) => !isNaN(id));
  
        const uniqueRmContactIds = [...new Set(rmContactIds)];
  
        if (uniqueRmContactIds.length > 0) {
          queryBuilder.andWhere(
            subQuery => {
              return `inventory.id IN (${subQuery
                .subQuery()
                .select('DISTINCT alloc.program_room_inventory_map_id')
                .from('room_allocation', 'alloc')
                .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                .where('reg.rm_contact IN (:...rmContactIds)')
                .andWhere('alloc.deleted_at IS NULL')
                .andWhere('reg.deleted_at IS NULL')
                .andWhere('regApp.approval_status = :approvedStatus')
                .getQuery()})`;
            },
            { rmContactIds: uniqueRmContactIds, approvedStatus: ApprovalStatusEnum.APPROVED }
          );
        }
      }
  
      // Filter by city (array of city values)
      if (filter.city && Array.isArray(filter.city) && filter.city.length > 0) {
        const cityValues = filter.city
          .filter((city) => city && typeof city === 'string' && city.trim() !== '')
          .map((city) => city.trim());
  
        if (cityValues.length > 0) {
          queryBuilder.andWhere(
            subQuery => {
              return `inventory.id IN (${subQuery
                .subQuery()
                .select('DISTINCT alloc.program_room_inventory_map_id')
                .from('room_allocation', 'alloc')
                .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                .where('(LOWER(reg.city) IN (:...cityValues) OR LOWER(reg.other_city_name) IN (:...cityValues))')
                .andWhere('alloc.deleted_at IS NULL')
                .andWhere('reg.deleted_at IS NULL')
                .andWhere('regApp.approval_status = :approvedStatus')
                .getQuery()})`;
            },
            { cityValues: cityValues.map((city) => city.toLowerCase()), approvedStatus: ApprovalStatusEnum.APPROVED }
          );
        }
      }
  
      // Legacy filter support - single IDs
      if (filter.floorId) {
        queryBuilder.andWhere('floor.id = :floorId', { floorId: filter.floorId });
      }
  
      if (filter.blockId) {
        queryBuilder.andWhere('block.id = :blockId', { blockId: filter.blockId });
      }
  
      if (filter.venueId) {
        queryBuilder.andWhere('venue.id = :venueId', { venueId: filter.venueId });
      }
  
      // Filter by reserved for
      if (filter.reservedFor) {
        queryBuilder.andWhere('LOWER(inventory.reservedFor) LIKE LOWER(:reservedFor)', {
          reservedFor: `%${filter.reservedFor}%`,
        });
      }
  
      // Filter by number of HDBs (noOfHdbs)
      if (filter.noOfHdbs && Array.isArray(filter.noOfHdbs)) {
        queryBuilder.andWhere(
          new Brackets(qb => {
            filter.noOfHdbs.forEach((hdbFilter, index) => {
              if (hdbFilter?.min && hdbFilter?.max) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'registration', 'registration.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = registration.id')
                      .where(`registration.no_of_hdbs BETWEEN :minHdb${index} AND :maxHdb${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('registration.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`minHdb${index}`]: hdbFilter.min,
                    [`maxHdb${index}`]: hdbFilter.max,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (hdbFilter?.min) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'registration', 'registration.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = registration.id')
                      .where(`registration.no_of_hdbs >= :minHdb${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('registration.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`minHdb${index}`]: hdbFilter.min,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (hdbFilter?.max) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'registration', 'registration.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = registration.id')
                      .where(`registration.no_of_hdbs <= :maxHdb${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('registration.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`maxHdb${index}`]: hdbFilter.max,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (hdbFilter?.minExcluded !== undefined) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'registration', 'registration.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = registration.id')
                      .where(`registration.no_of_hdbs > :minExcludedHdb${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('registration.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`minExcludedHdb${index}`]: hdbFilter.minExcluded,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (hdbFilter?.maxExcluded !== undefined) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'registration', 'registration.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = registration.id')
                      .where(`registration.no_of_hdbs < :maxExcludedHdb${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('registration.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`maxExcludedHdb${index}`]: hdbFilter.maxExcluded,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (hdbFilter?.eq !== undefined) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'registration', 'registration.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = registration.id')
                      .where(`registration.no_of_hdbs = :eqHdb${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('registration.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`eqHdb${index}`]: hdbFilter.eq,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              }
            });
          })
        );
      }
  
      // Filter by pairedStatus (Yes/No)
      if (filter.pairedStatus !== undefined) {
        if (filter.pairedStatus?.toLowerCase() === 'yes') {
          queryBuilder.andWhere(
            subQuery => {
              return `inventory.id IN (${subQuery
                .subQuery()
                .select('DISTINCT alloc.program_room_inventory_map_id')
                .from('room_allocation', 'alloc')
                .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                .innerJoin('registration_pair_map', 'pairMap', 'pairMap.registration_id = reg.id AND pairMap.deleted_at IS NULL')
                .innerJoin('registration_pair', 'pair', 'pair.id = pairMap.registration_pair_id AND pair.deleted_at IS NULL')
                .where('alloc.deleted_at IS NULL AND reg.deleted_at IS NULL AND pair.id IS NOT NULL')
                .andWhere('regApp.approval_status = :approvedStatus')
                .getQuery()})`;
            },
            { approvedStatus: ApprovalStatusEnum.APPROVED }
          );
        } else if (filter.pairedStatus?.toLowerCase() === 'no') {
          queryBuilder.andWhere(
            subQuery => {
              return `inventory.id IN (${subQuery
                .subQuery()
                .select('DISTINCT alloc.program_room_inventory_map_id')
                .from('room_allocation', 'alloc')
                .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                .leftJoin('registration_pair_map', 'pairMap', 'pairMap.registration_id = reg.id AND pairMap.deleted_at IS NULL')
                .leftJoin('registration_pair', 'pair', 'pair.id = pairMap.registration_pair_id AND pair.deleted_at IS NULL')
                .where('alloc.deleted_at IS NULL AND reg.deleted_at IS NULL AND pair.id IS NULL')
                .andWhere('regApp.approval_status = :approvedStatus')
                .getQuery()})`;
            },
            { approvedStatus: ApprovalStatusEnum.APPROVED }
          );
        }
      }
  
      // Filter by departureDatetime (array of ranges)
      if (filter.departureDatetime && Array.isArray(filter.departureDatetime)) {
        queryBuilder.andWhere(
          new Brackets(qb => {
            filter.departureDatetime.forEach((dateFilter, index) => {
              if (dateFilter?.min && dateFilter?.max) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .innerJoin('hdb_registration_travel_plan', 'travel', 'travel.registration_id = reg.id')
                      .where(`travel.departure_datetime BETWEEN :minDepartureDate${index} AND :maxDepartureDate${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`minDepartureDate${index}`]: dateFilter.min,
                    [`maxDepartureDate${index}`]: dateFilter.max,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (dateFilter?.min) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .innerJoin('hdb_registration_travel_plan', 'travel', 'travel.registration_id = reg.id')
                      .where(`travel.departure_datetime >= :minDepartureDate${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`minDepartureDate${index}`]: dateFilter.min,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (dateFilter?.max) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .innerJoin('hdb_registration_travel_plan', 'travel', 'travel.registration_id = reg.id')
                      .where(`travel.departure_datetime <= :maxDepartureDate${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`maxDepartureDate${index}`]: dateFilter.max,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (dateFilter?.minExcluded !== undefined) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .innerJoin('hdb_registration_travel_plan', 'travel', 'travel.registration_id = reg.id')
                      .where(`travel.departure_datetime > :minExcludedDepartureDate${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`minExcludedDepartureDate${index}`]: dateFilter.minExcluded,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (dateFilter?.maxExcluded !== undefined) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .innerJoin('hdb_registration_travel_plan', 'travel', 'travel.registration_id = reg.id')
                      .where(`travel.departure_datetime < :maxExcludedDepartureDate${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`maxExcludedDepartureDate${index}`]: dateFilter.maxExcluded,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              } else if (dateFilter?.eq !== undefined) {
                qb.orWhere(
                  subQuery => {
                    return `inventory.id IN (${subQuery
                      .subQuery()
                      .select('DISTINCT alloc.program_room_inventory_map_id')
                      .from('room_allocation', 'alloc')
                      .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                      .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                      .innerJoin('hdb_registration_travel_plan', 'travel', 'travel.registration_id = reg.id')
                      .where(`travel.departure_datetime = :eqDepartureDate${index}`)
                      .andWhere('alloc.deleted_at IS NULL')
                      .andWhere('reg.deleted_at IS NULL')
                      .andWhere('regApp.approval_status = :approvedStatus')
                      .getQuery()})`;
                  },
                  {
                    [`eqDepartureDate${index}`]: dateFilter.eq,
                    approvedStatus: ApprovalStatusEnum.APPROVED
                  }
                );
              }
            });
          })
        );
      }
  
      // Add filter for travel type (domestic/international)
      if (filter.travelType) {
        const travelTypes = Array.isArray(filter.travelType) ? filter.travelType : [filter.travelType];
        const travelTypeConditions: string[] = [];
  
        travelTypes.forEach((type: string) => {
          if (type.toLowerCase() === 'domestic') {
            travelTypeConditions.push('travelPlan.isDomestic = true');
          } else if (type.toLowerCase() === 'international') {
            travelTypeConditions.push('travelPlan.isDomestic = false');
          }
        });
  
        if (travelTypeConditions.length > 0) {
          const travelTypeFilterCondition = travelTypeConditions.join(' OR ');
          queryBuilder.andWhere(
            subQuery => {
              return `inventory.id IN (${subQuery
                .subQuery()
                .select('DISTINCT alloc.program_room_inventory_map_id')
                .from('room_allocation', 'alloc')
                .innerJoin('hdb_program_registration', 'reg', 'reg.id = alloc.registration_id')
                .innerJoin('hdb_registration_approval', 'regApp', 'regApp.registration_id = reg.id')
                .innerJoin('hdb_registration_travel_plan', 'travelPlan', 'travelPlan.registration_id = reg.id AND travelPlan.deleted_at IS NULL')
                .where(`(${travelTypeFilterCondition})`)
                .andWhere('alloc.deleted_at IS NULL')
                .andWhere('reg.deleted_at IS NULL')
                .andWhere('regApp.approval_status = :approvedStatus')
                .getQuery()})`;
            },
            { approvedStatus: ApprovalStatusEnum.APPROVED }
          );
        }
      }
    }
  
    return queryBuilder;
  }

  /**
   * Parse age range string to min and max age values
   * Handles formats like "18-25", "26-35", "56+"
   * @param range - Age range string (e.g., "18-25", "56+")
   * @returns [minAge, maxAge] tuple. maxAge is null for open-ended ranges like "56+"
   */
  private parseAgeRange(range: string): [number, number | null] {
    if (range.includes('+')) {
      // Handle "56+" format
      const minAge = parseInt(range.replace('+', ''));
      return [minAge, null];
    } else if (range.includes('-')) {
      // Handle "18-25" format
      const [min, max] = range.split('-').map(s => parseInt(s.trim()));
      return [min, max];
    }
    
    // Fallback - treat as single age
    const age = parseInt(range);
    return [age, age];
  }

  /**
   * Retrieves room inventory data with pagination
   * @param limit - Number of records per page or 'all' to get all records
   * @param offset - Offset for pagination or 'all' to skip pagination
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param searchText - Optional search text for room number or label
   * @param filter - Optional advanced filter object with additional criteria
   * @param selectFields - Optional array of field paths to select
   * @param includeRelations - Optional flags to control which relations to join and load
   * @returns Promise with room inventory data and total count
   */
  async findRoomInventoryWithPagination(
    limit: number | 'all',
    offset: number | 'all',
    programId?: number,
    subProgramId?: number,
    searchText?: string,
    filter?: any,
    selectFields?: string[],
    includeRelations?: {
      room?: boolean;
      floor?: boolean;
      block?: boolean;
      venue?: boolean;
      address?: boolean;
      allocations?: boolean;
      registration?: boolean;
      rmUser?: boolean;
      travelPlan?: boolean;
      regPair?: boolean;
      allocatedProgram?: boolean;
    }
  ): Promise<{ data: ProgramRoomInventoryMap[]; totalRecords: number }> {
    this.logger.debug('Fetching room inventory data with pagination', {
      limit,
      offset,
      programId,
      subProgramId,
      searchText,
      filter,
      selectFields,
      includeRelations
    });
    console.log('Filter in findRoomInventoryWithPagination:', filter);
    try {
      const queryBuilder = this.buildBaseQueryBuilder(
        programId,
        subProgramId,
        searchText,
        filter,
        selectFields,
        includeRelations
      );

      // Get total count for pagination
      const totalRecords = await queryBuilder.getCount();

      // Apply pagination and get results
      let queryWithOrder = queryBuilder.orderBy('room.roomNumber', 'ASC');
      
      // Handle pagination - skip if limit or offset is 'all'
      if (limit !== 'all' && offset !== 'all') {
        queryWithOrder = queryWithOrder.skip(offset).take(limit);
      }
      
      const data = await queryWithOrder.getMany();

      this.logger.debug('Room inventory data fetched successfully', {
        totalRecords,
        currentPageRecords: data.length
      });

      return { data, totalRecords };

    } catch (error) {
      this.logger.error('Failed to fetch room inventory data', error.stack);
      throw error;
    }
  }

  /**
   * Calculates basic metrics for room inventory KPIs
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param searchText - Optional search text filter
   * @returns Promise with basic metrics
   */
  async calculateBasicMetrics(
    programId?: number,
    subProgramId?: number,
    searchText?: string
  ): Promise<{
    totalRooms: number;
    totalCapacity: number;
    totalRemainingOccupancy: number;
  }> {
    this.logger.debug('Calculating basic metrics for room inventory');

    try {
      const baseQuery = this.repository
        .createQueryBuilder('inventory')
        .leftJoin('inventory.room', 'room')
        .where('inventory.deletedAt IS NULL')
        .andWhere('room.deletedAt IS NULL');

      // Apply same filters as main query
      if (programId) {
        baseQuery.andWhere('inventory.programId = :programId', { programId });
      }

      if (subProgramId) {
        baseQuery.andWhere('inventory.subProgramId = :subProgramId', { subProgramId });
      }

      if (searchText) {
        baseQuery.andWhere(
          '(LOWER(room.roomNumber) LIKE LOWER(:searchText) OR LOWER(room.label) LIKE LOWER(:searchText))',
          { searchText: `%${searchText}%` }
        );
      }

      const result = await baseQuery
        .select([
          'COUNT(inventory.id) as totalRooms',
          'SUM(room.occupancy) as totalCapacity',
          'SUM(inventory.remainingOccupancy) as totalRemainingOccupancy'
        ])
        .getRawOne();

      return {
        totalRooms: parseInt(result.totalRooms) || 0,
        totalCapacity: parseInt(result.totalCapacity) || 0,
        totalRemainingOccupancy: parseInt(result.totalRemainingOccupancy) || 0
      };

    } catch (error) {
      this.logger.error('Failed to calculate basic metrics', error.stack);
      throw error;
    }
  }

  /**
   * Calculates status breakdown for room inventory KPIs
   * Returns raw status counts from database
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @returns Promise with raw status breakdown data from database
   */
  async calculateStatusBreakdown(
    programId?: number,
    subProgramId?: number,
  ): Promise<any[]> {
    this.logger.debug('Calculating status breakdown for room inventory');

    try {
      const baseQuery = this.repository
        .createQueryBuilder('inventory')
        .leftJoin('inventory.room', 'room')
        .where('inventory.deletedAt IS NULL')
        .andWhere('room.deletedAt IS NULL');

      // Apply only program and subProgram filters for KPIs
      if (programId) {
        baseQuery.andWhere('inventory.programId = :programId', { programId });
      }

      if (subProgramId) {
        baseQuery.andWhere('inventory.subProgramId = :subProgramId', { subProgramId });
      }

      const result = await baseQuery
        .select([
          'inventory.roomStatus as status',
          'COUNT(inventory.id) as count'
        ])
        .groupBy('inventory.roomStatus')
        .getRawMany();

      return result;

    } catch (error) {
      this.logger.error('Failed to calculate status breakdown', error.stack);
      throw error;
    }
  }

  /**
   * Get distinct venues for filter options
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param venueIds - Optional array of venue IDs to filter
   * @param blockIds - Optional array of block IDs to filter
   * @param floorIds - Optional array of floor IDs to filter
   * @returns Promise<any[]> Array of distinct venues with label and array of venue IDs as value
   */
  async getDistinctVenues(
    programId?: number, 
    subProgramId?: number,
    venueIds?: number[],
    blockIds?: number[],
    floorIds?: number[]
  ): Promise<any[]> {
    this.logger.debug('Fetching distinct venues for filter options', {
      programId,
      subProgramId,
      venueIds,
      blockIds,
      floorIds
    });

    try {
      const queryBuilder = this.repository
        .createQueryBuilder('inventory')
        .leftJoin('inventory.room', 'room')
        .leftJoin('room.floor', 'floor')
        .leftJoin('floor.block', 'block')
        .leftJoin('block.venue', 'venue')
        .select([
          'venue.label as label',
          'ARRAY_AGG(DISTINCT venue.id ORDER BY venue.id ASC) as ids'
        ])
        .where('inventory.deletedAt IS NULL')
        .andWhere('room.deletedAt IS NULL')
        .andWhere('floor.deletedAt IS NULL')
        .andWhere('block.deletedAt IS NULL')
        .andWhere('venue.deletedAt IS NULL')
        .andWhere('venue.id IS NOT NULL')
        .groupBy('venue.label');

      if (programId) {
        queryBuilder.andWhere('inventory.programId = :programId', { programId });
      }

      if (subProgramId) {
        queryBuilder.andWhere('inventory.subProgramId = :subProgramId', { subProgramId });
      }

      // Apply additional filters based on selected venues, blocks, floors
      if (venueIds && venueIds.length > 0) {
        queryBuilder.andWhere('venue.id IN (:...venueIds)', { venueIds });
      }

      if (blockIds && blockIds.length > 0) {
        queryBuilder.andWhere('block.id IN (:...blockIds)', { blockIds });
      }

      if (floorIds && floorIds.length > 0) {
        queryBuilder.andWhere('floor.id IN (:...floorIds)', { floorIds });
      }

      const rawResult = await queryBuilder
        .orderBy('venue.label', 'ASC')
        .getRawMany();

      // Convert PostgreSQL array to JavaScript array and use ids as value
      const result = rawResult.map(row => ({
        value: Array.isArray(row.ids) ? row.ids : [],
        label: row.label
      }));

      this.logger.debug('Distinct venues fetched successfully', { count: result.length });
      return result;

    } catch (error) {
      this.logger.error('Failed to fetch distinct venues', error.stack);
      return [];
    }
  }

  /**
   * Get distinct blocks for filter options
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param venueIds - Optional array of venue IDs to filter
   * @param blockIds - Optional array of block IDs to filter
   * @param floorIds - Optional array of floor IDs to filter
   * @returns Promise<any[]> Array of distinct blocks with label and array of block IDs as value
   */
  async getDistinctBlocks(
    programId?: number, 
    subProgramId?: number,
    venueIds?: number[],
    blockIds?: number[],
    floorIds?: number[]
  ): Promise<any[]> {
    this.logger.debug('Fetching distinct blocks for filter options', {
      programId,
      subProgramId,
      venueIds,
      blockIds,
      floorIds
    });

    try {
      const queryBuilder = this.repository
        .createQueryBuilder('inventory')
        .leftJoin('inventory.room', 'room')
        .leftJoin('room.floor', 'floor')
        .leftJoin('floor.block', 'block')
        .leftJoin('block.venue', 'venue')
        .select([
          'block.label as label',
          'ARRAY_AGG(DISTINCT block.id ORDER BY block.id ASC) as ids'
        ])
        .where('inventory.deletedAt IS NULL')
        .andWhere('room.deletedAt IS NULL')
        .andWhere('floor.deletedAt IS NULL')
        .andWhere('block.deletedAt IS NULL')
        .andWhere('block.id IS NOT NULL')
        .groupBy('block.label');

      if (programId) {
        queryBuilder.andWhere('inventory.programId = :programId', { programId });
      }

      if (subProgramId) {
        queryBuilder.andWhere('inventory.subProgramId = :subProgramId', { subProgramId });
      }

      // Apply additional filters based on selected venues, blocks, floors
      if (venueIds && venueIds.length > 0) {
        queryBuilder.andWhere('venue.id IN (:...venueIds)', { venueIds });
      }

      if (blockIds && blockIds.length > 0) {
        queryBuilder.andWhere('block.id IN (:...blockIds)', { blockIds });
      }

      if (floorIds && floorIds.length > 0) {
        queryBuilder.andWhere('floor.id IN (:...floorIds)', { floorIds });
      }

      const rawResult = await queryBuilder
        .orderBy('block.label', 'ASC')
        .getRawMany();

      // Convert PostgreSQL array to JavaScript array and use ids as value
      const result = rawResult.map(row => ({
        value: Array.isArray(row.ids) ? row.ids : [],
        label: row.label
      }));

      this.logger.debug('Distinct blocks fetched successfully', { count: result.length });
      return result;

    } catch (error) {
      this.logger.error('Failed to fetch distinct blocks', error.stack);
      return [];
    }
  }

  /**
   * Get distinct floors for filter options
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param venueIds - Optional array of venue IDs to filter
   * @param blockIds - Optional array of block IDs to filter
   * @param floorIds - Optional array of floor IDs to filter
   * @returns Promise<any[]> Array of distinct floors with label and array of floor IDs as value
   */
  async getDistinctFloors(
    programId?: number, 
    subProgramId?: number,
    venueIds?: number[],
    blockIds?: number[],
    floorIds?: number[]
  ): Promise<any[]> {
    this.logger.debug('Fetching distinct floors for filter options', {
      programId,
      subProgramId,
      venueIds,
      blockIds,
      floorIds
    });

    try {
      const queryBuilder = this.repository
        .createQueryBuilder('inventory')
        .leftJoin('inventory.room', 'room')
        .leftJoin('room.floor', 'floor')
        .leftJoin('floor.block', 'block')
        .leftJoin('block.venue', 'venue')
        .select([
          'floor.label as label',
          'ARRAY_AGG(DISTINCT floor.id ORDER BY floor.id ASC) as ids'
        ])
        .where('inventory.deletedAt IS NULL')
        .andWhere('room.deletedAt IS NULL')
        .andWhere('floor.deletedAt IS NULL')
        .andWhere('floor.id IS NOT NULL')
        .groupBy('floor.label');

      if (programId) {
        queryBuilder.andWhere('inventory.programId = :programId', { programId });
      }

      if (subProgramId) {
        queryBuilder.andWhere('inventory.subProgramId = :subProgramId', { subProgramId });
      }

      // Apply additional filters based on selected venues, blocks, floors
      if (venueIds && venueIds.length > 0) {
        queryBuilder.andWhere('venue.id IN (:...venueIds)', { venueIds });
      }

      if (blockIds && blockIds.length > 0) {
        queryBuilder.andWhere('block.id IN (:...blockIds)', { blockIds });
      }

      if (floorIds && floorIds.length > 0) {
        queryBuilder.andWhere('floor.id IN (:...floorIds)', { floorIds });
      }

      const rawResult = await queryBuilder
        .orderBy('floor.label', 'ASC')
        .getRawMany();

      // Convert PostgreSQL array to JavaScript array and use ids as value
      const result = rawResult.map(row => ({
        value: Array.isArray(row.ids) ? row.ids : [],
        label: row.label
      }));

      this.logger.debug('Distinct floors fetched successfully', { count: result.length });
      return result;

    } catch (error) {
      this.logger.error('Failed to fetch distinct floors', error.stack);
      return [];
    }
  }

  /**
   * Get occupancy ranges for filter options
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param venueIds - Optional array of venue IDs to filter
   * @param blockIds - Optional array of block IDs to filter
   * @param floorIds - Optional array of floor IDs to filter
   * @returns Promise<any[]> Array of occupancy statistics
   */
  async getOccupancyRanges(
    programId?: number, 
    subProgramId?: number,
    venueIds?: number[],
    blockIds?: number[],
    floorIds?: number[]
  ): Promise<any[]> {
    this.logger.debug('Fetching occupancy ranges for filter options', {
      programId,
      subProgramId,
      venueIds,
      blockIds,
      floorIds
    });

    try {
      const queryBuilder = this.repository
        .createQueryBuilder('inventory')
        .leftJoin('inventory.room', 'room')
        .leftJoin('room.floor', 'floor')
        .leftJoin('floor.block', 'block')
        .leftJoin('block.venue', 'venue')
        .select([
          'MIN(room.occupancy) as minOccupancy',
          'MAX(room.occupancy) as maxOccupancy',
          'AVG(room.occupancy) as avgOccupancy'
        ])
        .where('inventory.deletedAt IS NULL')
        .andWhere('room.deletedAt IS NULL')
        .andWhere('room.occupancy IS NOT NULL');

      if (programId) {
        queryBuilder.andWhere('inventory.programId = :programId', { programId });
      }

      if (subProgramId) {
        queryBuilder.andWhere('inventory.subProgramId = :subProgramId', { subProgramId });
      }

      // Apply additional filters based on selected venues, blocks, floors
      if (venueIds && venueIds.length > 0) {
        queryBuilder.andWhere('venue.id IN (:...venueIds)', { venueIds });
      }

      if (blockIds && blockIds.length > 0) {
        queryBuilder.andWhere('block.id IN (:...blockIds)', { blockIds });
      }

      if (floorIds && floorIds.length > 0) {
        queryBuilder.andWhere('floor.id IN (:...floorIds)', { floorIds });
      }

      const result = await queryBuilder.getRawOne();

      this.logger.debug('Occupancy ranges fetched successfully', result);
      return result ? [result] : [];

    } catch (error) {
      this.logger.error('Failed to fetch occupancy ranges', error.stack);
      return [];
    }
  }

  /**
   * Get distinct room types for filter options
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param venueIds - Optional array of venue IDs to filter
   * @param blockIds - Optional array of block IDs to filter
   * @param floorIds - Optional array of floor IDs to filter
   * @returns Promise<any[]> Array of distinct room types
   */
  async getDistinctRoomTypes(
    programId?: number, 
    subProgramId?: number,
    venueIds?: number[],
    blockIds?: number[],
    floorIds?: number[]
  ): Promise<any[]> {
    this.logger.debug('Fetching distinct room types for filter options', {
      programId,
      subProgramId,
      venueIds,
      blockIds,
      floorIds
    });

    try {
      const queryBuilder = this.repository
        .createQueryBuilder('inventory')
        .leftJoin('inventory.room', 'room')
        .leftJoin('room.floor', 'floor')
        .leftJoin('floor.block', 'block')
        .leftJoin('block.venue', 'venue')
        .distinct(true)
        .select([
          'room.roomType as value',
          'room.roomType as label',
        ])
        .where('inventory.deletedAt IS NULL')
        .andWhere('room.deletedAt IS NULL')
        .andWhere('room.roomType IS NOT NULL')
        .andWhere('room.roomType != :empty', { empty: '' });

      if (programId) {
        queryBuilder.andWhere('inventory.programId = :programId', { programId });
      }

      if (subProgramId) {
        queryBuilder.andWhere('inventory.subProgramId = :subProgramId', { subProgramId });
      }

      // Apply additional filters based on selected venues, blocks, floors
      if (venueIds && venueIds.length > 0) {
        queryBuilder.andWhere('venue.id IN (:...venueIds)', { venueIds });
      }

      if (blockIds && blockIds.length > 0) {
        queryBuilder.andWhere('block.id IN (:...blockIds)', { blockIds });
      }

      if (floorIds && floorIds.length > 0) {
        queryBuilder.andWhere('floor.id IN (:...floorIds)', { floorIds });
      }

      const result = await queryBuilder
        .orderBy('room.roomType', 'ASC')
        .getRawMany();

      this.logger.debug('Distinct room types fetched successfully', { count: result.length });
      return result;

    } catch (error) {
      this.logger.error('Failed to fetch distinct room types', error.stack);
      return [];
    }
  }

  /**
   * Get distinct bed types for filter options
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @param venueIds - Optional array of venue IDs to filter
   * @param blockIds - Optional array of block IDs to filter
   * @param floorIds - Optional array of floor IDs to filter
   * @returns Promise<any[]> Array of distinct bed types
   */
  async getDistinctBedTypes(
    programId?: number, 
    subProgramId?: number,
    venueIds?: number[],
    blockIds?: number[],
    floorIds?: number[]
  ): Promise<any[]> {
    this.logger.debug('Fetching distinct bed types for filter options', {
      programId,
      subProgramId,
      venueIds,
      blockIds,
      floorIds
    });

    try {
      const queryBuilder = this.repository
        .createQueryBuilder('inventory')
        .leftJoin('inventory.room', 'room')
        .leftJoin('room.floor', 'floor')
        .leftJoin('floor.block', 'block')
        .leftJoin('block.venue', 'venue')
        .distinct(true)
        .select([
          'room.bedType as value',
          'room.bedType as label',
        ])
        .where('inventory.deletedAt IS NULL')
        .andWhere('room.deletedAt IS NULL')
        .andWhere('room.bedType IS NOT NULL')
        .andWhere('room.bedType != :empty', { empty: '' });

      if (programId) {
        queryBuilder.andWhere('inventory.programId = :programId', { programId });
      }

      if (subProgramId) {
        queryBuilder.andWhere('inventory.subProgramId = :subProgramId', { subProgramId });
      }

      // Apply additional filters based on selected venues, blocks, floors
      if (venueIds && venueIds.length > 0) {
        queryBuilder.andWhere('venue.id IN (:...venueIds)', { venueIds });
      }

      if (blockIds && blockIds.length > 0) {
        queryBuilder.andWhere('block.id IN (:...blockIds)', { blockIds });
      }

      if (floorIds && floorIds.length > 0) {
        queryBuilder.andWhere('floor.id IN (:...floorIds)', { floorIds });
      }

      const result = await queryBuilder
        .orderBy('room.bedType', 'ASC')
        .getRawMany();

      this.logger.debug('Distinct bed types fetched successfully', { count: result.length });
      return result;

    } catch (error) {
      this.logger.error('Failed to fetch distinct bed types', error.stack);
      return [];
    }
  }

  /**
   * Get distinct occupancy values for filter options
   * @param programId - Optional program ID filter
   * @param subProgramId - Optional sub program ID filter
   * @returns Promise<any[]> Array of distinct occupancy values
   */
  async getDistinctOccupancyValues(programId?: number, subProgramId?: number): Promise<any[]> {
    this.logger.debug('Fetching distinct occupancy values for filter options');

    try {
      const queryBuilder = this.repository
        .createQueryBuilder('inventory')
        .leftJoin('inventory.room', 'room')
        .distinct(true)
        .select([
          'CASE ' +
          'WHEN room.occupancy = 1 THEN \'Single (1 person)\' ' +
          'WHEN room.occupancy = 2 THEN \'Double (2 persons)\' ' +
          'WHEN room.occupancy = 3 THEN \'Triple (3 persons)\' ' +
          'WHEN room.occupancy = 4 THEN \'Quad (4 persons)\' ' +
          'WHEN room.occupancy >= 5 THEN CONCAT(room.occupancy, \' persons\') ' +
          'ELSE CONCAT(room.occupancy, \' person(s)\') ' +
          'END as value',
          'CASE ' +
          'WHEN room.occupancy = 1 THEN \'Single (1 person)\' ' +
          'WHEN room.occupancy = 2 THEN \'Double (2 persons)\' ' +
          'WHEN room.occupancy = 3 THEN \'Triple (3 persons)\' ' +
          'WHEN room.occupancy = 4 THEN \'Quad (4 persons)\' ' +
          'WHEN room.occupancy >= 5 THEN CONCAT(room.occupancy, \' persons\') ' +
          'ELSE CONCAT(room.occupancy, \' person(s)\') ' +
          'END as label'
        ])
        .where('inventory.deletedAt IS NULL')
        .andWhere('room.deletedAt IS NULL')
        .andWhere('room.occupancy IS NOT NULL')
        .andWhere('room.occupancy > 0');

      if (programId) {
        queryBuilder.andWhere('inventory.programId = :programId', { programId });
      }

      if (subProgramId) {
        queryBuilder.andWhere('inventory.subProgramId = :subProgramId', { subProgramId });
      }

      const result = await queryBuilder
        .orderBy('room.occupancy', 'ASC')
        .getRawMany();

      this.logger.debug('Distinct occupancy values fetched successfully', { count: result.length });
      return result;

    } catch (error) {
      this.logger.error('Failed to fetch distinct occupancy values', error.stack);
      return [];
    }
  }
}