import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, IsNull, Not, DataSource, ILike, EntityManager, In, Between, Brackets } from 'typeorm';
import { CreateRegistrationDto } from './dto/create-registration.dto';
import {
  Program,
  ProgramSession,
  ProgramQuestion,
  Question,
  ProgramRegistration,
  RegistrationApproval,
  RegistrationCustomResponse,
  RegistrationQuestionAnswer,
  ProgramRegistrationFailure,
  FormSection,
  RegistrationTravelInfo,
  RegistrationTravelPlan,
  RegistrationInvoiceDetail,
  RegistrationPaymentDetail,
  Preference,
  ProgramType,
  ProgramRegistrationSwap,
  ProgramRegistrationRmRating,
  ProgramRegistrationRecommendations,
  RegistrationApprovalTrack,
  User,
  LookupData,
} from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { RegistrationBasicStatusEnum } from 'src/common/enum/registration-basic-status.enum';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { decrementSeatCounts, incrementSeatCounts } from '../common/utils/seat-count.util';
import { TravelStatusEnum } from 'src/common/enum/travel-status.enum';
import { PreferenceService } from 'src/preference/preference.service';
import { UpdatePreferenceDto } from 'src/preference/dto/update-preference.dto';
import { GenderEnum } from 'src/common/enum/gender.enum';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { InvoiceStatusEnum } from 'src/common/enum/invoice-status.enum';
import { CommonStatus } from 'src/common/enum/common-status.enum';
import { formatTimeIST, getAgeFromDOB, getDaysCountBetweenDates, getParentalFormStatus, toSnakeCase, sortSwapRequestedPrograms, getTravelOverAllStatus, replaceStringPlaceholders, shouldExcludeFromSignedUrl, formatGoodiesStatus, formatBooleanValue, formatRatriaPillar, formatSizeSelection, sortSizes, sortRatriaPillar, getRegistrationStatusDateTime } from 'src/common/utils/common.util';
import { GOODIES_SORT_FIELDS } from 'src/common/constants/constants';
import { SwapRequestStatus } from 'src/common/enum/swap-request-status-enum';
import { LOOKUP_CATEGORY, MajorCities, specialLookupKeys } from 'src/common/constants/constants';
import { UserTypeEnum } from 'src/common/enum/user-type.enum';
import { ProgramRegistrationRepository } from 'src/program-registration/program-registration.repository';
import { CLEAR_REGISTRATION_MESSAGES, ROLE_KEYS, registrationTravelPlanMessages } from 'src/common/constants/strings-constants';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { UserRepository } from 'src/user/user.repository';
import { CommunicationService } from 'src/communication/communication.service';
import { SwapType } from 'src/common/enum/swap-type-enum';
import { SwapRequirementEnum } from 'src/common/enum/swap-requirement.enum';
import { TravelTypeEnum } from 'src/common/enum/travel-type.enum';
import { awsConfig } from 'src/common/config/config';
import { PersonTypeEnum } from 'src/common/enum/person-type.enum';
import { CommunicationTemplatesKeysEnum } from 'src/common/enum/communication-template-keys.enum';
import { AwsS3Service } from 'src/common/services/awsS3.service';
import { SIGNED_URL_COLUMN_MAPPINGS } from 'src/common/constants/constants';

@Injectable()
export class RegistrationRepository {
  constructor(
    @InjectRepository(Program) private readonly programRepo: Repository<Program>,
    @InjectRepository(ProgramSession) private readonly sessionRepo: Repository<ProgramSession>,
    @InjectRepository(ProgramQuestion) private readonly programQuestionRepo: Repository<ProgramQuestion>,
    @InjectRepository(Question) private readonly questionRepo: Repository<Question>,
    @InjectRepository(ProgramRegistration) private readonly registrationRepo: Repository<ProgramRegistration>,
    @InjectRepository(RegistrationApproval) private readonly approvalRepo: Repository<RegistrationApproval>,
    @InjectRepository(RegistrationApprovalTrack) private readonly approvalTrackRepo: Repository<RegistrationApprovalTrack>,
    @InjectRepository(RegistrationCustomResponse) private readonly customResponseRepo: Repository<RegistrationCustomResponse>,
    @InjectRepository(RegistrationQuestionAnswer) private readonly questionAnswerRepo: Repository<RegistrationQuestionAnswer>,
    @InjectRepository(ProgramRegistrationFailure) private readonly failureRepo: Repository<ProgramRegistrationFailure>,
    @InjectRepository(Preference) private readonly preferenceRepo: Repository<Preference>,
    @InjectRepository(ProgramRegistrationSwap) private readonly swapRepo: Repository<ProgramRegistrationSwap>,
    @InjectRepository(ProgramRegistrationRmRating) private readonly rmRatingRepo: Repository<ProgramRegistrationRmRating>,
    @InjectRepository(RegistrationTravelInfo) private readonly travelInfoRepo: Repository<RegistrationTravelInfo>,
    @InjectRepository(RegistrationTravelPlan) private readonly travelPlanRepo: Repository<RegistrationTravelPlan>,
    @InjectRepository(ProgramRegistrationRecommendations) private readonly recommendationRepo: Repository<ProgramRegistrationRecommendations>,
    @InjectRepository(LookupData) private readonly lookupDataRepo: Repository<LookupData>,
    @InjectRepository(ProgramRegistrationSwap) private readonly swapRequestRepo: Repository<ProgramRegistrationSwap>,
    private readonly preferenceService: PreferenceService,
    private readonly dataSource: DataSource,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
    private readonly programRegistrationRepository: ProgramRegistrationRepository,
    private readonly userRepository: UserRepository,
    private readonly communicationService: CommunicationService,
    private readonly awsS3Service: AwsS3Service,
  ) {}

  /**
   * Get registration counts for each sub-program of a given program.
   * @param programId - The primary program ID.
   * @returns Array of objects: [{ programId, programName, registrationCount }]
   */
  async getSubProgramRegistrationCounts(
    programId: number,
  ): Promise<{ programId: number; programName: string; registrationCount: number }[]> {
    // Use a single query to get counts for all sub-programs
    const query = this.programRepo
      .createQueryBuilder('sub_program')
      .select('sub_program.id', 'programId')
      .leftJoin(
        'program_registrations',
        'registration',
        'registration.program_id = sub_program.id AND registration.deleted_at IS NULL',
      )
      .addSelect('sub_program.name', 'programName')
      .addSelect('COUNT(registration.id)', 'regisstrationCount')
      .where('sub_program.primary_program_id = :programId', { programId })
      .andWhere('sub_program.deleted_at IS NULL')
      .groupBy('sub_program.id')
      .addGroupBy('sub_program.name');

    const result = await query.getRawMany();

    return result.map((row) => ({
      programId: Number(row.programId),
      programName: row.programName,
      registrationCount: Number(row.registrationCount),
    }));
  }

/**
 * Get registration counts for a specific dimension
 * @param programId The ID of the program
 * @param dimension The dimension to get registration counts for
 * @param subProgramId The ID of the sub-program (optional)
 * @returns The registration counts for the specified dimension
 */
  async getRegistrationCountsOfDimension({programId, dimension, subProgramId, rmId}: {programId: number, dimension: string, subProgramId?: number, rmId?: number}){
    try{

      const subProgramsIds = subProgramId ? [subProgramId] : (await this.getSubPrograms(programId)).map(sp => sp.id);

      switch (dimension) {
        case 'programRegistration':
         const registrationQuery = this.registrationRepo
         .createQueryBuilder('registration')
         .innerJoin('hdb_registration_approval', 'approval', 'approval.registration_id = registration.id')
         .select([
          `COUNT(DISTINCT CASE WHEN approval.approval_status = 'approved' THEN registration.id END) AS "blessedSeekers"`,
          `COUNT(DISTINCT CASE WHEN approval.approval_status = 'approved' AND registration.registration_status = 'pending' THEN registration.id END) AS "pendingDetails"`,
          `COUNT(DISTINCT registration.id) AS "totalRegistrations"`,
          `COUNT(DISTINCT CASE WHEN approval.approval_status = 'pending' THEN registration.id END) AS "newRegistrations"`,          
        ])
        .addSelect(`COUNT(DISTINCT CASE WHEN ((payment.payment_status IN ('online_pending', 'offline_pending')) OR (payment.id IS NULL AND registration.allocated_program_id IS NOT NULL)) AND registration.is_free_seat = false THEN registration.id END) AS "paymentPending"`,)
        .addSelect(`COUNT(DISTINCT CASE WHEN ((payment.payment_status IN ('online_pending', 'offline_pending')) OR (payment.id IS NULL AND registration.allocated_program_id IS NOT NULL)) AND registration.is_free_seat = false THEN registration.id END) AS "paymentPending"`,)
        .addSelect(`COUNT(DISTINCT CASE WHEN (travelInfo.travel_info_status != 'completed' OR travelPlan.travel_plan_status != 'completed' OR travelInfo.travel_info_status IS NULL OR travelPlan.travel_plan_status IS NULL) AND registration.allocated_program_id IS NOT NULL THEN registration.id END)`, 'travelPending')
        .addSelect(`COUNT(DISTINCT CASE WHEN invoice.invoice_status = 'invoice_pending' THEN registration.id END)`, 'invoicePending') 
        .leftJoin('registration.paymentDetails', 'payment')
        .leftJoin('hdb_registration_travel_plan', 'travelPlan', 'travelPlan.registrationId = registration.id')
        .leftJoin('hdb_registration_travel_info', 'travelInfo', 'travelInfo.registrationId = registration.id')
        .leftJoin('registration.invoiceDetails', 'invoice')
        .where('registration.program_id = :programId', { programId })
        .andWhere('registration.registration_status != :status', { status: RegistrationStatusEnum.SAVE_AS_DRAFT })
        .andWhere('travelInfo.deleted_at IS NULL')
        .andWhere('travelPlan.deleted_at IS NULL')
        .andWhere('registration.deleted_at IS NULL');
        if (subProgramId) {
          registrationQuery.andWhere('registration.allocated_program_id = :subProgramId', { subProgramId });
        }
        if(rmId){
          registrationQuery.andWhere('registration.rm_contact = :rmId', { rmId });
        }
        const registrationCounts = await registrationQuery.getRawOne()

        const swapRequestQuery = this.swapRepo
        .createQueryBuilder('swapRequest')
        .select('COUNT(swapRequest.id)', 'totalSwapRequests')
        .leftJoin('hdb_program_registration', 'registration', 'registration.id = swapRequest.program_registration_id')
        .leftJoin('hdb_registration_approval', 'approval', 'approval.registration_id = registration.id')
        .andWhere('registration.deleted_at IS NULL')

        if(subProgramId){
          swapRequestQuery.where('registration.allocated_program_id = :subProgramId', { subProgramId })
          .andWhere('swapRequest.status = :status', { status: 'active' });
         } else {      
            swapRequestQuery.where('registration.program_id = :programId', { programId })
            .andWhere('swapRequest.status = :status', { status: 'active' })
        }
        swapRequestQuery.andWhere('registration.registration_status NOT IN (:...statuses)', { statuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED] })
        .andWhere('approval.approval_status NOT IN (:...approvalStatuses)', { approvalStatuses: [ApprovalStatusEnum.REJECTED, ApprovalStatusEnum.ON_HOLD, ApprovalStatusEnum.CANCELLED] });
    
        if(rmId){
          swapRequestQuery.andWhere('registration.rm_contact = :rmId', { rmId });
        }
        const swapRequestCounts = await swapRequestQuery.getRawOne();
        registrationCounts.swapRequest = swapRequestCounts.totalSwapRequests;
        return registrationCounts;

        case 'rating':
       const ratingQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('sub_registration.full_name', 'fullName')
          .addSelect('sub_registration.id', 'programRegistrationId')
          .addSelect('sub_registration.no_of_hdbs', 'noOfHdbs')
          .addSelect('sub_registration.user_profile_url', 'profileUrl')
          .addSelect('sub_registration.gender', 'gender')
          .addSelect('sub_registration.city', 'city')
          .addSelect('sub_registration.registration_date', 'createdAt')
          .addSelect('EXTRACT(YEAR FROM AGE(sub_registration.dob))', 'age')
          .leftJoin('sub_registration.ratings', 'rating')
          .where('sub_registration.program_id = :programId', { programId })
          .andWhere('sub_registration.deleted_at IS NULL')
          .andWhere('rating.id IS NULL')
          .andWhere('sub_registration.registration_status != :status', { status: RegistrationStatusEnum.SAVE_AS_DRAFT })

          if(rmId){
            ratingQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          const totalUnRatedRegistrations = await ratingQuery.getCount();
          ratingQuery.take(5);
          const unratedRegistrations = await ratingQuery.getRawMany();
          return { unratedRegistrations, totalUnRatedRegistrations }
        
        case 'programBlessedSeekers':
          const subProgramQuery = this.programRepo
          .createQueryBuilder('sub_program')
          .select('sub_program.name', 'displayName')
          .addSelect('sub_program.id', 'id')
          .addSelect('COUNT(DISTINCT registration.id)', 'count')
          .leftJoin('hdb_program_registration', 'registration', 'registration.allocated_program_id = sub_program.id')
          .leftJoin('hdb_registration_approval', 'approval', 'approval.registration_id = registration.id')
          .where('sub_program.primary_program_id = :programId', { programId })
          .andWhere('approval.approval_status = :approvalStatus', { approvalStatus: ApprovalStatusEnum.APPROVED })
          .andWhere('registration.program_id = :programId', { programId })
          .andWhere('sub_program.deleted_at IS NULL')
          .andWhere('registration.deleted_at IS NULL')
          .groupBy('sub_program.id')
          .addGroupBy('sub_program.name')

          if(rmId){
            subProgramQuery.andWhere('registration.rm_contact = :rmId', { rmId });
          }
          const subProgramCounts = await subProgramQuery.getRawMany();
          return subProgramCounts;

        case 'programApprovalStatus':
          const programRegistrationStatusQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('COUNT(DISTINCT sub_registration.id)', 'count')
          .addSelect('approval.approval_status', 'approvalStatus')
          .innerJoin('hdb_registration_approval', 'approval', 'approval.registration_id = sub_registration.id')
          .where('sub_registration.program_id = :programId', { programId: programId })
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('approval.approval_status')

          if(rmId){
            programRegistrationStatusQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          const programRegistrationStatusCounts = await programRegistrationStatusQuery.getRawMany()

        // add swap requests count to the result
        const swapRequestCountQuery = this.swapRepo
          .createQueryBuilder('swapRequest')
          .select('COUNT(swapRequest.id)', 'swapRequestCount')
          .leftJoin('hdb_program_registration', 'registration', 'registration.id = swapRequest.program_registration_id')
          .where('registration.program_id = :programId', { programId })
          .andWhere('swapRequest.status = :status', { status: SwapRequestStatus.ACTIVE })
          .andWhere('registration.deleted_at IS NULL')
          .andWhere('registration.registration_status NOT IN (:...statuses)', { statuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED] });
        if (rmId) {
          swapRequestCountQuery.andWhere('registration.rm_contact = :rmId', { rmId });
        }
        const swapRequestCountResult = await swapRequestCountQuery.getRawOne();
        programRegistrationStatusCounts.push({
          count: swapRequestCountResult.swapRequestCount,
          approvalStatus: 'swap_request',
        });
        return programRegistrationStatusCounts;
        
        case 'preferredVsBlessed':

        const preferencesQuery = this.programRepo
          .createQueryBuilder('sub_program')
          .select('sub_program.name', 'displayName')
          .addSelect('sub_program.id', 'programId')
          .addSelect('COUNT(DISTINCT preference.id)', 'totalPreferences')
          .addSelect('COUNT(CASE WHEN preference.preferred_program_id = registration.allocated_program_id THEN registration.id END)', 'totalBlessedSeekers')
          .innerJoin('hdb_preference', 'preference', 'preference.preferred_program_id = sub_program.id AND preference.priority_order = 1')
          .innerJoin('hdb_program_registration', 'registration', 'registration.id = preference.registration_id AND preference.priority_order = 1')
          .where('sub_program.primary_program_id = :programId', { programId })
          .andWhere('sub_program.deleted_at IS NULL')
          .andWhere('registration.deleted_at IS NULL')
          .groupBy('sub_program.id')
          .addGroupBy('sub_program.name')
          .orderBy('sub_program.id','ASC');
        
        if(rmId){
          preferencesQuery.andWhere('registration.rm_contact = :rmId', { rmId });
        }
        
        const preferencesCounts = await preferencesQuery.getRawMany();
        return preferencesCounts;

        case 'detailedData':
        const detailedDataQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('sub_program.name', 'hdbsMsds')
          .addSelect('sub_program.id', 'programId')
          .addSelect('COUNT(*)', 'totalSeekers')
          .addSelect('COUNT(CASE WHEN sub_registration.no_of_hdbs = 0 THEN 1 END)', 'firstTimers')
          .addSelect('COUNT(CASE WHEN sub_registration.no_of_hdbs BETWEEN 1 AND 10 THEN 1 END)', 'oneToTenHdbsMsds')
          .addSelect('COUNT(CASE WHEN sub_registration.no_of_hdbs > 10 THEN 1 END)', 'greaterThanTenHdbsMsds')
          .addSelect(`COUNT(CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN 1 END)`, 'organisation')
          .addSelect(`COUNT(CASE WHEN sub_registration.gender = '${GenderEnum.MALE}' THEN 1 END)`, 'male')
          .addSelect(`COUNT(CASE WHEN sub_registration.gender = '${GenderEnum.FEMALE}' THEN 1 END)`, 'female')
          .leftJoin('sub_registration.allocatedProgram', 'sub_program')
          .leftJoin('sub_registration.user', 'user')
          .where('sub_registration.allocated_program_id IN (:...allocatedProgramIds)', { allocatedProgramIds: subProgramsIds })
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('sub_registration.allocated_program_id')
          .addGroupBy('sub_program.name')
          .addGroupBy('sub_program.id');
          if(rmId){
            detailedDataQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }

          const result  = await detailedDataQuery.getRawMany();

        return result;
        
        case 'gender':
          const genderQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('COUNT(DISTINCT sub_registration.id)', 'count')
          .addSelect('sub_registration.gender', 'gender')
          .innerJoin('hdb_registration_approval', 'approval', 'approval.registration_id = sub_registration.id')
          .where('sub_registration.program_id = :programId', { programId: programId })
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('sub_registration.gender');
          const genderCounts = await genderQuery.getRawMany();
          return genderCounts;

        case 'demographics':
        const ageGenderQuery = this.registrationRepo.createQueryBuilder('sub_registration')
          .select('EXTRACT(YEAR FROM AGE(sub_registration.dob))', 'age')
          .addSelect('sub_registration.gender', 'gender')
          .addSelect('COUNT(DISTINCT sub_registration.id)', 'count')

          if(subProgramId){
           // Demographics for a specific sub-program
            ageGenderQuery.where('sub_registration.allocated_program_id = :subProgramId', { subProgramId: subProgramId });
          }
          else {
            // Demographics for all sub-programs under the main program
            ageGenderQuery.where('sub_registration.program_id = :programId', { programId: programId });
            ageGenderQuery.andWhere('sub_registration.registration_status != :status', { status: RegistrationStatusEnum.SAVE_AS_DRAFT });
          }
          if(rmId){
            ageGenderQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          ageGenderQuery
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('age')
          .addGroupBy('sub_registration.gender');

        const rawAgesGender = await ageGenderQuery.getRawMany();

        return rawAgesGender;
              
        case 'payment':
          const paymentQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('COUNT(DISTINCT sub_registration.id)', 'count')
          .addSelect('payment.payment_status', 'paymentStatus')
          .leftJoin('sub_registration.paymentDetails', 'payment')

          if(subProgramId){
            // Payment counts for a specific sub-program
            paymentQuery.where('sub_registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId: subProgramId });
          }
          else {
            // Payment counts for all sub-programs under the main program
            paymentQuery.where('sub_registration.allocated_program_id IN (:...allocatedProgramIds)', { allocatedProgramIds: subProgramsIds });
          }
          if(rmId){
            paymentQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          paymentQuery
          .andWhere('sub_registration.deleted_at IS NULL')
          .groupBy('payment.payment_status');

          const paymentCounts = await paymentQuery.getRawMany();
          return paymentCounts;

        case 'invoice':
          const invoiceQuery = this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('COUNT(DISTINCT sub_registration.id)', 'count')
          .addSelect('invoice.invoice_status', 'invoiceStatus')
          .leftJoin('sub_registration.invoiceDetails', 'invoice')
          .leftJoin('sub_registration.paymentDetails', 'payment')
          .where('payment.payment_status IN (:...paymentStatuses)', { paymentStatuses: [PaymentStatusEnum.OFFLINE_COMPLETED, PaymentStatusEnum.ONLINE_COMPLETED] });

          if(subProgramId){
          // Invoice counts for a specific sub-program
          invoiceQuery.andWhere('sub_registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId: subProgramId });
          }
          else {
          // Invoice counts for all sub-programs under the main program
          invoiceQuery.andWhere('sub_registration.program_id = :programId', { programId: programId });
          }
          if(rmId){
          invoiceQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          invoiceQuery
          .andWhere('sub_registration.deleted_at IS NULL')
          .andWhere('invoice.deleted_at IS NULL')
          .groupBy('invoice.invoice_status');

          const totalCountQuery = invoiceQuery
          totalCountQuery.andWhere('invoice.invoice_status != :invoiceStatus', { invoiceStatus: InvoiceStatusEnum.SAVE_AS_DRAFT });
          const totalInvoiceCount = await totalCountQuery.getCount();
          
          const invoiceStatusCounts = await invoiceQuery.getRawMany();
          return { invoiceStatusCounts : invoiceStatusCounts, totalInvoiceCount :  totalInvoiceCount};
        case 'travelPlan':
          const travelPlanQuery = this.registrationRepo
            .createQueryBuilder('sub_registration')
            .select('COUNT(DISTINCT sub_registration.id)', 'count')
            .addSelect('travelPlan.travel_type', 'travelType')
            .leftJoin('hdb_registration_travel_plan', 'travelPlan', 'travelPlan.registration_id = sub_registration.id')
            .andWhere('sub_registration.deleted_at IS NULL')
            .groupBy('travelPlan.travel_type');

          if (subProgramId) {
            travelPlanQuery.where('sub_registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId: subProgramId });
          }

          if (rmId) {
            travelPlanQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          const travelPlanCounts = await travelPlanQuery.getRawMany();
          return travelPlanCounts;
        
        case 'travelStatus':
          const travelStatusQuery =  this.registrationRepo
          .createQueryBuilder('sub_registration')
          .select('COUNT(DISTINCT sub_registration.id)', 'totalRegistrations')
          .addSelect(`COUNT(CASE WHEN travelInfo.travel_info_status = 'completed' AND travelPlan.travel_plan_status = 'completed' THEN travelInfo.id END)`, 'completedCount')
          .addSelect(
            `COUNT(CASE WHEN travelInfo.travel_info_status != 'completed' OR travelPlan.travel_plan_status != 'completed' OR travelInfo.travel_info_status IS NULL OR travelPlan.travel_plan_status IS NULL THEN sub_registration.id END)`,
            'pendingCount'
          )
          .leftJoin('hdb_registration_travel_plan', 'travelPlan', 'travelPlan.registrationId = sub_registration.id')
          .leftJoin('hdb_registration_travel_info', 'travelInfo', 'travelInfo.registrationId = sub_registration.id')
          .where('sub_registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId: subProgramId })
          .andWhere('sub_registration.deleted_at IS NULL')
          // .groupBy('sub_registration.id');

          if(rmId){
            travelStatusQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          const travelStatusCounts = await travelStatusQuery.getRawOne();
          return travelStatusCounts;
          
        case 'location':
          const cities = MajorCities
          const locationQuery = this.registrationRepo
            .createQueryBuilder('sub_registration')
            .select(`CASE 
              WHEN sub_registration.city IN (:...majorCities) THEN sub_registration.city
              ELSE 'Others'
            END`, 'location')
            .addSelect('COUNT(DISTINCT sub_registration.id)', 'count')
            .where('sub_registration.deleted_at IS NULL')
            .andWhere('sub_registration.registration_status != :status', { status: RegistrationStatusEnum.SAVE_AS_DRAFT })

            if(programId){
              locationQuery.andWhere('sub_registration.program_id = :programId', { programId });
            }else{
              locationQuery.andWhere('sub_registration.allocated_program_id IN (:...allocatedProgramIds)', { allocatedProgramIds: subProgramsIds })
            }
            locationQuery.groupBy('location');

          if(rmId){
            locationQuery.andWhere('sub_registration.rm_contact = :rmId', { rmId });
          }
          const locationCounts = await locationQuery.setParameter('majorCities', cities).getRawMany();
          return locationCounts;

        case 'goodies':
        case 'goodiesData':
          const goodiesQuery = this.registrationRepo
            .createQueryBuilder('registration')
            .select('sub_program.name', 'hdbMsd')
            .addSelect('sub_program.id', 'programId')
            .addSelect('COUNT(DISTINCT registration.id)', 'totalOffered')
            .addSelect('COUNT(DISTINCT registration.id)', 'totalRegistrations')
            .addSelect('COUNT(CASE WHEN goodies.notebook = true THEN 1 END)', 'noteBook')
            .addSelect('COUNT(CASE WHEN goodies.flask = true THEN 1 END)', 'flask')
            .addSelect('COUNT(CASE WHEN goodies.ratria_pillar_leonia = true THEN 1 END)', 'ratriaPillarTotal')
            .addSelect(`COUNT(CASE WHEN goodies.ratria_pillar_leonia = true AND LOWER(goodies.ratria_pillar_location) = 'leonia' THEN 1 END)`, 'ratriaPillarLeonia')  
            .addSelect(`COUNT(CASE WHEN goodies.ratria_pillar_leonia = true AND LOWER(goodies.ratria_pillar_location) != 'leonia' THEN 1 END)`, 'ratriaPillarOtherLocations')
            .addSelect('COUNT(CASE WHEN goodies.tshirt = true THEN 1 END)', 'tShirtTotal')
            .addSelect('COUNT(CASE WHEN goodies.jacket = true THEN 1 END)', 'jacketTotal')
            .addSelect(
              `COUNT(CASE WHEN goodies.id IS NOT NULL THEN 1 END)`,
              'goodiesComplete'
            )
            .addSelect(
              `COUNT(CASE WHEN goodies.id IS NULL THEN 1 END)`,
              'goodiesPending'
            )
            .leftJoin('registration.allocatedProgram', 'sub_program')
            .leftJoin('hdb_program_registration_goodies', 'goodies', 'goodies.registration_id = registration.id AND goodies.deleted_at IS NULL')
            .where('registration.allocated_program_id IN (:...allocatedProgramIds)', { allocatedProgramIds: subProgramsIds })
            .andWhere('registration.deleted_at IS NULL')
            .andWhere('registration.registration_status NOT IN (:...excludeStatuses)', { 
              excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED] 
            })
            .groupBy('registration.allocated_program_id')
            .addGroupBy('sub_program.name')
            .addGroupBy('sub_program.id')
            .addGroupBy('sub_program.total_seats')
            .orderBy('sub_program.id', 'ASC');
          
          if (rmId) {
            goodiesQuery.andWhere('registration.rm_contact = :rmId', { rmId });
          }

          try {
            const goodiesCounts = await goodiesQuery.getRawMany();
            return goodiesCounts;
          } catch (error) {
            console.error('Error executing goodies query:', error);
            throw error;
          }

        case 'goodiesTshirt':
          // get tshirt sizes dynamically from lookup data of category TSHIRT_SIZE
          const tshirtSizes = await this.getActiveTshirtSizes();

          // Build size select clauses
          const sizeSelects = tshirtSizes.flatMap(size => [
            `COUNT(CASE WHEN registration.gender = 'Male' AND goodies.tshirt = true AND goodies.tshirt_size = '${size.lookupKey}' THEN 1 END) AS "male${size.lookupKey}"`,
            `COUNT(CASE WHEN registration.gender = 'Female' AND goodies.tshirt = true AND goodies.tshirt_size = '${size.lookupKey}' THEN 1 END) AS "female${size.lookupKey}"`,
            `COUNT(CASE WHEN goodies.id IS NULL THEN 1 END) AS "noValue${size.lookupKey}"`
          ]).join(', ');

          const tshirtQuery = this.registrationRepo
            .createQueryBuilder('registration')
            .select('sub_program.id', 'programId')
            .addSelect('sub_program.name', 'hdbMsd')
            .addSelect(sizeSelects)
            .addSelect(`COUNT(CASE WHEN registration.gender = 'Male' AND goodies.id IS NOT NULL AND goodies.tshirt = false THEN 1 END)`, 'maleNA')
            .addSelect(`COUNT(CASE WHEN registration.gender = 'Female' AND goodies.id IS NOT NULL AND goodies.tshirt = false THEN 1 END)`, 'femaleNA')
            .addSelect(`COUNT(CASE WHEN registration.gender = 'Male' AND goodies.id IS NULL THEN 1 END)`, 'maleNoValue')
            .addSelect(`COUNT(CASE WHEN registration.gender = 'Female' AND goodies.id IS NULL THEN 1 END)`, 'femaleNoValue')
            .leftJoin('registration.allocatedProgram', 'sub_program')
            .leftJoin('hdb_program_registration_goodies', 'goodies', 'goodies.registration_id = registration.id AND goodies.deleted_at IS NULL')
            .where('registration.allocated_program_id IN (:...allocatedProgramIds)', { allocatedProgramIds: subProgramsIds })
            .andWhere('registration.deleted_at IS NULL')
            .andWhere('registration.registration_status NOT IN (:...excludeStatuses)', { 
              excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED] 
            })
            .groupBy('registration.allocated_program_id')
            .addGroupBy('sub_program.name')
            .addGroupBy('sub_program.id')
            .orderBy('sub_program.id', 'ASC');

          if (rmId) {
            tshirtQuery.andWhere('registration.rm_contact = :rmId', { rmId });
          }

          try {
            return await tshirtQuery.getRawMany();
          } catch (error) {
            this.logger.error('Error executing goodiesTshirt query:', error);
            throw error;
          }

        case 'goodiesJacket':
          // get jacket sizes dynamically from lookup data of category JACKET_SIZE
          const jacketSizes = await this.getActiveJacketSizes();

          // Build size select clauses
          const jacketSizeSelects = jacketSizes.flatMap(size => [
            `COUNT(CASE WHEN registration.gender = 'Male' AND goodies.jacket = true AND goodies.jacket_size = '${size.lookupKey}' THEN 1 END) AS "male${size.lookupKey}"`,
            `COUNT(CASE WHEN registration.gender = 'Female' AND goodies.jacket = true AND goodies.jacket_size = '${size.lookupKey}' THEN 1 END) AS "female${size.lookupKey}"`,
            `COUNT(CASE WHEN goodies.id IS NULL THEN 1 END) AS "noValue${size.lookupKey}"`
          ]).join(', ');

          const jacketQuery = this.registrationRepo
            .createQueryBuilder('registration')
            .select('sub_program.id', 'programId')
            .addSelect('sub_program.name', 'hdbMsd')
            .addSelect(jacketSizeSelects)
            .addSelect(`COUNT(CASE WHEN registration.gender = 'Male' AND goodies.id IS NOT NULL AND goodies.jacket = false THEN 1 END)`, 'maleNA')
            .addSelect(`COUNT(CASE WHEN registration.gender = 'Female' AND goodies.id IS NOT NULL AND goodies.jacket = false THEN 1 END)`, 'femaleNA')
            .addSelect(`COUNT(CASE WHEN registration.gender = 'Male' AND goodies.id IS NULL THEN 1 END)`, 'maleNoValue')
            .addSelect(`COUNT(CASE WHEN registration.gender = 'Female' AND goodies.id IS NULL THEN 1 END)`, 'femaleNoValue')
            .leftJoin('registration.allocatedProgram', 'sub_program')
            .leftJoin('hdb_program_registration_goodies', 'goodies', 'goodies.registration_id = registration.id AND goodies.deleted_at IS NULL')
            .where('registration.allocated_program_id IN (:...allocatedProgramIds)', { allocatedProgramIds: subProgramsIds })
            .andWhere('registration.deleted_at IS NULL')
            .andWhere('registration.registration_status NOT IN (:...excludeStatuses)', { 
              excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED] 
            })
            .groupBy('registration.allocated_program_id')
            .addGroupBy('sub_program.name')
            .addGroupBy('sub_program.id')
            .orderBy('sub_program.id', 'ASC');

          if (rmId) {
            jacketQuery.andWhere('registration.rm_contact = :rmId', { rmId });
          }

          try {
            return await jacketQuery.getRawMany();
          } catch (error) {
            this.logger.error('Error executing goodiesJacket query:', error);
            throw error;
          }

        default:
          this.logger.warn(`Invalid dimension provided: ${dimension}`);
          throw new InifniBadRequestException(`Invalid dimension: ${dimension}`); 
        }
    }catch(error){
      throw new Error('Failed to fetch registration statistics');
    }
  }

  async getActiveTshirtSizes(){
    return await this.lookupDataRepo.find({
      where: { lookupCategory: 'TSHIRT_SIZE', lookupStatus: CommonStatus.ACTIVE },
      select: ['lookupKey', 'lookupOrder'],
      order: { lookupOrder: 'ASC' }
    });
  }

  async getActiveJacketSizes(){
    return await this.lookupDataRepo.find({
      where: { lookupCategory: 'TSHIRT_SIZE', lookupStatus: CommonStatus.ACTIVE },
      select: ['lookupKey', 'lookupOrder'],
      order: { lookupOrder: 'ASC' }
    });
  }

  async getSubPrograms(programId : number): Promise<{ id: number; name: string; code: string; totalSeats: number }[]> {
    try {
      if (!programId || programId <= 0) {
        this.logger.warn(`Invalid programId provided: ${programId}`);
        throw new InifniBadRequestException(`Invalid programId: ${programId}`);
      }

      const subPrograms = await this.programRepo.find({
        where: { primaryProgramId: programId, deletedAt: IsNull() },
        select: ['id', 'name', 'code', 'totalSeats', 'startsAt'],
        order: { groupDisplayOrder: 'ASC' },
      });

      this.logger.log(`Successfully retrieved ${subPrograms.length} sub-programs for programId: ${programId}`);
      
      if (subPrograms.length === 0) {
        this.logger.warn(`No sub-programs found for programId: ${programId}`);
      } else {
        this.logger.debug(`Sub-programs for programId ${programId}:`, 
          subPrograms.map(sp => `${sp.name} (${sp.code}) - ID: ${sp.id}, Seats: ${sp.totalSeats}`));
      }

      return subPrograms;
    } catch (error) {
      this.logger.error(`Error fetching sub-programs for programId ${programId}: ${error.message}`, error.stack);
      throw new InifniNotFoundException(ERROR_CODES.PROGRAM_NOTFOUND, null, null, programId.toString());
    }
  }
  
  // ===== USER REGISTRATION QUERIES =====

  async findExistingUserRegistration(userId: number, programId: number, programSessionId?: number): Promise<ProgramRegistration | null> {
    return await this.registrationRepo.findOne({
      where: { 
        userId, 
        program: { id: programId }, 
        programSession: programSessionId ? { id: programSessionId } : undefined,
        deletedAt: IsNull(),
      },
    });
  }

  // ===== PROGRAM AND SESSION QUERIES =====

  async findProgramWithType(programId: number): Promise<Program | null> {
    return await this.programRepo.findOne({
      where: { id: programId },
      relations: ['type'],
    });
  }

  async findSessionForProgram(sessionId: number, programId: number): Promise<ProgramSession | null> {
    return await this.sessionRepo.findOne({
      where: { id: sessionId },
      relations: ['program'],
    });
  }

  // ===== QUESTION QUERIES =====

  async findProgramQuestions(programId: number, sessionId?: number) {
    return await this.programQuestionRepo.find({
      where: {
        program: { id: programId },
        programSession: sessionId ? { id: sessionId } : IsNull(),
        deletedAt: IsNull(),
      },
      relations: ['question', 'question.formSection'],
    });
  }

  async findFormSection(formSectionId: number): Promise<FormSection | null> {
    return await this.dataSource.getRepository(FormSection).findOne({
      where: { id: formSectionId },
    });
  }

  // ===== REGISTRATION QUERIES =====

  async findRegistrationWithRelations(registrationId: number) {
    return await this.registrationRepo.findOne({
      where: { id: registrationId, deletedAt: IsNull() },
      relations: ['program', 'program.type', 'programSession', 'createdBy', 'updatedBy', 'user'],
    });
  }

  async findPendingApproval(registrationId: number): Promise<RegistrationApproval | null> {
    return await this.approvalRepo.findOne({
      where: { 
        registrationId: registrationId,
        approvalStatus: ApprovalStatusEnum.PENDING,
        deletedAt: IsNull(),
      },
      order: { createdAt: 'DESC' },
    });
  }

  async findQuestionAnswer(
    registrationId: number,
    questionId: number,
  ): Promise<RegistrationQuestionAnswer | null> {
    return await this.questionAnswerRepo.findOne({
      where: { registrationId, questionId },
    });
  }
  
  async findLastWaitlistRegistration(programId: number, sessionId?: number) {
    const whereCondition: any = {
      waitingListSeqNumber: Not(IsNull()),
    };
    
    if (sessionId) {
      whereCondition.programSession = { id: sessionId };
    } else {
      whereCondition.program = { id: programId };
    }
    
    return await this.registrationRepo.findOne({
      where: whereCondition,
      order: { waitingListSeqNumber: 'DESC' },
    });
  }

  async findLastDirectRegistration(programId: number, sessionId?: number) {
    const whereCondition: any = {
      registrationSeqNumber: Not(IsNull()),
    };

    if (sessionId) {
      whereCondition.programSession = { id: sessionId };
    } else {
      whereCondition.program = { id: programId };
    }

    return await this.registrationRepo.findOne({
      where: whereCondition,
      order: { registrationSeqNumber: 'DESC' },
    });
  }

  /**
   * Get the next registration sequence number based on existing registrationSeqNumber values
   * Extracts numeric part from registrationSeqNumber and increments it
   */
  private async getNextRegistrationSeqNumber(programId: number, sessionId?: number): Promise<number> {
    const whereCondition: any = {
      registrationSeqNumber: Not(IsNull()),
    };

    if (sessionId) {
      whereCondition.programSession = { id: sessionId };
    } else {
      whereCondition.program = { id: programId };
    }

    const lastRegistration = await this.registrationRepo.findOne({
      where: whereCondition,
      order: { registrationSeqNumber: 'DESC' },
    });

    if (!lastRegistration?.registrationSeqNumber) {
      return 1;
    }

    // Extract numeric part from registrationSeqNumber (e.g., "ABC123" -> 123)
    const numericPart = await this.extractNumericFromSeqNumber(lastRegistration.registrationSeqNumber, programId);
    return numericPart + 1;
  }

  /**
   * Get the next waitlist sequence number based on existing waitListRegistrationSeqNumber values
   * Extracts numeric part from waitListRegistrationSeqNumber and increments it
   */
  private async getNextWaitlistSeqNumber(programId: number, sessionId?: number): Promise<number> {
    const whereCondition: any = {
      waitListRegistrationSeqNumber: Not(IsNull()),
    };

    if (sessionId) {
      whereCondition.programSession = { id: sessionId };
    } else {
      whereCondition.program = { id: programId };
    }

    const lastWaitlistRegistration = await this.registrationRepo.findOne({
      where: whereCondition,
      order: { waitListRegistrationSeqNumber: 'DESC' },
    });

    if (!lastWaitlistRegistration?.waitListRegistrationSeqNumber) {
      return 1;
    }

    // Extract numeric part from waitListRegistrationSeqNumber
    const numericPart = await this.extractNumericFromSeqNumber(
      lastWaitlistRegistration.waitListRegistrationSeqNumber,
      programId,
      true,
    );
    return numericPart + 1;
  }

  /**
   * Helper method to extract numeric part from sequence number
   * e.g., "ABC123" -> 123, "XYZ001" -> 1
   */
  private async extractNumericFromSeqNumber(
    seqNumber: string,
    programId: number,
    isWaitlist = false,
  ): Promise<number> {
    if (programId) {
      const program = await this.programRepo.findOne({ where: { id: programId } });
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString(),
        );
      }
      const prefix = isWaitlist ? `WL${program.code}` : program.code;
      const escapedPrefix = prefix.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
      seqNumber = seqNumber.replace(new RegExp(`^${escapedPrefix}`), '');
    }
    const match = seqNumber.match(/\d+$/);
    return match ? parseInt(match[0], 10) : 0;
  }

  /**
   * Generate formatted sequence number with program code prefix
   */
  private generateFormattedSeqNumber(program: Program, numericSeq: number, isWaitlist: boolean = false): string {
    const prefix = isWaitlist ? `WL${program.code}` : program.code;
    return `${prefix}${numericSeq.toString().padStart(3, '0')}`;
  }

  // ===== REGISTRATION CREATION =====

  async createRegistration(params: {
    dto: CreateRegistrationDto,
    userId: number,
    isSelfRegister: boolean,
    program: Program,
    session: ProgramSession | null,
    registrationLevel: string,
    seatInfo: any,
    processedAnswers: any,
    registrationOutcome: any,
    registrationStatus?: RegistrationStatusEnum,
    allocatedProgram?: Program | null,
    allocatedSession?: ProgramSession | null
  }, manager: EntityManager) {
    this.logger.log('Creating new registration...', params.dto);

    // Create base registration record
    const registrationData = this.buildRegistrationData(params);
    const registration = manager.create(ProgramRegistration, registrationData);
    await manager.save(ProgramRegistration, registration);
    const regId = registration.id;

    // Set audit fields after creation when ID is available
    await manager.update(ProgramRegistration, regId, {
      auditRefId: regId,
      parentRefId: regId,
    });

    // Save custom responses
    await this.saveCustomResponses(manager, params.processedAnswers.customResponses, regId);

    // Store grouped answers to specific table locations
    if (params.processedAnswers.groupedDirectLocationAnswers.size > 0) {
      await this.storeQuestionResponsesGrouped(
        params.processedAnswers.groupedDirectLocationAnswers,
        regId,
        manager,
        registration,
        params.registrationStatus,
      );
    }

    // Store question-answer mappings
    await this.upsertQuestionAnswers(manager, regId, params.processedAnswers.qaMappings, params.userId);

    // Handle registration outcome
    const result = await this.handleRegistrationOutcome(manager, regId, params);

    return result;
  }

  // ===== REGISTRATION UPDATE =====

  async updateRegistrationData(params: {
    registration: any;
    processedAnswers: any;
    userId: number;
    registrationStatus?: RegistrationStatusEnum;
  }, manager: EntityManager) {
    // Update or create custom responses
    await this.updateCustomResponses(manager, params.processedAnswers.customResponses, params.registration.id, params.userId);

    // Store answers in designated table locations
    if (params.processedAnswers.groupedDirectLocationAnswers.size > 0) {
      await this.storeQuestionResponsesGrouped(params.processedAnswers.groupedDirectLocationAnswers, params.registration.id, manager, params.registration, params.registrationStatus, params.processedAnswers.travelUpdateStatus, params.userId);
    }

    // Update question-answer mappings
    await this.upsertQuestionAnswers(manager, params.registration.id, params.processedAnswers.qaMappings, params.userId);

    return { 
      registrationId: params.registration.id,
      updated: true,
    };
  }

  async cancelRegistration(
    registrationId: number,
    cancellationDate: Date,
    cancelledBy: number,
    cancellationReason?: string,
    cancellationComments?: string) {
    if (!registrationId) {
      throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_ID_REQUIRED);
    }

    try {
      await this.dataSource.manager.transaction(async (manager) => {
        const registration = await manager.findOne(ProgramRegistration, {
        where: { id: registrationId, deletedAt: IsNull() },
        relations: ['allocatedProgram', 'program'],
      });

      if (!registration) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          registrationId.toString(),
        );
      }

      if (registration.registrationStatus === RegistrationStatusEnum.CANCELLED) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_ALREADY_CANCELLED,
          null,
          null,
          registrationId.toString(),
        );
      }

        await this.adjustSeatCountsOnCancel(manager, registration);

        await this.updateApprovalOnCancel(manager, registrationId, cancelledBy);

        await manager.update(ProgramRegistration, registrationId, {
        cancellationDate,
        cancelledBy,
        cancellationReason,
        cancellationComments,
        registrationStatus: RegistrationStatusEnum.CANCELLED,
        allocatedProgram: null,
        allocatedSession: null,
        auditRefId: registrationId,
        parentRefId: registrationId,
    
  });
      });
      return { registrationId, cancelled: true };
    } catch (error) {
      this.logger.error('Registration cancellation transaction failed', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_CANCEL_FAILED, error);
    }
  }

  private async adjustSeatCountsOnCancel(
    manager: EntityManager,
    registration: ProgramRegistration,
  ) {
    if (registration?.allocatedProgram) {
      await decrementSeatCounts(manager, registration.allocatedProgram.id);
      await decrementSeatCounts(manager, registration.program.id);
    }
  }

  private async updateApprovalOnCancel(
    manager: EntityManager,
    registrationId: number,
    cancelledBy: number,
  ) {
    const approval = await manager.findOne(RegistrationApproval, {
      where: { registrationId, deletedAt: IsNull() },
      order: { createdAt: 'DESC' },
    });

    if (!approval) {
      return;
    }

    await manager.update(
      RegistrationApproval,
      { id: approval.id },
      {
        approvalStatus: ApprovalStatusEnum.CANCELLED,
        updatedBy: { id: cancelledBy } as any,
        auditRefId: approval.id,
        parentRefId: approval.registrationId,
      },
    );

    const approvalTrack = manager.create(RegistrationApprovalTrack, {
      registrationId,
      approvalId: approval.id,
      approvalStatus: ApprovalStatusEnum.CANCELLED,
      allocatedProgramId: null,
      allocatedSessionId: null,
      createdBy: { id: cancelledBy } as any,
      updatedBy: { id: cancelledBy } as any,
    });

    await manager.save(RegistrationApprovalTrack, approvalTrack);

    this.logger.log(
      `Updated approval status to CANCELLED for registration ${registrationId}`,
    );
  }

  // ===== FAILURE TRACKING =====

  async createRegistrationFailure(program: Program, session: ProgramSession | null, dto: CreateRegistrationDto, userId: number, reason: string) {
    const failure = this.failureRepo.create({
      programId: dto.programId,
      programSessionId: session?.id || null,
      payload: dto as any,
      failureReason: reason,
      createdBy: { id: userId } as any,
      updatedBy: { id: userId } as any,
      program,
      programSession: session,
    });
    const saved = await this.failureRepo.save(failure);
    
    // Set auditRefId and parentRefId after save
    await this.failureRepo.update(saved.id, {
      auditRefId: saved.id,
    });
    
    return saved;
  }

  // ===== SEARCH AND LISTING =====
  async findRegistrations(
    limit: number | 'all',
    offset: number | 'all',
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    userRoles?: string[],
    filterRegistrationsByUserId?: number | null,
    sortKey: string = 'id',
    sortOrder: 'ASC' | 'DESC' = 'DESC',
    requiredRelations?: string[], // Optional parameter for relation optimization
    communicationTemplateKey: CommunicationTemplatesKeysEnum | null = null,
    selectFields?: (keyof ProgramRegistration | string)[],
    relationSelect?: Record<string, string[]>,
  ) {
    try {
      const whereClause: any = { deletedAt: IsNull() };
      
      // Track all ID constraints to intersect them later
      const idConstraints: number[][] = [];

            // if (!(userRoles?.includes('shoba') || userRoles?.includes('admin'))) {
      if (!parsedFilters?.createdBy) {
        whereClause.registrationStatus = Not(RegistrationStatusEnum.SAVE_AS_DRAFT)
      }
      if (communicationTemplateKey === CommunicationTemplatesKeysEnum.FIRST_TIMER) {
        whereClause.noOfHDBs = 0;
      }
           // }
      // Apply program/session filters
      if (programId) {
        const program = await this.programRepo.findOne({ where: { id: programId } });
        if (program?.registrationLevel === 'program') {
          whereClause.program = { id: programId };
          whereClause.programSession = IsNull();
        } else if (program?.registrationLevel === 'session') {
          if (programSessionId) {
            whereClause.programSession = { id: programSessionId };
          } else {
            whereClause.program = { id: programId };
          }
        } else {
          whereClause.program = { id: programId };
        }
      } else if (programSessionId) {
        whereClause.programSession = { id: programSessionId };
      }

      if (parsedFilters?.registrationStatus) {
        if (Array.isArray(parsedFilters.registrationStatus)) {
          whereClause.registrationStatus = In(parsedFilters.registrationStatus);
        } else {
          whereClause.registrationStatus = parsedFilters.registrationStatus;
        }
      }

      if (parsedFilters?.createdBy) {
        whereClause.createdBy = { id: parsedFilters.createdBy };
      }

      if (parsedFilters?.organisation) {
        const orgValues = Array.isArray(parsedFilters.organisation)
          ? parsedFilters.organisation
          : [parsedFilters.organisation];

        // If both 'yes' and 'no' are present, do not filter by userType (include all)
        if (orgValues.includes('yes') && orgValues.includes('no')) {
          // No filter needed, include all user types
        } else if (orgValues.includes('yes')) {
          whereClause.user = { ...whereClause.user, userType: UserTypeEnum.ORG };
        } else if (orgValues.includes('no')) {
          whereClause.user = { ...whereClause.user, userType: Not(UserTypeEnum.ORG) };
        }
      }
      if (parsedFilters?.approvalStatus) {
        whereClause.approvals = {
          approvalStatus: Array.isArray(parsedFilters.approvalStatus) 
            ? In(parsedFilters.approvalStatus)
            : parsedFilters.approvalStatus,
          deletedAt: IsNull(),
        };
      }

      // Special handling for swap request filters - apply similar logic as getTotalSwapRequests
      if (parsedFilters?.kpiCategory === 'swapRequests' || parsedFilters?.kpiFilter === 'swapRequests') {
        // For swap requests, we need registrations that:
        // 1. Have active swap requests
        // 2. Have allocated_program_id IS NOT NULL
        // 3. Optionally filter by specific allocated_program_id
        whereClause.swapsRequests = {
          status: SwapRequestStatus.ACTIVE,
          type: SwapType.WantsSwap
        };
        whereClause.allocatedProgram = Not(IsNull());
        
        // If there's a specific allocated program filter, apply it
        if (parsedFilters?.kpiFilter.startsWith('swapRequests_program_')) {
          const allocatedProgramId = parseInt(parsedFilters.kpiFilter.replace('swapRequests_program_', ''));
          parsedFilters.swapPreferredProgramId = allocatedProgramId;
        }
      }
      // handle for swap demand
      if (parsedFilters?.kpiCategory === 'swapDemand' || parsedFilters?.kpiFilter === 'swapDemand') {
        // For swap demand, we need registrations that:
        // 1. Have swap demands in ON_HOLD status with swap requirement SWAP_DEMAND
        // 2. Unlike swap requests (only one active), swap demands can have multiple ON_HOLD entries per registration
        // 3. We need to pick the latest swap demand (highest ID) per registration to get the preferred program
        
        // Use a subquery to get only the latest swap demand per registration
        const latestSwapDemandsSubQuery = this.swapRequestRepo
          .createQueryBuilder('s')
          .select('DISTINCT ON (s.program_registration_id) s.program_registration_id', 'program_registration_id')
          .where('s.status = :status', { status: SwapRequestStatus.ON_HOLD })
          .andWhere('s.swap_requirement = :requirement', { requirement: SwapRequirementEnum.SWAP_DEMAND })
          .orderBy('s.program_registration_id', 'ASC')
          .addOrderBy('s.id', 'DESC');

        const latestSwapDemandRegistrationIds = await latestSwapDemandsSubQuery.getRawMany().then(rows => 
          rows.map(r => Number(r.program_registration_id))
        );

        if (latestSwapDemandRegistrationIds.length > 0) {
          idConstraints.push([...new Set(latestSwapDemandRegistrationIds)]);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }

        // Apply approval status filter for swap demands
        whereClause.approvals = {
          approvalStatus: In([ApprovalStatusEnum.ON_HOLD]),
          deletedAt: IsNull(),
        };
        
        // If there's a specific allocated program filter, apply it
        if (parsedFilters?.kpiFilter.startsWith('onHold_program_')) {
          const allocatedProgramId = parseInt(parsedFilters.kpiFilter.replace('onHold_program_', ''));
          // For swap demands, set a special flag to handle latest swap demand filtering
          parsedFilters.swapDemandPreferredProgramId = allocatedProgramId;
        }
      }

      // if kpiCategory is regPending || kpiFilter is regPending_program_{id}, filter by that program
      if (parsedFilters?.kpiCategory === 'regPending' || parsedFilters?.kpiFilter === 'regPending') {

        whereClause.approvals = {
          approvalStatus: In([ApprovalStatusEnum.ON_HOLD, ApprovalStatusEnum.REJECTED]),
          deletedAt: IsNull(),
        };

        if (parsedFilters?.kpiFilter.startsWith('regPending_program_')) {
          const programId = parseInt(parsedFilters.kpiFilter.replace('regPending_program_', ''));
          parsedFilters.pendingProgramId = programId;
        }
      }

      if (parsedFilters?.mahatriaChoice) {
        whereClause.preferences = {
          preferredProgram: IsNull(),
          preferredSession: IsNull(),
        }
        whereClause.allocatedProgram = IsNull();
        whereClause.allocatedSession = IsNull();
        whereClause.approvals = {
          approvalStatus: In([ApprovalStatusEnum.ON_HOLD, ApprovalStatusEnum.PENDING, ApprovalStatusEnum.REJECTED]),
        };
        whereClause.registrationStatus = Not(RegistrationStatusEnum.SAVE_AS_DRAFT);
        whereClause.deletedAt = IsNull();
      }

      // Existing preferences-based filters
      if (parsedFilters?.preferredProgramId || parsedFilters?.preferredSessionId) {
        const preferenceSubQuery = this.preferenceRepo
          .createQueryBuilder('preference')
          .select('preference.registration_id')
          .where('preference.priority_order = 1')
          .andWhere('preference.deleted_at IS NULL');

        if (parsedFilters.preferredProgramId) {
          preferenceSubQuery.andWhere('preference.preferred_program_id = :preferredProgramId', {
            preferredProgramId: parsedFilters.preferredProgramId,
          });
        }
        if (parsedFilters.preferredSessionId) {
          preferenceSubQuery.andWhere('preference.preferred_session_id = :preferredSessionId', {
            preferredSessionId: parsedFilters.preferredSessionId,
          });

          // In Seat Allocation for subPrograms KPIs blessed and registration save-as-draft seekers should be excluded.
          if (parsedFilters.kpiCategory == 'all') {
            preferenceSubQuery
              .leftJoin('preference.registration', 'registration')
              .leftJoin('registration.approvals', 'approval');

            if (Array.isArray(parsedFilters.approvalStatus))
            {
              preferenceSubQuery.andWhere('approval.approval_status IN (:...approvalStatuses)', {
                approvalStatuses: parsedFilters.approvalStatus,
              });
            }
            else if (parsedFilters.approvalStatus) {
              preferenceSubQuery.andWhere('approval.approval_status = :approvalStatus', {
                approvalStatus: parsedFilters.approvalStatus,
              });
            }
          }
        }
        
        const preferenceRegistrationIds = await preferenceSubQuery.getRawMany().then(rows => rows.map(r => r.registration_id));
        if (preferenceRegistrationIds.length > 0) {
          idConstraints.push([...new Set(preferenceRegistrationIds)]);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      // Allocated filters
      if (parsedFilters.allocatedProgramId) {
        whereClause.allocatedProgram = { id: parsedFilters.allocatedProgramId };
      }

      if (parsedFilters.allocatedSessionId) {
        whereClause.allocatedSession = { id: parsedFilters.allocatedSessionId };
      }

      if (filterRegistrationsByUserId) {
        whereClause.rmContactUser = { id: filterRegistrationsByUserId };
      }
    
      if (parsedFilters?.gender) {
        if (Array.isArray(parsedFilters.gender)) {
          whereClause.gender = In(parsedFilters.gender);
        } else {
          whereClause.gender = parsedFilters.gender;
        }
      }

      // Age filter - collect IDs instead of overriding
      if (parsedFilters?.age) {
        const ageRanges = Array.isArray(parsedFilters.age) ? parsedFilters.age : [parsedFilters.age];
        const ageRegistrationIds: number[] = [];
        
        for (const range of ageRanges) {
          const currentDate = new Date();
          let minAge: number = 0, maxAge: number = 0;

          switch (range) {
            case '0-20':
              minAge = 0;
              maxAge = 20;
              break;
            case '21-30':
              minAge = 21;
              maxAge = 30;
              break;
            case '31-50':
              minAge = 31;
              maxAge = 50;
              break;
            case '51-65':
              minAge = 51;
              maxAge = 65;
              break;
            case '>65':
              minAge = 66;
              maxAge = 200; // Arbitrary large number for upper limit
              break;
            default:
              continue;
          }

          const maxBirthDate = new Date(currentDate.getFullYear() - minAge, currentDate.getMonth(), currentDate.getDate());
          const minBirthDate = new Date(currentDate.getFullYear() - maxAge - 1, currentDate.getMonth(), currentDate.getDate());
          

          const ageSubQuery = this.registrationRepo
            .createQueryBuilder('reg')
            .select('reg.id', 'id')
            .where('reg.dob BETWEEN :minBirthDate AND :maxBirthDate', {
              minBirthDate: minBirthDate.toISOString().split('T')[0],
              maxBirthDate: maxBirthDate.toISOString().split('T')[0]
            })
            .andWhere('reg.deleted_at IS NULL');

          const ageResults = await ageSubQuery.getRawMany();
          ageRegistrationIds.push(...ageResults.map(r => Number(r.id)));
        }

        // Always set a restrictive filter if no matches found
        if (ageRegistrationIds.length > 0) {
          idConstraints.push([...new Set(ageRegistrationIds)]);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      //  Number of HDBs filter - collect IDs instead of overriding  
      if (parsedFilters?.numberOfHdbs) {
        const hdbRanges = Array.isArray(parsedFilters.numberOfHdbs) ? parsedFilters.numberOfHdbs : [parsedFilters.numberOfHdbs];
        const hdbRegistrationIds: number[] = [];

        for (const range of hdbRanges) {
          let minHdb: number = 0, maxHdb: number = 0;
          let isRange = false;

          if (typeof range === 'string') {
            if (range.includes('-')) {
            // e.g. '0-5'
              const [min, max] = range.split('-').map(Number);
              if (!isNaN(min) && !isNaN(max)) {
                minHdb = min;
                maxHdb = max;
                isRange = true;
              }
            } else if (range.startsWith('>')) {
              minHdb = parseInt(range.replace('>', ''), 10) + 1;
            maxHdb = 1000; // Arbitrary upper limit
              isRange = true;
            } else if (range.startsWith('<')) {
              minHdb = 0;
              maxHdb = parseInt(range.replace('<', ''), 10) - 1;
              isRange = true;
            } else if (range.startsWith('=')) { // '=10'
              minHdb = maxHdb = parseInt(range.replace('=', ''), 10);
              isRange = true;
            } else if (!isNaN(Number(range))) {
              minHdb = maxHdb = Number(range);
              isRange = true;
            }
          } else if (typeof range === 'number') {
            minHdb = maxHdb = range;
            isRange = true;
          }

          if (!isRange) continue;

          const hdbSubQuery = this.registrationRepo
            .createQueryBuilder('reg')
            .select('reg.id', 'id')
            .where('reg.no_of_hdbs BETWEEN :minHdb AND :maxHdb', {
              minHdb,
              maxHdb,
            })
            .andWhere('reg.deleted_at IS NULL');

          const hdbResults = await hdbSubQuery.getRawMany();
          hdbRegistrationIds.push(...hdbResults.map(r => Number(r.id)));
        }

      // Always set a restrictive filter if no matches found
        if (hdbRegistrationIds.length > 0) {
          idConstraints.push([...new Set(hdbRegistrationIds)]);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      // Birthday List Range filter - collect IDs based on birthday falling within program date ranges
      if (parsedFilters?.birthdayListRange) {
        const programIds = Array.isArray(parsedFilters.birthdayListRange) ? parsedFilters.birthdayListRange : [parsedFilters.birthdayListRange];
        const birthdayRegistrationIds: number[] = [];

        // First, get all programs with their date ranges in a single query
        const numericProgramIds = programIds
          .map(id => Number(id))
          .filter(id => !isNaN(id));

        if (numericProgramIds.length > 0) {
          const programs = await this.programRepo.find({
            where: { 
              id: In(numericProgramIds), 
              deletedAt: IsNull(),
              startsAt: Not(IsNull()),
              endsAt: Not(IsNull())
            },
            select: ['id', 'startsAt', 'endsAt']
          });

          if (programs.length > 0) {
            // Build birthday conditions for all programs
            const birthdayConditions: string[] = [];
            const parameters: any = {};

            programs.forEach((program, index) => {
              const programStartDate = new Date(program.startsAt);
              const programEndDate = new Date(program.endsAt);
              
              // Extract month and day from program dates to match against birthday (ignoring year)
              const startMonth = programStartDate.getMonth() + 1; // getMonth() returns 0-11, so add 1
              const startDay = programStartDate.getDate();
              const endMonth = programEndDate.getMonth() + 1;
              const endDay = programEndDate.getDate();

              const paramPrefix = `prog${index}`;
              parameters[`${paramPrefix}StartMonth`] = startMonth;
              parameters[`${paramPrefix}StartDay`] = startDay;
              parameters[`${paramPrefix}EndMonth`] = endMonth;
              parameters[`${paramPrefix}EndDay`] = endDay;

              // Handle cross-year date ranges (e.g., Dec 25 to Jan 5)
              if (startMonth > endMonth || (startMonth === endMonth && startDay > endDay)) {
                // Cross-year range: birthday should be >= start date OR <= end date
                birthdayConditions.push(
                  `((EXTRACT(MONTH FROM reg.dob) > :${paramPrefix}StartMonth) OR ` +
                  `(EXTRACT(MONTH FROM reg.dob) = :${paramPrefix}StartMonth AND EXTRACT(DAY FROM reg.dob) >= :${paramPrefix}StartDay) OR ` +
                  `(EXTRACT(MONTH FROM reg.dob) < :${paramPrefix}EndMonth) OR ` +
                  `(EXTRACT(MONTH FROM reg.dob) = :${paramPrefix}EndMonth AND EXTRACT(DAY FROM reg.dob) <= :${paramPrefix}EndDay))`
                );
              } else {
                // Same year range: birthday should be between start and end dates
                birthdayConditions.push(
                  `((EXTRACT(MONTH FROM reg.dob) > :${paramPrefix}StartMonth) OR ` +
                  `(EXTRACT(MONTH FROM reg.dob) = :${paramPrefix}StartMonth AND EXTRACT(DAY FROM reg.dob) >= :${paramPrefix}StartDay)) AND ` +
                  `((EXTRACT(MONTH FROM reg.dob) < :${paramPrefix}EndMonth) OR ` +
                  `(EXTRACT(MONTH FROM reg.dob) = :${paramPrefix}EndMonth AND EXTRACT(DAY FROM reg.dob) <= :${paramPrefix}EndDay))`
                );
              }
            });

            // Execute single query with all birthday conditions
            if (birthdayConditions.length > 0) {
              const birthdaySubQuery = this.registrationRepo
                .createQueryBuilder('reg')
                .select('reg.id', 'id')
                .where('reg.deleted_at IS NULL')
                .andWhere('reg.dob IS NOT NULL')
                .andWhere(`(${birthdayConditions.join(' OR ')})`, parameters);

              const birthdayResults = await birthdaySubQuery.getRawMany();
              birthdayRegistrationIds.push(...birthdayResults.map(r => Number(r.id)));
            }
          }
        }

        // Always set a restrictive filter if no matches found
        if (birthdayRegistrationIds.length > 0) {
          idConstraints.push([...new Set(birthdayRegistrationIds)]);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      // Experience tags filter - collect IDs instead of overriding
      if (parsedFilters?.experienceTags) {
        const experienceTagFilters = Array.isArray(parsedFilters.experienceTags) ? parsedFilters.experienceTags : [parsedFilters.experienceTags];
        
        const tagValues = experienceTagFilters.filter(tag => tag !== 'No Experience Tags');
        
        let allExperienceIds: number[] = [];

        // Get registrations with specific experience tags
        if (tagValues.length > 0) {
          const experienceQuery = this.registrationRepo
            .createQueryBuilder('registration')
            .leftJoin('registration.user', 'user')
            .leftJoin('user.programExperiences', 'experience', 'experience.deleted_at IS NULL')
            .leftJoin('experience.lookupData', 'experienceLookupData', 'experienceLookupData.lookup_status = :activeStatus')
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .andWhere('experienceLookupData.lookup_key IN (:...tagValues)', { tagValues })
            .setParameter('activeStatus', CommonStatus.ACTIVE);

          const experienceResults = await experienceQuery.getRawMany();
          allExperienceIds.push(...experienceResults.map(r => Number(r.id)));
        }


        const specialTags = tagValues.filter(tag => specialLookupKeys.includes(tag));

        // Handle special tags by checking subprogram types that start with PST_
        if (specialTags.length > 0) {
          // Map tags to PST_ subprogram types
          const pstSubprogramTypes: string[] = [];
          
          for (const tag of specialTags) {
            switch (tag) {
              case 'HDB':
                pstSubprogramTypes.push('PST_HDB');
                break;
              case 'MSD':
                pstSubprogramTypes.push('PST_MSD');
                break;
              case 'ENTRAINMENT_25':
                pstSubprogramTypes.push('PST_ENTRAINMENT');
                break;
              case 'TAT_2025':
                pstSubprogramTypes.push('PST_TAT');
                break;
            }
          }

          if (pstSubprogramTypes.length > 0) {
            // Query registrations where user has participated in subprograms with PST_ types
            const participationQuery = this.registrationRepo
              .createQueryBuilder('registration')
              .leftJoin('registration.user', 'user')
              .leftJoin('user.participationSummary', 'ups', 'ups.deleted_at IS NULL')
              .select('registration.id', 'id')
              .where('registration.deleted_at IS NULL')
              .andWhere('ups.subProgramType IN (:...pstTypes)', { pstTypes: pstSubprogramTypes })
              .andWhere('ups.id IS NOT NULL')
              .groupBy('registration.id');

            const participationResults = await participationQuery.getRawMany();
            allExperienceIds.push(...participationResults.map(r => Number(r.id)));
          }
        }

        // Remove duplicates and add to constraints
        const uniqueExperienceIds = [...new Set(allExperienceIds)];
        if (uniqueExperienceIds.length > 0) {
          idConstraints.push(uniqueExperienceIds);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      if (parsedFilters?.location) {
        if (Array.isArray(parsedFilters.location)) {
          whereClause.city = In(parsedFilters.location);
        } else {
          whereClause.city = parsedFilters.location;
        }
      }

      // Filter if registration is blessed with free seat
      if (parsedFilters?.freeSeat) {
        const freeSeatValues = Array.isArray(parsedFilters.freeSeat)
          ? parsedFilters.freeSeat
          : [parsedFilters.freeSeat];

        if (freeSeatValues.includes('yes') && freeSeatValues.includes('no')) {
          // Only true or false, exclude null
          whereClause.isFreeSeat = In([true, false]);
        } else if (freeSeatValues.includes('yes')) {
          whereClause.isFreeSeat = true;
        } else if (freeSeatValues.includes('no')) {
          whereClause.isFreeSeat = false;
        }
      }

      //  Payment status filter - collect IDs instead of overriding
      if (parsedFilters?.paymentStatus) {
        const paymentStatuses = Array.isArray(parsedFilters.paymentStatus)
          ? parsedFilters.paymentStatus
          : [parsedFilters.paymentStatus];

        const includesPending = paymentStatuses.includes('payment_pending');
        const includesCompleted = paymentStatuses.includes('payment_completed');
        const includesNoPayment = paymentStatuses.includes('no_payment');

        const allPaymentIds: number[] = [];

        // Get pending payment IDs
        if (includesPending) {
          const pendingIdsRaw = await this.registrationRepo
            .createQueryBuilder('registration')
            .leftJoin(
              'registration.paymentDetails',
              'payment',
              'payment.deleted_at IS NULL'
            )
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .andWhere(
              '(payment.payment_status IN (:...pending) OR (payment.id IS NULL AND registration.allocated_program_id IS NOT NULL AND registration.is_free_seat = false))',
              { pending: [PaymentStatusEnum.ONLINE_PENDING, PaymentStatusEnum.OFFLINE_PENDING] }
            )
            .getRawMany();

          const pendingIds = pendingIdsRaw.map((row: any) => Number(row.id));
          allPaymentIds.push(...pendingIds);
        }

        // Get completed payment IDs
        if (includesCompleted) {
          const completedIdsRaw = await this.registrationRepo
            .createQueryBuilder('registration')
            .innerJoin(
              'registration.paymentDetails',
              'payment',
              'payment.deleted_at IS NULL'
            )
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .andWhere('payment.payment_status IN (:...completed)', {
              completed: [PaymentStatusEnum.ONLINE_COMPLETED, PaymentStatusEnum.OFFLINE_COMPLETED]
            })
            .getRawMany();

          const completedIds = completedIdsRaw.map((row: any) => Number(row.id));
          allPaymentIds.push(...completedIds);
        }

        // Get no payment (free seat) IDs
        if (includesNoPayment) {
          const noPaymentIdsRaw = await this.registrationRepo
            .createQueryBuilder('registration')
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .andWhere('registration.is_free_seat = true')
            .getRawMany();

          const noPaymentIds = noPaymentIdsRaw.map((row: any) => Number(row.id));
          allPaymentIds.push(...noPaymentIds);
        }

        // Remove duplicates and add to constraints
        const uniquePaymentIds = [...new Set(allPaymentIds)];
        if (uniquePaymentIds.length > 0) {
          idConstraints.push(uniquePaymentIds);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      if (parsedFilters?.paymentMode) {
        const paymentModes = Array.isArray(parsedFilters.paymentMode)
          ? parsedFilters.paymentMode
          : [parsedFilters.paymentMode];

        if (paymentModes.length > 0) {
          whereClause.paymentDetails = {
            ...whereClause.paymentDetails,
            paymentMode: In(paymentModes),
            deletedAt: IsNull(),
          };
        }
      }

      if (parsedFilters?.invoiceStatus) {
        const invoiceStatuses = Array.isArray(parsedFilters.invoiceStatus) ? parsedFilters.invoiceStatus : [parsedFilters.invoiceStatus];
        
        const includesPending = invoiceStatuses.includes('invoice_pending');
        const includesCompleted = invoiceStatuses.includes('invoice_completed');

        // Helper function to get invoice IDs based on status
        const getInvoiceIds = async (isPending: boolean) => {
          const query = this.registrationRepo
            .createQueryBuilder('registration')
            .leftJoin('registration.invoiceDetails', 'invoice', 'invoice.deleted_at IS NULL')
            .leftJoin('registration.paymentDetails', 'payment', 'payment.deleted_at IS NULL')
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .andWhere('payment.payment_status IN (:...paymentStatus)', { 
              paymentStatus: [PaymentStatusEnum.ONLINE_COMPLETED, PaymentStatusEnum.OFFLINE_COMPLETED] 
            });

          if (isPending) {
            query.andWhere('(invoice.invoice_status IN (:...pending) OR invoice.id IS NULL)', {
              pending: [InvoiceStatusEnum.INVOICE_PENDING, InvoiceStatusEnum.DRAFT]
            });
          } else {
            query.andWhere('invoice.invoice_status = :completed', {
              completed: InvoiceStatusEnum.INVOICE_COMPLETED
            });
          }

          const results = await query.getRawMany();
          return results.map((row: any) => Number(row.id));
        };

        const allInvoiceIds: number[] = [];

        if (includesPending) {
          const pendingIds = await getInvoiceIds(true);
          allInvoiceIds.push(...pendingIds);
        }

        if (includesCompleted) {
          const completedIds = await getInvoiceIds(false);
          allInvoiceIds.push(...completedIds);
        }

        // Remove duplicates and add to constraints
        const uniqueIds = [...new Set(allInvoiceIds)];
        if (uniqueIds.length > 0) {
          idConstraints.push(uniqueIds);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      if (parsedFilters?.travelPlan) {
        const travelStatuses = Array.isArray(parsedFilters.travelPlan)
          ? parsedFilters.travelPlan
          : [parsedFilters.travelPlan];
        const mappedStatuses: string[] = [];

        travelStatuses.forEach(status => {
          switch (status) {
            case 'travel_completed':
              mappedStatuses.push(TravelStatusEnum.COMPLETED);
              break;
            case 'travel_pending':
              mappedStatuses.push(TravelStatusEnum.PENDING);
              break;
          }
        });

        if (mappedStatuses.length > 0) {
          whereClause.travelPlans = {
            travelPlanStatus: In(mappedStatuses),
            deletedAt: IsNull(),
          };
        }
      }

      //  Travel status filter - collect IDs instead of overriding
      if (parsedFilters?.travelStatus) {
        const statusValues = Array.isArray(parsedFilters.travelStatus)
          ? parsedFilters.travelStatus
          : [parsedFilters.travelStatus];

        const includesPending = statusValues.includes('travel_pending');
        const includesCompleted = statusValues.includes('travel_completed');

        // Helper function to get travel IDs based on status
        const getTravelIds = async (isPending: boolean) => {
          const query = this.registrationRepo
            .createQueryBuilder('registration')
            .leftJoin('registration.travelPlans', 'travelPlan', 'travelPlan.deleted_at IS NULL')
            .leftJoin('registration.travelInfo', 'travelInfo', 'travelInfo.deleted_at IS NULL')
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL');

          if (isPending) {
            query.andWhere('registration.allocated_program_id IS NOT NULL');
            query.andWhere(
              '(travelInfo.travel_info_status != :completed ' +
                'OR travelPlan.travel_plan_status != :completed ' +
                'OR travelInfo.travel_info_status IS NULL ' +
                'OR travelPlan.travel_plan_status IS NULL)',
              { completed: TravelStatusEnum.COMPLETED }
            );
          } else {
            query.andWhere('travelPlan.travel_plan_status = :completed', { completed: TravelStatusEnum.COMPLETED })
                 .andWhere('travelInfo.travel_info_status = :completed', { completed: TravelStatusEnum.COMPLETED });
          }

          const results = await query.getRawMany();
          return results.map((row: any) => Number(row.id));
        };

        const allTravelIds: number[] = [];

        if (includesPending) {
          const pendingIds = await getTravelIds(true);
          allTravelIds.push(...pendingIds);
        }

        if (includesCompleted) {
          const completedIds = await getTravelIds(false);
          allTravelIds.push(...completedIds);
        }

        // Remove duplicates and add to constraints
        const uniqueIds = [...new Set(allTravelIds)];
        if (uniqueIds.length > 0) {
          idConstraints.push(uniqueIds);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      if(parsedFilters.programPreference) {
        whereClause.preferences = {
          preferredProgram: {id: parsedFilters.programPreference},
          priorityOrder: 1,
          deletedAt: IsNull(),
        }
        delete parsedFilters.programPreference;
      }

      if (parsedFilters?.swapRequests) {
        const swapTypes = Array.isArray(parsedFilters.swapRequests) ? parsedFilters.swapRequests : [parsedFilters.swapRequests];

        if (swapTypes.length > 0) {
          whereClause.swapsRequests = {
            type: In(swapTypes),
            status : SwapRequestStatus.ACTIVE,
          };
          whereClause.approvals = {
            approvalStatus: In([ApprovalStatusEnum.APPROVED, ApprovalStatusEnum.PENDING]),
          };
        }
      }

      if(parsedFilters?.swapPreferredProgramId) {
        /**
         * Get registration IDs where the first inserted program matches the swapPreferredProgramId
         * Using QueryBuilder with ROW_NUMBER() window function to replicate the exact SQL logic
         */
        try {
          // Create the main query with window function using raw SQL within QueryBuilder
          const swapRequestIds = await this.dataSource
            .createQueryBuilder()
            .select('ranked.swap_request_id', 'swap_request_id')
            .from(subQuery => {
              return subQuery
                .select('srp.*')
                .addSelect('ROW_NUMBER() OVER (PARTITION BY srp.swap_request_id ORDER BY srp.id ASC)', 'rn')
                .from('hdb_swap_requested_program', 'srp');
            }, 'ranked')
            .where('ranked.rn = 1')
            .andWhere('ranked.program_id = :programId', { programId: parsedFilters.swapPreferredProgramId })
            .getRawMany();
          
          this.logger.debug(`Retrieved ${swapRequestIds.length} swap request IDs for preferred program ${parsedFilters.swapPreferredProgramId}`);
          
          const swapRequestIdValues = swapRequestIds.map(row => row.swap_request_id);
          
          if (swapRequestIdValues.length > 0) {
            whereClause.swapsRequests = {
              ...whereClause.swapsRequests,
              id: In(swapRequestIdValues),
            };
            this.logger.debug(`Applied swap request filter with ${swapRequestIdValues.length} request IDs`);
          } else {
            // No matches found - add empty constraint to ensure no results
            idConstraints.push([]);
            this.logger.debug(`No swap request found for preferred program ${parsedFilters.swapPreferredProgramId}`);
          }
        } catch (error) {
          this.logger.error(`Error filtering by swap preferred program ${parsedFilters.swapPreferredProgramId}:`, error?.stack);
          // On error, add empty constraint to prevent incorrect results
          idConstraints.push([]);
        }
      }
      
      // Handle pendingProgramId for rejected/on_hold - Add to idConstraints for intersection
      if (parsedFilters?.pendingProgramId) {
        /**
         * Get registration IDs where:
         * 1. Registration has REJECTED status with preference for pendingProgramId (priority 1)
         * 2. Registration has ON_HOLD status with latest swap demand for pendingProgramId
         */
        try {
          const pendingProgramId = parsedFilters.pendingProgramId;
          
          // Preference-based subquery (priority 1, REJECTED)
          const prefSubQuery = this.preferenceRepo
            .createQueryBuilder('preference')
            .select('preference.registration_id')
            .leftJoin('preference.registration', 'registration')
            .leftJoin('registration.approvals', 'approval')
            .where('preference.priority_order = 1')
            .andWhere('preference.preferred_program_id = :pendingProgramId', { pendingProgramId })
            .andWhere('preference.deleted_at IS NULL')
            .andWhere('approval.approval_status = :rejectedStatus', { rejectedStatus: ApprovalStatusEnum.REJECTED })
            .andWhere('approval.deleted_at IS NULL');
          const prefIds = await prefSubQuery.getRawMany().then(rows => rows.map(r => r.registration_id));

          // Swap demand-based subquery (latest ON_HOLD, SWAP_DEMAND)
          const latestSwapIdSubQuery = `(SELECT sr.id FROM hdb_program_registration_swap sr WHERE sr.program_registration_id = reg.id AND sr.status = '${SwapRequestStatus.ON_HOLD}' AND sr.swap_requirement = '${SwapRequirementEnum.SWAP_DEMAND}' ORDER BY sr.id DESC LIMIT 1)`;
          const firstRequestedProgramSubQuery = `(SELECT srp.program_id FROM hdb_swap_requested_program srp WHERE srp.swap_request_id = ${latestSwapIdSubQuery} ORDER BY srp.id ASC LIMIT 1)`;
          
          const swapDemandRegistrations = await this.registrationRepo
            .createQueryBuilder('reg')
            .leftJoin('reg.approvals', 'approval')
            .select('reg.id', 'id')
            .where(`(${latestSwapIdSubQuery}) IS NOT NULL`)
            .andWhere(`(${firstRequestedProgramSubQuery}) = :pendingProgramId`, { pendingProgramId })
            .andWhere('reg.deleted_at IS NULL')
            .andWhere('approval.approval_status = :onHoldStatus', { onHoldStatus: ApprovalStatusEnum.ON_HOLD })
            .andWhere('approval.deleted_at IS NULL')
            .getRawMany();
          const swapDemandIds = swapDemandRegistrations.map(row => Number(row.id));

          // Combine both sets
          const allPendingIds = Array.from(new Set([...prefIds, ...swapDemandIds]));
          
          // Add to idConstraints for intersection with other filters
          if (allPendingIds.length > 0) {
            idConstraints.push(allPendingIds);
            this.logger.debug(`Applied pendingProgramId filter with ${allPendingIds.length} registration IDs`);
          } else {
            idConstraints.push([]); // Empty array means no matches
            this.logger.debug(`No registrations found for pendingProgramId ${pendingProgramId}`);
          }
        } catch (error) {
          this.logger.error(`Error filtering by pendingProgramId ${parsedFilters.pendingProgramId}:`, error?.stack);
          idConstraints.push([]); // No matches on error
        }
      }
      

      // Handle swap demand preferred program filter (for onHold_program_ filters)
      if(parsedFilters?.swapDemandPreferredProgramId) {
        /**
         * Get registration IDs where the latest swap demand's first requested program matches the specified program
         */
        try {
          // Subquery for latest ON_HOLD swap demand per registration
          const latestSwapIdSubQuery = `(SELECT sr.id FROM hdb_program_registration_swap sr WHERE sr.program_registration_id = reg.id AND sr.status = '${SwapRequestStatus.ON_HOLD}' AND sr.swap_requirement = '${SwapRequirementEnum.SWAP_DEMAND}' ORDER BY sr.id DESC LIMIT 1)`;
          // Subquery for first requested program for the latest swap
          const firstRequestedProgramSubQuery = `(SELECT srp.program_id FROM hdb_swap_requested_program srp WHERE srp.swap_request_id = ${latestSwapIdSubQuery} ORDER BY srp.id ASC LIMIT 1)`;
          // Main query to get registration IDs
          const registrationsWithPreferredProgram = await this.registrationRepo
            .createQueryBuilder('reg')
            .select('reg.id', 'id')
            .where(`(${latestSwapIdSubQuery}) IS NOT NULL`)
            .andWhere(`(${firstRequestedProgramSubQuery}) = :preferredProgramId`, { preferredProgramId: parsedFilters.swapDemandPreferredProgramId })
            .andWhere('reg.deleted_at IS NULL')
            .getRawMany();
          const registrationIdValues = registrationsWithPreferredProgram.map(row => Number(row.id));
          if (registrationIdValues.length > 0) {
            idConstraints.push([...new Set(registrationIdValues)]);
            this.logger.debug(`Applied swap demand filter with ${registrationIdValues.length} registration IDs`);
          } else {
            idConstraints.push([]);
            this.logger.debug(`No registration found with latest swap demand for preferred program ${parsedFilters.swapDemandPreferredProgramId}`);
          }
        } catch (error) {
          this.logger.error(`Error filtering by swap demand preferred program ${parsedFilters.swapDemandPreferredProgramId}:`, error?.stack);
          idConstraints.push([]);
        }
      }

      if (parsedFilters?.preferredRoomMate) {
        const roomMateValues = Array.isArray(parsedFilters.preferredRoomMate)
          ? parsedFilters.preferredRoomMate.map((v: string) => v.toLowerCase())
          : [parsedFilters.preferredRoomMate.toLowerCase()];

        if (roomMateValues.includes('yes') && roomMateValues.includes('no')) {
          // No filter needed, include all
        } else if (roomMateValues.includes('yes')) {
          whereClause.preferredRoomMate = Not(IsNull());
        } else if (roomMateValues.includes('no')) {
          whereClause.preferredRoomMate = IsNull();
        }
      }

      // RM Rating filter - collect IDs instead of overriding
      if (parsedFilters?.rmRating) {
        const ratingFilters = Array.isArray(parsedFilters.rmRating)
          ? parsedFilters.rmRating
          : [parsedFilters.rmRating];

        const ratingIdsSet = new Set<number>();

        for (const ratingFilter of ratingFilters) {
          const query = this.registrationRepo
            .createQueryBuilder('registration')
            .leftJoin('registration.ratings', 'rating', 'rating.deleted_at IS NULL')
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .groupBy('registration.id');

          if (ratingFilter === 'unrated') {
            query.having('COUNT(rating.id) = 0');
          } else if (ratingFilter === 'only5' || ratingFilter === '5') {
            query.having('AVG(rating.rating) = 5');
          } else {
            const threshold = parseInt(ratingFilter);
            if (!isNaN(threshold)) {
              query.having('AVG(rating.rating) >= :threshold', { threshold });
            }
          }

          const ids = await query.getRawMany();
          ids.forEach((r: any) => ratingIdsSet.add(Number(r.id)));
        }

        if (ratingIdsSet.size > 0) {
          idConstraints.push([...ratingIdsSet]);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      // RM Contact filter
      if (parsedFilters?.rmContact) {
        // Array of IDs or single ID
        const rmContactIds = Array.isArray(parsedFilters.rmContact)
          ? parsedFilters.rmContact.map((id: any) => Number(id))
          : [Number(parsedFilters.rmContact)];

        whereClause.rmContactUser = { id: In(rmContactIds) };
      }

      // Goodies status filter - collect IDs instead of overriding
      if (parsedFilters?.goodiesStatus) {
        const statusValues = Array.isArray(parsedFilters.goodiesStatus)
          ? parsedFilters.goodiesStatus
          : [parsedFilters.goodiesStatus];

        const includesPending = statusValues.includes('goodies_pending');
        const includesCompleted = statusValues.includes('goodies_completed');

        // Helper function to get goodies IDs based on status
        const getGoodiesIds = async (isPending: boolean) => {
          const query = this.registrationRepo
            .createQueryBuilder('registration')
            .leftJoin('registration.goodies', 'goodies', 'goodies.deleted_at IS NULL')
            .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .andWhere('registration.allocated_program_id IS NOT NULL')
            .andWhere('approval.approval_status = :approvalStatus', { approvalStatus: ApprovalStatusEnum.APPROVED });

          if (isPending) {
            query.andWhere('goodies.id IS NULL');
          } else {
            query.andWhere('goodies.id IS NOT NULL');
          }

          const results = await query.getRawMany();
          return results.map((row: any) => Number(row.id));
        };

        const allGoodiesIds: number[] = [];

        if (includesPending) {
          const pendingIds = await getGoodiesIds(true);
          allGoodiesIds.push(...pendingIds);
        }

        if (includesCompleted) {
          const completedIds = await getGoodiesIds(false);
          allGoodiesIds.push(...completedIds);
        }

        // Remove duplicates and add to constraints
        const uniqueIds = [...new Set(allGoodiesIds)];
        if (uniqueIds.length > 0) {
          idConstraints.push(uniqueIds);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      // Recommendation filter - collect IDs instead of overriding
      if (parsedFilters?.recommendation) {
        const recommendationFilters = Array.isArray(parsedFilters.recommendation)
          ? parsedFilters.recommendation
          : [parsedFilters.recommendation];

        const hasNone = recommendationFilters.includes('none');
        const keys = recommendationFilters.filter((k) => k !== 'none');

        let ids: number[] = [];

        if (hasNone) {
          const noneQuery = this.registrationRepo
            .createQueryBuilder('registration')
            .leftJoin('registration.recommendation', 'rec')
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .groupBy('registration.id')
            .having('(COUNT(CASE WHEN rec.id IS NOT NULL THEN 1 END) = 0 OR COUNT(CASE WHEN rec.is_recommended = true THEN 1 END) = 0)');

          const noneIds = await noneQuery.getRawMany();
          ids.push(...noneIds.map((r: any) => Number(r.id)));
        }

        if (keys.length > 0) {
          const keyQuery = this.registrationRepo
            .createQueryBuilder('registration')
            .innerJoin('registration.recommendation', 'rec')
            .select('registration.id', 'id')
            .where('registration.deleted_at IS NULL')
            .andWhere('rec.is_recommended = true')
            .andWhere('rec.recommendation_key IN (:...recKeys)', { recKeys: keys })
            .groupBy('registration.id');

          const keyIds = await keyQuery.getRawMany();
          ids.push(...keyIds.map((r: any) => Number(r.id)));
        }

        ids = [...new Set(ids)]; // Remove duplicates
        if (ids.length > 0) {
          idConstraints.push(ids);
        } else {
          idConstraints.push([]); // Empty array means no matches
        }
      }

      // Preferred program filter - collect IDs instead of overriding
      if (parsedFilters?.preferredProgram) {
        const preferredPrograms = Array.isArray(parsedFilters.preferredProgram) ? parsedFilters.preferredProgram : [parsedFilters.preferredProgram];
        
        if (preferredPrograms.length > 0) {
          const preferenceSubQuery = this.preferenceRepo
            .createQueryBuilder('preference')
            .select('preference.registration_id')
            .where('preference.preferred_program_id IN (:...preferredPrograms)', {
              preferredPrograms: preferredPrograms
            })
            .andWhere('preference.priority_order = 1')
            .andWhere('preference.deleted_at IS NULL');

          const preferenceResults = await preferenceSubQuery.getRawMany();
          const registrationIds = preferenceResults.map(r => r.registration_id);
          
          if (registrationIds.length > 0) {
            idConstraints.push(registrationIds);
          } else {
            idConstraints.push([]); // Empty array means no matches
          }
        }
      }

      if (parsedFilters?.programStatus) {
        whereClause.program = {
          ...whereClause.program,
          status: Array.isArray(parsedFilters.programStatus)
            ? In(parsedFilters.programStatus)
            : parsedFilters.programStatus,
          deletedAt: IsNull(),
        };
      }

      if (parsedFilters?.programType) {
        whereClause.program = {
          ...whereClause.program,
          type: {
            key: parsedFilters.programType,
            deletedAt: IsNull(),
          },
          deletedAt: IsNull(),
        };
      }

      if (parsedFilters?.dateRange?.startDate && parsedFilters?.dateRange?.endDate) {
        // Convert DD/MM/YYYY to YYYY-MM-DD if needed
        const parseDate = (dateStr: string) => {
          if (/^\d{2}\/\d{2}\/\d{4}$/.test(dateStr)) {
            const [day, month, year] = dateStr.split('/');
            return `${year}-${month}-${day}`;
          }
          return dateStr;
        };
        whereClause.registrationDate = Between(
          parseDate(parsedFilters.dateRange.startDate),
          parseDate(parsedFilters.dateRange.endDate)
        );
      }

      if (parsedFilters?.excludeCancelledRegistrations) {
        whereClause.registrationStatus = Not(RegistrationStatusEnum.CANCELLED);
      }

      // Handle any remaining unknown filters
      if (parsedFilters) {
        const handledFilters = [
          'registrationStatus',
          'createdBy',
          'approvalStatus',
          'mahatriaChoice',
          'preferredProgramId',
          'preferredSessionId',
          'allocatedProgramId',
          'allocatedSessionId',
          'gender',
          'age',
          'numberOfHdbs',
          'birthdayListRange',
          'experienceTags',
          'location',
          'paymentStatus',
          'paymentMode',
          'invoiceStatus',
          'travelPlan',
          'travelStatus',
          'swapRequests',
          'preferredProgram',
          'programStatus',
          'programType',
          'rmRating',
          'preferredRoomMate',
          'dateRange',
          'organisation',
          'swapPreferredProgramId',
          'swapDemandPreferredProgramId',
          'recommendation',
          'freeSeat',
          'excludeCancelledRegistrations',
          'rmContact',
          'goodiesStatus',
          'view',
          'swapDemand',
          'pendingProgramId',
          'regPending'
        ];

        delete parsedFilters.kpiCategory;
        delete parsedFilters.kpiFilter;

        for (const [key, value] of Object.entries(parsedFilters)) {
          if (handledFilters.includes(key)) {
            continue;
          }
          this.logger.debug(`Processing unknown filter - ${key}:`, value);
          if (Array.isArray(value)) {
            whereClause[key] = In(value);
          } else if (typeof value === 'object' && value !== null) {
            whereClause[key] = value;
          } else {
            whereClause[key] = value;
          }
        }
      }

      //Apply ID intersection logic
      if (idConstraints.length > 0) {
        // Check if any constraint is empty (no matches)
        const hasEmptyConstraint = idConstraints.some(constraint => constraint.length === 0);
        
        if (hasEmptyConstraint) {
          // If any filter returns no matches, the intersection is empty
          whereClause.id = In([-1]); // No matches
        } else {
          // Find intersection of all ID constraints
          let intersectedIds = idConstraints[0];
          for (let i = 1; i < idConstraints.length; i++) {
            const constraintIds = idConstraints[i].filter(id => !isNaN(id) && id !== null && id !== undefined);
            intersectedIds = intersectedIds.filter(id => constraintIds.includes(id));
          }
          whereClause.id = intersectedIds.length > 0 ? In(intersectedIds) : In([-1]);
        }
      }

      // Apply search text
      let finalWhereClause: any = whereClause;
      if (searchText?.trim()) {
        const searchTerm = ILike(`%${searchText.trim()}%`);
        
        // Create search conditions that properly inherit all base filters
        const createSearchCondition = (searchField: string, searchTarget?: any) => {
          const condition = { ...whereClause };
          
          if (searchTarget) {
            // For nested searches (like program.name), merge with existing program filters
            if (searchField === 'program') {
              condition.program = {
                ...condition.program, // Keep existing program filters
                ...searchTarget, // Add search criteria
              };
            }
          } else {
            // For direct field searches
            condition[searchField] = searchTerm;
          }
          
          return condition;
        };

        // Only include program/programType search if parsedFilters.createdBy is present
        if (parsedFilters?.createdBy) {
          finalWhereClause = [
            createSearchCondition('fullName'),
            createSearchCondition('emailAddress'),
            createSearchCondition('mobileNumber'),
            createSearchCondition('program', {
              name: searchTerm,
              deletedAt: IsNull(),
            }),
            createSearchCondition('program', {
              type: {
                name: searchTerm,
                deletedAt: IsNull(),
              },
              deletedAt: IsNull(),
            }),
          ];
        } else {
          finalWhereClause = [
            createSearchCondition('fullName'),
            // createSearchCondition('emailAddress'),
            createSearchCondition('mobileNumber'),
          ];
        }
      }
      // Define default relations (used when requiredRelations is not provided)
      const defaultRelations = [
        'program',
        'program.type',
        'programSession',
        'invoiceDetails',
        'invoiceDetails.updatedBy',
        'paymentDetails',
        'paymentDetails.editRequests',
        'travelInfo',
        'travelPlans',
        'user',
        'user.programExperiences',
        'user.programExperiences.lookupData',
        'user.profileExtension',
        'approvals',
        'approvals.tracks',
        'allocatedProgram',
        'allocatedSession',
        'swapsRequests',
        'swapsRequests.currentProgram',
        'swapsRequests.requestedPrograms',
        'swapsRequests.swapRequestedProgram',
        'swapsRequests.swapRequestedProgram.program',
        'ratings',
        'rmContactUser',
        'preferences',
        'preferences.preferredProgram',
        'preferences.preferredSession',
        'recommendation',
        'goodies',
        'createdBy',
        'updatedBy',
        'seekerProgramExperience'
      ];

      // Determine required relations based on filters and sorting
      let relations: string[] = [];
      if (requiredRelations && requiredRelations.length > 0) {
        relations = [...requiredRelations];
      } else {
        relations = [...defaultRelations];
      }

      // Dynamically append relations based on parsed filters
      const additionalRelations = this.getRequiredRelationsForFilters(parsedFilters, sortKey);
      relations = [...new Set([...relations, ...additionalRelations])];
      
      // Map frontend sortKey to database field name
      const sortKeyMapping: Record<string, string> = {
        'seekerName': 'fullName',
        'age': 'dob',
        'location': 'city',
        'numberOfHDBs': 'noOfHDBs',
        'lastHdbMsd': 'lastHdbAttended',  
        'blessedWith': 'allocatedProgram.id',
        'blessedWithProgram': 'allocatedProgramId',
        'blessedDate': 'approvals.approvalDate',
        'cancelledDate': 'cancellationDate',
        'holdDate': 'approvals.updatedAt',
        'swapDemandDate': 'approvals.updatedAt',
        'swapRequestDate': 'swapsRequests.createdAt',
        'allStatusDate': 'allStatusDate',
        'pendingDate': 'approvals.updatedAt',
        'averageRating': 'averageRating',
        'profileUrl': 'profileUrl',
        'mobileNumber': 'mobileNumber',
        'emailAddress': 'emailAddress',
        'fullName': 'fullName',
        'gender': 'gender',
        'city': 'city',
        'noOfHDBs': 'noOfHDBs',
        'id': 'id',
        'dob': 'dob',
        'rmContact': 'rmContactUser.orgUsrName',
        'travelPlanStatus': 'travelPlans.travelPlanStatus',
        'goodiesStatus': 'goodies.goodiesStatus',
        'flask': 'goodies.flask',
        'notebook':'goodies.notebook',
        'ratriaPillar': 'goodies.ratriaPillars',
        'jacket': 'goodies.jacket',
        'tshirt': 'goodies.tshirt'
      };
      
      // Convert sortKey to database field name
      const dbSortKey = sortKeyMapping[sortKey] || sortKey;
      
      // Check if sorting by nested relation field or requires custom sorting logic
      const isNestedSort = this.requiresCustomSorting(dbSortKey);
      
      let data;
      let totalCountForNestedSort = 0;

      this.logger.log(`Sorting configuration: sortKey=${sortKey}, dbSortKey=${dbSortKey}, isNestedSort=${isNestedSort}`);

      if (isNestedSort) {
        this.logger.debug('Using custom nested sorting with in-memory processing');
        
        // Ensure required relations for sorting are included
        const sortRelations = this.getRequiredRelationsForSorting(dbSortKey);
        relations = [...new Set([...relations, ...sortRelations])];
        
        // First, get all data without sorting to preserve complex filtering
        const unsortedData = await this.commonDataService.getWithRelFields(
          this.registrationRepo,
          selectFields,
          finalWhereClause,
          'all', // Get all data first for accurate sorting
          'all',
          undefined, // No sorting at DB level
          undefined,
          relations,
          relationSelect
        );
        
        // Store total count for pagination calculation
        totalCountForNestedSort = unsortedData.length;
        
        // Now sort the data in memory using custom sorting logic
        const sortedData = this.sortDataByNestedField(unsortedData, dbSortKey, sortOrder);
        
        // Apply pagination after sorting
        if (limit !== 'all' && offset !== 'all') {
          const startIndex = offset as number;
          const endIndex = startIndex + (limit as number);
          data = sortedData.slice(startIndex, endIndex);
        } else {
          data = sortedData;
        }
      } else {
        this.logger.debug('Using database-level sorting');
        
        // Ensure required relations for nested field sorting are included
        const sortRelations = this.getRequiredRelationsForSorting(dbSortKey);
        relations = [...new Set([...relations, ...sortRelations])];
        
        // Use simple nested sorting to avoid TypeORM nulls configuration issues
        this.logger.debug(`Using simple nested sorting at DB level for: ${dbSortKey}`);
        data = await this.commonDataService.getWithRelFields(
          this.registrationRepo,
          selectFields,
          finalWhereClause,
          limit,
          offset,
          { [dbSortKey]: sortOrder },
          undefined,
          relations,
          relationSelect
        );
      }
      data.forEach((registration: any) => {
        registration?.paymentDetails?.forEach(payment => {
          if (payment.editRequests && payment.editRequests.length > 0) {
            // Keep only the editRequest with the highest id
            const maxIdRequest = payment.editRequests.reduce((max: any, curr: any) => {
              return (!max || curr.id > max.id) ? curr : max;
            }, null);
            payment.editRequests = maxIdRequest ? [maxIdRequest] : [];
          }
        });
        
        // Sort swap requested programs by insertion order
        sortSwapRequestedPrograms(registration);
      });
      

      // Use stored averageRating from database instead of manual calculation
      const dataWithAvgRating = data.map((registration: any) => {
        // The averageRating is already fetched from database, just format it if needed
        if (registration.averageRating !== null && registration.averageRating !== undefined) {
          registration.averageRating = Number(registration.averageRating).toFixed(2);
        } else {
          registration.averageRating = null;
        }
        return registration;
      });

      // Handle pagination response based on limit/offset type
      let paginationInfo;
      
      if (limit === 'all' || offset === 'all') {
        // Unlimited query - return simplified pagination
        paginationInfo = {
          totalPages: 1,
          pageNumber: 1,
          pageSize: null,
          totalRecords: dataWithAvgRating.length,
          numberOfRecords: dataWithAvgRating.length,
          queryType: 'unlimited',
          isUnlimited: true,
        };
      } else {
        // Regular pagination - get total count
        let total;
        if (isNestedSort) {
          // For nested sort, we already have the total count from in-memory processing
          total = totalCountForNestedSort;
        } else {
          // For regular sort, get count from database
          total = await this.registrationRepo.count({ where: finalWhereClause });
        }
        
        paginationInfo = {
          totalPages: Math.ceil(total / (limit)),
          pageNumber: Math.floor((offset) / (limit)) + 1,
          pageSize: limit,
          totalRecords: total,
          numberOfRecords: dataWithAvgRating.length,
          queryType: 'paginated',
          isUnlimited: false,
        };
      }

      return {
        data: dataWithAvgRating,
        pagination: paginationInfo,
      };

    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Helper method to sort data by nested field in memory
   */
  private sortDataByNestedField(data: any[], sortField: string, sortOrder: 'ASC' | 'DESC'): any[] {
    return data.sort((a, b) => {
      let primaryValueA, primaryValueB;
      let secondaryValueA, secondaryValueB;
      
      // Normalize sort order to uppercase for consistent comparison
      const normalizedSortOrder = sortOrder.toUpperCase();
      
      // Handle specific nested field cases with secondary sorting
      if (sortField === 'rmContactUser.orgUsrName') {
        // If rmContactUser.orgUsrName is "Other", use otherInfinitheismContact for final sort
        const rmContactA = a.rmContactUser?.orgUsrName || '';
        const rmContactB = b.rmContactUser?.orgUsrName || '';
        
        if (rmContactA.toLowerCase() === 'other' || rmContactB.toLowerCase() === 'other') {
          // Use otherInfinitheismContact directly for sorting when either is "Other"
          primaryValueA = rmContactA.toLowerCase() === 'other' ? (a.otherInfinitheismContact || '') : rmContactA;
          primaryValueB = rmContactB.toLowerCase() === 'other' ? (b.otherInfinitheismContact || '') : rmContactB;
          secondaryValueA = '';
          secondaryValueB = '';
        } else {
          // Primary sort: rmContactUser.orgUsrName
          primaryValueA = rmContactA;
          primaryValueB = rmContactB;
          
          // Secondary sort: otherInfinitheismContact
          secondaryValueA = a.otherInfinitheismContact || '';
          secondaryValueB = b.otherInfinitheismContact || '';
        }
        
      } else if (sortField === 'allocatedProgram.id') {
        if (normalizedSortOrder === 'ASC') {
          // For ASC: Primary level (approval status) first, then secondary level (allocatedProgram.id)
          primaryValueA = a?.approvals?.[0]?.approvalStatus || '';
          primaryValueB = b?.approvals?.[0]?.approvalStatus || '';
          secondaryValueA = a.allocatedProgram?.id || 0;
          secondaryValueB = b.allocatedProgram?.id || 0;
        } else {
          // For DESC:  primary level (approval status) the Secondary level (allocatedProgram.id) first
          secondaryValueA = a.allocatedProgram?.id || 0;
          secondaryValueB = b.allocatedProgram?.id || 0;
          primaryValueA = a?.approvals?.[0]?.approvalStatus || '';
          primaryValueB = b?.approvals?.[0]?.approvalStatus || '';
        }
        
      } else if (sortField === 'travelPlans.travelPlanStatus') {
        primaryValueA = getTravelOverAllStatus(a);
        primaryValueB = getTravelOverAllStatus(b);
        secondaryValueA = '';
        secondaryValueB = '';
        
      } else if (sortField === GOODIES_SORT_FIELDS.GOODIES_STATUS) {
        primaryValueA = formatGoodiesStatus(a?.goodies[0], a?.allocatedProgram?.id, a?.approvals?.[0]?.approvalStatus)
        primaryValueB = formatGoodiesStatus(b?.goodies[0], b?.allocatedProgram?.id, b?.approvals?.[0]?.approvalStatus)
        secondaryValueA = '';
        secondaryValueB = '';
      } else if (sortField === GOODIES_SORT_FIELDS.FLASK) {
        primaryValueA = a?.goodies?.[0] ? formatBooleanValue(a.goodies[0].flask) : '';
        primaryValueB = b?.goodies?.[0] ? formatBooleanValue(b.goodies[0].flask) : '';
        secondaryValueA = '';
        secondaryValueB = '';
      } else if (sortField === GOODIES_SORT_FIELDS.NOTEBOOK) {
        primaryValueA = a?.goodies?.[0] ? formatBooleanValue(a.goodies[0].notebook) : '';
        primaryValueB = b?.goodies?.[0] ? formatBooleanValue(b.goodies[0].notebook) : '';
        secondaryValueA = '';
        secondaryValueB = '';
      } else if (sortField === GOODIES_SORT_FIELDS.RATRIA_PILLARS) {
        primaryValueA = a?.goodies?.[0]?.ratriaPillarLeonia !== undefined 
          ? formatRatriaPillar(a.goodies[0].ratriaPillarLeonia, a.goodies[0].ratriaPillarLocation, a.goodies[0].ratriaPillarOtherLocation)
          : '';
        primaryValueB = b?.goodies?.[0]?.ratriaPillarLeonia !== undefined 
          ? formatRatriaPillar(b.goodies[0].ratriaPillarLeonia, b.goodies[0].ratriaPillarLocation, b.goodies[0].ratriaPillarOtherLocation)
          : '';
        secondaryValueA = '';
        secondaryValueB = '';
        
        // Use custom Ratria Pillar sorting instead of string comparison
        const pillarComparison = sortRatriaPillar(primaryValueA, primaryValueB);
        return normalizedSortOrder === 'ASC' ? pillarComparison : -pillarComparison;
      } else if (sortField === GOODIES_SORT_FIELDS.JACKET) {
        primaryValueA = a?.goodies?.[0]?.jacket !== undefined 
          ? formatSizeSelection(a.goodies[0].jacket, a.goodies[0].jacketSize)
          : '';
        primaryValueB = b?.goodies?.[0]?.jacket !== undefined 
          ? formatSizeSelection(b.goodies[0].jacket, b.goodies[0].jacketSize)
          : '';
        secondaryValueA = '';
        secondaryValueB = '';
        
        // Use custom size sorting instead of string comparison
        const sizeComparison = sortSizes(primaryValueA, primaryValueB);
        return normalizedSortOrder === 'ASC' ? sizeComparison : -sizeComparison;
        
      } else if (sortField === GOODIES_SORT_FIELDS.TSHIRT) {
        primaryValueA = a?.goodies?.[0]?.tshirt !== undefined 
          ? formatSizeSelection(a.goodies[0].tshirt, a.goodies[0].tshirtSize)
          : '';
        primaryValueB = b?.goodies?.[0]?.tshirt !== undefined 
          ? formatSizeSelection(b.goodies[0].tshirt, b.goodies[0].tshirtSize)
          : '';
        secondaryValueA = '';
        secondaryValueB = '';
        
        // Use custom size sorting instead of string comparison
        const sizeComparison = sortSizes(primaryValueA, primaryValueB);
        return normalizedSortOrder === 'ASC' ? sizeComparison : -sizeComparison;
      } else if (sortField === 'allStatusDate') {
        // Custom handling for allStatusDate field - using shared utility function
        const approvalRecordA = a?.approvals?.[0];
        const latestSwapRequestA = a?.swapsRequests && a.swapsRequests.length > 0 
          ? [...a.swapsRequests].sort((c, d) => d.id - c.id)[0] 
          : null;
        
        const approvalRecordB = b?.approvals?.[0];
        const latestSwapRequestB = b?.swapsRequests && b.swapsRequests.length > 0 
          ? [...b.swapsRequests].sort((c, d) => d.id - c.id)[0] 
          : null;
        
        primaryValueA = getRegistrationStatusDateTime(a, approvalRecordA, latestSwapRequestA);
        primaryValueB = getRegistrationStatusDateTime(b, approvalRecordB, latestSwapRequestB);
        secondaryValueA = '';
        secondaryValueB = '';
        // Primary comparison
        let comparison;
        if (normalizedSortOrder === 'ASC') {
          comparison = new Date(primaryValueA).getTime() - new Date(primaryValueB).getTime();
        } else {
          comparison = new Date(primaryValueB).getTime() - new Date(primaryValueA).getTime();
        }
        return comparison;
      } else {
        // Generic nested field handling (no secondary sort)
        const fieldParts = sortField.split('.');
        primaryValueA = fieldParts.reduce((obj, key) => obj?.[key], a) || '';
        primaryValueB = fieldParts.reduce((obj, key) => obj?.[key], b) || '';
        secondaryValueA = '';
        secondaryValueB = '';
      }
      
      // Convert to string and handle null/undefined for primary sort
      const primaryStrA = String(primaryValueA || '').toLowerCase();
      const primaryStrB = String(primaryValueB || '').toLowerCase();
      
      // Primary comparison
      let primaryComparison;
      if (normalizedSortOrder === 'ASC') {
        primaryComparison = primaryStrA.localeCompare(primaryStrB);
      } else {
        primaryComparison = primaryStrB.localeCompare(primaryStrA);
      }
      
      // If primary values are equal, use secondary sort
      if (primaryComparison === 0 && secondaryValueA !== '' && secondaryValueB !== '') {
        const secondaryStrA = String(secondaryValueA || '').toLowerCase();
        const secondaryStrB = String(secondaryValueB || '').toLowerCase();
        
        if (normalizedSortOrder === 'ASC') {
          return secondaryStrA.localeCompare(secondaryStrB);
        } else {
          return secondaryStrB.localeCompare(secondaryStrA);
        }
      }
      
      return primaryComparison;
    });
  }

  /**
   * Helper method to determine required relations based on parsed filters and sort key
   */
  private getRequiredRelationsForFilters(parsedFilters: Record<string, any>, sortKey: string): string[] {
    const relations: string[] = [];
    
    // Add relations based on filters
    if (parsedFilters?.approvalStatus) {
      relations.push('approvals');
    }
    
    if (parsedFilters?.paymentStatus || parsedFilters?.paymentMode) {
      relations.push('paymentDetails');
    }
    
    if (parsedFilters?.invoiceStatus) {
      relations.push('invoiceDetails', 'invoiceDetails.updatedBy');
    }
    
    if (parsedFilters?.travelStatus || parsedFilters?.travelPlan) {
      relations.push('travelInfo', 'travelPlans');
    }
    
    if (parsedFilters?.goodiesStatus) {
      relations.push('goodies', 'approvals');
    }
    
    if (parsedFilters?.swapRequests || parsedFilters?.swapPreferredProgramId || parsedFilters?.swapDemandPreferredProgramId) {
      relations.push('swapsRequests', 'swapsRequests.swapRequestedProgram', 'swapsRequests.swapRequestedProgram.program');
    }
    
    if (parsedFilters?.preferredProgramId || parsedFilters?.preferredSessionId || parsedFilters?.preferredProgram) {
      relations.push('preferences', 'preferences.preferredProgram', 'preferences.preferredSession');
    }
    
    if (parsedFilters?.allocatedProgramId || parsedFilters?.allocatedSessionId) {
      relations.push('allocatedProgram', 'allocatedSession');
    }
    
    if (parsedFilters?.rmContact || parsedFilters?.rmRating) {
      relations.push('rmContactUser', 'ratings');
    }
    
    if (parsedFilters?.recommendation) {
      relations.push('recommendation');
    }
    
    if (parsedFilters?.experienceTags) {
      relations.push('user', 'user.programExperiences', 'user.programExperiences.lookupData', 'user.participationSummary');
    }
    
    if (parsedFilters?.organisation) {
      relations.push('user');
    }
    
    if (parsedFilters?.createdBy) {
      relations.push('createdBy');
    }
    
    if (parsedFilters?.programStatus || parsedFilters?.programType) {
      relations.push('program', 'program.type');
    }
    
    // Additional missing filters
    if (parsedFilters?.mahatriaChoice) {
      relations.push('preferences');
    }
    
    // Age, numberOfHdbs, birthdayListRange, and location use direct fields on registration table - no additional relations needed
    // (These filters use reg.dob, reg.no_of_hdbs, and reg.city respectively)
    
    if (parsedFilters?.freeSeat) {
      relations.push('allocatedProgram');
    }
    
    if (parsedFilters?.preferredRoomMate) {
      // No additional relations needed - this is a direct field on registration
    }
    
    if (parsedFilters?.pendingProgramId) {
      relations.push('preferences', 'preferences.preferredProgram');
    }
    
    // KPI category filters
    if (parsedFilters?.kpiCategory === 'swapRequests' || parsedFilters?.kpiFilter === 'swapRequests' || 
        parsedFilters?.kpiFilter?.startsWith('swapRequests_program_')) {
      relations.push('swapsRequests', 'swapsRequests.swapRequestedProgram', 'swapsRequests.swapRequestedProgram.program');
    }
    
    if (parsedFilters?.kpiCategory === 'swapDemand' || parsedFilters?.kpiFilter === 'swapDemand' || 
        parsedFilters?.kpiFilter?.startsWith('onHold_program_')) {
      relations.push('preferences', 'preferences.preferredProgram', 'approvals');
    }
    
    if (parsedFilters?.kpiCategory === 'regPending' || parsedFilters?.kpiFilter === 'regPending' || 
        parsedFilters?.kpiFilter?.startsWith('regPending_program_')) {
      relations.push('preferences', 'preferences.preferredProgram', 'approvals');
    }
    
    // Add relations based on sort key
    const sortKeyMapping: Record<string, string[]> = {
      'rmContact': ['rmContactUser'],
      'travelPlanStatus': ['travelInfo', 'travelPlans'],
      'goodiesStatus': ['goodies', 'allocatedProgram', 'approvals'],
      'flask': ['goodies'],
      'notebook': ['goodies'],
      'ratriaPillar': ['goodies'],
      'jacket': ['goodies'],
      'tshirt': ['goodies'],
      'blessedWith': ['allocatedProgram', 'approvals'],
      'allocatedProgram.id': ['allocatedProgram', 'approvals'],
      'blessedDate': ['approvals'],
      'holdDate': ['approvals'],
      'swapDemandDate': ['approvals'],
      'swapRequestDate': ['swapsRequests'],
      'pendingDate': ['approvals'],
      'allStatusDate': ['approvals', 'swapsRequests']
    };
    
    if (sortKeyMapping[sortKey]) {
      relations.push(...sortKeyMapping[sortKey]);
    }
    
    return relations;
  }

  /**
   * Helper method to determine if a sort key requires custom sorting logic
   * With enhanced nested sorting in CommonDataService, most fields can be handled at DB level
   */
  private requiresCustomSorting(sortKey: string): boolean {
    // Only use custom sorting for fields that require complex business logic
    const customSortFields = [
      'rmContactUser.orgUsrName',
      'allocatedProgram.id',
      'travelPlans.travelPlanStatus',
      'goodies.goodiesStatus',
      'goodies.ratriaPillars',
      'goodies.jacket',
      'goodies.tshirt',
      'allStatusDate'
    ];
    
    return customSortFields.includes(sortKey);
  }

  /**
   * Helper method to get required relations for sorting
   */
  private getRequiredRelationsForSorting(sortKey: string): string[] {
    const relationMapping: Record<string, string[]> = {
      'rmContactUser.orgUsrName': ['rmContactUser'],
      'allocatedProgram.id': ['allocatedProgram', 'approvals'],
      'approvals.approvalDate': ['approvals'],
      'travelPlans.travelPlanStatus': ['travelInfo', 'travelPlans'],
      'goodies.goodiesStatus': ['goodies', 'allocatedProgram', 'approvals'],
      'goodies.flask': ['goodies'],
      'goodies.notebook': ['goodies'],
      'goodies.ratriaPillars': ['goodies'],
      'goodies.jacket': ['goodies'],
      'goodies.tshirt': ['goodies'],
      'holdDate': ['approvals'],
      'swapDemandDate': ['approvals'],
      'swapRequestDate': ['swapsRequests'],
      'blessedDate': ['approvals'],
      'pendingDate': ['approvals'],
      'allStatusDate': ['approvals', 'swapsRequests']
    };
    
    return relationMapping[sortKey] || [];
  }

  async findRegistrationById(id: number) {
    try {
      const registration = await this.commonDataService.findOneById(
        this.registrationRepo,
        id,
        true,
        [
          'program',
          'program.type',
          'programSession',
          'invoiceDetails',
          'paymentDetails',
          'paymentDetails.editRequests',
          'travelInfo',
          'travelPlans',
          'user',
          'user.programExperiences',
          'user.programExperiences.lookupData',
          'user.profileExtension',
          'approvals',
          'preferences',
          'preferences.preferredProgram',
          'preferences.preferredSession',
          'ratings',
          'allocatedProgram',
          'allocatedSession',
          'swapsRequests',
          'swapsRequests.currentProgram',
          'swapsRequests.requestedPrograms',
          'swapsRequests.swapRequestedProgram',
          'swapsRequests.swapRequestedProgram.program',
          'rmContactUser',
          'goodies',
          'recommendation'
        ]
      );
      registration?.paymentDetails?.forEach(payment => {
        if (payment.editRequests && payment.editRequests.length > 0) {
          // Keep only the editRequest with the highest id
          const maxIdRequest = payment.editRequests.reduce((max: any, curr: any) => {
            return (!max || curr.id > max.id) ? curr : max;
          }, null);
          payment.editRequests = maxIdRequest ? [maxIdRequest] : [];
        }
      });
      
      // Sort swap requested programs by insertion order
      sortSwapRequestedPrograms(registration);
      
      if (!registration) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          id.toString(),
        );
      }
      return registration;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  async findQuestionResponsesByRegistrationId(registrationId: number) {
    try {
      const answers = await this.questionAnswerRepo.find({
        where: { registrationId },
        relations: [
          'question',
          'question.formSection',
          'registration',
          'registration.program',
        ],
        order: { id: 'ASC' },
      });

      const programId = answers[0].registration?.program?.id;
      let mapping: any = undefined;
      if (programId) {
        mapping = await this.programQuestionRepo.find({
          where: {
            program: { id: programId },
            deletedAt: IsNull(),
          },
          relations: ['question'],
        });
      }

      const responsePromises = answers.map(async a => {

        const section =  a.question?.formSection || null;
        const order = mapping?.find(m => m.question.id === a.questionId)?.displayOrder || 0;

        return {
          questionId: a.questionId,
          questionLabel: a.question?.label,
          config: a.question?.config,
          questionType: a.question?.type,
          sectionId: section?.id,
          sectionName: section?.name,
          sectionOrder: section?.displayOrder || 0,
          displayOrder: order,
          sectionKey: section?.key,
          answer: a.answerValue,
          questionBindingKey: a.question?.bindingKey
        };
      });

      const responses = await Promise.all(responsePromises);
      responses.sort((a, b) => {
        if (a.sectionOrder !== b.sectionOrder) {
          return (a.sectionOrder ?? 0) - (b.sectionOrder ?? 0);
        }
        return (a.displayOrder ?? 0) - (b.displayOrder ?? 0);
      });
      return responses;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_QUESTION_RESPONSE_GET_FAILED, error);
    }
  }

  // ===== KPI QUERY METHODS FOR SERVICE LAYER =====

  async findGroupedPrograms(primaryProgramId: number) {
    return await this.programRepo.find({
      where: { 
        primaryProgramId: primaryProgramId,
        deletedAt: IsNull()
      },
      select: ['id', 'name', 'totalSeats','startsAt','totalBedCount'],
      order: { groupDisplayOrder: 'ASC' }
    });
  }

  async getApprovalMetrics(programId: number, parsedFilters?: Record<string, any>, searchText?: string) {
    this.logger.debug(`Getting approval metrics for programId: ${programId}`);
    
    try {
      if (!programId) {
        this.logger.warn('Invalid programId provided for getApprovalMetrics');
        throw new InifniBadRequestException('Invalid programId provided');
      }

      const approvalQuery = this.dataSource
        .getRepository(RegistrationApproval)
        .createQueryBuilder('approval')
        .leftJoin('approval.registration', 'registration')
        .leftJoin('registration.user', 'user')
        .select([
          `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.ON_HOLD}' THEN 1 END) AS "holdCount"`,
          `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.REJECTED}' THEN 1 END) AS "rejectedCount"`,
          `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.ON_HOLD}' AND user.userType = '${UserTypeEnum.ORG}' THEN 1 END) AS "holdOrgCount"`,
          `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.REJECTED}' AND user.userType = '${UserTypeEnum.ORG}' THEN 1 END) AS "rejectedOrgCount"`
        ])
        .addSelect('registration.gender AS "gender"')
        .where('registration.program_id = :programId', { programId })
        .andWhere('approval.deleted_at IS NULL')
        .andWhere('registration.deleted_at IS NULL')
        .groupBy('registration.gender');

      const results = await approvalQuery.getRawMany();
      this.logger.debug(`Retrieved ${results.length} approval metrics for programId: ${programId}`);
      
      return results;
    } catch (error) {
      this.logger.error('Error in getApprovalMetrics', error?.stack);
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  async getProgramPreferencesMetrics(programId: number, approvalStatus: string) {
    try {
      // Validate inputs
      if (!programId || !approvalStatus) {
        throw new Error('programId and approvalStatus are required');
      }

      // First, get all sub-programs under the given programId
      const subPrograms = await this.programRepo.find({
        where: {
          primaryProgramId: programId,
          deletedAt: IsNull(),
        },
        select: ['id', 'name'],
        order: { groupDisplayOrder: 'ASC' }
      });

      if (subPrograms.length === 0) {
        return [];
      }

      const subProgramIds = subPrograms.map(program => program.id);

      // Optimized single query to get preference counts
      // This query counts registrations that have a first priority preference for each sub-program
      const preferenceCountQuery = this.dataSource
        .getRepository(Preference)
        .createQueryBuilder('preference')
        .leftJoin('preference.registration', 'registration')
        .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
        .select([
          'preference.preferred_program_id AS "preferredProgramId"',
          'COUNT(DISTINCT preference.registration_id) AS "count"'
        ])
        .where('preference.preferred_program_id IN (:...subProgramIds)', { subProgramIds })
        .andWhere('preference.priority_order = 1') // Only first priority preferences
        .andWhere('preference.deleted_at IS NULL')
        .andWhere('registration.deleted_at IS NULL')
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
        .andWhere('approval.approval_status = :approvalStatus', { approvalStatus })
        .setParameters({
          excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
        })
        .groupBy('preference.preferred_program_id');

      const preferenceCounts = await preferenceCountQuery.getRawMany();

      // Create a map for quick lookup
      const countMap = preferenceCounts.reduce((acc, item) => {
        acc[item.preferredProgramId] = parseInt(item.count, 10);
        return acc;
      }, {} as Record<number, number>);

      // Map results to include all sub-programs (even those with 0 count)
      return subPrograms.map((program) => ({
        programId: program.id,
        programName: program.name,
        count: countMap[program.id] || 0,
      }));

    } catch (error) {
      this.logger.error(`Error in getProgramPreferencesMetrics for programId: ${programId}, approvalStatus: ${approvalStatus}`, error?.stack);
      throw new Error(`Failed to get program preferences metrics: ${error.message}`);
    }
  }

  async getBlessedCount(programId: number) {
    try {
      const result = await this.dataSource
        .getRepository(ProgramRegistration)
        .createQueryBuilder('registration')
        .leftJoin('registration.user', 'user')
        .leftJoin('registration.swapsRequests', 'swapsRequests', 'swapsRequests.status = :swapStatus')
        .setParameter('swapStatus', SwapRequestStatus.ACTIVE)
        .select([
          'COUNT(DISTINCT registration.id) AS "blessedCount"',
          `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.MALE}' THEN registration.id END) AS "maleCount"`,
          `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.FEMALE}' THEN registration.id END) AS "femaleCount"`,
          `COUNT(DISTINCT CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "orgCount"`,
          `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.MALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "maleOrganisationUserCount"`,
          `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.FEMALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "femaleOrganisationUserCount"`,
          `COUNT(CASE WHEN registration.preferredRoomMate IS NOT NULL THEN 1 END) AS "roomMatePreferenceCount"`,
          `COUNT(DISTINCT CASE WHEN swapsRequests.id IS NOT NULL THEN registration.id END) AS "swapRequestsCount"`
        ])
        .where('registration.program_id = :programId', { programId })
        .andWhere('registration.allocated_program_id IS NOT NULL')
        .andWhere('registration.deleted_at IS NULL')
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
        .setParameters({
          excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
        })
        .getRawOne();

      return result;
    } catch (error) {
      this.logger.error(`Error in getBlessedCount for programId: ${programId}`, error?.stack);
      throw new Error(`Failed to get blessed count: ${error.message}`);
    }
  }

  async getMahatriaChoiceMetrics(programId: number, parsedFilters?: Record<string, any>, searchText?: string) {
    let mahatriaChoiceUnallocatedQuery = this.dataSource
      .getRepository(ProgramRegistration)
      .createQueryBuilder('registration')
      .leftJoin('registration.preferences', 'preference')
      .leftJoin('registration.user', 'user')
      .select([
        'COUNT(DISTINCT registration.id) AS "count"',
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.MALE}' THEN registration.id END) AS "maleCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.FEMALE}' THEN registration.id END) AS "femaleCount"`,
        `COUNT(DISTINCT CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "orgCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.MALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "maleOrganizationUserCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.FEMALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "femaleOrganizationUserCount"`
      ])
      .where('registration.program_id = :programId', { programId })
      .andWhere('registration.allocated_program_id IS NULL')
      .andWhere('preference.id IS NULL') // No preferences exist
      .andWhere('registration.deleted_at IS NULL')
      .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
      .setParameters({
        excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
      });

    mahatriaChoiceUnallocatedQuery = this.applyFiltersToQuery(mahatriaChoiceUnallocatedQuery, {
      approvalStatus: ApprovalStatusEnum.PENDING,
    }, searchText || '', 'registration');

    return await mahatriaChoiceUnallocatedQuery.getRawOne();
  }

  async getAllocatedMetrics(programId: number, allocatedProgramId: number, parsedFilters?: Record<string, any>, searchText?: string) {
    let allocatedQuery = this.dataSource
      .getRepository(ProgramRegistration)
      .createQueryBuilder('registration')
      .leftJoin('registration.user', 'user')
      .leftJoin('registration.swapsRequests', 'swapsRequests', 'swapsRequests.status = :swapStatus')
      .setParameter('swapStatus', SwapRequestStatus.ACTIVE)
      .select([
        'COUNT(registration.id) AS "allocatedCount"',
        `COUNT(CASE WHEN registration.gender = '${GenderEnum.MALE}' THEN 1 END) AS "maleCount"`,
        `COUNT(CASE WHEN registration.gender = '${GenderEnum.FEMALE}' THEN 1 END) AS "femaleCount"`,
        `COUNT(CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN 1 END) AS "orgCount"`,
        `COUNT(CASE WHEN registration.gender = '${GenderEnum.MALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN 1 END) AS "maleOrganizationUserCount"`,
        `COUNT(CASE WHEN registration.gender = '${GenderEnum.FEMALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN 1 END) AS "femaleOrganizationUserCount"`,
        `COUNT(CASE WHEN registration.preferredRoomMate IS NOT NULL THEN 1 END) AS "roomMatePreferenceCount"`,
        `COUNT(DISTINCT CASE WHEN swapsRequests.id IS NOT NULL THEN registration.id END) AS "swapRequestsCount"`,
      ])
      .where('registration.program_id = :programId', { programId })
      .andWhere('registration.allocated_program_id = :allocatedProgramId', {
        allocatedProgramId: allocatedProgramId
      })
      .andWhere('registration.deleted_at IS NULL')
      .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
      .setParameters({
        excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
      });

    allocatedQuery = this.applyFiltersToQuery(allocatedQuery, {
      approvalStatus: ApprovalStatusEnum.APPROVED,
    }, searchText, 'registration');

    return await allocatedQuery.getRawOne();
  }

  async getNonAllocatedMetrics(programId: number, parsedFilters?: Record<string, any>, searchText?: string) {
    let nonAllocatedQuery = this.dataSource
      .getRepository(Preference)
      .createQueryBuilder('preference')
      .leftJoin('preference.registration', 'registration')
      .leftJoin('registration.user', 'user')
      .select([
        'COUNT(DISTINCT preference.id) AS "count"',
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.MALE}' THEN preference.id END) AS "maleCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.FEMALE}' THEN preference.id END) AS "femaleCount"`,
        `COUNT(DISTINCT CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN preference.id END) AS "orgCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.MALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN preference.id END) AS "maleOrganizationUserCount"`,
        `COUNT(DISTINCT CASE WHEN registration.gender = '${GenderEnum.FEMALE}' AND user.userType = '${UserTypeEnum.ORG}' THEN preference.id END) AS "femaleOrganizationUserCount"`
      ])
      .where('preference.preferred_program_id = :programId', {
        programId: programId
      })
      .andWhere('preference.priority_order = 1')
      .andWhere('registration.allocated_program_id IS NULL')
      .andWhere('preference.deleted_at IS NULL')
      .andWhere('registration.deleted_at IS NULL')
      .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
      .setParameters({
        excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
      });

    nonAllocatedQuery = this.applyFiltersToQuery(nonAllocatedQuery, {
      approvalStatus: ApprovalStatusEnum.PENDING,
    }, searchText, 'registration');

    return await nonAllocatedQuery.getRawOne();
  }

  async getTotalRegistrations(programId: number, parsedFilters?: Record<string, any>, searchText?: string) {
    this.logger.debug(`Getting total registrations for programId: ${programId}`);
    
    try {
      if (!programId) {
        this.logger.warn('Invalid programId provided for getTotalRegistrations');
        throw new InifniBadRequestException('Invalid programId provided');
      }

      let totalRegistrationsQuery = this.registrationRepo
        .createQueryBuilder('registration')
        .where('registration.program_id = :programId', { programId })
        .andWhere('registration.deleted_at IS NULL')
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
        .setParameters({
          excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
        });

      totalRegistrationsQuery = this.applyFiltersToQuery(totalRegistrationsQuery, parsedFilters, searchText, 'registration');

      const count = await totalRegistrationsQuery.getCount();
      this.logger.debug(`Retrieved total registration count: ${count} for programId: ${programId}`);
      
      return count;
    } catch (error) {
      this.logger.error('Error in getTotalRegistrations', error?.stack);
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  async getTotalBlessedRegistrations(programId: number, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let totalBlessedRegistrationsQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
      .leftJoin('registration.paymentDetails', 'payment', 'payment.deleted_at IS NULL')
      .leftJoin('registration.invoiceDetails', 'invoice', 'invoice.deleted_at IS NULL')
      .leftJoin('registration.travelInfo', 'travelInfo', 'travelInfo.deleted_at IS NULL')
      .where('registration.program_id = :programId', { programId })
      .andWhere('registration.allocated_program_id IS NOT NULL')  // Key condition for blessed
      .andWhere('approval.approval_status = :approvalStatus', { approvalStatus: ApprovalStatusEnum.APPROVED })
      .andWhere('registration.deleted_at IS NULL');

    totalBlessedRegistrationsQuery = this.applyUserFilterToQuery(totalBlessedRegistrationsQuery, filterRegistrationsByUserId ?? null);
    totalBlessedRegistrationsQuery = this.applyFiltersToQuery(totalBlessedRegistrationsQuery, parsedFilters, searchText, 'registration');

    const [query, parameters] = totalBlessedRegistrationsQuery.getQueryAndParameters();
    this.logger.log('Blessed Query Parameters:', parameters);
    this.logger.log('Blessed Query Filters:', parsedFilters);

    return await totalBlessedRegistrationsQuery.getCount();
  }

  async getTotalBlessedRegistrationsWithOrg(programId: number, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let totalBlessedRegistrationsQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
      .leftJoinAndSelect('registration.user', 'user')
      .leftJoin('registration.paymentDetails', 'payment', 'payment.deleted_at IS NULL')
      .leftJoin('registration.invoiceDetails', 'invoice', 'invoice.deleted_at IS NULL')
      .leftJoin('registration.travelInfo', 'travelInfo', 'travelInfo.deleted_at IS NULL')
      .select([
        `COUNT(DISTINCT registration.id) AS "totalCount"`,
        `COUNT(DISTINCT CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "orgCount"`
      ])
      .where('registration.program_id = :programId', { programId })
      .andWhere('registration.allocated_program_id IS NOT NULL')  // Key condition for blessed
      .andWhere('approval.approval_status = :approvalStatus', { approvalStatus: ApprovalStatusEnum.APPROVED })
      .andWhere('registration.deleted_at IS NULL');

    totalBlessedRegistrationsQuery = this.applyUserFilterToQuery(totalBlessedRegistrationsQuery, filterRegistrationsByUserId ?? null);
    totalBlessedRegistrationsQuery = this.applyFiltersToQuery(totalBlessedRegistrationsQuery, parsedFilters, searchText, 'registration');

    const [query, parameters] = totalBlessedRegistrationsQuery.getQueryAndParameters();
    this.logger.log('Blessed with Org Query Parameters:', parameters);

    return await totalBlessedRegistrationsQuery.getRawOne();
  }
  
  async getTotalAllocatedRegistrations(programId: number, allocatedProgramId: number, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let totalAllocatedRegistrationsQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
      .leftJoin('registration.paymentDetails', 'payment', 'payment.deleted_at IS NULL')
      .leftJoin('registration.invoiceDetails', 'invoice', 'invoice.deleted_at IS NULL')
      .leftJoin('registration.travelInfo', 'travelInfo', 'travelInfo.deleted_at IS NULL')
      .where('registration.program_id = :programId', { programId })
      .andWhere('registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId })
      .andWhere('approval.approval_status = :approvalStatus', { approvalStatus: ApprovalStatusEnum.APPROVED })
      .andWhere('registration.deleted_at IS NULL')
      .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
      .setParameters({
        excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
      });

    totalAllocatedRegistrationsQuery = this.applyUserFilterToQuery(totalAllocatedRegistrationsQuery, filterRegistrationsByUserId ?? null);
    totalAllocatedRegistrationsQuery = this.applyFiltersToQuery(totalAllocatedRegistrationsQuery, parsedFilters, searchText, 'registration');

    const [query, parameters] = totalAllocatedRegistrationsQuery.getQueryAndParameters();
    this.logger.log('Query Parameters:', parameters);

    return await totalAllocatedRegistrationsQuery.getCount();
  }


    async getTotalAllocatedRegistrationsWithOrg(programId: number, allocatedProgramId: number, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let totalAllocatedRegistrationsQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
      .leftJoinAndSelect('registration.user', 'user')
      .leftJoin('registration.paymentDetails', 'payment', 'payment.deleted_at IS NULL')
      .leftJoin('registration.invoiceDetails', 'invoice', 'invoice.deleted_at IS NULL')
      .leftJoin('registration.travelInfo', 'travelInfo', 'travelInfo.deleted_at IS NULL')
      .select([
        `COUNT(DISTINCT registration.id) AS "totalCount"`,
        `COUNT(DISTINCT CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "orgCount"`
      ])
      .where('registration.program_id = :programId', { programId })
      .andWhere('registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId })
      .andWhere('approval.approval_status = :approvalStatus', { approvalStatus: ApprovalStatusEnum.APPROVED })
      .andWhere('registration.deleted_at IS NULL')
      .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
      .setParameters({
        excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
      });

    totalAllocatedRegistrationsQuery = this.applyUserFilterToQuery(totalAllocatedRegistrationsQuery, filterRegistrationsByUserId ?? null);
    totalAllocatedRegistrationsQuery = this.applyFiltersToQuery(totalAllocatedRegistrationsQuery, parsedFilters, searchText, 'registration');

    const [query, parameters] = totalAllocatedRegistrationsQuery.getQueryAndParameters();
    this.logger.log('Query Parameters:', parameters);

    return await totalAllocatedRegistrationsQuery.getRawOne();
  }
  
  async getTotalSwapRequests(programId: number, allocatedProgramId?: number | null, filterRegistrationsByUserId?: number | null) {
    try {
      if (!programId || programId <= 0) {
        this.logger.warn(`Invalid programId provided to getTotalSwapRequests: ${programId}`);
        throw new InifniBadRequestException(`Invalid programId: ${programId}`);
      }

      // Subquery for latest ACTIVE swap demand per registration
      const latestSwapIdSubQuery = `(SELECT sr.id FROM hdb_program_registration_swap sr WHERE sr.program_registration_id = registration.id AND sr.status = '${SwapRequestStatus.ACTIVE}' AND sr.type = '${SwapType.WantsSwap}' ORDER BY sr.id DESC LIMIT 1)`;
      // Subquery for first requested program for the latest swap
      const firstRequestedProgramSubQuery = `(SELECT srp.program_id FROM hdb_swap_requested_program srp WHERE srp.swap_request_id = ${latestSwapIdSubQuery} ORDER BY srp.id ASC LIMIT 1)`;
      let qb = this.dataSource
        .getRepository('hdb_program_registration')
        .createQueryBuilder('registration')
        .leftJoin('users', 'user', `user.id = registration.user_id`)
        .leftJoin('hdb_registration_approval', 'approval', `approval.registration_id = registration.id AND approval.deleted_at IS NULL`)
        .where('registration.program_id = :programId', { programId })
        .andWhere('registration.allocated_program_id IS NOT NULL')
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)', { excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED] })
        .andWhere('approval.approval_status NOT IN (:...excludeApprovalStatuses)', { excludeApprovalStatuses: [ApprovalStatusEnum.CANCELLED, ApprovalStatusEnum.REJECTED, ApprovalStatusEnum.ON_HOLD] })
        .andWhere('registration.deleted_at IS NULL')
        .andWhere(`(${latestSwapIdSubQuery}) IS NOT NULL`);
      if (allocatedProgramId) {
        qb.andWhere(`(${firstRequestedProgramSubQuery}) = :allocatedProgramId`, { allocatedProgramId });
        this.logger.debug(`Added allocatedProgramId filter: ${allocatedProgramId}`);
      }
      if (filterRegistrationsByUserId) {
        qb.andWhere('registration.rm_contact = :rmId', { rmId: filterRegistrationsByUserId });
        this.logger.debug(`Added RM filter: ${filterRegistrationsByUserId}`);
      }
      const count = await qb.getCount();
      this.logger.log(`Successfully retrieved ${count} swap demand registrations for programId: ${programId}, allocatedProgramId: ${allocatedProgramId}`);
      return count;
    } catch (error) {
      this.logger.error(`Error in getTotalSwapRequests for programId ${programId}, allocatedProgramId ${allocatedProgramId}: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  async getTotalSwapRequestsWithOrgCount(programId: number, allocatedProgramId?: number | null, filterRegistrationsByUserId?: number | null) {
    try {
      if (!programId || programId <= 0) {
        this.logger.warn(`Invalid programId provided to getTotalSwapRequestsWithOrgCount: ${programId}`);
        throw new InifniBadRequestException(`Invalid programId: ${programId}`);
      }

      // Subquery for latest ACTIVE swap demand per registration
      const latestSwapIdSubQuery = `(SELECT sr.id FROM hdb_program_registration_swap sr WHERE sr.program_registration_id = registration.id AND sr.status = '${SwapRequestStatus.ACTIVE}' AND sr.type = '${SwapType.WantsSwap}' ORDER BY sr.id DESC LIMIT 1)`;
      // Subquery for first requested program for the latest swap
      const firstRequestedProgramSubQuery = `(SELECT srp.program_id FROM hdb_swap_requested_program srp WHERE srp.swap_request_id = ${latestSwapIdSubQuery} ORDER BY srp.id ASC LIMIT 1)`;
      let qb = this.dataSource
        .getRepository('hdb_program_registration')
        .createQueryBuilder('registration')
        .leftJoin('users', 'user', `user.id = registration.user_id`)
        .leftJoin('hdb_registration_approval', 'approval', `approval.registration_id = registration.id AND approval.deleted_at IS NULL`)
        .select([
          `COUNT(DISTINCT registration.id) AS "totalCount"`,
          `COUNT(DISTINCT CASE WHEN user.user_type = 'Org' THEN registration.id END) AS "orgCount"`
        ])
        .where('registration.program_id = :programId', { programId })
        .andWhere('registration.allocated_program_id IS NOT NULL')
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)', { excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED] })
        .andWhere('approval.approval_status NOT IN (:...excludeApprovalStatuses)', { excludeApprovalStatuses: [ApprovalStatusEnum.CANCELLED, ApprovalStatusEnum.REJECTED, ApprovalStatusEnum.ON_HOLD] })
        .andWhere('registration.deleted_at IS NULL')
        .andWhere(`(${latestSwapIdSubQuery}) IS NOT NULL`);

      if (allocatedProgramId) {
        // need to check for requested program matching allocatedProgramId
        qb.andWhere(`(${firstRequestedProgramSubQuery}) = :allocatedProgramId`, { allocatedProgramId });
        this.logger.debug(`Added allocatedProgramId filter: ${allocatedProgramId}`);
      }
      if (filterRegistrationsByUserId) {
        qb.andWhere('registration.rm_contact = :rmId', { rmId: filterRegistrationsByUserId });
        this.logger.debug(`Added RM filter: ${filterRegistrationsByUserId}`);
      }
      const result = await qb.getRawOne();
      this.logger.log(`Successfully retrieved swap demand registration counts for programId: ${programId}, allocatedProgramId: ${allocatedProgramId}`);
      return result;
    } catch (error) {
      this.logger.error(`Failed to retrieve swap request counts for programId: ${programId}, allocatedProgramId: ${allocatedProgramId}`, error.stack);
      throw new InifniNotFoundException(`Failed to retrieve swap request counts`);
    }
  }

  async getTotalOnHoldRegistrationsWithSwapPreference(
    programId: number, 
    prefferedProgramId?: number | null, 
    parsedFilters?: Record<string, any>,
    searchText?: string,
    filterRegistrationsByUserId?: number | null
  ) {
    try {
      if (!programId) {
        this.logger.warn(`Invalid programId provided to getTotalOnHoldRegistrationsWithSwapPreference: ${programId}`);
        throw new InifniBadRequestException(`Invalid programId: ${programId}`);
      }

      // Use TypeORM query builder with subqueries for latest swap and first requested program
      const registrationAlias = 'registration';
      const userAlias = 'user';
      const approvalAlias = 'approval';
      // Subquery for latest ON_HOLD swap demand per registration
      const latestSwapIdSubQuery = `(SELECT sr.id FROM hdb_program_registration_swap sr WHERE sr.program_registration_id = registration.id AND sr.status = '${SwapRequestStatus.ON_HOLD}' AND sr.swap_requirement = '${SwapRequirementEnum.SWAP_DEMAND}' ORDER BY sr.id DESC LIMIT 1)`;

      // Subquery for first requested program for the latest swap
      const firstRequestedProgramSubQuery = `(SELECT srp.program_id FROM hdb_swap_requested_program srp WHERE srp.swap_request_id = ${latestSwapIdSubQuery} ORDER BY srp.id ASC LIMIT 1)`;

      let qb = this.dataSource
        .getRepository('hdb_program_registration')
        .createQueryBuilder(registrationAlias)
        .leftJoin('users', userAlias, `${userAlias}.id = ${registrationAlias}.user_id`)
        .leftJoin('hdb_registration_approval', approvalAlias, `${approvalAlias}.registration_id = ${registrationAlias}.id AND ${approvalAlias}.deleted_at IS NULL`)
        .select([
          `COUNT(DISTINCT ${registrationAlias}.id) AS "totalCount"`,
          `COUNT(DISTINCT CASE WHEN ${userAlias}.user_type = 'Org' THEN ${registrationAlias}.id END) AS "orgCount"`
        ])
        .where(`${registrationAlias}.program_id = :programId`, { programId })
        .andWhere(`${approvalAlias}.approval_status = :approvalStatus`, { approvalStatus: ApprovalStatusEnum.ON_HOLD })
        .andWhere(`${registrationAlias}.registration_status NOT IN (:...excludeStatuses)`, {
          excludeStatuses: [
            RegistrationStatusEnum.SAVE_AS_DRAFT,
            RegistrationStatusEnum.CANCELLED,
            RegistrationStatusEnum.REJECTED,
          ],
        })
        .andWhere(`${registrationAlias}.deleted_at IS NULL`)
        .andWhere(`(${latestSwapIdSubQuery}) IS NOT NULL`);

      if (prefferedProgramId) {
        qb.andWhere(`(${firstRequestedProgramSubQuery}) = :prefferedProgramId`, { prefferedProgramId });
        this.logger.debug(`Added preferredProgramId filter for swap demand preference: ${prefferedProgramId}`);
      }

      if (filterRegistrationsByUserId) {
        qb.andWhere(`${registrationAlias}.rm_contact = :rmContact`, { rmContact: filterRegistrationsByUserId });
        this.logger.debug(`Added RM filter: ${filterRegistrationsByUserId}`);
      }

      const result = await qb.getRawOne();

      this.logger.log(`Successfully retrieved on-hold registrations with swap demand preference for programId: ${programId}, prefferedProgramId: ${prefferedProgramId}`);

      return result || { totalCount: '0', orgCount: '0' };
    } catch (error) {
      this.logger.error(`Failed to retrieve on-hold registrations with swap demand preference for programId: ${programId}, prefferedProgramId: ${prefferedProgramId}`, error.stack);
      throw new InifniNotFoundException(`Failed to retrieve on-hold registrations with swap demand preference`);
    }
  }

  async getSeatsMetrics(programId?: number | null, programSessionId?: number | null, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
      .leftJoinAndSelect('registration.user', 'user')
      .select([
        `COUNT(DISTINCT registration.id) AS "all"`,
        `COUNT(CASE WHEN registration.registration_status = '${RegistrationStatusEnum.WAITLISTED}' THEN 1 END) AS "waitlisted"`,
        `COUNT(CASE WHEN registration.registration_status = '${RegistrationStatusEnum.CANCELLED}' THEN 1 END) AS "cancelled"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.PENDING}' THEN 1 END) AS "pending"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.REJECTED}' THEN 1 END) AS "rejected"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.ON_HOLD}' THEN 1 END) AS "onHold"`,
        `COUNT(CASE WHEN approval.approval_status = '${ApprovalStatusEnum.APPROVED}' THEN 1 END) AS "approved"`,
        `COUNT(CASE WHEN registration.registration_status = '${RegistrationStatusEnum.COMPLETED}' THEN 1 END) AS "completed"`,
        `COUNT(DISTINCT CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "allOrgCount"`,
        `COUNT(DISTINCT CASE WHEN approval.approval_status = '${ApprovalStatusEnum.PENDING}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "pendingOrgCount"`,
        `COUNT(DISTINCT CASE WHEN approval.approval_status = '${ApprovalStatusEnum.APPROVED}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "approvedOrgCount"`,
        `COUNT(DISTINCT CASE WHEN approval.approval_status = '${ApprovalStatusEnum.REJECTED}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "rejectedOrgCount"`,
        `COUNT(DISTINCT CASE WHEN approval.approval_status = '${ApprovalStatusEnum.ON_HOLD}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "onHoldOrgCount"`,
        `COUNT(DISTINCT CASE WHEN registration.registration_status = '${RegistrationStatusEnum.CANCELLED}' AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "cancelledOrgCount"`,
        `COUNT(DISTINCT CASE WHEN approval.approval_status IN ('${ApprovalStatusEnum.ON_HOLD}', '${ApprovalStatusEnum.REJECTED}') AND user.userType = '${UserTypeEnum.ORG}' THEN registration.id END) AS "registrationPendingOrgCount"`,
        `COUNT(DISTINCT CASE WHEN approval.approval_status IN ('${ApprovalStatusEnum.ON_HOLD}', '${ApprovalStatusEnum.REJECTED}') THEN registration.id END) AS "registrationPendingCount"`
      ])
      .where('registration.deleted_at IS NULL')
      .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
      .setParameters({
        excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT],
      });

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }

    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);
    // baseQuery = this.applyFiltersToQuery(baseQuery, parsedFilters, searchText, 'registration');
    return await baseQuery.getRawOne();
  }

  async getPaymentsMetrics(programId?: number | null, programSessionId?: number | null, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.paymentDetails', 'payment', 'payment.deleted_at IS NULL')
      .select([
        `COUNT(CASE WHEN payment.payment_status NOT IN ('${PaymentStatusEnum.DRAFT}', '${PaymentStatusEnum.SAVE_AS_DRAFT}') AND registration.is_free_seat = false THEN payment.id END) AS "all"`,
        `COUNT(CASE WHEN payment.payment_status IN ('${PaymentStatusEnum.ONLINE_PENDING}', '${PaymentStatusEnum.OFFLINE_PENDING}') AND registration.is_free_seat = false THEN payment.id END) AS "paymentPending"`,
        `COUNT(CASE WHEN payment.payment_status IN ('${PaymentStatusEnum.ONLINE_COMPLETED}', '${PaymentStatusEnum.OFFLINE_COMPLETED}') AND registration.is_free_seat = false THEN payment.id END) AS "paymentCompleted"`
      ])
      .where('registration.deleted_at IS NULL')
      .andWhere('payment.deleted_at IS NULL')
      .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
      .setParameters({
        excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
      });

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }

    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);
    baseQuery = this.applyFiltersToQuery(baseQuery, parsedFilters, searchText, 'registration');

    return await baseQuery.getRawOne();
  }

  async getInvoicesMetrics(programId?: number | null, programSessionId?: number | null, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.invoiceDetails', 'invoice', 'invoice.deleted_at IS NULL')
      .select([
        `COUNT(CASE WHEN invoice.invoice_status NOT IN ('${InvoiceStatusEnum.DRAFT}', '${InvoiceStatusEnum.SAVE_AS_DRAFT}') THEN invoice.id END) AS "all"`,
        `COUNT(CASE WHEN invoice.invoice_status = '${InvoiceStatusEnum.INVOICE_PENDING}' THEN invoice.id END) AS "invoicePending"`,
        `COUNT(CASE WHEN invoice.invoice_status = '${InvoiceStatusEnum.INVOICE_COMPLETED}' THEN invoice.id END) AS "invoiceCompleted"`
      ])
      .where('registration.deleted_at IS NULL')
      .andWhere('invoice.deleted_at IS NULL')
      .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
      .setParameters({
        excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
      });

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }

    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);
    baseQuery = this.applyFiltersToQuery(baseQuery, parsedFilters, searchText, 'registration');

    return await baseQuery.getRawOne();
  }

  async getTravelMetrics(programId?: number | null, programSessionId?: number | null, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.travelInfo', 'travelInfo', 'travelInfo.deleted_at IS NULL')
      .leftJoinAndSelect('registration.travelPlans', 'travelPlan', 'travelPlan.deleted_at IS NULL')
      .select([
        `COUNT(DISTINCT CASE WHEN travelInfo.travel_info_status IN ('${TravelStatusEnum.PENDING}', '${TravelStatusEnum.COMPLETED}') THEN registration.id END) AS "all"`,
        `COUNT(DISTINCT CASE WHEN travelInfo.travel_info_status = '${TravelStatusEnum.PENDING}' THEN registration.id END) AS "travelPending"`,
        `COUNT(DISTINCT CASE WHEN travelInfo.travel_info_status = '${TravelStatusEnum.COMPLETED}' THEN registration.id END) AS "travelCompleted"`,
        `COUNT(DISTINCT CASE WHEN travelPlan.travel_plan_status = '${TravelStatusEnum.PENDING}' THEN registration.id END) AS "logisticsPending"`,
        `COUNT(DISTINCT CASE WHEN travelPlan.travel_plan_status = '${TravelStatusEnum.COMPLETED}' THEN registration.id END) AS "logisticsCompleted"`
      ])
      .where('registration.deleted_at IS NULL')
      .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
      .setParameters({
        excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
      });

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }

    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);
    baseQuery = this.applyFiltersToQuery(baseQuery, parsedFilters, searchText, 'registration');

    return await baseQuery.getRawOne();
  }

  // ===== BLESSED-SPECIFIC METRICS METHODS =====

  async getBlessedPaymentsMetrics(programId?: number | null, programSessionId?: number | null, allocatedProgramId?: number | null, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.paymentDetails', 'payment', 'payment.deleted_at IS NULL')
      .leftJoinAndSelect('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
      .select([
        `COUNT(CASE WHEN (payment.payment_status IN ('${PaymentStatusEnum.ONLINE_PENDING}', '${PaymentStatusEnum.OFFLINE_PENDING}', '${PaymentStatusEnum.FAILED}') OR payment.id IS NULL) AND registration.is_free_seat = false  THEN registration.id END) AS "paymentPending"`,
        `COUNT(CASE WHEN payment.payment_status IN ('${PaymentStatusEnum.ONLINE_COMPLETED}', '${PaymentStatusEnum.OFFLINE_COMPLETED}') THEN registration.id END) AS "paymentCompleted"`
      ])
      .where('registration.deleted_at IS NULL')
      .andWhere('registration.allocated_program_id IS NOT NULL')
      .andWhere('approval.approval_status = :approvalStatus', { approvalStatus: ApprovalStatusEnum.APPROVED });

    // Filter by specific allocated program ID if provided
    if (allocatedProgramId) {
      baseQuery.andWhere('registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId });
    }

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }

    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);
    baseQuery = this.applyFiltersToQuery(baseQuery, parsedFilters, searchText, 'registration');

    const result = await baseQuery.getRawOne();
    return result;
  }

  async getBlessedInvoicesMetrics(programId?: number | null, programSessionId?: number | null, allocatedProgramId?: number | null, filterRegistrationsByUserId?: number | null) {
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.invoiceDetails', 'invoice', 'invoice.deleted_at IS NULL')
      .leftJoinAndSelect('registration.paymentDetails', 'payment', 'payment.deleted_at IS NULL')
      .select([
        `COUNT(CASE WHEN (invoice.invoice_status IN ('${InvoiceStatusEnum.INVOICE_PENDING}', '${InvoiceStatusEnum.DRAFT}')  OR invoice.id IS NULL) THEN registration.id END) AS "invoicePending"`,
        `COUNT(CASE WHEN invoice.invoice_status = '${InvoiceStatusEnum.INVOICE_COMPLETED}' THEN registration.id END) AS "invoiceCompleted"`
      ])
      .where('registration.deleted_at IS NULL')
      .andWhere('payment.payment_status IN (:...paymentStatus)', { paymentStatus: [PaymentStatusEnum.ONLINE_COMPLETED, PaymentStatusEnum.OFFLINE_COMPLETED] })
      .andWhere('registration.allocated_program_id IS NOT NULL')

    // Filter by specific allocated program ID if provided
    if (allocatedProgramId) {
      baseQuery.andWhere('registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId });
    }

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }
    
    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);

    return await baseQuery.getRawOne();
  }

  async getBlessedTravelMetrics(programId?: number | null, programSessionId?: number | null, allocatedProgramId?: number | null, parsedFilters?: Record<string, any>, searchText?: string, filterRegistrationsByUserId?: number | null) {
    let baseQuery = this.registrationRepo
      .createQueryBuilder('registration')
      .leftJoinAndSelect('registration.travelInfo', 'travelInfo', 'travelInfo.deleted_at IS NULL')
      .leftJoinAndSelect('registration.travelPlans', 'travelPlan', 'travelPlan.deleted_at IS NULL')
      .leftJoinAndSelect('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
      .select([
        `COUNT(DISTINCT CASE WHEN (travelInfo.travel_info_status = '${TravelStatusEnum.PENDING}' OR travelPlan.travel_plan_status = '${TravelStatusEnum.PENDING}' OR travelInfo.id IS NULL OR travelPlan.id IS NULL) AND approval.approval_status = '${ApprovalStatusEnum.APPROVED}' THEN registration.id END) AS "travelPending"`,
        `COUNT(DISTINCT CASE WHEN travelInfo.travel_info_status = '${TravelStatusEnum.COMPLETED}' AND travelPlan.travel_plan_status = '${TravelStatusEnum.COMPLETED}' AND approval.approval_status = '${ApprovalStatusEnum.APPROVED}' THEN registration.id END) AS "travelCompleted"`
      ])
      .where('registration.deleted_at IS NULL')
      .andWhere('registration.allocated_program_id IS NOT NULL')
      .andWhere('approval.approval_status = :approvalStatus', { approvalStatus: ApprovalStatusEnum.APPROVED });

    // Filter by specific allocated program ID if provided
    if (allocatedProgramId) {
      baseQuery.andWhere('registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId });
    }

    if (programId) {
      baseQuery.andWhere('registration.program_id = :programId', { programId });
    }
    if (programSessionId) {
      baseQuery.andWhere('registration.program_session_id = :programSessionId', { programSessionId });
    }

    baseQuery = this.applyUserFilterToQuery(baseQuery, filterRegistrationsByUserId ?? null);
    baseQuery = this.applyFiltersToQuery(baseQuery, parsedFilters, searchText, 'registration');

    return await baseQuery.getRawOne();
  }

  // ===== HELPER METHODS FOR FILTERS =====

  /**
   * Apply filters and search criteria to a query builder
   */
  private applyFiltersToQuery(
    queryBuilder: any,
    parsedFilters: Record<string, any> = {},
    searchText: string = '',
    tableAlias: string = 'registration'
  ): any {
    // registrationStatus filter
    const regStatus = parsedFilters?.registrationStatus;
    if (regStatus) {
      if (Array.isArray(regStatus)) {
        const validStatuses = regStatus.filter((s) => typeof s === 'string');
        if (validStatuses.length > 0) {
          queryBuilder.andWhere(`${tableAlias}.registration_status IN (:...registrationStatuses)`, {
            registrationStatuses: validStatuses
          });
        }
      } else if (typeof regStatus === 'string') {
        queryBuilder.andWhere(`${tableAlias}.registration_status = :registrationStatus`, {
          registrationStatus: regStatus
        });
      }
    }
  
    // createdBy filter
    if (parsedFilters?.createdBy) {
      queryBuilder.andWhere(`${tableAlias}.created_by = :createdBy`, {
        createdBy: parsedFilters.createdBy
      });
    }
  
    // allocatedProgramId filter (for blessed constraint)
    if (parsedFilters?.allocatedProgramId) {
      if (parsedFilters.allocatedProgramId === 'IS_NOT_NULL') {
        queryBuilder.andWhere(`${tableAlias}.allocated_program_id IS NOT NULL`);
      } else if (typeof parsedFilters.allocatedProgramId === 'number') {
        queryBuilder.andWhere(`${tableAlias}.allocated_program_id = :allocatedProgramId`, {
          allocatedProgramId: parsedFilters.allocatedProgramId
        });
      }
    }

    // approvalStatus filter
    const approvalStatus = parsedFilters?.approvalStatus;
    if (approvalStatus) {
      queryBuilder.leftJoin(`${tableAlias}.approvals`, 'approval_filter');
  
      if (Array.isArray(approvalStatus)) {
        const validApprovalStatuses = approvalStatus.filter((s) => typeof s === 'string');
        if (validApprovalStatuses.length > 0) {
          queryBuilder.andWhere('approval_filter.approval_status IN (:...approvalStatuses)', {
            approvalStatuses: validApprovalStatuses
          });
        }
      } else if (typeof approvalStatus === 'string') {
        queryBuilder.andWhere('approval_filter.approval_status = :approvalStatus', {
          approvalStatus
        });
      }
  
      queryBuilder.andWhere('approval_filter.deleted_at IS NULL');
    }

    // paymentDetails filter
    if (parsedFilters?.paymentDetails) {
      const paymentStatus = parsedFilters.paymentDetails.paymentStatus;
      if (paymentStatus) {
        if (Array.isArray(paymentStatus)) {
          const validPaymentStatuses = paymentStatus.filter((s) => typeof s === 'string');
          if (validPaymentStatuses.length > 0) {
            queryBuilder.andWhere('payment.payment_status IN (:...paymentStatuses)', {
              paymentStatuses: validPaymentStatuses
            });
            // Ensure we only include records that actually have payment details
            queryBuilder.andWhere('payment.id IS NOT NULL');
          }
        } else if (typeof paymentStatus === 'string') {
          queryBuilder.andWhere('payment.payment_status = :paymentStatus', {
            paymentStatus
          });
          // Ensure we only include records that actually have payment details
          queryBuilder.andWhere('payment.id IS NOT NULL');
        }
      }
    }

    // invoiceDetails filter
    if (parsedFilters?.invoiceDetails) {
      const invoiceStatus = parsedFilters.invoiceDetails.invoiceStatus;
      if (invoiceStatus) {
        if (Array.isArray(invoiceStatus)) {
          const validInvoiceStatuses = invoiceStatus.filter((s) => typeof s === 'string');
          if (validInvoiceStatuses.length > 0) {
            queryBuilder.andWhere('invoice.invoice_status IN (:...invoiceStatuses)', {
              invoiceStatuses: validInvoiceStatuses
            });
          }
        } else if (typeof invoiceStatus === 'string') {
          queryBuilder.andWhere('invoice.invoice_status = :invoiceStatus', {
            invoiceStatus
          });
        }
      }
    }

    // travelInfo filter
    if (parsedFilters?.travelInfo) {
      const travelInfoStatus = parsedFilters.travelInfo.travelInfoStatus;
      if (travelInfoStatus) {
        if (Array.isArray(travelInfoStatus)) {
          const validTravelStatuses = travelInfoStatus.filter((s) => typeof s === 'string');
          if (validTravelStatuses.length > 0) {
            queryBuilder.andWhere('travelInfo.travel_info_status IN (:...travelStatuses)', {
              travelStatuses: validTravelStatuses
            });
          }
        } else if (typeof travelInfoStatus === 'string') {
          queryBuilder.andWhere('travelInfo.travel_info_status = :travelInfoStatus', {
            travelInfoStatus
          });
        }
      }
    }
  
    // preferred program/session filters
    if (parsedFilters?.preferredProgramId || parsedFilters?.preferredSessionId) {
      queryBuilder.leftJoin(`${tableAlias}.preferences`, 'preference_filter');
      queryBuilder.andWhere('preference_filter.priority_order = 1');
      queryBuilder.andWhere('preference_filter.deleted_at IS NULL');
  
      if (parsedFilters.preferredProgramId) {
        queryBuilder.andWhere('preference_filter.preferred_program_id = :preferredProgramId', {
          preferredProgramId: parsedFilters.preferredProgramId
        });
      }
  
      if (parsedFilters.preferredSessionId) {
        queryBuilder.andWhere('preference_filter.preferred_session_id = :preferredSessionId', {
          preferredSessionId: parsedFilters.preferredSessionId
        });
      }
    }
  
    // searchText
    if (searchText && searchText.trim() !== '') {
      const searchTerm = `%${searchText.trim()}%`;
      queryBuilder.andWhere(
        `(${tableAlias}.full_name ILIKE :searchTerm OR ${tableAlias}.email_address ILIKE :searchTerm OR ${tableAlias}.mobile_number ILIKE :searchTerm)`,
        { searchTerm }
      );
    }
  
    return queryBuilder;
  }  

  private applyUserFilterToQuery(queryBuilder: any, userId: number | null): any {
    if (userId) {
      queryBuilder.andWhere('registration.rm_contact = :rmContactUserId', {
        rmContactUserId: userId
      });
    }
    return queryBuilder;
  }

  // ===== USER HISTORY QUERIES =====

  async findLastRegistrationByUserId(userId: number) {
    try {
      const registration = await this.registrationRepo.findOne({
        where: { userId, deletedAt: IsNull() },
        order: { registrationDate: 'DESC' },
        relations: [
          'invoiceDetails',
          'paymentDetails',
          'travelInfo',
          'travelPlans',
        ],
      });
      return registration;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  async findLastInvoiceDetailByUserId(userId: number) {
    try {
      return await this.dataSource
        .getRepository(RegistrationInvoiceDetail)
        .createQueryBuilder('invoice')
        .innerJoin('invoice.registration', 'registration')
        .where('registration.userId = :userId', { userId })
        .andWhere('invoice.deletedAt IS NULL')
        .andWhere('registration.deletedAt IS NULL')
        .orderBy('COALESCE(invoice.updatedAt, invoice.createdAt)', 'DESC')
        .getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_INVOICE_DETAIL_GET_FAILED, error);
    }
  }

  async findLastPaymentDetailByUserId(userId: number) {
    try {
      return await this.dataSource
        .getRepository(RegistrationPaymentDetail)
        .createQueryBuilder('payment')
        .innerJoin('payment.registration', 'registration')
        .where('registration.userId = :userId', { userId })
        .andWhere('payment.deletedAt IS NULL')
        .andWhere('registration.deletedAt IS NULL')
        .orderBy('COALESCE(payment.updatedAt, payment.createdAt)', 'DESC')
        .getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_PAYMENT_DETAIL_GET_FAILED, error);
    }
  }

  async findLastTravelPlanByUserId(userId: number) {
    try {
      return await this.dataSource
        .getRepository(RegistrationTravelPlan)
        .createQueryBuilder('plan')
        .innerJoin('plan.registration', 'registration')
        .where('registration.userId = :userId', { userId })
        .andWhere('plan.deletedAt IS NULL')
        .andWhere('registration.deletedAt IS NULL')
        .orderBy('COALESCE(plan.updatedAt, plan.createdAt)', 'DESC')
        .getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_GET_FAILED, error);
    }
  }

  async findLastTravelInfoByUserId(userId: number) {
    try {
      return await this.dataSource
        .getRepository(RegistrationTravelInfo)
        .createQueryBuilder('info')
        .innerJoin('info.registration', 'registration')
        .where('registration.userId = :userId', { userId })
        .andWhere('info.deletedAt IS NULL')
        .andWhere('registration.deletedAt IS NULL')
        .orderBy('COALESCE(info.updatedAt, info.createdAt)', 'DESC')
        .getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_GET_FAILED, error);
    }
  }

  /**
   * Find the approval status of a registration
   * @param registrationId - The ID of the registration
   * @returns The latest approval status of the registration
   */
  async findApprovalStatus(registrationId: number): Promise<RegistrationApproval | null> {
    try {
      return await this.approvalRepo.findOne({
        where: { 
          registrationId: registrationId,
          deletedAt: IsNull()
        },
        order: { createdAt: 'DESC' },
      });
    } catch (error) {
      this.logger.error('Failed to find approval status', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  private buildRegistrationData(params: any): any {
    const {
      dto,
      userId,
      isSelfRegister,
      registrationLevel,
      session,
      registrationStatus,
    } = params;
    
    const registrationData: any = {
      createdBy: { id: userId } as any,
      updatedBy: { id: userId } as any,
      registrationStatus: registrationStatus ?? RegistrationStatusEnum.DRAFT,
      basicDetailsStatus: RegistrationBasicStatusEnum.COMPLETED,
      registrationDate: registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT ? null : new Date().toISOString(),
    };

    // Set the appropriate ID field based on registration level
    if (registrationLevel === 'program') {
      registrationData.programId = dto.programId;
      registrationData.program = { id: dto.programId } as any;
      registrationData.programSessionId = null;
      registrationData.programSession = null;
    } else if (registrationLevel === 'session') {
      registrationData.programSessionId = session.id;
      registrationData.programSession = { id: session.id } as any;
      registrationData.programId = null;
      registrationData.program = null;
    }

    // If self registration, set userId on registration
    if (isSelfRegister) {
      registrationData.userId = userId;
    }

    return registrationData;
  }

  private async saveCustomResponses(manager: EntityManager, customResponses: RegistrationCustomResponse[], registrationId: number): Promise<void> {
    for (const cr of customResponses) {
      cr.registrationId = registrationId;
      cr.registration = { id: registrationId } as any;
      await manager.save(RegistrationCustomResponse, cr);
    }
  }

  private async updateCustomResponses(manager: EntityManager, customResponses: RegistrationCustomResponse[], registrationId: number, userId: number): Promise<void> {
    for (const cr of customResponses) {
      const existing = await manager.findOne(RegistrationCustomResponse, {
        where: { registrationId: registrationId, questionId: cr.questionId },
      });
      if (existing) {
        await manager.update(RegistrationCustomResponse, existing.id, {
          responseValue: cr.responseValue,
          updatedBy: { id: userId } as any,
        });
      } else {
        cr.registrationId = registrationId;
        cr.registration = { id: registrationId } as any;
        await manager.save(RegistrationCustomResponse, cr);
      }
    }
  }

  private async handleRegistrationOutcome(manager: EntityManager, registrationId: number, params: any) {
    const { registrationOutcome, seatInfo, dto, program, session, registrationLevel, userId, registrationStatus } = params;
    if (registrationStatus == RegistrationStatusEnum.SAVE_AS_DRAFT) {
      await manager.update(ProgramRegistration, registrationId, {
        registrationStatus: RegistrationStatusEnum.SAVE_AS_DRAFT,
        basicDetailsStatus: RegistrationBasicStatusEnum.DRAFT,
        auditRefId: registrationId,
        parentRefId: registrationId,
      });
      return {
        registrationId,   
        waitlisted: false,
        pendingApproval: false,
        registrationLevel,
        registrationStatus: RegistrationStatusEnum.SAVE_AS_DRAFT,
        basicDetailsStatus: RegistrationBasicStatusEnum.DRAFT,
      }
    }

    switch (registrationOutcome.type) {
      case 'waitlist':
        return await this.handleWaitlistRegistration(manager, registrationId, program, session?.id, seatInfo, registrationLevel);
      
      case 'no_space':
        await this.createRegistrationFailure(program, session, dto, userId, 'NO_SPACE');
        throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_NO_SPACE);
      
      case 'pending_approval':
        return await this.handleApprovalRegistration(manager, registrationId, program, session?.id, userId, registrationLevel);
      
      case 'direct_registration':
        return await this.handleDirectRegistration(manager, registrationId, program, session, registrationLevel, registrationOutcome, seatInfo);
      
      default:
        throw new Error('Unknown registration outcome type');
    }
  }

  private async handleWaitlistRegistration(
    manager: EntityManager,
    registrationId: number,
    program: Program,
    sessionId: number | undefined,
    seatInfo: any,
    registrationLevel: string,
  ) {
    // Get next waitlist sequence number
    const nextWaitlistSeqNumber = await this.getNextWaitlistSeqNumber(program.id, sessionId);
    
    // Get next registration sequence number
    const nextRegSeqNumber = await this.getNextRegistrationSeqNumber(program.id, sessionId);
    
    // Generate formatted sequence numbers
    const formattedRegNumber = this.generateFormattedSeqNumber(program, nextRegSeqNumber, false);
    const formattedWaitlistNumber = this.generateFormattedSeqNumber(program, nextWaitlistSeqNumber, true);

    await manager.update(ProgramRegistration, registrationId, {
      waitingListSeqNumber: nextWaitlistSeqNumber,
      registrationSeqNumber: formattedRegNumber,
      waitListRegistrationSeqNumber: formattedWaitlistNumber, // New column
      basicDetailsStatus: RegistrationBasicStatusEnum.WAITLISTED,
      registrationStatus: RegistrationStatusEnum.WAITLISTED,
      auditRefId: registrationId,
      parentRefId: registrationId,
    });

    return {
      registrationId,
      registrationSeqNumber: formattedRegNumber,
      waitListRegistrationSeqNumber: formattedWaitlistNumber,
      waitlisted: true,
      waitlistPosition: nextWaitlistSeqNumber,
      registrationLevel,
      basicDetailsStatus: RegistrationBasicStatusEnum.COMPLETED,
      registrationStatus: RegistrationStatusEnum.WAITLISTED,
      participantRemainingSeats: seatInfo.totalSeats - seatInfo.filledSeats,
    };
  }

  private async handleApprovalRegistration(
    manager: EntityManager,
    registrationId: number,
    program: Program,
    sessionId: number | undefined,
    userId: number,
    registrationLevel: string,
  ) {
    const nextRegSeqNumber = await this.getNextRegistrationSeqNumber(program.id, sessionId);
    const formattedNumber = this.generateFormattedSeqNumber(program, nextRegSeqNumber, false);

    let approvalType = ApprovalStatusEnum.PENDING;
    const existingApproval = await manager.findOne(RegistrationApproval, {
      where: { registrationId: registrationId, deletedAt: IsNull() },
    });

    if (existingApproval) {
      this.logger.log(`Approval record already exists for registration ${registrationId}, skipping creation.`);
      approvalType = existingApproval.approvalStatus ?? ApprovalStatusEnum.PENDING; // Assign from existing if present
    } else {
      await manager.update(ProgramRegistration, registrationId, {
        registrationSeqNumber: formattedNumber,
        basicDetailsStatus: RegistrationBasicStatusEnum.COMPLETED,
        registrationStatus: RegistrationStatusEnum.PENDING,
        registrationDate: new Date().toISOString(),
        auditRefId: registrationId,
        parentRefId: registrationId,
      });

      approvalType = ApprovalStatusEnum.PENDING;
      const approval = manager.create(RegistrationApproval, {
        registrationId: registrationId,
        approvalStatus: ApprovalStatusEnum.PENDING,
        approvalType: approvalType,
        createdBy: { id: userId } as any,
        updatedBy: { id: userId } as any,
        registration: { id: registrationId } as any,
      });
      await manager.save(RegistrationApproval, approval);
    }

    return {
      registrationId,
      registrationSeqNumber: formattedNumber,
      pendingApproval: true,
      approvalType: approvalType,
      registrationLevel,
      registrationStatus: RegistrationStatusEnum.PENDING,
      basicDetailsStatus: RegistrationBasicStatusEnum.COMPLETED,
    };
  }

  private async handleDirectRegistration(
    manager: EntityManager,
    registrationId: number,
    program: Program,
    session: ProgramSession | null,
    registrationLevel: string,
    registrationOutcome: any,
    seatInfo: any,
  ) {
    const nextRegSeqNumber = await this.getNextRegistrationSeqNumber(program.id, session?.id);
    const formattedNumber = this.generateFormattedSeqNumber(program, nextRegSeqNumber, false);

    await manager.update(ProgramRegistration, registrationId, {
      registrationSeqNumber: formattedNumber,
      registrationStatus: registrationOutcome.finalStatus,
      auditRefId: registrationId,
      parentRefId: registrationId,
    });

    if (!registrationOutcome.requiresPayment) {
      await incrementSeatCounts(manager, program.id, session?.id);
    }

    return {
      registrationId,
      registrationSeqNumber: formattedNumber,
      registrationLevel,
      registrationStatus: registrationOutcome.finalStatus,
      basicDetailsStatus: RegistrationBasicStatusEnum.COMPLETED,
      remainingSeats: seatInfo.waitlistTriggerCount - seatInfo.filledSeats,
      participantRemainingSeats: seatInfo.totalSeats - seatInfo.filledSeats,
    };
  }
  private async storeQuestionResponsesGrouped(
    groupedAnswers: Map<string, { columnName: string; value: any; question: Question }[]>,
    registrationId: number,
    manager: EntityManager,
    registration: ProgramRegistration,
    registrationStatus?: RegistrationStatusEnum,
    travelUpdateStatus?: TravelStatusEnum,
    userId?: number,
  ): Promise<void> {
    const regWithProgram = await manager.findOne(ProgramRegistration, {
      where: { id: registrationId },
      relations: ['program'],
    });

    const programId = regWithProgram?.program?.id ?? registration.program?.id;
    const programInvolvesTravel = regWithProgram?.program?.involvesTravel ?? registration.program?.involvesTravel;
    const programRequiresPayment = regWithProgram?.program?.requiresPayment ?? registration.program?.requiresPayment;
    this.logger.log(`Program ID: ${programId}, Involves Travel: ${programInvolvesTravel}, Requires Payment: ${programRequiresPayment}`);

    const travelInfoTable = this.dataSource.getMetadata(RegistrationTravelInfo).tableName;
    const travelPlanTable = this.dataSource.getMetadata(RegistrationTravelPlan).tableName;
    const paymentDetailTable = this.dataSource.getMetadata(RegistrationPaymentDetail).tableName;
    const invoiceDetailTable = this.dataSource.getMetadata(RegistrationInvoiceDetail).tableName;

        // Track if we need to create approval record after all updates are complete
    let shouldCreateApproval = false;

    this.logger.log(`Checking completion for registration ${registrationId}...`);
    
    // Get current registration status - use fresh query to ensure we get latest data
    const currentRegistration = await manager.findOne(ProgramRegistration, {
      where: { id: registrationId },
      relations: ['paymentDetails', 'travelInfo', 'travelPlans'],
    });
      this.logger.log(`Current registration status: ${currentRegistration?.registrationStatus}`);

    // Move completion check outside the loop and modify it to handle transaction context
    const checkAndUpdateRegistrationCompletion = async (currentTableName?: string, currentTableAnswers?: { columnName: string; value: any; question: Question }[]) => {
      if (registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT) {
        return;
      }
      
      // Only check completion for pending registrations
      // if (currentRegistration?.registrationStatus !== RegistrationStatusEnum.PENDING) {
      //   return;
      // }
      
      this.logger.log(`Registration ${registrationId} is pending, checking completion criteria...`);
      let shouldComplete = false;
      const isFreeSeat = currentRegistration?.isFreeSeat === true;

      // Helper function to check if current table indicates completion
      const isCurrentTableCompleted = (tableName: string) => {
        if (!currentTableName) return false;
        // If we're currently processing travel info table, consider travel info as completed
        if (tableName === travelInfoTable) {
          return currentTableName === travelInfoTable;
        }
        // If travelUpdateStatus is present, treat travel plan as completed if travelUpdateStatus is completed
        if (tableName === travelPlanTable) {
          // If travelUpdateStatus is set, treat as completed if currentTableName matches
          return travelUpdateStatus ? travelUpdateStatus === TravelStatusEnum.COMPLETED : false;
        }
        // If we're currently processing payment table, consider payment as completed
        if (tableName === paymentDetailTable) {
          return currentTableName === paymentDetailTable;
        }
        
        return false;
      };

      if (programInvolvesTravel && programRequiresPayment) {
        this.logger.log(`Program ${programId} involves travel and requires payment, checking both...`);
        
        // Check payment completion - look for any valid payment status OR if we're currently processing payment
        let paymentCompleted = isFreeSeat? true : false;
        if (isCurrentTableCompleted(paymentDetailTable)) {
          paymentCompleted = true;
          this.logger.log(`Payment completion assumed from current table processing`);
        } else {
          const paymentRecord = await manager.findOne(RegistrationPaymentDetail, {
            where: {
              registrationId,
              paymentStatus: In([
                PaymentStatusEnum.ONLINE_COMPLETED,
                PaymentStatusEnum.OFFLINE_COMPLETED,
              ]),
            },
          });
          paymentCompleted = !!paymentRecord || isFreeSeat;
        }

        // Check travel completion - look for completed travel info/plan OR if we're currently processing travel
        let travelCompleted = false;
        let travelInfoCompleted = false;
        let travelPlanCompleted = false;
        if (isCurrentTableCompleted(travelInfoTable)) {
          // When currently processing travel info table, check if InternationalId field has value
          const hasInternationalId = currentTableAnswers?.some(answer => 
            answer.columnName === 'international_id_picture_url' && 
            answer.value && 
            answer.value.trim() !== ''
          ) || false;
          travelInfoCompleted = hasInternationalId;
          travelPlanCompleted = !!(await manager.findOne(RegistrationTravelPlan, {
            where: { registrationId, travelPlanStatus: TravelStatusEnum.COMPLETED },
          }));
          this.logger.log(`Travel info completion based on InternationalId presence: ${hasInternationalId}`);
        } else if (isCurrentTableCompleted(travelPlanTable)) {
          travelPlanCompleted = true;
          travelInfoCompleted = !!(await manager.findOne(RegistrationTravelInfo, {
            where: { registrationId, travelInfoStatus: TravelStatusEnum.COMPLETED },
          }));
          this.logger.log(`Travel plan completion assumed from current table processing`);
        } else {
          travelInfoCompleted = !!(await manager.findOne(RegistrationTravelInfo, {
            where: { registrationId, travelInfoStatus: TravelStatusEnum.COMPLETED },
          }));

          travelPlanCompleted = !!(await manager.findOne(RegistrationTravelPlan, {
            where: { registrationId, travelPlanStatus: TravelStatusEnum.COMPLETED },
          }));
        }
          travelCompleted = !!travelInfoCompleted && !!travelPlanCompleted;
          this.logger.log(`Payment completed: ${paymentCompleted}, Travel completed: ${travelCompleted}`);
          shouldComplete = paymentCompleted && travelCompleted;

      } else if (programInvolvesTravel) {
        this.logger.log(`Program ${programId} involves travel, checking travel info and plan...`);
        
        // Check travel completion - look for completed travel info/plan OR if we're currently processing travel
        let travelCompleted = false;
        let travelInfoCompleted = false;
        let travelPlanCompleted = false;
        if (isCurrentTableCompleted(travelInfoTable)) {
          // When currently processing travel info table, check if InternationalId field has value
          const hasInternationalId = currentTableAnswers?.some(answer => 
            answer.columnName === 'international_id_picture_url' && 
            answer.value && 
            answer.value.trim() !== ''
          ) || false;
          travelInfoCompleted = hasInternationalId;
          travelPlanCompleted = !!(await manager.findOne(RegistrationTravelPlan, {
            where: { registrationId, travelPlanStatus: TravelStatusEnum.COMPLETED },
          }));
          this.logger.log(`Travel info completion based on InternationalId presence: ${hasInternationalId}`);
        } else if (isCurrentTableCompleted(travelPlanTable)) {
          travelInfoCompleted = !!(await manager.findOne(RegistrationTravelInfo, {
            where: { registrationId, travelInfoStatus: TravelStatusEnum.COMPLETED },
          }));
          travelPlanCompleted = true;
          this.logger.log(`Travel plan completion assumed from current table processing`);
        } else {
          travelInfoCompleted = !!(await manager.findOne(RegistrationTravelInfo, {
            where: { registrationId, travelInfoStatus: TravelStatusEnum.COMPLETED },
          }));
          travelPlanCompleted = !!(await manager.findOne(RegistrationTravelPlan, {
            where: { registrationId, travelPlanStatus: TravelStatusEnum.COMPLETED },
          }));
        }
        travelCompleted = !!travelInfoCompleted && !!travelPlanCompleted;
      
        this.logger.log(`Travel completed: ${travelCompleted}`);
        shouldComplete = travelCompleted;
      } else if (programRequiresPayment) {
        this.logger.log(`Program ${programId} requires payment, checking payment details...`);
        
        // Check payment completion - look for any valid payment status OR if we're currently processing payment
        let paymentCompleted = isFreeSeat? true : false;
        if (isCurrentTableCompleted(paymentDetailTable)) {
          paymentCompleted = true;
          this.logger.log(`Payment completion assumed from current table processing`);
        } else {
          const paymentRecord = await manager.findOne(RegistrationPaymentDetail, {
            where: {
              registrationId,
              paymentStatus: In([
                PaymentStatusEnum.ONLINE_COMPLETED,
                PaymentStatusEnum.OFFLINE_COMPLETED,
              ]),
            },
          });
          paymentCompleted = !!paymentRecord || isFreeSeat;
        }
        
        this.logger.log(`Payment completed: ${paymentCompleted}`);
        shouldComplete = paymentCompleted;
      } else {
        shouldComplete = true;
      }
      
      this.logger.log(`Should complete registration ${registrationId}: ${shouldComplete}`);
      if (shouldComplete) {
        await manager.update(
          ProgramRegistration,
          { id: registrationId },
          { 
            registrationStatus: RegistrationStatusEnum.COMPLETED,
            auditRefId: registrationId,
            parentRefId: registrationId,
          },
        );
        this.logger.log(`Registration ${registrationId} marked as completed`);
      } else {
        // get the existing registration status to avoid overwriting with PENDING if it's already rejected or pending or cancelled
        let existingStatus = currentRegistration?.registrationStatus;
        if (existingStatus === RegistrationStatusEnum.SAVE_AS_DRAFT 
        || existingStatus === RegistrationStatusEnum.DRAFT) {
          // update it to pending only if it's currently draft or save_as_draft
          existingStatus = RegistrationStatusEnum.PENDING;
          this.logger.log(`Registration ${registrationId} updated from ${currentRegistration?.registrationStatus} to PENDING due to incomplete criteria`);
        } else if (existingStatus === RegistrationStatusEnum.COMPLETED) {
          // If registration is currently completed but criteria are no longer met, revert to pending
          existingStatus = RegistrationStatusEnum.PENDING;
          this.logger.log(`Registration ${registrationId} reverted from COMPLETED to PENDING due to incomplete criteria`);
        } else {
          // retain existing status (e.g., REJECTED, CANCELLED)
          this.logger.log(`Registration ${registrationId} retains existing status: ${existingStatus}`);
        }
        
        await manager.update(
          ProgramRegistration,
          { id: registrationId },
          { 
            registrationStatus: existingStatus,
            auditRefId: registrationId,
            parentRefId: registrationId,
          },
        );
      }
    };
    // Helper function to determine person type based on age and program limits
    const determinePersonType = async (dob: Date, programId: number): Promise<PersonTypeEnum | null> => {
      try {
        // Get program with age limits
        const programWithLimits = await manager.findOne(Program, {
          where: { id: programId },
          // select: ['elderMinAge', 'childMaxAge']
        });

        if (!programWithLimits) {
          this.logger.warn(`Program ${programId} not found for age validation`);
          return null;
        }

        const { elderMinAge, childMaxAge } = programWithLimits;
        
        // Calculate age from DOB
        const age = getAgeFromDOB(dob);

        this.logger.log(`Age calculated: ${age}, elderMinAge: ${elderMinAge}, childMaxAge: ${childMaxAge}`);

        // Determine person type based on age limits
        if (childMaxAge !== null && age <= childMaxAge) {
          return PersonTypeEnum.CHILD;
        } else if (elderMinAge !== null && age >= elderMinAge) {
          return PersonTypeEnum.ELDER;
        } else {
          return PersonTypeEnum.ADULT;
        }
      } catch (error) {
        this.logger.error('Error determining person type:', error);
        return null;
      }
    };

    // 8106042366

    // Process all table updates first
    for (const [tableName, answers] of groupedAnswers.entries()) {
      // Handle preferences via service
      if (tableName === this.dataSource.getMetadata(Preference).tableName) {
        let parsedFilters;
        if (answers[0]?.value) {
          parsedFilters = decodeURIComponent(answers[0].value);
          parsedFilters = JSON.parse(parsedFilters);
        }
  
        await this.handlePreferencesWithService(
          manager,
          parsedFilters,
          registrationId,
          programId,
          registration.createdBy.id,
          registration.updatedBy.id
        );
        continue;
      }
  
      // Get the entity name from the table name
      const entityMetadata = manager.connection.entityMetadatas.find(
        metadata => metadata.tableName === tableName || `hdb_${metadata.tableName}` === tableName
      );
      if (!entityMetadata) {
        throw new Error(`Entity metadata not found for table: ${tableName}`);
      }
  
      const repository = manager.getRepository(entityMetadata.target);
      if (!repository) {
        throw new Error(`Repository not found for table: ${tableName}`);
      }
  
      const isRegistrationTable = tableName === this.dataSource.getMetadata(ProgramRegistration).tableName;
      const existing = await manager.findOne(repository.target, {
        where: isRegistrationTable ? { id: registrationId } : { registrationId: registrationId },
      });

  

    // if existing record then check for table name and corresponding status if it's sttaus is draft or save_as_draft then only new status as save_as_draft will be set
    if (registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT) {
      // Map table names to their status fields and enums
      const tableStatusMap = {
        [this.dataSource.getMetadata(ProgramRegistration).tableName]: {
          field: 'registrationStatus',
          enums: RegistrationStatusEnum,
          error: 'Cannot update registration with status other than draft or save_as_draft',
        },
        [travelInfoTable]: {
          field: 'travelInfoStatus',
          enums: TravelStatusEnum,
          error: 'Cannot update travel info with status other than draft or save_as_draft',
        },
        [travelPlanTable]: {
          field: 'travelPlanStatus',
          enums: TravelStatusEnum,
          error: 'Cannot update travel plan with status other than draft or save_as_draft',
        },
        [paymentDetailTable]: {
          field: 'paymentStatus',
          enums: PaymentStatusEnum,
          error: 'Cannot update payment detail with status other than draft or save_as_draft',
        },
        [invoiceDetailTable]: {
          field: 'invoiceStatus',
          enums: InvoiceStatusEnum,
          error: 'Cannot update invoice detail with status other than draft or save_as_draft',
        },
      };

      // Determine the key for the current table
      const tableKey = isRegistrationTable
        ? this.dataSource.getMetadata(ProgramRegistration).tableName
        : tableName;

      const statusConfig = tableStatusMap[tableKey];

      if (statusConfig) {
        const currentStatus = existing?.[statusConfig.field];
        if (
          !currentStatus || (
          currentStatus === statusConfig.enums.DRAFT ||
          currentStatus === statusConfig.enums.SAVE_AS_DRAFT)
        ) {
          this.logger.log("here in save as draft");
        } else {
          throw new InifniBadRequestException(
            ERROR_CODES.REGISTRATION_INVALID_STATUS,
            null,
            null,
            statusConfig.error
          );
        }
      }
    }
    
      
      // Data transformation logic
      if (tableName === travelInfoTable || tableName === travelPlanTable) {
        answers.forEach((a) => {
          if (a.columnName === 'travel_type' || a.columnName === 'return_travel_type') {
            if (typeof a.value === 'string') {
              a.value = toSnakeCase(a.value.trim());
            }
            if (a.value.trim() === '') {
              a.value = null;
            }
          } else if (a.value.trim() === '') {
            a.value = null; // Ensure empty strings are set to null
          }
        });
      } else {
        answers.forEach((a) => {
          if (a.value.trim() === '') {
            a.value = null; // Ensure empty strings are set to null
          }
        });
      }
      // Payment mode transformation
      if (tableName === paymentDetailTable) {
        answers.forEach((a) => {
          if (a.columnName === 'payment_mode') {
            if (typeof a.value === 'string') {
              a.value = a.value.toLowerCase().includes('online') ? 'online' : 'offline';
            }
          }
        });
      }

      // Determine appropriate status based on table and registrationStatus
      const getStatusForTable = async (tableName: string, registrationStatus?: RegistrationStatusEnum) => {
        if (!registrationStatus) {
          return null;
        }

        if ([RegistrationStatusEnum.SAVE_AS_DRAFT].includes(registrationStatus)) {
          if (tableName === travelInfoTable) {
            return TravelStatusEnum.SAVE_AS_DRAFT;
          } else if (tableName === travelPlanTable) {
            return TravelStatusEnum.SAVE_AS_DRAFT;
          } else if (tableName === paymentDetailTable) {
            return PaymentStatusEnum.SAVE_AS_DRAFT;
          } else if (tableName === invoiceDetailTable) {
            return InvoiceStatusEnum.SAVE_AS_DRAFT;
          }
          return null;
        }
        return null;
      };

      // Map database column names to entity property names
      const columnMap = new Map<string, string>();
      entityMetadata.columns.forEach((col) => columnMap.set(col.databaseName, col.propertyName));

      const entityData: any = {};
      for (const ans of answers) {
        const prop = columnMap.get(ans.columnName);
        if (prop) {
          entityData[prop] = ans.value;
        }
      }

      // Generate and store signed URLs for file upload questions
      await this.generateAndStoreSignedUrls(entityData, answers, tableName, columnMap);

      // Special handling for registration table - determine person type if DOB is being updated
      if (isRegistrationTable && entityData.dob && programId) {
        try {
          const personType = await determinePersonType(new Date(entityData.dob), programId);
          if (personType) {
            entityData.personType = personType;
            entityData.parentalFormStatus = getParentalFormStatus(personType, { parentalFormStatus: currentRegistration?.parentalFormStatus });
          } else {
            this.logger.warn('Person type could not be determined, leaving unchanged');
          }
        } catch (error) {
          this.logger.error('Error setting person type:', error);
          // Don't throw error, just log and continue without setting person type
        }
      }

      // Add status fields based on table type and registrationStatus
        if (tableName === travelInfoTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          // Check if InternationalId binding key has value to determine completion status
          let travelInfoStatus = status ?? TravelStatusEnum.PENDING;
          if (!status) {
            // Check if internationalIdPictureUrl field has a value (not empty or null)
            const hasInternationalId = entityData.internationalIdPictureUrl && 
                                      entityData.internationalIdPictureUrl.trim() !== '';
            travelInfoStatus = hasInternationalId ? TravelStatusEnum.COMPLETED : TravelStatusEnum.PENDING;
            this.logger.log(
              `Setting travel_info_status to ${travelInfoStatus} based on InternationalId presence: ${hasInternationalId}`,
            );
          }
          entityData.travelInfoStatus = travelInfoStatus;
        } else if (tableName === travelPlanTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
        entityData.travelPlanStatus =
          status ?? (travelUpdateStatus ?? TravelStatusEnum.PENDING);
        this.logger.log(
          `Setting travel_plan_status to ${
            travelUpdateStatus ?? TravelStatusEnum.PENDING
          } status for travel plan table ${status}`,
        );
        } else if (tableName === paymentDetailTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          if (status) {
          entityData.paymentStatus = status;
          }
        } else if (tableName === invoiceDetailTable) {
          const status = await getStatusForTable(tableName, registrationStatus);
          if (status) {
          entityData.invoiceStatus = status;
          }
        }

      if (existing || isRegistrationTable) {
        // UPDATE path using TypeORM save to trigger audit logs
        const primaryKey = entityMetadata.primaryColumns[0].propertyName;
        entityData[primaryKey] = isRegistrationTable ? registrationId : existing?.id;
        // Check if entity has audit fields before trying to set them
        const hasAuditRefId = entityMetadata.columns.some((col) => col.propertyName === 'auditRefId');
        const hasParentRefId = entityMetadata.columns.some((col) => col.propertyName === 'parentRefId');
        
        if (hasAuditRefId && hasParentRefId) {
          // For update: use existing parentRefId if available, otherwise determine from entity
          let parentRefId = existing?.parentRefId;
          if (!parentRefId) {
            // Create a temporary entity with the data to determine parentRefId
            const tempEntity = { ...entityData, registrationId };
            parentRefId = this.determineParentRefId(tempEntity);
          }
          
          entityData.auditRefId = isRegistrationTable ? registrationId : existing?.auditRefId || existing?.id;
          entityData.parentRefId = parentRefId;
        }
        await manager.save(repository.target, repository.create(entityData));

        if (
          isRegistrationTable &&
          registrationStatus != RegistrationStatusEnum.SAVE_AS_DRAFT &&
          registration.registrationStatus != RegistrationStatusEnum.PENDING &&
          registration.registrationStatus != RegistrationStatusEnum.COMPLETED &&
          registration.registrationStatus != RegistrationStatusEnum.DRAFT &&
          registration.registrationStatus != RegistrationStatusEnum.REJECTED &&
          registration.registrationStatus != RegistrationStatusEnum.CANCELLED &&
          registration.registrationStatus != RegistrationStatusEnum.ON_HOLD
        ) {
          shouldCreateApproval = true;
          this.logger.log('Flagged to create approval record after batch update', { registrationId, registrationStatus } );
        }
      } else {
        // INSERT path using TypeORM save to trigger audit logs
        entityData.registrationId = registrationId;
        await manager.save(repository.target, repository.create(entityData));
        this.logger.log('Batch created new record', { 
          tableName, 
          registrationId, 
          columns: Object.keys(entityData),
          registrationStatus,
        });
      }
    // Create approval record after all registration data has been updated
    // This ensures the approval record reflects the latest registration changes
    if (shouldCreateApproval) {
      this.logger.log('Creating approval record after all registration updates are complete', {
        registrationId,
        registrationStatus,
        originalStatus: registration.registrationStatus
      });

      await this.handleApprovalRegistration(
        manager,
        registrationId,
        registration.program,
        registration.programSession?.id,
        registration.updatedBy.id,
        registration.program.registrationLevel
      );
    }
      
      await checkAndUpdateRegistrationCompletion(tableName, answers);
      // if current table being updated is travel plan execute one sample function
      if (tableName === travelPlanTable) {
        await this.sendCommunication(registrationId, manager, currentRegistration?.travelPlans,userId);
      }
    }

  }

  /**
   * Generate and store signed URLs for file upload questions
   * @param entityData - The entity data object to update
   * @param answers - Array of answers containing column name and value
   * @param tableName - Name of the table being updated
   * @param columnMap - Map of database column names to entity property names
   */
  private async generateAndStoreSignedUrls(
    entityData: any,
    answers: { columnName: string; value: any; question: Question }[],
    tableName: string,
    columnMap: Map<string, string>,
  ): Promise<void> {
    try {
      // Iterate through all signed URL mappings
      for (const [bindingKey, mapping] of Object.entries(SIGNED_URL_COLUMN_MAPPINGS)) {
        // Check if this mapping belongs to the current table
        if (mapping.tableName !== tableName) {
          continue;
        }

        // Find the answer that corresponds to this plain URL column
        const answer = answers.find(ans => ans.columnName === mapping.plainUrlColumn);
        
        if (!answer || !answer.value) {
          continue;
        }

        const plainUrl = answer.value;

        // Check if URL should be excluded from signed URL generation (zoho, entrainment domains)
        if (shouldExcludeFromSignedUrl(plainUrl)) {
          this.logger.log(`Skipping signed URL generation for excluded domain: ${plainUrl}`);
          // Use the plain URL as signed URL for excluded domains
          const signedUrlPropertyName = columnMap.get(mapping.signedUrlColumn);
          if (signedUrlPropertyName) {
            entityData[signedUrlPropertyName] = plainUrl;
          }
          continue;
        }

        try {
          const signedUrl = await this.awsS3Service.generateLongLivedSignedUrl(plainUrl);
          
          // Find the property name for the signed URL column using columnMap
          const signedUrlPropertyName = columnMap.get(mapping.signedUrlColumn);

          if (signedUrlPropertyName) {
            entityData[signedUrlPropertyName] = signedUrl;
            this.logger.log(`Generated signed URL for ${bindingKey} (${mapping.plainUrlColumn})`);
          }
        } catch (error) {
          this.logger.error(`Failed to generate signed URL for ${bindingKey}:`, error);
          // On error, use plain URL as fallback
          const signedUrlPropertyName = columnMap.get(mapping.signedUrlColumn);
          if (signedUrlPropertyName) {
            entityData[signedUrlPropertyName] = plainUrl;
          }
        }
      }
    } catch (error) {
      this.logger.error('Error in generateAndStoreSignedUrls:', error);
      // Don't throw error to prevent registration from failing
    }
  }

  async sendCommunication(
    registrationId: number,
    manager: EntityManager,
    existingTravelPlans?: RegistrationTravelPlan[],
    userId?: number
  ) {
    try {
      const registrationDetails = await manager.findOne(ProgramRegistration, { 
        where: { id: registrationId }, 
        relations: ['program','allocatedProgram', 'rmContactUser', 'travelPlans'],
      });
      if (!userId) {
        this.logger.warn(`No user found for ID: ${userId}`);
        handleKnownErrors(ERROR_CODES.USER_NOTFOUND, new Error(`User not found for ID: ${userId}`));
      }
      const userInfo = await manager.findOne(User, { where: { id: userId } });

      if (!userInfo) {
        this.logger.warn(`No user found for ID: ${userId}`);
        handleKnownErrors(ERROR_CODES.USER_NOTFOUND, new Error(`User not found for ID: ${userId}`));
      }

      if (!registrationDetails || !registrationDetails.travelPlans || registrationDetails.travelPlans.length === 0) {
        this.logger.warn(`No registration found for ID: ${registrationId}`);
        return;
      }
      if (
        registrationDetails.allocatedProgram?.startsAt &&
        (getDaysCountBetweenDates(
          new Date(),
          registrationDetails.allocatedProgram?.startsAt,
        ) <= 3) && userInfo.role !== ROLE_KEYS.SHOBA
      ) {

        if (!existingTravelPlans || existingTravelPlans.length === 0) { // send new travel plan communication
          this.logger.log('No existing travel plans found this is the first travel plan, sending communication to coordinators');
          await this.sendOnNewTravelPlanSavesMessage(registrationDetails, userInfo);
        } else if (
          (registrationDetails?.travelPlans[0]?.travelType !== existingTravelPlans[0]?.travelType) || 
          (registrationDetails?.travelPlans[0]?.returnTravelType !== existingTravelPlans[0]?.returnTravelType)) {
          this.logger.log('Return/Onward travel type change detected, sending communication to coordinators');
          await this.sendOnTravelModeChangeMessage(registrationDetails, userInfo);
        } else if (registrationDetails?.travelPlans[0]?.travelType === TravelTypeEnum.FLIGHT  && existingTravelPlans[0]?.travelType === registrationDetails?.travelPlans[0]?.travelType) {
          if (
            (registrationDetails?.travelPlans[0]?.airlineName !== existingTravelPlans[0]?.airlineName) || 
            (registrationDetails?.travelPlans[0]?.flightNumber !== existingTravelPlans[0]?.flightNumber) || 
            (registrationDetails?.travelPlans[0]?.arrivalDatetime !== existingTravelPlans[0]?.arrivalDatetime)
          ) {
            await this.sendOnFlightTravelPlanChangeMessage(registrationDetails, existingTravelPlans[0], userInfo, true);
            this.logger.log('Flight details changed detected for flight travel plan, sending communication to coordinators');
          } else {
            this.logger.log('No changes detected for flight travel plan to send communication');
          }
        } else if (registrationDetails?.travelPlans[0]?.returnTravelType === TravelTypeEnum.FLIGHT  && existingTravelPlans[0]?.returnTravelType === registrationDetails?.travelPlans[0]?.returnTravelType) {
          if (
            (registrationDetails?.travelPlans[0]?.departureAirlineName !== existingTravelPlans[0]?.departureAirlineName) || 
            (registrationDetails?.travelPlans[0]?.departureFlightNumber !== existingTravelPlans[0]?.departureFlightNumber) || 
            (registrationDetails?.travelPlans[0]?.departureDatetime !== existingTravelPlans[0]?.departureDatetime)
          ) {
            await this.sendOnFlightTravelPlanChangeMessage(registrationDetails, existingTravelPlans[0], userInfo, false);
            this.logger.log('Return flight details changed detected for flight return travel plan, sending communication to coordinators');
          } else {
            this.logger.log('No changes detected for flight return travel plan to send communication');
          }
        }
      } else {
        this.logger.log(
          'no need to send communication'
        );
      }
    } catch (error) {
      this.logger.log('Error sending communication:', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_COMMUNICATION_FAILED, error);
    }
  }

  async sendOnFlightTravelPlanChangeMessage(
    registrationDetails: ProgramRegistration,
    existingPlan: RegistrationTravelPlan,
    userInfo: User,
    isOnWard: boolean
  ) {
    try {
      const coordinators = await this.userRepository.getUsersByRoleKey(ROLE_KEYS.SHOBA);
      if (!coordinators || coordinators.length === 0) {
        this.logger.warn('No coordinators found for communication');
        return;
      }
      let params: Array<{ name: string; value: string }[]> = [];
      if (isOnWard) {
        params.push([
          {
            name: 'o_onward_airline',
            value: existingPlan?.airlineName,
          },
          {
            name: 'o_onward_fligh',
            value: existingPlan?.flightNumber,
          },
          {
            name: 'o_onward_time',
            value: formatTimeIST(existingPlan?.arrivalDatetime.toISOString()),
          },
          {
            name: 'n_onward_airline',
            value: registrationDetails?.travelPlans[0]?.airlineName,
          },
          {
            name: 'n_onward_fligh',
            value: registrationDetails?.travelPlans[0]?.flightNumber,
          },
          {
            name: 'n_onward_time',
            value: formatTimeIST(registrationDetails?.travelPlans[0]?.arrivalDatetime.toISOString()),
          },
        ]);
      } else {
        params.push([
          {
            name: 'o_return_airline',
            value: existingPlan?.departureAirlineName,
          },
          {
            name: 'o_return_fligh',
            value: existingPlan?.departureFlightNumber,
          },
          {
            name: 'o_return_time',
            value: formatTimeIST(existingPlan?.departureDatetime.toISOString()),
          },
          {
            name: 'n_return_airline',
            value: registrationDetails?.travelPlans[0]?.departureAirlineName,
          },
          {
            name: 'n_return_fligh',
            value: registrationDetails?.travelPlans[0]?.departureFlightNumber,
          },
          {
            name: 'n_return_time',
            value: formatTimeIST(registrationDetails?.travelPlans[0]?.departureDatetime.toISOString()),
          },
        ]);
      }
      for (const coordinator of coordinators) {
        if (coordinator?.countryCode && coordinator?.phoneNumber) {
          const messageData: SendTemplateMessageDto = {
            whatsappNumber: (coordinator?.countryCode + coordinator?.phoneNumber).replace('+', ''), // Remove + prefix
            templateName: isOnWard ? process.env.WATI_CO_ORDINATOR_ONWARD_TRAVEL_PLAN_CHANGE_TEMPLATE_ID || '' : process.env.WATI_CO_ORDINATOR_RETURN_TRAVEL_PLAN_CHANGE_TEMPLATE_ID || '',
            broadcastName: isOnWard ? process.env.WATI_CO_ORDINATOR_ONWARD_TRAVEL_PLAN_CHANGE_TEMPLATE_ID || '' : process.env.WATI_CO_ORDINATOR_RETURN_TRAVEL_PLAN_CHANGE_TEMPLATE_ID || '',
            parameters: [
              ...params.flat(),
              { name: 'reg_name', value: registrationDetails.fullName },
              { name: 'hdb_msd', value: registrationDetails.allocatedProgram?.name ?? '' },
              { name: 'login_user', value: userInfo?.firstName && userInfo?.lastName ? userInfo?.firstName + ' ' + userInfo?.lastName : userInfo?.fullName },
              
            ],
            trackinfo: {
              registrationId: registrationDetails.id,
            },
          };
          try {
            await this.communicationService.sendTemplateMessage(messageData);
          } catch (error) {
            this.logger.error('Error sending new travel plan saves message:', error);
          }
        } else {
          this.logger.log("No travel change send to coordinator");
        }
      }
    } catch (error) {
      this.logger.error('Error sending onward travel plan change message:', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_COMMUNICATION_FAILED, error);
    }
  }

  async sendOnTravelModeChangeMessage(registrationDetails: ProgramRegistration, userInfo: User) {
    try {
      const coordinators = await this.userRepository.getUsersByRoleKey(ROLE_KEYS.SHOBA);
      if (!coordinators || coordinators.length === 0) {
        this.logger.warn('No coordinators found for communication');
        return;
      }
      for (const coordinator of coordinators) {
          if (coordinator?.countryCode && coordinator?.phoneNumber) {
            const messageData: SendTemplateMessageDto = {
              whatsappNumber: (coordinator?.countryCode + coordinator?.phoneNumber).replace('+', ''), // Remove + prefix
              templateName: process.env.WATI_CO_ORDINATOR_TRAVEL_PLAN_CHANGE_TEMPLATE_ID || '',
              broadcastName: process.env.WATI_CO_ORDINATOR_TRAVEL_PLAN_CHANGE_TEMPLATE_ID || '',
              parameters: [
                { name: 'reg_name', value: registrationDetails.fullName },
                { name: 'reg_id', value: registrationDetails.id.toString() },
                { name: 'coordinator_name', value: (coordinator?.firstName && coordinator?.lastName) ? coordinator?.firstName + ' ' + coordinator?.lastName : coordinator?.fullName ?? coordinator?.legalFullName ?? "Infinitheist" },
                {
                  name: 'rm_name',
                  value:
                    userInfo?.firstName &&
                    userInfo?.lastName
                      ? userInfo?.firstName +
                        ' ' +
                        userInfo?.lastName
                      : userInfo?.fullName,
                },
              ],
              trackinfo: {
                registrationId: registrationDetails.id,
              },
            };
            try {
              await this.communicationService.sendTemplateMessage(messageData);
            } catch (error) {
              this.logger.error('Error sending new travel plan saves message:', error);
            }
          } else {
            this.logger.log("No travel change send to coordinator");
          }
        }
    } catch (error) {
      this.logger.error('Error sending travel mode change message:', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_COMMUNICATION_FAILED, error);
    }
  }

  async sendOnNewTravelPlanSavesMessage(registrationDetails: ProgramRegistration, userInfo: User) {
    try {
      const coordinators = await this.userRepository.getUsersByRoleKey(ROLE_KEYS.SHOBA);
      if (!coordinators || coordinators.length === 0) {
        this.logger.warn('No coordinators found for communication');
        return;
      }
      for (const coordinator of coordinators) {
        if (coordinator?.countryCode && coordinator?.phoneNumber) {
          const messageData: SendTemplateMessageDto = {
            whatsappNumber: (coordinator?.countryCode + coordinator?.phoneNumber).replace('+', ''), // Remove + prefix
            templateName: process.env.WATI_CO_ORDINATOR_NEW_TRAVEL_PLAN_MADE_TEMPLATE_ID || '',
            broadcastName: process.env.WATI_CO_ORDINATOR_NEW_TRAVEL_PLAN_MADE_TEMPLATE_ID || '',
            parameters: [
              { name: 'reg_name', value: registrationDetails.fullName },
              { name: 'reg_id', value: registrationDetails.id.toString() },
              { name: 'coordinator_name', value: (coordinator?.firstName && coordinator?.lastName) ? coordinator?.firstName + ' ' + coordinator?.lastName : coordinator?.fullName ?? coordinator?.legalFullName ?? "Infinitheist" },
              {
                  name: 'rm_name',
                  value:
                    userInfo?.firstName &&
                    userInfo?.lastName
                      ? userInfo?.firstName +
                        ' ' +
                        userInfo?.lastName
                      : userInfo?.fullName,
                },
            ],
            trackinfo: {
              registrationId: registrationDetails.id,
            },
          };
          try {
            await this.communicationService.sendTemplateMessage(messageData);
          } catch (error) {
            this.logger.error('Error sending new travel plan saves message:', error);
          }
        } else {
          this.logger.log("No travel change send to coordinator");
        }
      }
    } catch (error) {
      this.logger.error('Error sending new travel plan saves message:', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_COMMUNICATION_FAILED, error);
    }
  }
/**
   * Handle preferences using the preference service within a transaction
   * This replaces the direct database manipulation with proper service layer usage
   */
  private async handlePreferencesWithService(
    manager: EntityManager,
    preferences: any[],
    registrationId: number,
    programId: number,
    createdBy: number,
    updatedBy: number
  ): Promise<void> {
    try {
      // Validate preferences data
      if (!preferences || !Array.isArray(preferences) || preferences.length === 0) {
        this.logger.warn('No preferences provided or invalid preferences format', { 
          registrationId, 
          programId 
        });
        // If no preferences, we can still update the registration with an empty preference set
        const updatePreferenceDto: UpdatePreferenceDto = {
          programId: programId,
          preferences: [],
          updatedBy: updatedBy,
          createdBy: createdBy,
        };

        await this.preferenceService.update(
          registrationId,
          updatePreferenceDto,
          manager,
        );

        this.logger.log('No preferences to update, cleared preferences for registration', { 
          registrationId, 
          programId 
        });
        return;
      }

      // Prepare UpdatePreferenceDto for the preference service
      const updatePreferenceDto: UpdatePreferenceDto = {
        programId: programId,
        preferences: preferences.map((p, index) => ({
          sessionId: parseInt(p.sessionId),
          priorityOrder: p.priorityOrder ?? index + 1,
        })),
        updatedBy: updatedBy,
        createdBy: createdBy,
      };

      // Use the preference service's update method with the transaction manager
      // This will handle soft deletion of existing preferences and creation of new ones
      const result = await this.preferenceService.update(
        registrationId, 
        updatePreferenceDto, 
        manager
      );

      this.logger.log('Preferences handled via preference service', { 
        registrationId, 
        programId, 
        preferencesCount: preferences.length,
        createdPreferences: result.length
      });

    } catch (error) {
      this.logger.error('Error handling preferences via preference service', error);
      throw error;
    }
  }

  async upsertQuestionAnswers(
    manager: EntityManager,
    registrationId: number,
    answers: { questionId: number; answerValue: any }[],
    userId: number,
  ): Promise<RegistrationQuestionAnswer[]> {
    const savedAnswers: RegistrationQuestionAnswer[] = [];
    for (const ans of answers) {
      let entity = await manager.findOne(RegistrationQuestionAnswer, {
        where: { registrationId, questionId: ans.questionId },
      });
      if (entity) {
        entity.answerValue = ans.answerValue;
        entity.updatedBy = { id: userId } as any;
        entity.auditRefId = entity.id; // Ensure auditRefId is set correctly
        entity.parentRefId = registrationId;
      } else {
        entity = manager.create(RegistrationQuestionAnswer, {
          registrationId,
          questionId: ans.questionId,
          answerValue: ans.answerValue,
          createdBy: { id: userId } as any,
          updatedBy: { id: userId } as any,
        });
      }
      const saved = await manager.save(RegistrationQuestionAnswer, entity);
      savedAnswers.push(saved);
    }
    return savedAnswers;
  }

  private async updatePreferenceFromAnswer(
    value: UpdatePreferenceDto,
    registrationId: number,
    manager: EntityManager,
  ): Promise<void> {
    try {
      const dto = value;
      await this.preferenceService.update(registrationId, dto, manager);
    } catch (error) {
      this.logger.error('Error updating preferences from registration', error);
      throw error;
    }
  }

  async softDeleteRegistration(id: number, userId: number, manager: EntityManager) {
    if (!id) {
      throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_ID_REQUIRED);
    }

    const registrationRepo = manager.getRepository(ProgramRegistration);
    const registration = await registrationRepo.findOne({
      where: { id, deletedAt: IsNull() },
      relations: [
        'approvals',
        'program',
        'programSession',
        'allocatedProgram',
        'allocatedSession',
      ],
    });

    if (!registration) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        id.toString(),
      );
    }

    const isApproved = registration.approvals?.some(
      (app) =>
        app.approvalStatus === ApprovalStatusEnum.APPROVED && !app.deletedAt,
    );

    if (
      registration.registrationStatus === RegistrationStatusEnum.COMPLETED ||
      isApproved
    ) {
      await decrementSeatCounts(
        manager,
        registration.program.id,
        registration.programSession?.id,
      );

      if (registration.allocatedProgram) {
        await decrementSeatCounts(
          manager,
          registration.allocatedProgram.id,
          registration.allocatedSession?.id,
        );
      }
    }

    await manager.update(ProgramRegistration, id, {
      deletedAt: new Date(),
      updatedBy: { id: userId } as any,
      auditRefId: id,
      parentRefId: id,
    });

    return { registrationId: id };
  }

  async findTravelPlanByRegistrationId(registrationId: number): Promise<RegistrationTravelPlan | null> {
    try {
      return await this.travelPlanRepo.findOne({
        where: { registrationId, deletedAt: IsNull() }
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_GET_FAILED, error);
    }
  } 

  async updateTravelPlanStatus(
    planId: number,
    status: TravelStatusEnum,
    userId: number
     ): Promise<void> {
    try { 
      // Get the existing travel plan to extract registrationId for parentRefId
      const existingPlan = await this.travelPlanRepo.findOne({
        where: { id: planId, deletedAt: IsNull() },
        relations: ['registration']
      });
      
      if (!existingPlan) {
        throw new Error(`Travel plan with id ${planId} not found.`);
      }
      
      await this.travelPlanRepo.update(
        { id: planId, deletedAt: IsNull() },
        { 
          travelPlanStatus: status, 
          updatedAt: new Date(), 
          updatedBy: { id: userId } as any,
          auditRefId: planId,
          parentRefId: existingPlan.registrationId
        },
      );
      this.logger.log(`Travel plan status updated for registration ${planId} to ${status}`);
    } catch (error) {
      this.logger.error('Failed to update travel plan status', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_UPDATE_FAILED, error);
    } 
  }

  async createTravelPlan(
    registrationId: number,
    travelPlanStatus: TravelStatusEnum,
    userId: number
  ): Promise<RegistrationTravelPlan> {
    try {
      const travelPlan = this.travelPlanRepo.create({
        registrationId,
        travelPlanStatus: travelPlanStatus,
        createdBy: { id: userId } as any,
        updatedBy: { id: userId } as any,
        travelType: null, // Default to null, can be set later
      });

      const savedPlan = await this.travelPlanRepo.save(travelPlan);

      // Update audit fields once ID is available
      if (savedPlan && savedPlan.id) {
        await this.travelPlanRepo.update(savedPlan.id, {
          auditRefId: savedPlan.id,
          parentRefId: registrationId,
        });
        savedPlan.auditRefId = savedPlan.id;
        savedPlan.parentRefId = registrationId;
      }

      return savedPlan;
    } catch (error) {
      this.logger.error('Failed to create travel plan', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_CREATE_FAILED, error);
    }
  }

  async findTravelInfoByRegistrationId(registrationId: number): Promise<RegistrationTravelInfo | null> {
    try {
      return await this.travelInfoRepo.findOne({
        where: { registrationId, deletedAt: IsNull() } 
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_GET_FAILED, error);
    }
  }
  async updateTravelInfoStatus(
    infoId: number,   
    status: TravelStatusEnum,
    userId: number
  ): Promise<void> {
    try {
      // Get the existing travel info to extract registrationId for parentRefId
      const existingInfo = await this.travelInfoRepo.findOne({
        where: { id: infoId, deletedAt: IsNull() },
        relations: ['registration']
      });
      
      if (!existingInfo) {
        throw new Error(`Travel info with id ${infoId} not found.`);
      }
      
      await this.travelInfoRepo.update(
        { id: infoId, deletedAt: IsNull() },
        { 
          travelInfoStatus: status, 
          updatedAt: new Date(), 
          updatedBy: { id: userId } as any,
          auditRefId: infoId,
          parentRefId: existingInfo.registrationId
        },
      );
      this.logger.log(`Travel info status updated for registration ${infoId} to ${status}`);
    } catch (error) {
      this.logger.error('Failed to update travel info status', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_UPDATE_FAILED, error);
    }
  }
  async createTravelInfo(
    registrationId: number,
    travelInfoStatus: TravelStatusEnum,   
    userId: number
  ): Promise<RegistrationTravelInfo> {
    try {
      const travelInfo = this.travelInfoRepo.create({
        registrationId,
        travelInfoStatus: travelInfoStatus,
        createdBy: { id: userId } as any,
        updatedBy: { id: userId } as any,
      });

      const savedInfo = await this.travelInfoRepo.save(travelInfo);

      // Update audit fields once ID is available
      if (savedInfo && savedInfo.id) {
        await this.travelInfoRepo.update(savedInfo.id, {
          auditRefId: savedInfo.id,
          parentRefId: registrationId,
        });
        savedInfo.auditRefId = savedInfo.id;
        savedInfo.parentRefId = registrationId;
      }

      return savedInfo;
    } catch (error) {
      this.logger.error('Failed to create travel info', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_INFO_CREATE_FAILED, error);
    }
  }

  /**
   * Resets travel details for a registration by clearing travel plan data and related question responses
   * - Clears ALL travel plan fields except IDs, status, and audit fields
   * - Removes travel plan related question answers from custom responses table
   * - Does NOT affect travel info table (as per requirements)
   * @param registrationId - The ID of the registration to reset travel details for
   * @param userId - The ID of the user performing the reset operation
   */
  async resetTravelDetails(registrationId: number, userId: number): Promise<void> {
    this.logger.log(registrationTravelPlanMessages.RESET_TRAVEL_DETAILS, { registrationId, userId });
    
    try {
      await this.dataSource.transaction(async (manager: EntityManager) => {
        
        const fetchedRegistration = await manager.findOne(ProgramRegistration, {
          where: { id: registrationId, deletedAt: IsNull() }
        });

        if (fetchedRegistration && fetchedRegistration.registrationStatus === RegistrationStatusEnum.COMPLETED) {
          await manager
            .createQueryBuilder()
            .update(ProgramRegistration)
            .set({
              registrationStatus: RegistrationStatusEnum.PENDING,
              updatedBy: { id: userId } as any,
              updatedAt: new Date(),
            })
            .where('id = :id', { id: registrationId })
            .execute();
            
          this.logger.log('Registration status reverted from COMPLETED to PENDING due to travel reset', { 
            registrationId, 
            previousStatus: RegistrationStatusEnum.COMPLETED,
            newStatus: RegistrationStatusEnum.PENDING 
          });
        }

        // 1. Reset travel plan fields (keep only essential fields)
        const existingTravelPlan = await manager.findOne(RegistrationTravelPlan, {
          where: { registrationId, deletedAt: IsNull() }
        });

        if (existingTravelPlan) {
          // Use TypeORM query builder to properly set fields to NULL
          await manager
            .createQueryBuilder()
            .update(RegistrationTravelPlan)
            .set({
              travelPlanStatus: 'pending',
              travelType: null,
              returnTravelType: null,
              airlineName: null,
              flightNumber: null,
              arrivalDatetime: null,
              departureLocation: null,
              departureDatetime: null,
              departureFlightNumber: null,
              departureAirlineName: null,
              arrivalFrom: null,
              departureTo: null,
              otherDeparture: null,
              otherArrival: null,
              pickupTime: null,
              pickupLocation: null,
              checkinAt: null,
              checkinTime: null,
              checkinLocation: null,
              onwardAdditionalInfo: null,
              returnAdditionalInfo: null,
              arrivalOtherAirlineName: null,
              departureOtherAirlineName: null,
              updatedBy: { id: userId } as any,
              updatedAt: new Date(),
              auditRefId: existingTravelPlan.id,
              parentRefId: registrationId
            })
            .where('id = :id', { id: existingTravelPlan.id })
            .execute();
          this.logger.log('Travel plan fields reset successfully', { travelPlanId: existingTravelPlan.id });
        }

        // 2. Reset travel plan related question answers in custom responses table
        // Find all travel plan related questions by binding keys and section
        const travelPlanQuestions = await manager
          .createQueryBuilder(Question, 'q')
          .innerJoin('q.formSection', 'fs')
          .select('q.id')
          .where('fs.key = :sectionKey', { sectionKey: 'FS_TRAVELPLAN' })
          .andWhere('q.bindingKey IN (:...bindingKeys)', {
            bindingKeys: [
              'travelPlanOnward', 'travelPlanReturn', 'travelDetails',
              'airlineNameOnward', 'airlineNameReturn', 'flightNumberOnward', 'flightNumberReturn',
              'comingFrom', 'goingTo', 'arrivalDateTime', 'departureDateTime',
              'onwardInfo', 'returnInfo', 'uploadOnwardJourneyTicket', 'uploadReturnJourneyTicket',
              'otherArrival', 'otherDeparture', 'otherAirlineNameOnward', 'otherAirlineNameReturn'
            ]
          })
          .getRawMany();
          
        // Extract question IDs and nullify corresponding question answers
        if (travelPlanQuestions.length > 0) {
          const questionIds = travelPlanQuestions.map(q => q.q_id);

        // Nullify travel plan related question answers
          await manager
            .createQueryBuilder()
            .update(RegistrationQuestionAnswer)
            .set({
              answerValue: null,
              updatedBy: { id: userId } as any,
              updatedAt: new Date()
            })
            .where('registrationId = :registrationId', { registrationId })
            .andWhere('questionId IN (:...questionIds)', { questionIds })
            .execute();
            
          this.logger.log('Travel plan question answers nullified', { 
            registrationId, 
            questionCount: questionIds.length 
          });
        }
      });

      this.logger.log(registrationTravelPlanMessages.TRAVEL_DETAILS_RESET, { registrationId });
    } catch (error) {
      this.logger.error('Failed to reset travel details', '', { registrationId, userId, error });
      handleKnownErrors(ERROR_CODES.REGISTRATION_TRAVEL_PLAN_UPDATE_FAILED, error);
    }
  }

  /**
   * Find save-as-draft registrations with search and basic filters
   */
  async findSaveDraftRegistrations(
    limit: number = 10,
    offset: number = 10,
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    filterRegistrationsByUserId?: number | null,
    sortKey: string = 'id',
    sortOrder: 'ASC' | 'DESC' = 'ASC',
    requiredRelations?: string[], // Optional parameter for relation optimization
    selectedFields?: string[], // Optional parameter for field selection
    relationsSelectedFields?: Record<string, string[]>, // Optional parameter for relation field selection
  ) {
    try {
      const whereClause: any = {
        deletedAt: IsNull(),
        registrationStatus: RegistrationStatusEnum.SAVE_AS_DRAFT,
      };

      if (programId) {
        const program = await this.programRepo.findOne({ where: { id: programId } });
        if (program?.registrationLevel === 'program') {
          whereClause.program = { id: programId };
          whereClause.programSession = IsNull();
        } else if (program?.registrationLevel === 'session') {
          if (programSessionId) {
            whereClause.programSession = { id: programSessionId };
          } else {
            whereClause.program = { id: programId };
          }
        } else {
          whereClause.program = { id: programId };
        }
      } else if (programSessionId) {
        whereClause.programSession = { id: programSessionId };
      }

      if (parsedFilters?.gender) {
        if (Array.isArray(parsedFilters.gender)) {
          whereClause.gender = In(parsedFilters.gender);
        } else {
          whereClause.gender = parsedFilters.gender;
        }
      }

      if (filterRegistrationsByUserId) {
        whereClause.rmContactUser = { id: filterRegistrationsByUserId };
      }

      let finalWhereClause: any = whereClause;

      // FIXED: Search functionality with role filtering
      if (searchText != null && searchText.trim() !== '') {
        const searchTerm = searchText.trim();

        // Create search conditions - each must include the role filter if present
        const searchConditions = [
          { ...whereClause, fullName: ILike(`%${searchTerm}%`) },
          { ...whereClause, emailAddress: ILike(`%${searchTerm}%`) },
          { ...whereClause, mobileNumber: ILike(`%${searchTerm}%`) },
        ];

        finalWhereClause = searchConditions;
      }

      // Define default relations for save draft registrations
      const defaultRelations = ['program', 'program.type', 'programSession',
         'rmContactUser', 'user', 'user.profileExtension', 
         'user.programExperiences', 'user.programExperiences.lookupData',
         'user.participationSummary'
      ];
      
      // Use requiredRelations if provided, otherwise use defaultRelations
      const relations = requiredRelations && requiredRelations.length > 0 ? requiredRelations : defaultRelations;

      // Map frontend sortKey to database field name
      const sortKeyMapping: Record<string, string> = {
        'seekerName': 'fullName',
        'age': 'dob',
        'location': 'city',
        'numberOfHDBs': 'noOfHDBs',
        'lastHdbMsd': 'lastHdbAttended',
        'blessedWith': 'allocatedProgramId',
        'paymentStatus': 'paymentDetails.paymentStatus',
        'travelPlanStatus': 'travelPlans.travelPlanStatus',
        'invoiceStatus': 'invoiceDetails.invoiceStatus',
        'averageRating': 'averageRating',
        'profileUrl': 'profileUrl',
        'mobileNumber': 'mobileNumber',
        'emailAddress': 'emailAddress',
        'fullName': 'fullName',
        'gender': 'gender',
        'city': 'city',
        'noOfHDBs': 'noOfHDBs',
        'id': 'id',
        'dob': 'dob'
      };
      
      // Convert sortKey to database field name
      const dbSortKey = sortKeyMapping[sortKey] || sortKey;

      const data = await this.commonDataService.getWithRelFields(
        this.registrationRepo,
        selectedFields,
        finalWhereClause,
        limit,
        offset,
        { [dbSortKey]: sortOrder },
        undefined,
        relations,
        relationsSelectedFields
      );

      // FIXED: Use finalWhereClause for count to match the data query conditions
      const total = await this.registrationRepo.count({ where: finalWhereClause });

      const paginationInfo =
      {
        totalPages: Math.ceil(total / (limit)),
        pageNumber: Math.floor((offset) / (limit)) + 1,
        pageSize: limit,
        totalRecords: total,
        numberOfRecords: data.length,
      };

      return { data, pagination: paginationInfo };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }


/**
 * 
 * @param programId - The ID of the program for which to fetch registration metrics
 * @returns - An object containing various registration metrics for the specified program
 */
async getAllRegistrationMetrics(programId: number): Promise<any> {
    try {
      const approvalStatusesForAllKpis = [
        ApprovalStatusEnum.REJECTED,
        ApprovalStatusEnum.PENDING,
        ApprovalStatusEnum.ON_HOLD,
      ];
      const allKpis = {
        totalRegistrations: 0,
        femaleRegistrations: 0,
        maleRegistrations: 0,
        femaleRegistrationsOrgCount: 0,
        maleRegistrationsOrgCount: 0,
        totalRegistrationsOrgCount: 0,
        mahatriaChoicesRegistrations: 0,
      };
      const totalAllCountQuery = this.registrationRepo
        .createQueryBuilder('registration')
        .leftJoin('registration.user', 'user')
        .select(`COUNT(registration.id)`, 'totalRegistrations')
        .addSelect(
          `COUNT(CASE WHEN user.user_type = '${UserTypeEnum.ORG}' THEN registration.id END)`,
          'totalRegistrationsOrgCount',
        )
        .where('registration.deletedAt IS NULL')
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
        .andWhere('registration.program_id = :programId', { programId })
        .setParameters({
          excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT],
        });

      const totalAll = await totalAllCountQuery.getRawOne();
      allKpis.totalRegistrations = parseInt(totalAll.totalRegistrations) || 0;
      allKpis.totalRegistrationsOrgCount =
        parseInt(totalAll.totalRegistrationsOrgCount) || 0;

      const allGenderCountQuery = this.registrationRepo
        .createQueryBuilder('registration')
        .leftJoin('registration.user', 'user')
        .select(
          `COUNT(CASE WHEN registration.gender = '${GenderEnum.FEMALE}' THEN registration.id END)`,
          'femaleRegistrations',
        )
        .addSelect(
          `COUNT(CASE WHEN registration.gender = '${GenderEnum.MALE}' THEN registration.id END)`,
          'maleRegistrations',
        )
        .addSelect(
          `COUNT(CASE WHEN registration.gender = '${GenderEnum.FEMALE}' AND user.user_type = '${UserTypeEnum.ORG}' THEN registration.id END)`,
          'femaleRegistrationsOrgCount',
        )
        .addSelect(
          `COUNT(CASE WHEN registration.gender = '${GenderEnum.MALE}' AND user.user_type = '${UserTypeEnum.ORG}' THEN registration.id END)`,
          'maleRegistrationsOrgCount',
        )
        .leftJoin('registration.approvals', 'approval')
        .where('registration.deletedAt IS NULL')
        .andWhere('registration.program_id = :programId', { programId })
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
        // .andWhere('approval.approval_status IN (:...statuses)')
        .setParameters({
          excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT],
          // statuses: approvalStatusesForAllKpis,
        });
      const allGenderCount = await allGenderCountQuery.getRawOne();
      allKpis.femaleRegistrations = parseInt(allGenderCount.femaleRegistrations) || 0;
      allKpis.maleRegistrations = parseInt(allGenderCount.maleRegistrations) || 0;
      allKpis.femaleRegistrationsOrgCount =
        parseInt(allGenderCount.femaleRegistrationsOrgCount) || 0;
      allKpis.maleRegistrationsOrgCount =
        parseInt(allGenderCount.maleRegistrationsOrgCount) || 0;

      const allMahatriaChoicesCountQuery = this.registrationRepo
        .createQueryBuilder('registration')
        .select('COUNT(registration.id)', 'mahatriaChoicesRegistrations')
        .addSelect(
          `COUNT(CASE WHEN user.user_type = '${UserTypeEnum.ORG}' THEN registration.id END)`,
          'mahatriaChoicesRegistrationsOrgCount',
        )
        .leftJoin('registration.user', 'user')
        .leftJoin('registration.approvals', 'approval')
        .leftJoin('registration.preferences', 'preference')
        // .where('registration.deletedAt IS NULL')
        .andWhere('registration.program_id = :programId', { programId })
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
        .andWhere('approval.approval_status IN (:...statuses)')
        .andWhere('preference.id IS NULL')
        .setParameters({
          excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
          statuses: approvalStatusesForAllKpis,
        });

      const allMahatriaChoicesCount = await allMahatriaChoicesCountQuery.getRawOne();
      allKpis.mahatriaChoicesRegistrations =
        parseInt(allMahatriaChoicesCount.mahatriaChoicesRegistrations) || 0;

      const subProgramPreferredRegistrationsQuery = await this.programRepo
        .createQueryBuilder('subProgram')
        .select('subProgram.id', 'subProgramId')
        .addSelect('subProgram.name', 'subProgramName')
        .addSelect('ARRAY_AGG(DISTINCT registration.id)', 'registrationIds')
        .addSelect(
          `COUNT(DISTINCT CASE WHEN user.user_type = '${UserTypeEnum.ORG}' THEN registration.id END)`,
          'orgCount'
        )
        .leftJoin('hdb_preference', 'preference', 'preference.preferred_program_id = subProgram.id ')
        .leftJoin(
          'hdb_program_registration',
          'registration',
          'registration.id = preference.registration_id',
        )
        .leftJoin('registration.approvals', 'approval')
        .leftJoin('registration.user', 'user')
        .where('subProgram.primary_program_id = :programId', { programId })
        .andWhere('registration.deletedAt IS NULL')
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
        .andWhere('approval.approval_status IN (:...statuses)')
        .andWhere('preference.priority_order = 1')
        .setParameters({
          statuses: approvalStatusesForAllKpis,
          excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED],
        })
        .groupBy('subProgram.id')
        .addGroupBy('subProgram.name');

      const subProgramPreferredRegistrations =
        await subProgramPreferredRegistrationsQuery.getRawMany();
      const subProgramMetrics: {
        subProgramId: number;
        subProgramName: string;
        count: number;
        orgCount: number;
      }[] = [];

      const allSubPrograms = await this.getSubPrograms(programId);

      for (const subProgram of allSubPrograms) {
        const preferred = subProgramPreferredRegistrations.filter(
          (item) => item.subProgramId === subProgram.id,
        )[0];

        const allIds = new Set<string>();
        if (preferred?.registrationIds) {
          for (const id of preferred.registrationIds) allIds.add(id);
        }
        subProgramMetrics.push({
          subProgramId: subProgram.id,
          subProgramName: subProgram.name,
          count: allIds.size,
          orgCount: preferred?.orgCount ? Number(preferred.orgCount) : 0,
        });
      }
      return {
        ...allKpis,
        subProgramMetrics,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

/**
 * Fetch male and female count of cancelled registrations of a program
 * @param programId - The ID of the program for which to fetch cancelled registration metrics
 * @returns - An object containing counts
 */
async getCancelledRegistrationMetrics(programId: number): Promise<any> {
    try {

      const cancelledRegistrations = await this.registrationRepo
        .createQueryBuilder('registration')
        .leftJoin('registration.user', 'user')
        .where('registration.registration_status = :status', { status: RegistrationStatusEnum.CANCELLED })
        .andWhere('registration.deletedAt IS NULL')
        .andWhere('registration.program_id = :programId', { programId })
        .select('COUNT(registration.id)', 'totalCancelledRegistrations')
        .addSelect('SUM(CASE WHEN registration.gender = :male THEN 1 ELSE 0 END)', 'maleCancelledRegistrations')
        .addSelect('SUM(CASE WHEN registration.gender = :female THEN 1 ELSE 0 END)', 'femaleCancelledRegistrations')
        .addSelect('SUM(CASE WHEN user.user_type = :org THEN 1 ELSE 0 END)', 'totalCancelledRegistrationsOrgCount')
        .addSelect('SUM(CASE WHEN user.user_type = :org AND registration.gender = :male THEN 1 ELSE 0 END)', 'maleCancelledRegistrationsOrgCount')
        .addSelect('SUM(CASE WHEN user.user_type = :org AND registration.gender = :female THEN 1 ELSE 0 END)', 'femaleCancelledRegistrationsOrgCount')
        .setParameters({ male: GenderEnum.MALE, female: GenderEnum.FEMALE, org: UserTypeEnum.ORG })
        .getRawOne();

      return {
        kpiCategory: 'cancelled',
        kpiFilter: 'cancelled',
        categoryLabel: 'Cancelled',
        totalCancelledRegistrations: Number(cancelledRegistrations.totalCancelledRegistrations) || 0,
        maleCancelledRegistrations: Number(cancelledRegistrations.maleCancelledRegistrations) || 0,
        femaleCancelledRegistrations: Number(cancelledRegistrations.femaleCancelledRegistrations) || 0,
        totalCancelledRegistrationsOrgCount: Number(cancelledRegistrations.totalCancelledRegistrationsOrgCount) || 0,
        maleCancelledRegistrationsOrgCount: Number(cancelledRegistrations.maleCancelledRegistrationsOrgCount) || 0,
        femaleCancelledRegistrationsOrgCount: Number(cancelledRegistrations.femaleCancelledRegistrationsOrgCount) || 0,
        programId: Number(programId),
        status: String(RegistrationStatusEnum.CANCELLED),
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  // update registration status
  async updateRegistrationStatus(
    manager: EntityManager,
    registrationId: number,
    status: RegistrationStatusEnum,
    userId: number
  ): Promise<void> {
    const registration = await manager.findOne(ProgramRegistration, {
      where: { id: registrationId, deletedAt: IsNull() },
      relations: ['program', 'programSession'],
    });

    if (!registration) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        registrationId.toString(),
      );
    }

    this.logger.log(`Updating registration ${registrationId} status to ${status}`);
    // Update the registration status
    await manager.update(ProgramRegistration, registrationId, {
      registrationStatus: status,
      updatedBy: { id: userId } as any,
      auditRefId: registrationId,
      parentRefId: registrationId,
    });

    this.logger.log(`Registration ${registrationId} status updated to ${status}`);
  }

  async updateApprovalStatus(
    registrationId: number,
    status: ApprovalStatusEnum,
    userId: number,
  ) {
    try {
      const approval = await this.approvalRepo.findOne({
        where: { registrationId, deletedAt: IsNull() },
        order: { createdAt: 'DESC' },
      });

      if (!approval) {
        throw new InifniNotFoundException(
          ERROR_CODES.REGISTRATION_APPROVAL_NOTFOUND,
          null,
          null,
          registrationId.toString(),
        );
      }
      // if the status is approved update approvalDate else same date
      const approvalDate = status === ApprovalStatusEnum.APPROVED ? new Date() : approval.approvalDate;

      await this.approvalRepo.update(
        { id: approval.id },
        {
          approvalStatus: status,
          approvalDate,
          updatedBy: { id: userId } as any,
          auditRefId: approval.id,
          parentRefId: approval.registrationId,
        },
      );
      return await this.approvalRepo.findOne({ where: { id: approval.id } });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }
  async findUniqueRegistrationsForPrograms(
    programId: number,
    allocatedProgramId?: number,
    limit?: number,
    offset?: number,
  ) {
    try {
      const registrations = await this.registrationRepo
        .createQueryBuilder('registration')
        .leftJoin('registration.program', 'program')
        .leftJoin('program.type', 'programType')
        .leftJoin('registration.approvals', 'approval') // Join with approvals
        .where('registration.program.id = :programId', { programId })
        .andWhere('registration.userId IS NOT NULL') // Ignore null userId registrations
        .andWhere('approval.approvalStatus = :approvedStatus', { approvedStatus: ApprovalStatusEnum.APPROVED }) // Only approved registrations
        .andWhere((qb) => {
          const subQuery = qb
            .subQuery()
            .select('COUNT(r.id)')
            .from(ProgramRegistration, 'r')
            .leftJoin('r.program', 'p')
            .leftJoin('p.type', 'pt')
            .where('r.userId = registration.userId')
            .andWhere('pt.id = programType.id') // Same program type
            .andWhere('r.program.id != :programId', { programId }) // Exclude the current program
            .andWhere('r.deletedAt IS NULL') // Exclude soft-deleted registrations
            .getQuery();
          return `(${subQuery}) = 0`; // Ensure no previous registrations for the same type
        })
        .andWhere('registration.deletedAt IS NULL') // Exclude soft-deleted registrations
        .orderBy('registration.id', 'DESC');

      if (allocatedProgramId) {
        registrations.andWhere('registration.allocated_program_id = :allocatedProgramId', { allocatedProgramId });
      }

      if (limit !== undefined && offset !== undefined) {
        registrations.take(limit).skip(offset);
      }
      const result = await registrations.getMany();
      return result;
    } catch (error) {
      this.logger.error('Error fetching registrations for unique programs by programId', error);
      throw new InifniBadRequestException(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

    private determineParentRefId(entity: any): number | null {
    // For User entity, parentRefId should be null as mentioned in the requirements
    const entityClassName = entity.constructor.name;
    if (entityClassName === 'User') {
      return null;
    }

    // Check if entity has a registrationId field (most common parent reference)
    if (entity.registrationId) {
      return entity.registrationId;
    }

    // Check if entity has a relation.id pattern (for entities like Preference that use relations)
    if (entity.registration?.id) {
      return entity.registration.id;
    }

    // Check if entity has a programRegistrationId field
    if (entity.programRegistrationId) {
      return entity.programRegistrationId;
    }

    // Check if entity has a programRegistration relation
    if (entity.programRegistration?.id) {
      return entity.programRegistration.id;
    }

    // For entities that don't have a clear parent, use their own ID as parentRefId
    // This follows the pattern seen in ProgramRegistration and other root entities
    if (entity.id) {
      return entity.id;
    }

    return null;
  }

  /**
   * Get subProgram-wise registration counts with approval status breakdown
   * Only for mahatria and shoba roles
   * @param programId - The primary program ID to get sub-program counts for
   * @returns Object with global counts and array of subProgram registration counts
   */
  async getSubProgramStatusCounts(
    programId: number,
  ): Promise<any> {
    try {
      this.logger.debug(`Getting subProgram status counts for programId: ${programId}`);

      // First, get all sub-programs under the given primary program
      const subPrograms = await this.programRepo.find({
        where: {
          primaryProgramId: programId,
          deletedAt: IsNull(),
        },
        select: ['id', 'name', 'totalSeats','startsAt'],
        order: { groupDisplayOrder: 'ASC' }
      });

      if (subPrograms.length === 0) {
        this.logger.debug(`No sub-programs found for programId: ${programId}`);
        return {
          0: { label: 'Hold', category: 'status', count: 0, startsAt: null },
          1: { label: 'Swap demand', category: 'status', count: 0, startsAt: null }
        };
      }

      const subProgramIds = subPrograms.map(program => program.id);

      // Get global holdCount and ytdCount for the entire program
      const globalCountsQuery = this.registrationRepo
        .createQueryBuilder('registration')
        .leftJoin('registration.approvals', 'approval')
        .select([
          'COUNT(CASE WHEN approval.approval_status = :rejected THEN 1 END) AS "globalHoldCount"',
          'COUNT(CASE WHEN approval.approval_status = :onHold THEN 1 END) AS "globalYtdCount"'
        ])
        .where('registration.program_id = :programId', { programId })
        .andWhere('registration.deleted_at IS NULL')
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)', { 
          excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED] 
        })
        .setParameters({
          rejected: ApprovalStatusEnum.REJECTED, // Hold
          onHold: ApprovalStatusEnum.ON_HOLD, // YTD
        });

      const globalCounts = await globalCountsQuery.getRawOne();
      
      // Build the query to get registration counts with approval status breakdown per sub-program
      const statusCountsQuery = this.registrationRepo
        .createQueryBuilder('registration')
        .leftJoin('registration.approvals', 'approval')
        .leftJoin('registration.allocatedProgram', 'allocatedProgram')
        .leftJoin('registration.user', 'user')
        .select([
          'allocatedProgram.id AS "subProgramId"',
          'allocatedProgram.name AS "subProgramName"',
          'allocatedProgram.starts_at AS "startsAt"',
          'allocatedProgram.total_seats AS "totalSeatsCount"',
          'COUNT(CASE WHEN approval.approval_status = :approved THEN 1 END) AS "approvedCount"',
          'COUNT(CASE WHEN user.user_type = :orgUserType THEN 1 END) AS "organisationUserCount"'
        ])
        .where('registration.allocated_program_id IN (:...subProgramIds)', { subProgramIds })
        .andWhere('registration.deleted_at IS NULL')
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)', { 
          excludeStatuses: [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED] 
        })
        .setParameters({
          approved: ApprovalStatusEnum.APPROVED,
          orgUserType: UserTypeEnum.ORG
        })
        .groupBy('allocatedProgram.id')
        .addGroupBy('allocatedProgram.name')
        .addGroupBy('allocatedProgram.total_seats');

      const results = await statusCountsQuery.getRawMany();
      
      // Transform results to include sub-programs that have zero registrations
      const statusCountsMap = new Map();
      results.forEach(result => {
        statusCountsMap.set(parseInt(result.subProgramId), {
          subProgramId: parseInt(result.subProgramId),
          subProgramName: result.subProgramName,
          totalSeatsCount: parseInt(result.totalSeatsCount || 0),
          startsAt: result.startsAt,
          label: result.subProgramName,
          count: parseInt(result.approvedCount || 0),
          organisationUserCount: parseInt(result.organisationUserCount || 0),
          category: 'subProgram'
        });
      });

      // Include sub-programs with zero registrations
      const subProgramCounts = subPrograms.map((subProgram, index) => {
        const data = statusCountsMap.get(subProgram.id) || {
          subProgramId: subProgram.id,
          subProgramName: subProgram.name,
          label: subProgram.name,
          startsAt: subProgram.startsAt,
          totalSeatsCount: subProgram.totalSeats || 0,
          count: 0,
          organisationUserCount: 0,
          category: 'subProgram'
        };
        return data;
      });

      const globalData = [
        { label: 'Hold', category: 'status', count: parseInt(globalCounts.globalHoldCount || 0) }, 
        { label: 'Swap demand', category: 'status', count: parseInt(globalCounts.globalYtdCount || 0) }
      ];

      // Combine all data with proper indexing
      const allData = [...subProgramCounts, ...globalData];

      this.logger.debug(`Retrieved ${subProgramCounts.length} subProgram status counts with global counts for programId: ${programId}`);
      return allData;

    } catch (error) {
      this.logger.error(`Error getting subProgram status counts for programId ${programId}:`, error);
      throw error;
    }
  }

  async updateRegistration(registrationId: number, updateData: Partial<ProgramRegistration>, userId: number): Promise<ProgramRegistration> {
    try {
      const registration = await this.registrationRepo.findOne({
        where: { id: registrationId, deletedAt: IsNull() },
      });

      if (!registration) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          registrationId.toString(),
        );
      }

      // Merge the existing registration with the update data
      const updatedRegistration = this.registrationRepo.merge(registration, updateData, {
        updatedBy: { id: userId } as any,
        auditRefId: registrationId,
        parentRefId: registrationId,
      });

      // Save the updated registration
      return await this.registrationRepo.save(updatedRegistration);
    } catch (error) {
      this.logger.error('Error updating registration', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  /**
   * Get basic registrations with minimal fields (id, userId, name)
   * Filtered by rmContact for relational_manager role and programId
   * @param rmContactId - Filter by RM contact ID (for relational_manager role)
   * @param programId - Program ID to filter registrations
   * @param searchText - Search text to filter by name, phone number, or email
   * @returns Array of comprehensive registration and user data
   */
  async getBasicRegistrations(rmContactId?: number, programId?: number, searchText?: string): Promise<{
    id: number;
    userId: number;
    userName: string | null;
    userLegalName: string | null;
    userFirstName: string | null;
    userLastName: string | null;
    userPhoneNumber: string | null;
    userCountryCode: string | null;
    userEmailAddress: string | null;
    userGender: string | null;
    userDob: Date | null;
    registrationName: string | null;
    registrationSeqNumber: string | null;
    registrationPhoneNumber: string | null;
    registrationEmailAddress: string | null;
    registrationGender: string | null;
    registrationDob: Date | null;
  }[]> {
    try {
      const whereClause: any = {
        deletedAt: IsNull(),
        registrationStatus: RegistrationStatusEnum.COMPLETED,
      };

      // If rmContactId is provided, filter by rmContact
      if (rmContactId) {
        whereClause.rmContactUser = { id: rmContactId };
      }

      // If programId is provided, filter by programId
      if (programId) {
        whereClause.program = { id: programId };
      }

      // Add search functionality
      let finalWhereClause: any = whereClause;
      if (searchText && searchText.trim()!== '') {
        const searchTerm = ILike(`%${searchText.trim()}%`);
        
        // Create search conditions that properly inherit all base filters
        const createSearchCondition = (userField: string) => {
          return {
            ...whereClause,
            user: {
              ...whereClause.user,
              [userField]: searchTerm,
            },
          };
        };

        finalWhereClause = [
          createSearchCondition('fullName'),
          createSearchCondition('firstName'),
          createSearchCondition('lastName'),
          createSearchCondition('legalFullName'),
          createSearchCondition('phoneNumber'),
          createSearchCondition('email'),
        ];
      }

      const registrations = await this.registrationRepo.find({
        where: finalWhereClause,
        relations: ['user'],
        select: {
          id: true,
          userId: true,
          registrationSeqNumber: true,
          user: {
            id: true,
            fullName: true,
            firstName: true,
            lastName: true,
            legalFullName: true,
            phoneNumber: true,
            countryCode: true,
            email: true,
          },
        },
        order: { id: 'DESC' },
      });

      return registrations.map(registration => {
        return {
          id: registration.id,
          // user data
          userId: registration.userId,
          userName: registration.user?.fullName ?? null,
          userLegalName: registration.user?.legalFullName ?? null,
          userFirstName: registration.user?.firstName ?? null,
          userLastName: registration.user?.lastName ?? null,
          userPhoneNumber: registration.user?.phoneNumber ?? null,
          userCountryCode: registration.user?.countryCode ?? null,
          userEmailAddress: registration.user?.email ?? null,
          userGender: registration.user?.gender ?? null,
          userDob: registration.user?.dob ?? null,
          // registration data
          registrationName: registration?.fullName ?? null,
          registrationSeqNumber: registration?.registrationSeqNumber ?? null,
          registrationPhoneNumber: registration?.mobileNumber ?? null,
          registrationEmailAddress: registration?.emailAddress ?? null,
          registrationGender: registration?.gender ?? null,
          registrationDob: registration?.dob ?? null
        };
      });
    } catch (error) {
      this.logger.error('Error getting basic registrations:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  async getExperienceTags() {
  // from lookupdata get seeker_program_experience tags
    try {
      const tags = await this.lookupDataRepo.find({
        where: {
          lookupCategory: LOOKUP_CATEGORY.SEEKER_PROGRAM_EXPERIENCE,
          lookupStatus: CommonStatus.ACTIVE,
        },
        order: { lookupOrder: 'ASC' },
      });
      return tags;
    } catch (error) {
      this.logger.error('Error fetching experience tags', error);
  }
}

async clearAllRegistrationsForProgram(programId: number, userId: number, manager: EntityManager): Promise<{ clearedCount: number }> {
    try {
      this.logger.log(replaceStringPlaceholders(CLEAR_REGISTRATION_MESSAGES.CLEARING_REGISTRATIONS, [programId.toString(), userId.toString()]));
      
      // Get count of active registrations before deletion
      const activeRegistrations = await manager
        .createQueryBuilder()
        .select('COUNT(*)', 'count')
        .from(ProgramRegistration, 'pr')
        .where('pr.programId = :programId', { programId })
        .andWhere('pr.deletedAt IS NULL')
        .getRawOne();
      
      const initialCount = parseInt(activeRegistrations.count) || 0;
      
      if (initialCount === 0) {
        this.logger.log(replaceStringPlaceholders(CLEAR_REGISTRATION_MESSAGES.NO_ACTIVE_REGISTRATIONS, [programId.toString()]));
        return { clearedCount: 0 };
      }

      // Use a single bulk update to soft delete all registrations
      const currentDate = new Date();
      const updateResult = await manager
        .createQueryBuilder()
        .update(ProgramRegistration)
        .set({
          deletedAt: currentDate,
          updatedAt: currentDate,
          updatedBy: { id: userId } as any
        })
        .where('programId = :programId', { programId })
        .andWhere('deletedAt IS NULL')
        .execute();

      const clearedCount = updateResult.affected || 0;
      this.logger.log(replaceStringPlaceholders(CLEAR_REGISTRATION_MESSAGES.SOFT_DELETED_REGISTRATIONS, [clearedCount.toString(), programId.toString()]));

      return { clearedCount };
    } catch (error) {
      this.logger.error('Error clearing registrations for program', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_DELETE_FAILED, error);
    }
  }
  async checkActiveWantsSwapRequestForRegistration(registrationId: number): Promise<boolean> {
    try {
      const activeSwapRequest = await this.swapRequestRepo.findOne({
        where: {
          programRegistrationId: registrationId,
          status: In([SwapRequestStatus.ACTIVE]),
          type: In([SwapType.WantsSwap]),
        },
      });
      return !!activeSwapRequest;
    } catch (error) {
      this.logger.error('Error checking active swap request for registration', error);
      handleKnownErrors(ERROR_CODES.SWAP_REQUEST_CHECK_FAILED, error);
    }
  }
  async checkActiveSwapRequestForRegistration(registrationId: number): Promise<boolean> {
    try {
      const activeSwapRequest = await this.swapRequestRepo.findOne({
        where: {
          programRegistrationId: registrationId,
          status: In([SwapRequestStatus.ACTIVE]),
        },
      });
      return !!activeSwapRequest;
    } catch (error) {
      this.logger.error('Error checking active swap request for registration', error);
      handleKnownErrors(ERROR_CODES.SWAP_REQUEST_CHECK_FAILED, error);
    }
  }

  async checkActiveCanShiftRequestForRegistration(registrationId: number): Promise<boolean> {
    try {
      const activeSwapRequest = await this.swapRequestRepo.findOne({
        where: {
          programRegistrationId: registrationId,
          status: In([SwapRequestStatus.ACTIVE]),
          type: In([SwapType.CanShift]),
        },
      });
      return !!activeSwapRequest;
    } catch (error) {
      this.logger.error('Error checking active can shift request for registration', error);
      handleKnownErrors(ERROR_CODES.SWAP_REQUEST_CHECK_FAILED, error);
    }
  }
  /**
   * Get total registrations with approval status ON_HOLD or REJECTED and preference or swap demand for a program
   */
  async getTotalPendingRegistrationsWithPreference(
    programId: number,
    pendingProgramId?: number | null,
    parsedFilters?: Record<string, any>,
    searchText?: string,
    filterRegistrationsByUserId?: number | null
  ) {
    try {
      if (!programId) {
        this.logger.warn(`Invalid programId provided to getTotalPendingRegistrationsWithPreference: ${programId}`);
        throw new InifniBadRequestException(`Invalid programId: ${programId}`);
      }

      // If pendingProgramId is not provided, return counts of ON_HOLD, REJECTED, and org users
      if (!pendingProgramId) {
        const qb = this.registrationRepo
          .createQueryBuilder('registration')
          .leftJoin('registration.user', 'user')
          .leftJoin('registration.approvals', 'approval')
          .select([
            'COUNT(DISTINCT CASE WHEN user.user_type = :orgType THEN registration.id END) AS orgCount',
            `COUNT(DISTINCT registration.id) AS totalCount`,
          ])
          .where('registration.program_id = :programId', { programId })
          .andWhere('registration.registration_status IN (:...statuses)', {
            statuses: [
              RegistrationStatusEnum.REJECTED,
              RegistrationStatusEnum.PENDING,
            ],
          })
          .andWhere('registration.deleted_at IS NULL')
          .andWhere('approval.approval_status IN (:onHold, :rejected)')
          .setParameters({
            onHold: ApprovalStatusEnum.ON_HOLD,
            rejected: ApprovalStatusEnum.REJECTED,
            orgType: 'Org',
          });
        if (filterRegistrationsByUserId) {
          qb.andWhere('registration.rm_contact = :rmContact', { rmContact: filterRegistrationsByUserId });
        }
        const result = await qb.getRawOne();
        return {
          totalCount: result.totalcount || '0',
          orgCount: result.orgcount || '0',
        };
      }

      // ...existing code for pendingProgramId provided...
      // Preference-based (priority 1, REJECTED)
      const prefSubQuery = this.preferenceRepo
        .createQueryBuilder('preference')
        .select('preference.registration_id')
        .leftJoin('preference.registration', 'registration')
        .leftJoin('registration.approvals', 'approval')
        .where('preference.priority_order = 1')
        .andWhere('preference.preferred_program_id = :pendingProgramId', { pendingProgramId })
        .andWhere('preference.deleted_at IS NULL')
        .andWhere('approval.approval_status IN (:...statuses)', { statuses: [ApprovalStatusEnum.REJECTED] });
      const prefIds = await prefSubQuery.getRawMany().then(rows => rows.map(r => r.registration_id));

      // Swap demand-based (latest ON_HOLD, SWAP_DEMAND)
      const swapDemandIds = await this.registrationRepo
        .createQueryBuilder('reg')
        .select('reg.id', 'id')
        .leftJoin('reg.approvals', 'approval')
        .where('reg.deleted_at IS NULL')
        .andWhere('approval.approval_status IN (:...statuses)', { statuses: [ApprovalStatusEnum.ON_HOLD] })
        .andWhere(`(SELECT srp.program_id FROM hdb_swap_requested_program srp WHERE srp.swap_request_id = (SELECT sr.id FROM hdb_program_registration_swap sr WHERE sr.program_registration_id = reg.id AND sr.status = :swapOnHoldStatus AND sr.swap_requirement = :swapDemandRequirement ORDER BY sr.id DESC LIMIT 1) ORDER BY srp.id ASC LIMIT 1) = :pendingProgramId`, {
          pendingProgramId,
          swapOnHoldStatus: SwapRequestStatus.ON_HOLD,
          swapDemandRequirement: SwapRequirementEnum.SWAP_DEMAND,
        })
        .getRawMany()
        .then(rows => rows.map(r => r.id));

      // Combine and deduplicate registration IDs
      const allIds = Array.from(new Set([...prefIds, ...swapDemandIds]));

      // Filter by programId, status, and rm_contact if needed
      let filteredIds = allIds;
      if (allIds.length > 0) {
        let regQb = this.registrationRepo
          .createQueryBuilder('registration')
          .leftJoin('registration.user', 'user')
          .select([
            'registration.id',
            'user.user_type'
          ])
          .where('registration.id IN (:...ids)', { ids: allIds })
          .andWhere('registration.program_id = :programId', { programId })
          .andWhere('registration.registration_status NOT IN (:...excludeStatuses)', {
            excludeStatuses: [
              RegistrationStatusEnum.SAVE_AS_DRAFT,
              RegistrationStatusEnum.CANCELLED,
            ],
          })
          .andWhere('registration.deleted_at IS NULL');
        if (filterRegistrationsByUserId) {
          regQb = regQb.andWhere('registration.rm_contact = :rmContact', { rmContact: filterRegistrationsByUserId });
        }
        const regRows = await regQb.getRawMany();
        filteredIds = regRows.map(r => r.registration_id || r.id);
        // Count org users
        const orgCount = regRows.filter(r => r.user_user_type === 'Org').length;
        this.logger.log(`Successfully retrieved pending registrations with preference for programId: ${programId}, pendingProgramId: ${pendingProgramId}`);
        return { totalCount: String(filteredIds.length), orgCount: String(orgCount) };
      } else {
        return { totalCount: '0', orgCount: '0' };
      }
    } catch (error) {
      this.logger.error(`Failed to retrieve pending registrations with preference for programId: ${programId}, pendingProgramId: ${pendingProgramId}`, error.stack);
      throw new InifniNotFoundException(`Failed to retrieve pending registrations with preference`);
    }
  }
}