import { Injectable } from '@nestjs/common';
import { DataSource, IsNull, QueryRunner, Repository } from 'typeorm';
import { InjectDataSource, InjectRepository } from '@nestjs/typeorm';
import { forwardRef, Inject } from '@nestjs/common';
import { CreateRegistrationApprovalDto } from './dto/create-registration-approval.dto';
import { UpdateRegistrationApprovalDto } from './dto/update-registration-approval.dto';
import { RegistrationApprovalRepository } from './registration-approval.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CommonDataService } from 'src/common/services/commonData.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import {
  registrationApprovalMessages,
  stateGstCodes,
  zeptoEmailCreadentials,
} from 'src/common/constants/strings-constants';
import { CommunicationService } from 'src/communication/communication.service';
import { SendSingleEmailDto } from 'src/communication/dto/email-communication.dto';
import {
  amountInWordsIndian,
  areGSTsFromSameState,
  deductDaysFromDate,
  formatDateIST,
  generatePaymentLink,
  getWeekName,
} from 'src/common/utils/common.util';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { UserRepository } from 'src/user/user.repository';
import { InvoiceTemplateDto, ProFormaInvoice } from 'src/common/templates/dtos/templates.dto';
import { INVOICE, UPDATED_PRO_FORMA_INVOICE } from 'src/common/templates/templates';
import axios from 'axios';
import { AwsS3Service } from 'src/common/services/awsS3.service';
import { v4 as uuidv4 } from 'uuid';
import {
  ProgramRegistration,
  RegistrationApproval,
  RegistrationApprovalTrack
} from 'src/common/entities';
import { SEEKER_FE_REG_PATH } from 'src/common/constants/constants';
import { PaymentModeEnum } from 'src/common/enum/payment-mode.enum';
import chromium from '@sparticuz/chromium-min';
import { RegistrationService } from 'src/registration/registration.service';
import { SubProgramTypeEnum } from 'src/common/enum/sub-program-type.enum';
import { RegistrationRepository } from 'src/registration/registration.repository';
import { Msg91Service } from 'src/common/services/msg91.service';
import { ConfigService } from '@nestjs/config';
import { ProgramRegistrationService } from 'src/program-registration/program-registration.service';
import { ProgramRoomInventoryMapService } from 'src/room-inventory/program-room-inventory-map.service';
import { AllocationClearingService } from 'src/common/services/allocation-clearing.service';

const puppeteer = require('puppeteer-core');

@Injectable()
export class RegistrationApprovalService {
  private repo: RegistrationApprovalRepository;
  private msg91BlessNotifyMessageTemplateId: string;

  constructor(
    @InjectDataSource()
    private readonly dataSource: DataSource,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepo: Repository<ProgramRegistration>,
    @InjectRepository(RegistrationApproval)
    private readonly registrationApprovalRepo: Repository<RegistrationApproval>,
    @InjectRepository(RegistrationApprovalTrack)
    private readonly approvalTrackRepo: Repository<RegistrationApprovalTrack>,
    private readonly logger: AppLoggerService,
    private readonly communicationService: CommunicationService,
    private readonly userRepo: UserRepository,
    private readonly commonService: CommonDataService,
    private readonly awsS3Service: AwsS3Service,
    @Inject(forwardRef(() => RegistrationService))
    private readonly registrationService: RegistrationService,
    @Inject(forwardRef(() => RegistrationRepository))
    private readonly registrationRepository: RegistrationRepository,
    private readonly msg91Service: Msg91Service,
    private readonly configService: ConfigService,
    @Inject(forwardRef(() => ProgramRegistrationService))
    private readonly programRegistrationService: ProgramRegistrationService,
    @Inject(forwardRef(() => ProgramRoomInventoryMapService))
    private readonly programRoomInventoryMapService: ProgramRoomInventoryMapService,
    private readonly allocationClearingService: AllocationClearingService,
  ) {
    this.repo = new RegistrationApprovalRepository(
      this.registrationApprovalRepo,
      this.commonService,
      this.logger,
      this.registrationRepo,
      this.approvalTrackRepo,
    );
    this.msg91BlessNotifyMessageTemplateId = this.configService.get<string>('MSG91_RM_BLESS_NOTIFY_TEMPLATE_ID') ?? '';
  }

  async create(dto: CreateRegistrationApprovalDto) {
    this.logger.log(registrationApprovalMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationApprovalRepository(
          manager.getRepository(RegistrationApproval),
          this.commonService,
          this.logger,
          manager.getRepository(ProgramRegistration),
          manager.getRepository(RegistrationApprovalTrack),
        );
        return await repository.createEntity(dto);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number, programId?: number, search?: string, filters: Record<string, any> = {}) {
    this.logger.log(registrationApprovalMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      return await this.repo.findAll(limit, offset, programId, search, filters);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(registrationApprovalMessages.FIND_ONE_REQUEST_RECEIVED);
    try {
      return await this.repo.findOne(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_FIND_BY_ID_FAILED, error);
    }
  }

  async update(id: number, dto: UpdateRegistrationApprovalDto) {
    this.logger.log(registrationApprovalMessages.UPDATE_REQUEST_RECEIVED);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationApprovalRepository(
          manager.getRepository(RegistrationApproval),
          this.commonService,
          this.logger,
          manager.getRepository(ProgramRegistration),
          manager.getRepository(RegistrationApprovalTrack),
        );
        return await repository.updateEntity(id, dto);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    }
  }

  async updateByRegistrationId(registrationId: number, dto: UpdateRegistrationApprovalDto) {
    this.logger.log('Update approval by registrationId', { registrationId, dto });
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    await queryRunner.startTransaction();
    try {
      const result = await this.repo.updateByRegistrationId(registrationId, dto, queryRunner);
      await queryRunner.commitTransaction();
      return result;
    } catch (error) {
      await queryRunner.rollbackTransaction();
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_SAVE_FAILED, error);
    } finally {
      await queryRunner.release();
    }
  }

  async remove(id: number) {
    this.logger.log(registrationApprovalMessages.DELETE_REQUEST_RECEIVED);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const repository = new RegistrationApprovalRepository(
          manager.getRepository(RegistrationApproval),
          this.commonService,
          this.logger,
          manager.getRepository(ProgramRegistration),
          manager.getRepository(RegistrationApprovalTrack),
        );
        return await repository.remove(id);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_DELETE_FAILED, error);
    }
  }

  async findOneByRegistrationId(registrationId: number) {
    this.logger.log('Find approval by registrationId', { registrationId });
    try {
      return await this.repo.findByRegistrationId(registrationId);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_APPROVAL_NOTFOUND, error);
    }
  }

  /**
   * Send MSG91 SMS notification to RM about registration approval status
   * @param rmContactUser - The RM contact user object with phone details
   * @param rmName - Name of the RM (Relationship Manager)
   * @param registrationName - Name of the person who registered
   * @param userPhoneNumber - Phone number of the registered user (formatted with country code)
   * @param blessedProgramNameWithDates - Program name with start and end dates
   */
  
  private async sendRmMsg91Notification(
    rmContactUser: { countryCode: string; phoneNumber: string },
    rmName: string,
    registrationName: string,
    userPhoneNumber: string,
    blessedProgramNameWithDates: string,
  ): Promise<void> {
    if (!rmContactUser?.countryCode || !rmContactUser?.phoneNumber) {
      this.logger.warn(`Missing RM contact details for user ${userPhoneNumber}`);
      return;
    }

    try {
      const phoneNumber = `${rmContactUser.countryCode}${rmContactUser.phoneNumber}`.replace('+', '');
      await this.msg91Service.sendSms(
        phoneNumber,
        { 
          var1: rmName,
          var2: registrationName,
          var3: userPhoneNumber,
          var4: blessedProgramNameWithDates
        },
        this.msg91BlessNotifyMessageTemplateId,
      );
    } catch (error) {
      this.logger.log(
        `Error sending MSG91 notification to RM for user ${userPhoneNumber}: ${error}`,
      );
    } finally {
      return;
    }
  }
  
  async sendApprovalEmailNotification(
    approvalId: number,
    trackHistory: RegistrationApprovalTrack[],
    _existingApprovalStatus?: ApprovalStatusEnum
  ) {
    try {
      const approval = await this.findOne(approvalId);
      const programRegistration = approval.registration;
      this.logger.log('Approval Details:', approval);
      let bccEmails: { emailAddress: string; name: string }[] = [];
      if (process.env.HDB_BCC_EMAIL) {
        bccEmails.push({
          emailAddress: process.env.HDB_BCC_EMAIL,
          name: process.env.HDB_BCC_EMAIL_NAME || 'HDB Team',
        });
      }
      try {
        const financeUsers = await this.userRepo.getUsersByRoleKey('ROLE_FINANCE_MANAGER');
        if (financeUsers && financeUsers.length > 0) {
          bccEmails = bccEmails.concat(
            financeUsers.map((u) => ({
              emailAddress: u.email,
              name: u.fullName || 'Finance Manager',
            })),
          );
        }
      } catch (err) {
        this.logger.error('Failed to fetch finance managers for BCC', err);
      }
      if (approval.approvalStatus === ApprovalStatusEnum.APPROVED) {
        if (programRegistration.isFreeSeat) {
          const approvedMergeInfo = {
            venue_name: programRegistration?.allocatedProgram?.venueNameInEmails ?? '',
            s_day: programRegistration?.allocatedProgram?.startsAt
              ? getWeekName(programRegistration?.allocatedProgram?.startsAt ?? new Date())
              : '-',
            e_day: getWeekName(programRegistration?.allocatedProgram?.endsAt ?? new Date()) || '',
            reg_name: programRegistration.fullName ?? 'Infinitheist',
            hdb_or_msd: programRegistration?.allocatedProgram?.name ?? 'HDB',
            hdb_dates: `${formatDateIST(programRegistration?.allocatedProgram?.startsAt?.toISOString() ?? '')} to ${formatDateIST(programRegistration?.allocatedProgram?.endsAt?.toISOString() ?? '')}`,
            hdb_no: programRegistration?.allocatedProgram?.name ?? 'HDB',
          };
          const templateKey = process.env.ZEPTO_BLESSED_NO_PAYMENT_HDB_EMAIL_TEMPLATE_ID;
          this.logger.log('Sending approval email for free seat with merge info', { approvedMergeInfo, templateKey });
          const emailData: SendSingleEmailDto = {
            templateKey,
            from: { address: zeptoEmailCreadentials.ZEPTO_EMAIL, name: programRegistration.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME },
            to: { emailAddress: programRegistration.emailAddress, name: programRegistration.fullName },
            bcc: bccEmails,
            mergeInfo: approvedMergeInfo,
            attachments: [],
            subject: '',
            trackinfo: {
              registrationId: programRegistration.id,
            },
          };
          await this.communicationService.sendSingleEmail(emailData);
          this.logger.log('Approval email sent successfully for free seat');
          const regMessageData: SendTemplateMessageDto = {
            whatsappNumber: programRegistration.mobileNumber.slice(0),
            templateName: process.env.WATI_BLESSED_NO_PAYMENT_HDB_TEMPLATE_ID || '',
            broadcastName: process.env.WATI_BLESSED_NO_PAYMENT_HDB_TEMPLATE_ID || '',
            parameters: [
              { name: 'reg_name', value: programRegistration?.fullName },
              { name: 'hdb_msd', value: programRegistration?.allocatedProgram?.name || 'HDB' },
              {
                name: 's_date',
                value: formatDateIST(
                  programRegistration?.allocatedProgram?.startsAt?.toISOString() ?? '',
                ),
              },
              {
                name: 'e_date',
                value: formatDateIST(
                  programRegistration?.allocatedProgram?.endsAt?.toISOString() ?? '',
                ),
              },
            ],
            trackinfo: {
              registrationId: programRegistration.id,
            },
          };
          await this.communicationService.sendTemplateMessage(regMessageData);
          this.logger.log('WhatsApp message sent successfully to registered user', { regMessageData });
          const rmDetails = await this.userRepo.getUserByField('id', programRegistration.rmContact);
          this.logger.log('RM Details:', rmDetails);
          const rmMessageData: SendTemplateMessageDto = {
            whatsappNumber: `${rmDetails.countryCode.slice(0)}${rmDetails.phoneNumber}`,
            templateName: process.env.WATI_BLESSED_HDB_RM_TEMPLATE_ID || '',
            broadcastName: process.env.WATI_BLESSED_HDB_RM_TEMPLATE_ID || '',
            parameters: [
              { name: 'reg_name', value: programRegistration.fullName },
              {
                name: 'rm_name',
                value:
                  rmDetails.firstName && rmDetails.lastName
                    ? rmDetails.firstName + ' ' + rmDetails.lastName
                    : rmDetails.fullName || '',
              },
              { name: 'reg_mobile', value: programRegistration.mobileNumber },
              { name: 'hdb_msd_variable', value: programRegistration?.allocatedProgram?.name || '-' },
              { name: 's_date', value: formatDateIST(programRegistration?.allocatedProgram?.startsAt?.toISOString() ?? '') },
              { name: 'e_date', value: formatDateIST(programRegistration?.allocatedProgram?.endsAt?.toISOString() ?? '') },
            ],
            trackinfo: {
              registrationId: programRegistration.id,
            },
          };
          this.logger.log('RM Message Data:', rmMessageData);
          await this.communicationService.sendTemplateMessage(rmMessageData);
          this.logger.log('WhatsApp messages sent successfully to rm', { rmDetails, rmMessageData });
          
          // Send MSG91 SMS notification to RM
          // Fetch the registered user details to get their phone number
          const registeredUser = await this.userRepo.getUserByField('id', programRegistration.userId);
          const userFormattedPhoneNumber = registeredUser?.countryCode && registeredUser?.phoneNumber 
            ? `${registeredUser.countryCode} ${registeredUser.phoneNumber}` 
            : null;
          
          await this.sendRmMsg91Notification(
            { countryCode: rmDetails.countryCode, phoneNumber: rmDetails.phoneNumber },
            rmDetails.orgUsrName || 
            (rmDetails.firstName && rmDetails.lastName
                    ? rmDetails.firstName + ' ' + rmDetails.lastName
                    : rmDetails.fullName || ''),
            programRegistration.fullName,
            userFormattedPhoneNumber ?? '+'.concat(programRegistration.mobileNumber.replace('+', '')),
            `${programRegistration?.allocatedProgram?.name} from ${formatDateIST(programRegistration?.allocatedProgram?.startsAt?.toISOString() ?? '')} to ${formatDateIST(programRegistration?.allocatedProgram?.endsAt?.toISOString() ?? '')}`
          );
        } else {
          let queryRunner: QueryRunner | null = null;
          let proformaInvoiceSeqNumber = programRegistration.proFormaInvoiceSeqNumber;
          try {
            if (!proformaInvoiceSeqNumber) {
              queryRunner = this.dataSource.createQueryRunner();
              await queryRunner.connect();
              await queryRunner.startTransaction();
              
              proformaInvoiceSeqNumber = await this.repo.generateProFormaInvoiceSequenceNumber(
                queryRunner.manager,
                programRegistration,
              );
              
              await queryRunner.commitTransaction();
            }
          } catch (error) {
            this.logger.error('Failed to generate proforma invoice sequence number', error);
            if (queryRunner) {
              await queryRunner.rollbackTransaction();
            }
            handleKnownErrors(ERROR_CODES.FAILED_TO_GENERATE_PROFORMA_INVOICE_SEQ_NUMBER, error);
          } finally {
            if (queryRunner) {
              await queryRunner.release();
            }
          }
          // save the proFormaInvoiceSeqNumber to the registration registrationrepo
          await this.registrationRepo.update(programRegistration.id, {
            proFormaInvoiceSeqNumber: proformaInvoiceSeqNumber,
            auditRefId: programRegistration.id,
            parentRefId: programRegistration.id,
          });
          this.logger.log('Proforma Invoice Sequence Number:', {proformaInvoiceSeqNumber});
          const approvedMergeInfo = {
            venue_name: programRegistration?.allocatedProgram?.venueNameInEmails ?? "",
            s_day: programRegistration?.allocatedProgram?.startsAt ? getWeekName(programRegistration?.allocatedProgram?.startsAt ?? new Date)  : "-",
            hdb_price: programRegistration?.allocatedProgram?.basePrice || "",
            payment_last_date: programRegistration?.allocatedProgram?.startsAt ? formatDateIST(deductDaysFromDate(programRegistration?.allocatedProgram?.startsAt, 10).toISOString() || "") : "-",
            cash_details: (programRegistration.allocatedProgram?.basePrice ?? 0) > 150 ? "" : "4. Cash: If you are paying by cash, after paying the cash, please click on the following link and fill in the details. Please note that this simple update by you will ensure that there is no omission or reconciliation issue owing to human error.",
            e_day: getWeekName(programRegistration?.allocatedProgram?.endsAt ?? new Date ) || "",
            reg_name: programRegistration.fullName ?? "Infinitheist",
            hdb_or_msd: programRegistration?.allocatedProgram?.name ?? "HDB",
            hdb_dates: `${ formatDateIST(programRegistration?.allocatedProgram?.startsAt.toISOString() ?? "")} to ${formatDateIST(programRegistration?.allocatedProgram?.endsAt.toISOString() ?? "")}`,
            hdb_no: programRegistration?.allocatedProgram?.name ?? "HDB",
            payment_online_link: generatePaymentLink(programRegistration.program.id, programRegistration.userId), //`${process.env.SEEKER_FE_BASE_URL}${SEEKER_FE_REG_PATH}${programRegistration.program.id}&paymentMode=${encodeURIComponent(PaymentModeEnum.ONLINE_RAZORPAY)}`,
            payment_back_transfer_link: generatePaymentLink(programRegistration.program.id, programRegistration.userId), // `${process.env.SEEKER_FE_BASE_URL}${SEEKER_FE_REG_PATH}${programRegistration.program.id}&paymentMode=${encodeURIComponent(PaymentModeEnum.DIRECT_BANK_TRANSFER)}`,
            payment_cheque_link: generatePaymentLink(programRegistration.program.id, programRegistration.userId), //`${process.env.SEEKER_FE_BASE_URL}${SEEKER_FE_REG_PATH}${programRegistration.program.id}&paymentMode=${encodeURIComponent(PaymentModeEnum.CHEQUE)}`,
          }
          const templateKey = process.env.ZEPTO_BLESSED_HDB_EMAIL_TEMPLATE_ID;

          const pdfBuffer = await this.generateProFormaInvoicePDF(programRegistration, proformaInvoiceSeqNumber);
          this.logger.log('Pro-forma invoice PDF generated', { pdfBufferUrl: pdfBuffer.url });
          const response = await axios.get(pdfBuffer.url, {
            responseType: 'arraybuffer',
            headers: {
              referer: process.env.BACKEND_URL,
            },
          });
          this.logger.log('PDF buffer retrieved successfully', {
            bufferLength: response.data.byteLength,
          });
          const fileData = Buffer.from(response.data, 'binary').toString('base64');

          const attachments = {
            filename: `pro-forma-invoice.pdf`, //`invoice-${invoiceData.invoice.invoiceSequenceNumber || invoiceData.invoice.id}.pdf`,
            content: fileData,
            contentType: 'application/pdf',
          }
          this.logger.log('Sending approval email with merge info', { approvedMergeInfo, templateKey} );
          const emailData: SendSingleEmailDto = {
            templateKey: templateKey,
            from: { address: zeptoEmailCreadentials.ZEPTO_EMAIL, name: programRegistration.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME },
            to: { emailAddress: programRegistration.emailAddress, name: programRegistration.fullName },
            bcc: bccEmails,
            mergeInfo: approvedMergeInfo,
            attachments: [
              {
                name: attachments.filename,
                content: attachments.content,
                mime_type: attachments.contentType,
              }
            ],
            subject: '',
            trackinfo: {
              registrationId: programRegistration.id,
            },
          };
          try {
            await this.communicationService.sendSingleEmail(emailData);
            this.logger.log('Approval email sent successfully');
          } catch (error) {
            this.logger.error('Error sending approval email', error);
          }

          // Send WhatsApp message to registred user
          const regMessageData: SendTemplateMessageDto = {
            whatsappNumber: programRegistration.mobileNumber.slice(0), // Remove + prefix
            templateName: process.env.WATI_BLESSED_HDB_TEMPLATE_ID || "",
            broadcastName: process.env.WATI_BLESSED_HDB_TEMPLATE_ID || "",
            parameters: [
              { name: 'reg_name', value: programRegistration?.fullName },
              { name: 'hdb_msd', value: programRegistration?.allocatedProgram?.name || "HDB" },
              {
                name: 'payment_link',
                value: generatePaymentLink(
                  programRegistration.program.id,
                  programRegistration.userId,
                  undefined,
                  undefined,
                  false,
                ),
              }, // `${SEEKER_FE_REG_PATH}${programRegistration.program.id}` },
              {
                name: 's_date',
                value: formatDateIST(
                  programRegistration?.allocatedProgram?.startsAt?.toISOString() ?? '',
                ),
              },
              {
                name: 'e_date',
                value: formatDateIST(
                  programRegistration?.allocatedProgram?.endsAt?.toISOString() ?? '',
                ),
              },
            ],
            trackinfo: {
              registrationId: programRegistration.id,
            },
          };
          try {
            await this.communicationService.sendTemplateMessage(regMessageData);
            this.logger.log('WhatsApp message sent successfully to registered user for blessed', { regMessageData });
          } catch (error) {
            this.logger.error(
              'Error sending WhatsApp message to registered user for blessed',
              error,
            );
          }

          // Send WhatsApp message to RM if available
          const rmDetails = await this.userRepo.getUserByField(
            'id',
            programRegistration.rmContact)

          this.logger.log('RM Details:', rmDetails);

          const rmMessageData: SendTemplateMessageDto = {
            whatsappNumber: `${rmDetails.countryCode.slice(0)}${rmDetails.phoneNumber}` ,
            templateName: process.env.WATI_BLESSED_HDB_RM_TEMPLATE_ID || "",
            broadcastName: process.env.WATI_BLESSED_HDB_RM_TEMPLATE_ID || "",
            parameters: [
              { name: 'reg_name', value: programRegistration.fullName },
              {
                name: 'rm_name',
                value:
                  rmDetails.firstName && rmDetails.lastName
                    ? rmDetails.firstName + ' ' + rmDetails.lastName
                    : rmDetails.fullName || '',
              },
              { name: 'reg_mobile', value: programRegistration.mobileNumber },
              { name: 'hdb_msd_variable', value: programRegistration?.allocatedProgram?.name || "-" },
              { name: 's_date', value: formatDateIST(programRegistration?.allocatedProgram?.startsAt?.toISOString() ?? "") },
              { name: 'e_date', value: formatDateIST(programRegistration?.allocatedProgram?.endsAt?.toISOString() ?? "") },
            ],
            trackinfo: {
              registrationId: programRegistration.id,
            },
          }
          this.logger.log('RM Message Data:', rmMessageData);
          try {
            await this.communicationService.sendTemplateMessage(rmMessageData);
            this.logger.log('WhatsApp messages sent successfully to rm', {
              rmDetails,
              rmMessageData,
            });
          } catch (error) {
            this.logger.error('Error sending WhatsApp message to RM', error);
          }

          // Send MSG91 SMS notification to RM
          const registeredUser = await this.userRepo.getUserByField('id', programRegistration.userId);
          const userFormattedPhoneNumber = registeredUser?.countryCode && registeredUser?.phoneNumber 
            ? `${registeredUser.countryCode} ${registeredUser.phoneNumber}` 
            : null;
          
          await this.sendRmMsg91Notification(
            { countryCode: rmDetails.countryCode, phoneNumber: rmDetails.phoneNumber },
            rmDetails.orgUsrName || 
            (rmDetails.firstName && rmDetails.lastName
              ? rmDetails.firstName + ' ' + rmDetails.lastName
              : rmDetails.fullName || ''),
            programRegistration.fullName,
            userFormattedPhoneNumber ?? '+'.concat(programRegistration.mobileNumber.replace('+', '')),
            `${programRegistration?.allocatedProgram?.name} from ${formatDateIST(programRegistration?.allocatedProgram?.startsAt?.toISOString() ?? '')} to ${formatDateIST(programRegistration?.allocatedProgram?.endsAt?.toISOString() ?? '')}`,
          );
          // Update the registration table with the proFormaInvoicePdfUrl
          await this.registrationRepo.update(programRegistration.id, {
            proFormaInvoicePdfUrl: pdfBuffer.url,
            auditRefId: programRegistration.id,
            parentRefId: programRegistration.id,
          });
          this.logger.log('Updated registration table with proFormaInvoicePdfUrl', { registrationId: programRegistration.id, proFormaInvoicePdfUrl: pdfBuffer.url });

       }
      } else if (approval.approvalStatus === ApprovalStatusEnum.REJECTED) {
        const ccEmails: { emailAddress: string; name: string }[] = [];
        ccEmails.push({
          emailAddress: programRegistration.rmContactUser.email,
          name: programRegistration.rmContactUser.orgUsrName,
        });
        const lastApprovedIndex = trackHistory
          .map((t) => t.approvalStatus)
          .lastIndexOf(ApprovalStatusEnum.APPROVED);
        const lastRejectedIndex = trackHistory
          .map((t) => t.approvalStatus)
          .lastIndexOf(ApprovalStatusEnum.REJECTED);
        const lastApprovedTrack = trackHistory[lastApprovedIndex];
        if (lastApprovedIndex !== -1 && lastRejectedIndex < lastApprovedIndex) {
          const rejectionMergeInfo = {
            reg_name: programRegistration.fullName || 'Infinitheist',
            hdb_msd: programRegistration?.allocatedProgram?.name ?? '',
            last_allocated_hdb_msd: lastApprovedTrack?.allocatedProgram?.name || 'HDB/MSD',
          };
          const templateKey =
            process.env.ZEPTO_HOLD_EMAIL_TEMPLATE_ID;
          this.logger.log('Sending rejection email with merge info', {
            rejectionMergeInfo,
            templateKey,
          });
          const emailData: SendSingleEmailDto = {
            templateKey: templateKey,
            from: {
              address: zeptoEmailCreadentials.ZEPTO_EMAIL,
              name:
                programRegistration.program.emailSenderName ||
                zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
            },
            to: {
              emailAddress: programRegistration.emailAddress,
              name: programRegistration.fullName,
            },
            cc: ccEmails,
            bcc: bccEmails,
            mergeInfo: rejectionMergeInfo,
            attachments: [],
            subject: '',
            trackinfo: {
              registrationId: programRegistration.id,
            },
          };
          await this.communicationService.sendSingleEmail(emailData);
          this.logger.log('Rejection email sent successfully', { emailData });
        } else if (
          (_existingApprovalStatus === ApprovalStatusEnum.PENDING ||
            _existingApprovalStatus === ApprovalStatusEnum.ON_HOLD) &&
          trackHistory.map((t) => t.approvalStatus).includes(ApprovalStatusEnum.APPROVED) === false
        ) {  // on initial rejection without blessing
          const rejectedMergeInfo = {
            reg_name: programRegistration.fullName || 'Infinitheist',
            reg_id: programRegistration.id,
            rm_name:
              programRegistration.rmContactUser.firstName &&
              programRegistration.rmContactUser.lastName
                ? (programRegistration.rmContactUser.firstName +
               ' ' +
                  programRegistration.rmContactUser.lastName)
                : (programRegistration.rmContactUser.orgUsrName ?? 'Infinitheist'),
            hdb_msd: programRegistration.program.name || 'HDB',
          };
          const emailData: SendSingleEmailDto = {
            templateKey: process.env.ZEPTO_RM_HOLD_EMAIL_TEMPLATE_ID,
            from: {
              address: zeptoEmailCreadentials.ZEPTO_EMAIL,
              name:
                programRegistration.program.emailSenderName ||
                zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
            },
            to: {
              emailAddress: programRegistration.rmContactUser.email,
              name: programRegistration.rmContactUser.orgUsrName,
            },
            mergeInfo: rejectedMergeInfo,
            attachments: [],
            subject: '',
            trackinfo: {
              registrationId: programRegistration.id,
            },
          };
          await this.communicationService.sendSingleEmail(emailData);
          this.logger.log('Rejection email sent successfully', { emailData });
        }
      }
      
    } catch (error) {
      this.logger.error('Error sending approval email notification', error);
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_EMAIL, error);
    }
  }
  async generateProFormaInvoicePDF(invoiceData: any, sequenceNumber: string):Promise<{ url: string; buffer: Buffer }> {
    try {
      // Launch Puppeteer
      const browser = await puppeteer.launch({
        args: [
          ...(Array.isArray(chromium.args) ? chromium.args : []),
          '--no-sandbox',
          '--disable-setuid-sandbox',
          '--disable-features=AudioServiceOutOfProcess',
          '--disable-gpu',
          '--disable-software-rasterize',
          '--disable-features=AudioServiceOutOfProcessKillAtHang',
          '--single-process',
          '--disable-software-rasterizer',
        ],
        defaultViewport: null,
        executablePath: await chromium.executablePath('https://github.com/Sparticuz/chromium/releases/download/v119.0.2/chromium-v119.0.2-pack.tar'),
        headless: true,
      });
      const page = await browser.newPage();
      // Set the HTML content
      this.logger.log('Generating pro-forma invoice template data', { invoiceData });
      const gstPercentage = invoiceData?.program?.gstNumber
        ? invoiceData?.proFormaGstNumber
          ? areGSTsFromSameState(invoiceData.program?.gstNumber, invoiceData?.proFormaGstNumber)
            ? Number(invoiceData?.program?.cgst ?? 0) + Number(invoiceData?.program?.sgst ?? 0)
            : Number(invoiceData?.program?.igst ?? 0)
          : Number(invoiceData?.program?.cgst ?? 0) + Number(invoiceData?.program?.sgst ?? 0) || 0
        : 0;
      const gstAmount = invoiceData.program?.gstNumber
        ? parseFloat(
            (Number(invoiceData.allocatedProgram?.basePrice) * (gstPercentage / 100)).toFixed(2),
          )
        : 0;

      const totalAmount = Math.round(
        (Number(invoiceData.allocatedProgram?.basePrice) || 0) + Number(gstAmount) || 0,
      );

      const invoiceTemplateData: InvoiceTemplateDto = {
        companyAddress: invoiceData?.program?.invoiceSenderAddress || 'Infinitheism',
        companyGST: invoiceData.program?.gstNumber,
        companyCIN: invoiceData.program?.invoiceSenderCin || '',
        invoiceNumber: sequenceNumber || '',
        currentDate: formatDateIST(new Date().toISOString()),
        placeOfSupply: invoiceData?.proFormaGstNumber
          ? stateGstCodes.find(
              (code) => code.code === invoiceData?.proFormaGstNumber.substring(0, 2),
            )?.stateName || null
          : null,
        seekerName: invoiceData?.proFormaInvoiceName,
        seekerAddress: invoiceData?.proFormaInvoiceAddress,
        seekerGstin: invoiceData?.proFormaGstNumber || '', //
        programName: invoiceData.allocatedProgram?.subProgramType === SubProgramTypeEnum.PST_MSD
            ? 'My Sacred Days - MSD'
            : invoiceData.allocatedProgram?.subProgramType === SubProgramTypeEnum.PST_HDB
              ? 'Higher Deeper Beyond - HDB'
              : invoiceData.allocatedProgram?.name || '',
        reportingTime: formatDateIST(
          String(invoiceData.allocatedProgram?.startsAt?.toISOString() || ''),
        ),
        checkOutTime: formatDateIST(
          String(invoiceData.allocatedProgram?.endsAt?.toISOString() || ''),
        ),
        amount: String(Math.round(Number(invoiceData.allocatedProgram?.basePrice))) || '0',
        igstCheck:
          invoiceData?.proFormaGstNumber && invoiceData.program?.gstNumber
            ? !areGSTsFromSameState(invoiceData.program?.gstNumber, invoiceData?.proFormaGstNumber)
            : false,
        igst: `${Math.round(Number(invoiceData?.allocatedProgram?.basePrice) * (Number(invoiceData.program.igst) / 100)) || 0}`,
        cgst: `${Math.round(Number(invoiceData?.allocatedProgram?.basePrice) * (Number(invoiceData.program.cgst) / 100)) || 0}`,
        sgst: `${Math.round(Number(invoiceData?.allocatedProgram?.basePrice) * (Number(invoiceData.program.sgst) / 100)) || 0}`,
        igstPercentage: Number(invoiceData.program?.igst) || 0,
        cgstPercentage: Number(invoiceData.program?.cgst) || 0,
        sgstPercentage: Number(invoiceData.program?.sgst) || 0,
        totalAmount: totalAmount,
        amountInWords: amountInWordsIndian(Number(totalAmount) || 0),
        qr: '', // 'https://delta-node-test.s3.ap-south-1.amazonaws.com/qr.png', // sample file, // Placeholder QR code URL
        irpDate: '',
        acknowledgeNumber: '', // '1234567890123456789',
        irn: '',
        companyPAN: invoiceData.program?.invoiceSenderPan || '',
      };

      await page.setContent(UPDATED_PRO_FORMA_INVOICE(invoiceTemplateData));


      
      // For now, return a placeholder buffer
      const pdfBuffer = await page.pdf({
        format: 'A4',
        printBackground: true,
      });      
      await browser.close();
      // const bucketName = process.env.AWS_S3_BUCKET_NAME || '';
      const key = `proforma-invoices/${invoiceData.program.code}/proforma_${invoiceData?.id || uuidv4()}.pdf`;
      const uploadedInvoice = await this.awsS3Service.uploadToS3(key, pdfBuffer, 'application/pdf');
      // const s3Url = await this.awsS3Service.getSignedUrl(key);
      // const s3Url = this.awsS3Service.getS3Url(key);
      return {url: uploadedInvoice, buffer: pdfBuffer};
      

    } catch (error) {
      this.logger.error('PDF generation failed', '', { invoiceId: invoiceData.id, error });
      throw error;
    }
  }

  /**
   * Generates a payment link for a specific program and user with optional payment mode.
   *
   * @param programId - The unique identifier of the program for which the payment link is generated.
   * @param userId - (Optional) The unique identifier of the user. Defaults to `undefined`.
   * @param paymentMode - (Optional) The payment mode to be included in the link. Should be of type `PaymentModeEnum`.
   * @returns The generated payment link as a string.
   */
  sendRefundCommunication(registrationId: number) {
    try {
      // await this.registrationService.sendRefundCommunication(registrationId);
    } catch (error) {
      this.logger.log('Error sending refund communication', { error, registrationId });
    }
  }

  async clearTravelDetails(
    existingStatus: ApprovalStatusEnum,
    currentStatus: ApprovalStatusEnum,
    registrationId: number,
    userId: number,
    existingAllocatedProgramId?: number,
  ) {
    if (
      (existingStatus === ApprovalStatusEnum.ON_HOLD ||
        existingStatus === ApprovalStatusEnum.CANCELLED ||
        existingStatus === ApprovalStatusEnum.REJECTED) &&
      currentStatus === ApprovalStatusEnum.APPROVED
    ) {
      try {
        await this.registrationRepository.resetTravelDetails(registrationId, userId);
      } catch (error) {
        this.logger.error('Error clearing travel details', '', { error, registrationId });
      }
    } else if (existingAllocatedProgramId) {
      const registration = await this.registrationRepo.findOne({
        where: { id: registrationId },
        relations: ['allocatedProgram', 'program'],
      });
      if (!registration) {
        this.logger.error('Registration not found while clearing travel details', '', {
          registrationId,
        });
        return;
      }
      if (
        registration.allocatedProgram?.id &&
        existingAllocatedProgramId !== registration.allocatedProgram?.id
      ) {
        try {
          await this.registrationRepository.resetTravelDetails(registrationId, userId);
        } catch (error) {
          this.logger.error('Error clearing travel details due to program change', '', {
            error,
            registrationId,
          });
        }
      } else {
        this.logger.log('No change in allocated program, travel details not cleared', {
          registrationId,
        });
      }
    }
  }

  async checkActiveWantsSwapRequestForRegistration(registrationId: number): Promise<boolean> {
    try {
      return await this.registrationRepository.checkActiveWantsSwapRequestForRegistration(registrationId);
    } catch (error) {
      this.logger.error('Error checking active swap request for registration', '', { error, registrationId });
      return false;
    }
  }

  async checkActiveSwapRequestForRegistration(registrationId: number): Promise<boolean> {
    try {
      return await this.registrationRepository.checkActiveSwapRequestForRegistration(registrationId);
    } catch (error) {
      this.logger.error('Error checking active swap request for registration', '', { error, registrationId });
      return false;
    }
  }

  async checkActiveCanShiftRequestForRegistration(registrationId: number): Promise<boolean> {
    try {
      return await this.registrationRepository.checkActiveCanShiftRequestForRegistration(registrationId);
    } catch (error) {
      this.logger.error('Error checking active can shift request for registration', '', { error, registrationId });
      return false;
    }
  }

  async handleOnHoldApproval(registration: any, userId: number) {
    try {
      await this.programRegistrationService.processOnHoldApproval(registration, userId);
    } catch (error) {
      this.logger.error('Error handling on-hold approval', '', { error, registrationId: registration.id});
    }
  }

  async handleRejectedApproval(registration: any, userId: number) {
    try {
      await this.programRegistrationService.processRejectedApproval(registration, userId);
    } catch (error) {
      this.logger.error('Error handling rejected approval', '', { error, registrationId: registration.id});
    }
  }

  async completeSwapRequestOnProgramAllocation(existingRegistration: any, userId: number) {
    try {
      await this.programRegistrationService.completeSwapRequestOnProgramAllocation(existingRegistration, userId);
    } catch (error) {
      this.logger.error('Error completing swap request on program allocation', '', { error, registrationId: existingRegistration.id  });
    }
  }
}
