import {
  Controller,
  Get,
  Post,
  Body,
  Param,
  Put,
  Delete,
  HttpStatus,
  ParseIntPipe,
  ValidationPipe,
  UsePipes,
  HttpCode,
  Query,
  Res,
  UseGuards,
  Req,
} from '@nestjs/common';
import { Response } from 'express';
import { QuestionService } from './question.service';
import { CreateQuestionDto } from './dto/create-question.dto';
import { UpdateQuestionDto } from './dto/update-question.dto';
import { ApiTags, ApiOperation, ApiResponse, ApiParam, ApiBody, ApiQuery, ApiBearerAuth, ApiSecurity } from '@nestjs/swagger';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { ResponseService } from 'src/common/response-handling/response-handler';
import ErrorHandler from 'src/common/response-handling/error-handling';
import { paginationConstants, questionConstMessages, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('question')
@Controller('question')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class QuestionController {
  constructor(
    private readonly questionService: QuestionService,
    private readonly responseService: ResponseService,
    private readonly errorHandler: ErrorHandler,
    private readonly logger: AppLoggerService
  ) {}

  /**
   * Creates a new question.
   * Validates the input using a validation pipe and ensures the question is created successfully.
   * @param createQuestionDto - Data transfer object containing question details.
   * @param res - Express Response object.
   * @returns A success message and the created question data.
   */
  @Post()
  @HttpCode(HttpStatus.CREATED)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: questionConstMessages.CREATE_NEW_QUESTION })
  @ApiBody({ type: CreateQuestionDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(@Body() createQuestionDto: CreateQuestionDto, @Req() req: any, @Res() res: Response) {
    this.logger.log(questionConstMessages.CREATE_QUESTION_REQUEST_RECEIVED, createQuestionDto);
    try {
      const userId = req.user?.id || createQuestionDto.createdBy;
      createQuestionDto.createdBy = userId;
      createQuestionDto.updatedBy = userId;
      const data = await this.questionService.create(createQuestionDto);
      await this.responseService.success(res, questionConstMessages.QUESTION_CREATED, data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res,error)
    }
  }

  /**
   * Retrieves all questions with optional pagination and search functionality.
   * @param limit - Number of records per page (default: 10).
   * @param offset - Offset for pagination (default: 0).
   * @param searchText - Optional search text to filter questions by name.
   * @param filters - Additional filters as a JSON string.
   * @param res - Express Response object.
   * @returns A paginated list of questions and metadata.
   */
  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: questionConstMessages.GET_ALL_QUESTIONS })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: paginationConstants.SEARCH_TEXT })
  @ApiQuery({ name: 'filters', type: String, required: false, description: paginationConstants.FILTERS })
  @ApiQuery({ name: 'formSectionId', type: Number, required: false, description: questionConstMessages.FILTER_QUESTIONS_FORM_SECTION_ID })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText: string = '',
    @Query('filters') filters: string = '',
    @Query('formSectionId') formSectionId: number | null = null,
    @Res() res: Response,
  ) {
    this.logger.log(questionConstMessages.FIND_ALL_QUESTIONS_REQUEST_RECEIVED, { limit, offset, searchText, filters, formSectionId });
    try {
      const parsedFilters = filters ? JSON.parse(decodeURIComponent(filters)) : {};
      const result = await this.questionService.findAll(limit, offset, searchText, parsedFilters, formSectionId);
      await this.responseService.success(res, questionConstMessages.QUESTIONS_RETRIEVED, result);
    } catch (error) {
      handleControllerError(res,error)
    }
  }

  /**
   * Retrieves a single question by its ID.
   * @param id - ID of the question to retrieve.
   * @param res - Express Response object.
   * @returns The question data if found.
   */
  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: questionConstMessages.GET_QUESTION_BY_ID })
  @ApiParam({ name: 'id', description: questionConstMessages.QUESTION_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(questionConstMessages.FIND_ONE_QUESTION_REQUEST_RECEIVED, { id });
    try {
      const data = await this.questionService.findOne(id);
      await this.responseService.success(res, questionConstMessages.QUESTIONS_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res,error)
    }
  }

  /**
   * Updates an existing question by its ID.
   * Validates the input using a validation pipe and ensures the question is updated successfully.
   * @param id - ID of the question to update.
   * @param updateQuestionDto - Data transfer object containing updated question details.
   * @param res - Express Response object.
   * @returns A success message and the updated question data.
   */
  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: questionConstMessages.UPDATE_QUESTION_BY_ID })
  @ApiParam({ name: 'id', description: questionConstMessages.QUESTION_ID, type: Number })
  @ApiBody({ type: UpdateQuestionDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async update(
    @Param('id', ParseIntPipe) id: number,
    @Body() updateQuestionDto: UpdateQuestionDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(questionConstMessages.UPDATE_QUESTION_REQUEST_RECEIVED, { id, updateQuestionDto });
    try {
      const userId = req.user?.id || updateQuestionDto.updatedBy;
      updateQuestionDto.updatedBy = userId;
      const data = await this.questionService.update(id, updateQuestionDto);
      await this.responseService.success(res, questionConstMessages.QUESTION_UPDATED, data);
    } catch (error) {
      handleControllerError(res,error)
    }
  }

  /**
   * Deletes a question by its ID.
   * @param id - ID of the question to delete.
   * @param res - Express Response object.
   * @returns A success message indicating the question was deleted.
   */
  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: questionConstMessages.DELETE_QUESTION_BY_ID })
  @ApiParam({ name: 'id', description: questionConstMessages.QUESTION_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Req() req: any, @Res() res: Response) {
    this.logger.log(questionConstMessages.DELETE_QUESTION_REQUEST_RECEIVED, { id });
    try {
      const userId = req.user;
      if (!userId) {
        return this.errorHandler.unauthorized(res);
      }
      await this.questionService.remove(id, userId);
      await this.responseService.success(res, questionConstMessages.QUESTION_DELETED);
    } catch (error) {
      handleControllerError(res,error)
    }
  }
}