import { Injectable } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { Repository, IsNull, In, ILike, EntityManager } from "typeorm";
import { ProgramSession, Program, User, Address } from "src/common/entities";
import { CreateProgramSessionDto } from "./dto/create-program-session.dto";
import { UpdateProgramSessionDto } from "./dto/update-program-session.dto";
import { CommonDataService } from "src/common/services/commonData.service";
import { AppLoggerService } from "src/common/services/logger.service";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";
import { programSessionConstMessages, programConstMessages, userConstMessages } from "src/common/constants/strings-constants";
import { ModeOfOperationEnum } from "src/common/enum/mode-of-operation.enum"; 

@Injectable()
export class ProgramSessionRepository {
  constructor(
    @InjectRepository(ProgramSession)
    private readonly sessionRepo: Repository<ProgramSession>,
    @InjectRepository(Program)
    private readonly programRepo: Repository<Program>,
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService
  ) {}

  async createSession(createDto: CreateProgramSessionDto, manager?: EntityManager): Promise<ProgramSession> {
    try {
      const { programId, name, code, displayOrder, createdBy, updatedBy } = createDto;

      // Get the appropriate repositories (transaction-aware or regular)
      const sessionRepo = manager ? manager.getRepository(ProgramSession) : this.sessionRepo;
      const programRepo = manager ? manager.getRepository(Program) : this.programRepo;
      const userRepo = manager ? manager.getRepository(User) : this.userRepo;

      // Validate program
      if (programId) {
        const program = await programRepo.findOne({
          where: { id: programId,deletedAt: IsNull() },
        });
        if (!program) {
          this.logger.error(programConstMessages.PROGRAM_NOT_FOUND_ID(programId));
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_NOTFOUND,
            null,
            null,
            programId.toString()
          );
        }
      }

      // Validate creator and updater
      const [creator, updater] = await Promise.all([
        userRepo.findOne({ where: { id: createdBy } }),
        userRepo.findOne({ where: { id: updatedBy } }),
      ]);

      if (!creator) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(createdBy));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_SESSION_CREATOR_NOTFOUND,
          null,
          null,
          createdBy.toString()
        );
      }
      if (!updater) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_SESSION_UPDATOR_NOTFOUND,
          null,
          null,
          updatedBy.toString()
        );
      }

      // Convert venueAddress DTO to Address entity if present
      let venueAddressEntity: Address | undefined = undefined;
      if (createDto.venueAddress) {
        venueAddressEntity = new Address(createDto.venueAddress);
      }

      const session = new ProgramSession({
        ...createDto,
        venueAddress: venueAddressEntity,
        status: createDto.status as any,
        modeOfOperation: createDto.modeOfOperation as ModeOfOperationEnum,
        creator,
        updater,
        checkinEndsAt: createDto.checkinEndsAt,
        checkoutEndsAt: createDto.checkoutEndsAt,
        logoUrl: createDto.logoUrl,
      });

      if (manager) {
        return await manager.save(ProgramSession, session);
      } else {
        return this.commonDataService.save(this.sessionRepo, session);
      }
    } catch (error) {
      this.logger.error('Error creating program session:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_SAVE_FAILED, error);
    }
  }

  async findAllSessions(
    limit: number,
    offset: number,
    searchText: string,
    programId: number[],
    parsedFilters: Record<string, any>,
    manager?: EntityManager
  ) {
    try {
      const whereClause: any = { deletedAt: IsNull() };
      if (programId?.length) whereClause.programId = In(programId);
      if (parsedFilters?.status) whereClause.status = parsedFilters.status;
      if (parsedFilters?.modeOfOperation) whereClause.modeOfOperation = parsedFilters.modeOfOperation;
      if (searchText) whereClause.name = ILike(`%${searchText}%`);

      const sessionRepo = manager ? manager.getRepository(ProgramSession) : this.sessionRepo;

      // Use transaction-aware repository if manager is provided
      let data;
      let total;
      
      if (manager) {
        data = await sessionRepo.find({
          select: ["id", "name", "code", "programId", "displayOrder", "startsAt", "endsAt", "status", "modeOfOperation"],
          where: whereClause,
          relations: ["program", "program.programQuestionMaps", "program.programQuestionMaps.question", "program.programQuestionMaps.question.questionOptionMaps", "program.programQuestionMaps.question.questionOptionMaps.option", "program.programQuestionMaps.question.formSection"],
          take: limit,
          skip: offset,
          order: { id: "ASC" },
        });
        
        total = await sessionRepo.count({ where: whereClause });
      } else {
        data = await this.commonDataService.get(
          this.sessionRepo,
          ["id", "name", "code", "programId", "displayOrder", "startsAt", "endsAt", "status", "modeOfOperation"],
          whereClause,
          limit,
          offset,
          { id: "ASC" },
          searchText ? { name: searchText } : undefined,
          ["program", "program.programQuestionMaps", "program.programQuestionMaps.question", "program.programQuestionMaps.question.questionOptionMaps", "program.programQuestionMaps.question.questionOptionMaps.option", "program.programQuestionMaps.question.formSection"],
          { program: ["id", "name"] }
        );
        
        total = await this.sessionRepo.count({ where: whereClause });
      }

      return {
        data,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      this.logger.error('Error finding program sessions:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_GET_FAILED, error);
    }
  }

  async findOneById(id: number, manager?: EntityManager): Promise<ProgramSession | null> {
    try {
      const repo = manager ? manager.getRepository(ProgramSession) : this.sessionRepo;
      
      if (manager) {
        return await repo.findOne({
          where: { id, deletedAt: IsNull() },
          relations: ["program", "creator", "updater", "program.programQuestionMaps", "program.programQuestionMaps.question", "program.programQuestionMaps.question.questionOptionMaps", "program.programQuestionMaps.question.questionOptionMaps.option", "program.programQuestionMaps.question.formSection"],
        });
      } else {
        return await this.commonDataService.findOneById(this.sessionRepo, id, true, [
          "program",
          "creator",
          "updater",
          "program.programQuestionMaps",
          "program.programQuestionMaps.question",
          "program.programQuestionMaps.question.questionOptionMaps",
          "program.programQuestionMaps.question.questionOptionMaps.option",
          "program.programQuestionMaps.question.formSection",
        ]);
      }
    } catch (error) {
      this.logger.error('Error finding program session by ID:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_FIND_BY_ID_FAILED, error);
    }
  }

  async updateSession(id: number, updateDto: UpdateProgramSessionDto, manager?: EntityManager): Promise<ProgramSession> {
    try {
      const sessionRepo = manager ? manager.getRepository(ProgramSession) : this.sessionRepo;
      const programRepo = manager ? manager.getRepository(Program) : this.programRepo;
      const userRepo = manager ? manager.getRepository(User) : this.userRepo;
      
      const session = await this.findOneById(id, manager);
      if (!session) {
        this.logger.error(programSessionConstMessages.SESSION_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      const { programId, name, code, displayOrder, updatedBy } = updateDto;

      // Validate program if provided
      if (programId) {
        const program = await programRepo.findOne({
          where: { id: programId, deletedAt: IsNull() },
        });
        if (!program) {
          this.logger.error(programConstMessages.PROGRAM_NOT_FOUND_ID(programId));
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_NOTFOUND,
            null,
            null,
            programId.toString()
          );
        }
        session.program = program;
      }

      // Validate updater
      const updater = await userRepo.findOne({ where: { id: updatedBy } });
      if (!updater) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_SESSION_UPDATOR_NOTFOUND,
          null,
          null,
          updatedBy.toString()
        );
      }

      Object.assign(session, updateDto);
      session.updater = updater;
      
      if (manager) {
        return await manager.save(ProgramSession, session);
      } else {
        return this.commonDataService.save(this.sessionRepo, session);
      }
    } catch (error) {
      this.logger.error('Error updating program session:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_SAVE_FAILED, error);
    }
  }

  async softDeleteSession(id: number, user: User, manager?: EntityManager): Promise<void> {
    try {
      const sessionRepo = manager ? manager.getRepository(ProgramSession) : this.sessionRepo;
      const session = await this.findOneById(id, manager);
      
      if (!session) {
        this.logger.error(programSessionConstMessages.SESSION_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      session.deletedAt = new Date();
      session.updatedBy = user.id;
      session.updater = user;
      
      if (manager) {
        await manager.save(ProgramSession, session);
      } else {
        await this.commonDataService.save(this.sessionRepo, session);
      }
      
      this.logger.log(`Successfully soft deleted program session ${id}`);
    } catch (error) {
      this.logger.error('Error soft deleting program session:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_DELETE_FAILED, error);
    }
  }

  /**
   * Soft deletes all sessions for a given program
   * This method is specifically designed to work within transactions
   */
  async softDeleteSessionsByProgramId(programId: number, user: User, manager?: EntityManager): Promise<void> {
    try {
      const sessionRepo = manager ? manager.getRepository(ProgramSession) : this.sessionRepo;
      
      // Find all active sessions for the program
      const sessions = await sessionRepo.find({
        where: { 
          programId, 
          deletedAt: IsNull() 
        }
      });


      if (sessions.length === 0) {
        this.logger.log(`No active sessions found for program ${programId}`);
        return;
      }

      // Update all sessions to mark them as deleted
      const updatedSessions = sessions.map(session => ({
        ...session,
        deletedAt: new Date(),
        updatedBy: user.id,
        updater: user,
      }));

      if (manager) {
        await manager.save(ProgramSession, updatedSessions);
      } else {
        await this.sessionRepo.save(updatedSessions);
      }

      this.logger.log(`Successfully soft deleted ${sessions.length} sessions for program ${programId}`);
    } catch (error) {
      this.logger.error(`Error soft deleting sessions for program ${programId}:`, error);
      throw error; // Re-throw to ensure transaction rollback
    }
  }

  /**
   * Finds sessions by program ID with transaction support
   */
  async findSessionsByProgramId(
    programId: number,
    manager?: EntityManager
  ): Promise<ProgramSession[]> {
    try {
      const sessionRepo = manager ? manager.getRepository(ProgramSession) : this.sessionRepo;
      
      return await sessionRepo.find({
        where: { 
          programId, 
          deletedAt: IsNull() 
        },
        relations: ["program", "creator", "updater"],
        order: { displayOrder: "ASC" }
      });
    } catch (error) {
      this.logger.error(`Error finding sessions for program ${programId}:`, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_GET_FAILED, error);
    }
  }

  /**
   * Updates status of all sessions for a given program
   */
  async updateSessionsStatusByProgramId(
    programId: number,
    status: string,
    updatedBy: number,
    manager?: EntityManager,
  ): Promise<void> {
    try {
      const sessionRepo = manager ? manager.getRepository(ProgramSession) : this.sessionRepo;
      const userRepo = manager ? manager.getRepository(User) : this.userRepo;

      const updater = await userRepo.findOne({ where: { id: updatedBy } });
      if (!updater) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_SESSION_UPDATOR_NOTFOUND,
          null,
          null,
          updatedBy.toString(),
        );
      }

      const sessions = await sessionRepo.find({
        where: { programId, deletedAt: IsNull() },
      });

      const updatedSessions = sessions.map((s) => ({
        ...s,
        status: status as any,
        updatedBy,
        updater,
      }));

      if (manager) {
        await manager.save(ProgramSession, updatedSessions);
      } else {
        await this.sessionRepo.save(updatedSessions);
      }
    } catch (error) {
      this.logger.error('Error updating session statuses:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_SAVE_FAILED, error);
    }
  }
}