import { Injectable } from '@nestjs/common';
import { ProgramRegistrationRecommendationRepository } from '../repositories/program-registration-recommendation.repository';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { ProgramRegistrationRecommendationsDto } from '../dto/program-registration-rm-ratings.dto';


@Injectable()
export class ProgramRegistrationRecommendationService {
  constructor(
    private readonly recommendationRepo: ProgramRegistrationRecommendationRepository,
  ) {}

  /**
   * Posts a new recommendation for a program registration.
   *
   * @param data - The recommendation data to post.
   * @returns The created recommendation entity.
   */
  async postRecommendation(programRegistrationId: number, data: ProgramRegistrationRecommendationsDto) {
    try {
      const { recommendationKey, isRecommended, recommendedBy, recommendationText, followUpCount } = data;

      // Create a new recommendation entity
      const recommendationEntity = this.recommendationRepo.create({
        recommendationKey,
        isRecommended,
        recommendedBy,
        registrationId: programRegistrationId,
        recommendationText,
        followUpCount: followUpCount || 0,
        createdAt: new Date(),
      });

      // Save the recommendation to the database
      const savedRecommendation = await this.recommendationRepo.save([recommendationEntity]);

      // Update audit fields to reference the entity ID
      if (savedRecommendation[0] && savedRecommendation[0].id) {
        await this.recommendationRepo.update(savedRecommendation[0].id, {
          auditRefId: savedRecommendation[0].id,
          parentRefId: savedRecommendation[0].registrationId,
        });
        // Reload the entity with updated fields
        const updatedRecommendation = await this.recommendationRepo.findOne(savedRecommendation[0].id);
        if (updatedRecommendation) {
          savedRecommendation[0] = updatedRecommendation;
        }
      }

      return {
        message: 'Recommendation posted successfully.',
        recommendation: savedRecommendation[0],
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RECOMMENDATION_SAVE_FAILED, error);
    }
  }

  /**
   * Updates an existing recommendation for a program registration.
   *
   * @param recommendationId - The ID of the recommendation to update.
   * @param data - The updated recommendation data.
   * @returns The updated recommendation entity.
   */
  async updateRecommendation(recommendationId: number, programRegistrationId:number, data: Partial<ProgramRegistrationRecommendationsDto>) {
    try {
      const { isRecommended, recommendationKey, recommendedBy, recommendationText, followUpCount } = data;

      // Find the existing recommendation
      const existingRecommendation = await this.recommendationRepo.findOne(recommendationId);
      if (!existingRecommendation) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_RECOMMENDATION_NOT_FOUND,
          new Error(`Recommendation with ID ${recommendationId} not found.`),
        );
      }

      // Update the recommendation entity
      const updatedRecommendation = await this.recommendationRepo.update(recommendationId, {
        isRecommended,
        recommendationKey,
        recommendedBy,
        recommendationText,
        followUpCount,
        registrationId: programRegistrationId,
        auditRefId: recommendationId,
        parentRefId: programRegistrationId,
        updatedAt: new Date(),
      });

      return {
        message: 'Recommendation updated successfully.',
        recommendation: updatedRecommendation,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RECOMMENDATION_UPDATE_FAILED, error);
    }
  }
}