import { Injectable } from '@nestjs/common';
import { Repository } from 'typeorm';
import { InjectRepository } from '@nestjs/typeorm';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { ProgramRegistrationRecommendations } from 'src/common/entities/program-registration-recommendations.entity';

@Injectable()
export class ProgramRegistrationRecommendationRepository {
  constructor(
    @InjectRepository(ProgramRegistrationRecommendations)
    private readonly repository: Repository<ProgramRegistrationRecommendations>,
  ) {}
  /**
   * Creates a new ProgramRegistrationRecommendations entity instance.
   * @param data - Partial data to create the entity.
   * @returns The created ProgramRegistrationRecommendations entity.
   */
  create(data: Partial<ProgramRegistrationRecommendations>): ProgramRegistrationRecommendations {
    try {
      const entity = this.repository.create({
        ...data,
      });
      return entity;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RECOMMENDATION_SAVE_FAILED, error);
    }
  }

  /**
   * Saves an array of ProgramRegistrationRecommendations entities to the database.
   * @param ratings - Array of ProgramRegistrationRecommendations entities to save.
   * @returns The saved ProgramRegistrationRecommendations entities.
   */
  async save(ratings: ProgramRegistrationRecommendations[]): Promise<ProgramRegistrationRecommendations[]> {
    try {
      return await this.repository.save(ratings);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RECOMMENDATION_SAVE_FAILED, error);
    }
  }

  /**
   * Finds a ProgramRegistrationRecommendations entity by its ID.
   * @param id - The ID of the ProgramRegistrationRecommendations entity to find.
   * @returns The found ProgramRegistrationRecommendations entity or undefined if not found.
   */
  async findOne(id: number): Promise<ProgramRegistrationRecommendations | null> {
    try {
      return await this.repository.findOne({ where: { id } });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RECOMMENDATION_NOT_FOUND, error);
    }
  }

  /**
   * Finds all ProgramRegistrationRecommendations entities by program registration ID.
   * @param programRegistrationId - The ID of the program registration to find ratings for.
   * @returns An array of ProgramRegistrationRecommendations entities.
   */
  async findAllByProgramRegistrationId(
    registrationId: number,
  ): Promise<ProgramRegistrationRecommendations[]> {
    try {
      return await this.repository.find({ where: { registrationId } });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RECOMMENDATION_NOT_FOUND, error);
    }
  }

  /**
   * Updates a ProgramRegistrationRecommendations entity with the given data.
   * @param id - The ID of the ProgramRegistrationRecommendations entity to update.
   * @param data - Partial data to update the entity.
   * @returns The updated ProgramRegistrationRecommendations entity.
   */
  async update(
    id: number,
    data: Partial<ProgramRegistrationRecommendations>,
  ): Promise<ProgramRegistrationRecommendations | null>  {
    try {
      // Get existing entity to extract registrationId for parentRefId
      const existingEntity = await this.findOne(id);
      
      // Ensure audit fields are always set for auditable entities
      const auditableUpdateData = {
        ...data,
        auditRefId: data.auditRefId || id,
        // Use registrationId from existing entity if available
        parentRefId: data.parentRefId || (existingEntity?.registrationId),
      };
      
      await this.repository.update(id, auditableUpdateData);
      return await this.findOne(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_RECOMMENDATION_UPDATE_FAILED, error);
    }
  }
}
