import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, EntityManager, Brackets, Not, IsNull } from 'typeorm';
import { ProgramRegistration } from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { CreateProgramRegistrationDto } from './dto/create-program-registration.dto';
import { UpdateProgramRegistrationDto } from './dto/update-program-registration.dto';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { GenderEnum } from 'src/common/enum/gender.enum';

@Injectable()
export class ProgramRegistrationRepository {
  constructor(
    @InjectRepository(ProgramRegistration)
    private readonly repo: Repository<ProgramRegistration>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  async createEntity(dto: CreateProgramRegistrationDto, manager?: EntityManager) {
    try {
      const entity = this.repo.create({
        ...dto,
        programSession: { id: dto.programSessionId } as any,
        user: dto.userId ? ({ id: dto.userId } as any) : undefined,
        rmContactUser: dto.rmContact ? ({ id: dto.rmContact } as any) : undefined,
        createdBy: { id: dto.createdBy } as any,
        updatedBy: { id: dto.updatedBy } as any,
      });
      const savedEntity = await this.commonDataService.save(this.repo, entity);
      
      // Set audit fields after creation when ID is available
      if (savedEntity && savedEntity.id) {
        const repository = manager ? manager.getRepository(ProgramRegistration) : this.repo;
        await repository.update(savedEntity.id, {
          auditRefId: savedEntity.id,
          // For ProgramRegistration, both auditRefId and parentRefId should be the same (entity's own ID)
          parentRefId: savedEntity.id,
        });
        savedEntity.auditRefId = savedEntity.id;
        savedEntity.parentRefId = savedEntity.id;
      }
      
      return savedEntity;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    try {
      const data = await this.commonDataService.get(
        this.repo,
        undefined,
        {},
        limit,
        offset,
        { id: 'ASC' },
        undefined,
        ['programSession', 'user'],
      );
      const total = await this.repo.count();
      return {
        data,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }
/**
 * Fetches all ProgramRegistration entities based on the provided where conditions and relations.
 * @param where - Partial object containing conditions to filter the ProgramRegistration entities.
 * @param relations - Array of relation names to be included in the query.
 * @returns An array of ProgramRegistration entities that match the conditions.
 */
   async getAll(where?: any, relations?: string[]) {
    try {
      return await this.commonDataService.get(
        this.repo,
        undefined,
        where || {},
        undefined,
        undefined,
        { id: 'ASC' },
        undefined,
        relations || ['programSession', 'user', 'program', 'allocatedProgram'],
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    try {
      const data = await this.commonDataService.findOneById(this.repo, id, false, ['programSession', 'user', 'program', 'allocatedProgram', 'rmContactUser']);
      if (!data) {
        throw new InifniNotFoundException(ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND, null, null, id.toString());
      }
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  async updateEntity(id: number, dto: UpdateProgramRegistrationDto, manager?: EntityManager) {
    try {
      const repository = manager ? manager.getRepository(ProgramRegistration) : this.repo;
      const entity = await repository.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND, null, null, id.toString());
      }
      Object.assign(entity, dto);
      if (dto.programSessionId !== undefined) entity.programSession = { id: dto.programSessionId } as any;
      if (dto.userId !== undefined) entity.user = dto.userId ? ({ id: dto.userId } as any) : undefined;
      if (dto.rmContact !== undefined) entity.rmContactUser = dto.rmContact ? ({ id: dto.rmContact } as any) : undefined;
      entity.updatedBy = { id: dto.updatedBy } as any;
      return manager ? await manager.save(entity) : await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  /**
   * Update a program registration entity
   * @param id - ID of the program registration to update
   * @param data - Partial data to update the program registration
   * @returns The updated program registration entity
   */
  async update(id: number, data: Partial<ProgramRegistration>) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
       handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
            null,
            null,
            id.toString()
          )
        );
      }
      
      // Always set audit fields for updates
      const updateData = {
        ...data,
        auditRefId: data.auditRefId || id,
        // For ProgramRegistration updates, parentRefId should remain the same as auditRefId (entity's own ID)
        parentRefId: data.parentRefId || id,
      };
      
      Object.assign(entity, updateData);
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  async remove(id: number, manager?: EntityManager) {
    try {
      const repository = manager ? manager.getRepository(ProgramRegistration) : this.repo;
      const entity = await repository.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND, null, null, id.toString());
      }
      await repository.remove(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_DELETE_FAILED, error);
    }
  }

  /**
   * Fetches all ProgramRegistration entities based on the provided where conditions and relations.
   * @param where - Partial object containing conditions to filter the ProgramRegistration entities.
   * @param relations - Array of relation names to be included in the query.
   * @returns An array of ProgramRegistration entities that match the conditions.
   */
  async findBy(where?: Partial<ProgramRegistration>, relations?: string[], limit?: number, offset?: number) {
    try {
      const filteredWhere = Object.fromEntries(
        Object.entries(where ?? {}).filter(([_, v]) => v !== null && v !== undefined)
      ) as any;
      return await this.repo.find({
        where: filteredWhere,
        relations,
        take: limit,
        skip: offset,
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Get distinct locations (cities) from approved registrations for a program
   * Fetches unique city names from registrations for filter options
   * Returns raw data without transformation - data massage should be done in the service layer
   * 
   * @param programId - Program ID to filter by
   * @param subProgramId - Optional sub program ID for more specific filtering
   * @param filters - Optional filter object containing additional criteria (gender, etc.)
   * @returns Promise<Array<{ city: string; otherCityName: string }>> - Array of raw location data
   */
  async getDistinctLocations(
    programId: number,
    subProgramId?: number,
    filters: Record<string, any> = {},
    hasAllocationJoin?: boolean,
  ): Promise<Array<{ city: string; otherCityName: string }>> {
    this.logger.log('Fetching distinct locations from program registrations', {
      programId,
      subProgramId,
      filters,
    });

    try {
      const queryBuilder = this.repo
        .createQueryBuilder('registration')
        .select(["TRIM(registration.city) AS city", "TRIM(registration.otherCityName) AS otherCityName"])
        .distinct(true)
        .leftJoin(
          'registration.approvals',
          'approval',
          'approval.deletedAt IS NULL',
        )
        .where('registration.deletedAt IS NULL')
        .andWhere('registration.program_id = :programId', { programId })
        .andWhere('approval.approvalStatus = :approvalStatus', {
          approvalStatus: ApprovalStatusEnum.APPROVED,
        })
        .andWhere('(registration.city IS NOT NULL OR registration.otherCityName IS NOT NULL)');
      
      // Apply join with room allocation if specified
      if (hasAllocationJoin) {
        queryBuilder.leftJoin(
          'room_allocation',
          'allocation',
          'allocation.registration_id = registration.id AND allocation.deleted_at IS NULL',
        );
      if( filters.roomAllocation !== undefined) {
          if (filters.roomAllocation === true || filters.roomAllocation === 'true') {
            queryBuilder.andWhere('allocation.id IS NOT NULL');
          } else if (filters.roomAllocation === false || filters.roomAllocation === 'false') {
            queryBuilder.andWhere('allocation.id IS NULL');
          }
        }
      }
      // Apply sub-program filter if provided
      if (subProgramId) {
        queryBuilder.andWhere('registration.allocated_program_id = :subProgramId', {
          subProgramId,
        });
      }

      // Apply additional filters if provided
      if (filters.gender) {
        queryBuilder.andWhere('registration.gender = :gender', {
          gender: filters.gender,
        });
      }

      const result = await queryBuilder.getRawMany();

      this.logger.log('Distinct locations fetched successfully from program registrations', {
        programId,
        subProgramId,
        count: result.length,
      });

      return result;
    } catch (error) {
      this.logger.error('Error fetching distinct locations from program registrations', error.stack, {
        programId,
        subProgramId,
        filters,
      });
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Fetch registrations by program with optional filters, pagination, and search.
   * @param programId - Program ID to filter by.
   * @param limit - Number of records per page.
   * @param offset - Offset for pagination.
   * @param searchText - Optional search text for filtering.
   * @param filters - Additional filters.
   * @param sort - Sorting options.
   * @param searchFields - Optional array of fields to search in
   * @param getPairAlong - Whether to include paired registrations in search
   * @param includeRelations - Optional flags to control which relations to join and load
   * @returns Paginated registrations data with metadata.
   */
  async getRegistrationsByProgram(
    programId: number,
    limit: number = 10,
    offset: number = 0,
    searchText?: string,
    filters: Record<string, any> = {},
    sort?: Array<Record<string, 'asc' | 'desc'>>,
    searchFields?: string[],
    getPairAlong: boolean = false,
    includeRelations?: {
      program?: boolean;
      rmUser?: boolean;
      travelPlan?: boolean;
      allocatedProgram?: boolean;
      allocation?: boolean;
      roomInventory?: boolean;
      room?: boolean;
      approval?: boolean;
      pairMap?: boolean;
      pair?: boolean;
    }
  ): Promise<{ data: any[]; total: number; pagination: any }> {
    this.logger.log('Fetching registrations by program from repository', {
      programId,
      limit,
      offset,
      searchText,
      filters,
      sort,
      includeRelations
    });
  
    try {
      // Default to loading all relations if not specified
      const relations = includeRelations || {
        program: true,
        rmUser: true,
        travelPlan: true,
        allocatedProgram: true,
        allocation: true,
        roomInventory: true,
        room: true,
        approval: true,
        pairMap: true,
        pair: true,
      };

      // Determine which joins are needed based on filters OR includeRelations
      const needAllocatedProgramJoin = filters.allocatedProgramId !== undefined || relations.allocatedProgram;
      const needAllocationJoin =
        filters.hasRoomAllocation !== undefined ||
        filters.roomInventoryId !== undefined ||
        needAllocatedProgramJoin ||
        relations.allocation;
      const needRoomInventoryJoin = filters.roomInventoryId !== undefined || relations.roomInventory;
      const needRoomJoin = filters.roomInventoryId !== undefined || relations.room;
      const needApprovalJoin = filters.approvalStatus !== undefined || relations.approval;
      const needPairMapJoin = relations.pairMap;
      const needPairJoin = relations.pair;
      const needProgramJoin = relations.program;
      const needRmUserJoin = relations.rmUser;
      const needTravelPlanJoin = relations.travelPlan;
  
      const selectFields = [
        'registration.id as "id"',
        'registration.registrationSeqNumber as "registrationSeqNumber"',
        'registration.fullName as "fullName"',
        'registration.gender as "gender"',
        'registration.mobileNumber as "mobileNumber"',
        'registration.emailAddress as "emailAddress"',
        'registration.dob as "dob"',
        'registration.registrationStatus as "registrationStatus"',
        'registration.registrationDate as "registrationDate"',
        'registration.preferredRoomMate as "preferredRoomMate"',
        'registration.city as "city"',
        'registration.otherCityName as "otherCityName"',
        'registration.countryName as "countryName"',
        'registration.profileUrl as "profileUrl"',
        'program.id as "programId"',
        'program.name as "programTitle"',
        'registration.noOfHDBs as "noOfHDBs"',
        'rmUser.fullName as "rmName"',
        'travelPlan.departureDatetime as "departureDatetime"',
        'travelPlan.returnTerminal as "returnTerminal"',
      ];
  
      if (needAllocatedProgramJoin) {
        selectFields.push('allocatedProgram.id as "allocatedProgramId"');
        selectFields.push('allocatedProgram.name as "allocatedProgramTitle"');
      }
  
      if (needAllocationJoin) {
        selectFields.push('allocation.id as "allocationId"');
        selectFields.push('allocation.bedPosition as "bedPosition"');
        selectFields.push('allocation.remarks as "allocationRemarks"');
      }
  
      if (needRoomInventoryJoin) {
        selectFields.push('roomInventory.id as "roomInventoryId"');
      }
  
      if (needRoomJoin) {
        selectFields.push('room.roomNumber as "roomNumber"');
        selectFields.push('room.label as "roomLabel"');
        selectFields.push('room.occupancy as "roomOccupancy"');
      }
  
      if (needPairMapJoin) {
        selectFields.push('pairMap.id as "pairMapId"');
        selectFields.push('pairMap.registrationPairId as "registrationPairId"');
      }
      if (needPairJoin) {
        selectFields.push('pair.id as "pairId"');
        selectFields.push('CASE WHEN pair.id IS NOT NULL THEN CONCAT(pairProgram.code, pair.pairCode, LPAD(pair.seq_number::text,3,\'0\')) ELSE NULL END as "pairCode"');
      }
  
      const queryBuilder = this.repo
        .createQueryBuilder('registration')
        .select(selectFields)
        .leftJoin('registration.program', 'program');
  
      if (needRmUserJoin) {
        queryBuilder.leftJoin('registration.rmContactUser', 'rmUser');
      }
  
      if (needTravelPlanJoin) {
        queryBuilder.leftJoin('registration.travelPlans', 'travelPlan');
      }
  
      if (needAllocatedProgramJoin) {
        queryBuilder.leftJoin('registration.allocatedProgram', 'allocatedProgram');
      }
  
      if (needAllocationJoin) {
        queryBuilder.leftJoin('registration.roomAllocations', 'allocation');
      }
  
      if (needRoomInventoryJoin) {
        queryBuilder.leftJoin(
          'program_room_inventory_map',
          'roomInventory',
          'roomInventory.id = allocation.program_room_inventory_map_id'
        );
      }
  
      if (needRoomJoin) {
        queryBuilder.leftJoin('room', 'room', 'room.id = roomInventory.room_id');
      }
  
      if (needApprovalJoin) {
        queryBuilder.leftJoin(
          'hdb_registration_approval',
          'approval',
          'approval.registration_id = registration.id AND approval.deleted_at IS NULL'
        );
      }
  
      if (needPairMapJoin) {
        queryBuilder.leftJoin('registration.registrationPairMaps', 'pairMap');
      }
      if (needPairJoin) {
        queryBuilder.leftJoin('pairMap.registrationPair', 'pair');
        if (needProgramJoin) {
          queryBuilder.leftJoin('registration.allocatedProgram', 'pairProgram');
        }
      }
  
      queryBuilder
        .where('registration.deletedAt IS NULL')
        .andWhere('registration.programId = :programId', { programId });
  
      // Handle search text
      if (searchText && searchText.trim() && searchFields && searchFields.length > 0) {
        const searchPattern = `%${searchText.trim()}%`;
        queryBuilder.andWhere(
          new Brackets(qb => {
            // Add conditions for each search field
            searchFields.forEach((field) => {
              qb.orWhere(`${field} ILIKE :searchText`, { searchText: searchPattern });
            });
  
            // If getPairAlong, also search in paired registrations
            if (getPairAlong) {
              qb.orWhere(
                subQuery => {
                  return `pair.id IS NOT NULL AND pair.id IN (${subQuery
                    .subQuery()
                    .select('paired.id')
                    .from('registration_pair', 'paired')
                    .innerJoin(
                      'registration_pair_map',
                      'pairMapSearch',
                      'pairMapSearch.registration_pair_id = paired.id AND pairMapSearch.deleted_at IS NULL'
                    )
                    .innerJoin(
                      'hdb_program_registration',
                      'registrationSearch',
                      'registrationSearch.id = pairMapSearch.registration_id AND registrationSearch.deleted_at IS NULL'
                    )
                    .where(
                      new Brackets(subQb => {
                        searchFields.forEach((field, index) => {
                          const fieldName = `registrationSearch.${field.split('.').pop()}`;
                          if (index === 0) {
                            subQb.where(`${fieldName} ILIKE :searchText`, { searchText: searchPattern });
                          } else {
                            subQb.orWhere(`${fieldName} ILIKE :searchText`, { searchText: searchPattern });
                          }
                        });
                      })
                    )
                    .getQuery()})`;
                },
                { searchText: searchPattern }
              );
            }
          })
        );
      }
  
      if (filters && Object.keys(filters).length > 0) {
        // Filter by return travel terminal
        if (
          filters.returnTravelTerminal &&
          typeof filters.returnTravelTerminal === 'string' &&
          filters.returnTravelTerminal.trim() !== ''
        ) {
          queryBuilder.andWhere(
            new Brackets(qb => {
              qb.where('travelPlan.return_terminal = :returnTerminal', {
                returnTerminal: filters.returnTravelTerminal,
              });
  
              if (getPairAlong) {
                qb.orWhere(
                  subQuery => {
                    return `pair.id IS NOT NULL AND pair.id IN (${subQuery
                      .subQuery()
                      .select('paired.id')
                      .from('registration_pair', 'paired')
                      .innerJoin(
                        'registration_pair_map',
                        'pairMapSearch',
                        'pairMapSearch.registration_pair_id = paired.id AND pairMapSearch.deleted_at IS NULL'
                      )
                      .innerJoin(
                        'hdb_program_registration',
                        'registrationSearch',
                        'registrationSearch.id = pairMapSearch.registration_id AND registrationSearch.deleted_at IS NULL'
                      )
                      .innerJoin(
                        'hdb_registration_travel_plan',
                        'travelPlanSearch',
                        'travelPlanSearch.registration_id = registrationSearch.id'
                      )
                      .where('travelPlanSearch.return_terminal = :returnTerminal')
                      .getQuery()})`;
                  },
                  { returnTerminal: filters.returnTravelTerminal }
                );
              }
            })
          );
        }
  
        // Filter by number of HDBs
        if (filters.noOfHdbs && Array.isArray(filters.noOfHdbs)) {
          queryBuilder.andWhere(
            new Brackets(qb => {
              filters.noOfHdbs.forEach((hdbFilter, index) => {
                if (hdbFilter?.min && hdbFilter?.max) {
                  qb.orWhere(`registration.no_of_hdbs BETWEEN :minHdb${index} AND :maxHdb${index}`, {
                    [`minHdb${index}`]: hdbFilter.min,
                    [`maxHdb${index}`]: hdbFilter.max,
                  });
                } else if (hdbFilter?.min) {
                  qb.orWhere(`registration.no_of_hdbs >= :minHdb${index}`, {
                    [`minHdb${index}`]: hdbFilter.min,
                  });
                } else if (hdbFilter?.max) {
                  qb.orWhere(`registration.no_of_hdbs <= :maxHdb${index}`, {
                    [`maxHdb${index}`]: hdbFilter.max,
                  });
                } else if (hdbFilter?.minExcluded !== undefined) {
                  qb.orWhere(`registration.no_of_hdbs > :minExcludedHdb${index}`, {
                    [`minExcludedHdb${index}`]: hdbFilter.minExcluded,
                  });
                } else if (hdbFilter?.maxExcluded !== undefined) {
                  qb.orWhere(`registration.no_of_hdbs < :maxExcludedHdb${index}`, {
                    [`maxExcludedHdb${index}`]: hdbFilter.maxExcluded,
                  });
                } else if (hdbFilter?.eq !== undefined) {
                  qb.orWhere(`registration.no_of_hdbs = :eqHdb${index}`, {
                    [`eqHdb${index}`]: hdbFilter.eq,
                  });
                }
              });
  
              // If getPairAlong, also check paired registrations
              if (getPairAlong) {
                qb.orWhere(
                  subQuery => {
                    const hdbSubQuery = subQuery
                      .subQuery()
                      .select('paired.id')
                      .from('registration_pair', 'paired')
                      .innerJoin(
                        'registration_pair_map',
                        'pairMapSearch',
                        'pairMapSearch.registration_pair_id = paired.id AND pairMapSearch.deleted_at IS NULL'
                      )
                      .innerJoin(
                        'hdb_program_registration',
                        'registrationSearch',
                        'registrationSearch.id = pairMapSearch.registration_id AND registrationSearch.deleted_at IS NULL'
                      )
                      .where(
                        new Brackets(subQb => {
                          filters.noOfHdbs.forEach((hdbFilter, index) => {
                            if (hdbFilter?.min && hdbFilter?.max) {
                              subQb.orWhere(
                                `registrationSearch.no_of_hdbs BETWEEN :minHdb${index} AND :maxHdb${index}`,
                                {
                                  [`minHdb${index}`]: hdbFilter.min,
                                  [`maxHdb${index}`]: hdbFilter.max,
                                }
                              );
                            } else if (hdbFilter?.min) {
                              subQb.orWhere(`registrationSearch.no_of_hdbs >= :minHdb${index}`, {
                                [`minHdb${index}`]: hdbFilter.min,
                              });
                            } else if (hdbFilter?.max) {
                              subQb.orWhere(`registrationSearch.no_of_hdbs <= :maxHdb${index}`, {
                                [`maxHdb${index}`]: hdbFilter.max,
                              });
                            } else if (hdbFilter?.minExcluded !== undefined) {
                              subQb.orWhere(`registrationSearch.no_of_hdbs > :minExcludedHdb${index}`, {
                                [`minExcludedHdb${index}`]: hdbFilter.minExcluded,
                              });
                            } else if (hdbFilter?.maxExcluded !== undefined) {
                              subQb.orWhere(`registrationSearch.no_of_hdbs < :maxExcludedHdb${index}`, {
                                [`maxExcludedHdb${index}`]: hdbFilter.maxExcluded,
                              });
                            } else if (hdbFilter?.eq !== undefined) {
                              subQb.orWhere(`registrationSearch.no_of_hdbs = :eqHdb${index}`, {
                                [`eqHdb${index}`]: hdbFilter.eq,
                              });
                            }
                          });
                        })
                      );
  
                    return `pair.id IS NOT NULL AND pair.id IN (${hdbSubQuery.getQuery()})`;
                  },
                  {} // Parameters are already included in subquery
                );
              }
            })
          );
        }
  
        // Filter by approval status
        if (filters.approvalStatus) {
          queryBuilder.andWhere('approval.approval_status = :approvalStatus', {
            approvalStatus: filters.approvalStatus,
          });
        }
  
        // Filter by allocated program ID
        if (filters.allocatedProgramId) {
          const allocatedProgramIds = Array.isArray(filters.allocatedProgramId)
            ? filters.allocatedProgramId
            : [filters.allocatedProgramId];
          queryBuilder.andWhere('registration.allocated_program_id IN (:...allocatedProgramIds)', {
            allocatedProgramIds,
          });
        }
  
        // Filter by registration status
        if (filters.registrationStatus) {
          queryBuilder.andWhere('registration.registrationStatus = :registrationStatus', {
            registrationStatus: filters.registrationStatus,
          });
        }
  
        // Filter by gender
        if (filters.gender) {
          const genderArray = Array.isArray(filters.gender)
            ? filters.gender
                .map((g: string) => g.toLowerCase())
                .filter((g) => g === 'male' || g === 'female')
                .map((g) => (g === 'male' ? GenderEnum.MALE : GenderEnum.FEMALE))
            : [filters.gender.toLowerCase() === 'male' ? GenderEnum.MALE : GenderEnum.FEMALE];
  
          if (genderArray.length > 0) {
            queryBuilder.andWhere(
              new Brackets(qb => {
                qb.where('registration.gender IN (:...gender)', { gender: genderArray });
  
                if (getPairAlong) {
                  qb.orWhere(
                    subQuery => {
                      return `pair.id IS NOT NULL AND pair.id IN (${subQuery
                        .subQuery()
                        .select('paired.id')
                        .from('registration_pair', 'paired')
                        .innerJoin(
                          'registration_pair_map',
                          'pairMapSearch',
                          'pairMapSearch.registration_pair_id = paired.id AND pairMapSearch.deleted_at IS NULL'
                        )
                        .innerJoin(
                          'hdb_program_registration',
                          'registrationSearch',
                          'registrationSearch.id = pairMapSearch.registration_id AND registrationSearch.deleted_at IS NULL'
                        )
                        .where('registrationSearch.gender IN (:...gender)')
                        .getQuery()})`;
                    },
                    { gender: genderArray }
                  );
                }
              })
            );
          }
        }
  
        // Filter by room allocation status
        if (filters.hasRoomAllocation !== undefined) {
          if (filters.hasRoomAllocation === true || filters.hasRoomAllocation === 'true') {
            queryBuilder.andWhere('allocation.id IS NOT NULL');
          } else if (filters.hasRoomAllocation === false || filters.hasRoomAllocation === 'false') {
            queryBuilder.andWhere('allocation.id IS NULL');
          }
        }
  
        // Filter by city
        if (filters.city) {
          queryBuilder.andWhere(
            new Brackets(qb => {
              qb.where('(registration.city IN (:...city) OR registration.otherCityName IN (:...city))', {
                city: filters.city,
              });
  
              if (getPairAlong) {
                qb.orWhere(
                  subQuery => {
                    return `pair.id IS NOT NULL AND pair.id IN (${subQuery
                      .subQuery()
                      .select('paired.id')
                      .from('registration_pair', 'paired')
                      .innerJoin(
                        'registration_pair_map',
                        'pairMapSearch',
                        'pairMapSearch.registration_pair_id = paired.id AND pairMapSearch.deleted_at IS NULL'
                      )
                      .innerJoin(
                        'hdb_program_registration',
                        'registrationSearch',
                        'registrationSearch.id = pairMapSearch.registration_id AND registrationSearch.deleted_at IS NULL'
                      )
                      .where('(LOWER(registrationSearch.city) IN (:...city) OR LOWER(registrationSearch.otherCityName) IN (:...city))')
                      .getQuery()})`;
                  },
                  { city: filters.city }
                );
              }
            })
          );
        }
  
        // Filter by country
        if (filters.country || filters.countryName) {
          queryBuilder.andWhere('registration.countryName = :countryName', {
            countryName: filters.country || filters.countryName,
          });
        }
  
        // Filter by room inventory ID
        if (filters.roomInventoryId) {
          queryBuilder.andWhere('roomInventory.id = :roomInventoryId', {
            roomInventoryId: filters.roomInventoryId,
          });
        }
  
        // Filter by age
        if (filters.age && Array.isArray(filters.age)) {
          queryBuilder.andWhere(
            new Brackets(qb => {
              filters.age.forEach((ageFilter, index) => {
                if (ageFilter?.min && ageFilter?.max) {
                  qb.orWhere(
                    `DATE_PART('year', AGE(registration.dob)) BETWEEN :minAge${index} AND :maxAge${index}`,
                    {
                      [`minAge${index}`]: ageFilter.min,
                      [`maxAge${index}`]: ageFilter.max,
                    }
                  );
                } else if (ageFilter?.min) {
                  qb.orWhere(
                    `DATE_PART('year', AGE(registration.dob)) >= :minAge${index}`,
                    {
                      [`minAge${index}`]: ageFilter.min,
                    }
                  );
                } else if (ageFilter?.max) {
                  qb.orWhere(
                    `DATE_PART('year', AGE(registration.dob)) <= :maxAge${index}`,
                    {
                      [`maxAge${index}`]: ageFilter.max,
                    }
                  );
                } else if (ageFilter?.minExcluded !== undefined) {
                  qb.orWhere(
                    `DATE_PART('year', AGE(registration.dob)) > :minExcludedAge${index}`,
                    {
                      [`minExcludedAge${index}`]: ageFilter.minExcluded,
                    }
                  );
                } else if (ageFilter?.maxExcluded !== undefined) {
                  qb.orWhere(
                    `DATE_PART('year', AGE(registration.dob)) < :maxExcludedAge${index}`,
                    {
                      [`maxExcludedAge${index}`]: ageFilter.maxExcluded,
                    }
                  );
                } else if (ageFilter?.eq !== undefined) {
                  qb.orWhere(
                    `DATE_PART('year', AGE(registration.dob)) = :eqAge${index}`,
                    {
                      [`eqAge${index}`]: ageFilter.eq,
                    }
                  );
                }
              });
  
              // If getPairAlong, also check paired registrations
              if (getPairAlong) {
                qb.orWhere(
                  subQuery => {
                    const ageSubQuery = subQuery
                      .subQuery()
                      .select('paired.id')
                      .from('registration_pair', 'paired')
                      .innerJoin(
                        'registration_pair_map',
                        'pairMapSearch',
                        'pairMapSearch.registration_pair_id = paired.id AND pairMapSearch.deleted_at IS NULL'
                      )
                      .innerJoin(
                        'hdb_program_registration',
                        'registrationSearch',
                        'registrationSearch.id = pairMapSearch.registration_id AND registrationSearch.deleted_at IS NULL'
                      )
                      .where(
                        new Brackets(subQb => {
                          filters.age.forEach((ageFilter, index) => {
                            if (ageFilter?.min && ageFilter?.max) {
                              subQb.orWhere(
                                `DATE_PART('year', AGE(registrationSearch.dob)) BETWEEN :minAge${index} AND :maxAge${index}`,
                                {
                                  [`minAge${index}`]: ageFilter.min,
                                  [`maxAge${index}`]: ageFilter.max,
                                }
                              );
                            } else if (ageFilter?.min) {
                              subQb.orWhere(
                                `DATE_PART('year', AGE(registrationSearch.dob)) >= :minAge${index}`,
                                {
                                  [`minAge${index}`]: ageFilter.min,
                                }
                              );
                            } else if (ageFilter?.max) {
                              subQb.orWhere(
                                `DATE_PART('year', AGE(registrationSearch.dob)) <= :maxAge${index}`,
                                {
                                  [`maxAge${index}`]: ageFilter.max,
                                }
                              );
                            } else if (ageFilter?.minExcluded !== undefined) {
                              subQb.orWhere(
                                `DATE_PART('year', AGE(registrationSearch.dob)) > :minExcludedAge${index}`,
                                {
                                  [`minExcludedAge${index}`]: ageFilter.minExcluded,
                                }
                              );
                            } else if (ageFilter?.maxExcluded !== undefined) {
                              subQb.orWhere(
                                `DATE_PART('year', AGE(registrationSearch.dob)) < :maxExcludedAge${index}`,
                                {
                                  [`maxExcludedAge${index}`]: ageFilter.maxExcluded,
                                }
                              );
                            } else if (ageFilter?.eq !== undefined) {
                              subQb.orWhere(
                                `DATE_PART('year', AGE(registrationSearch.dob)) = :eqAge${index}`,
                                {
                                  [`eqAge${index}`]: ageFilter.eq,
                                }
                              );
                            }
                          });
                        })
                      );
  
                    return `pair.id IS NOT NULL AND pair.id IN (${ageSubQuery.getQuery()})`;
                  },
                  {}
                );
              }
            })
          );
        }
  
        // Filter by preferred roommate
        if (
          filters.preferredRoommate &&
          Array.isArray(filters.preferredRoommate) &&
          filters.preferredRoommate.length > 0
        ) {
          const hasYes = filters.preferredRoommate.some((val) => val.toLowerCase() === 'yes');
          const hasNo = filters.preferredRoommate.some((val) => val.toLowerCase() === 'no');
  
          if (hasYes && !hasNo) {
            queryBuilder.andWhere(
              new Brackets(qb => {
                qb.where(
                  '(registration.preferredRoomMate IS NOT NULL AND registration.preferredRoomMate != \'\')',
                  {}
                );
  
                if (getPairAlong) {
                  qb.orWhere(
                    subQuery => {
                      return `pair.id IS NOT NULL AND pair.id IN (${subQuery
                        .subQuery()
                        .select('paired.id')
                        .from('registration_pair', 'paired')
                        .innerJoin(
                          'registration_pair_map',
                          'pairMapSearch',
                          'pairMapSearch.registration_pair_id = paired.id AND pairMapSearch.deleted_at IS NULL'
                        )
                        .innerJoin(
                          'hdb_program_registration',
                          'registrationSearch',
                          'registrationSearch.id = pairMapSearch.registration_id AND registrationSearch.deleted_at IS NULL'
                        )
                        .where('registrationSearch.preferredRoomMate IS NOT NULL')
                        .andWhere('registrationSearch.preferredRoomMate != :empty')
                        .getQuery()})`;
                    },
                    { empty: '' }
                  );
                }
              })
            );
          } else if (hasNo && !hasYes) {
            queryBuilder.andWhere(
              new Brackets(qb => {
                qb.where('(registration.preferredRoomMate IS NULL OR registration.preferredRoomMate = \'\')', {});
  
                if (getPairAlong) {
                  qb.orWhere(
                    subQuery => {
                      return `pair.id IS NOT NULL AND pair.id IN (${subQuery
                        .subQuery()
                        .select('paired.id')
                        .from('registration_pair', 'paired')
                        .innerJoin(
                          'registration_pair_map',
                          'pairMapSearch',
                          'pairMapSearch.registration_pair_id = paired.id AND pairMapSearch.deleted_at IS NULL'
                        )
                        .innerJoin(
                          'hdb_program_registration',
                          'registrationSearch',
                          'registrationSearch.id = pairMapSearch.registration_id AND registrationSearch.deleted_at IS NULL'
                        )
                        .where('(registrationSearch.preferredRoomMate IS NULL OR registrationSearch.preferredRoomMate = :empty)')
                        .getQuery()})`;
                    },
                    { empty: '' }
                  );
                }
              })
            );
          }
        }
  
        // Filter by paired status
        if (filters.pairedStatus !== undefined) {
          if (filters.pairedStatus?.toLowerCase() === 'yes') {
            queryBuilder.andWhere('pair.id IS NOT NULL');
          } else if (filters.pairedStatus?.toLowerCase() === 'no') {
            queryBuilder.andWhere('pair.id IS NULL');
          }
        }
  
        // Filter by departure datetime
        if (filters.departureDatetime && Array.isArray(filters.departureDatetime)) {
          queryBuilder.andWhere(
            new Brackets(qb => {
              filters.departureDatetime.forEach((dateFilter, index) => {
                if (dateFilter?.min && dateFilter?.max) {
                  qb.orWhere(
                    `travelPlan.departure_datetime BETWEEN :minDepartureDate${index} AND :maxDepartureDate${index}`,
                    {
                      [`minDepartureDate${index}`]: dateFilter.min,
                      [`maxDepartureDate${index}`]: dateFilter.max,
                    }
                  );
                } else if (dateFilter?.min) {
                  qb.orWhere(
                    `travelPlan.departure_datetime >= :minDepartureDate${index}`,
                    {
                      [`minDepartureDate${index}`]: dateFilter.min,
                    }
                  );
                } else if (dateFilter?.max) {
                  qb.orWhere(
                    `travelPlan.departure_datetime <= :maxDepartureDate${index}`,
                    {
                      [`maxDepartureDate${index}`]: dateFilter.max,
                    }
                  );
                }
              });
  
              // If getPairAlong, also check paired registrations
              if (getPairAlong) {
                qb.orWhere(
                  subQuery => {
                    const dateSubQuery = subQuery
                      .subQuery()
                      .select('paired.id')
                      .from('registration_pair', 'paired')
                      .innerJoin(
                        'registration_pair_map',
                        'pairMapSearch',
                        'pairMapSearch.registration_pair_id = paired.id AND pairMapSearch.deleted_at IS NULL'
                      )
                      .innerJoin(
                        'hdb_program_registration',
                        'registrationSearch',
                        'registrationSearch.id = pairMapSearch.registration_id AND registrationSearch.deleted_at IS NULL'
                      )
                      .innerJoin(
                        'hdb_registration_travel_plan',
                        'travelPlanSearch',
                        'travelPlanSearch.registration_id = registrationSearch.id'
                      )
                      .where(
                        new Brackets(subQb => {
                          filters.departureDatetime.forEach((dateFilter, index) => {
                            if (dateFilter?.min && dateFilter?.max) {
                              subQb.orWhere(
                                `travelPlanSearch.departure_datetime BETWEEN :minDepartureDate${index} AND :maxDepartureDate${index}`,
                                {
                                  [`minDepartureDate${index}`]: dateFilter.min,
                                  [`maxDepartureDate${index}`]: dateFilter.max,
                                }
                              );
                            } else if (dateFilter?.min) {
                              subQb.orWhere(
                                `travelPlanSearch.departure_datetime >= :minDepartureDate${index}`,
                                {
                                  [`minDepartureDate${index}`]: dateFilter.min,
                                }
                              );
                            } else if (dateFilter?.max) {
                              subQb.orWhere(
                                `travelPlanSearch.departure_datetime <= :maxDepartureDate${index}`,
                                {
                                  [`maxDepartureDate${index}`]: dateFilter.max,
                                }
                              );
                            }
                          });
                        })
                      );
  
                    return `pair.id IS NOT NULL AND pair.id IN (${dateSubQuery.getQuery()})`;
                  },
                  {}
                );
              }
            })
          );
        }
  
        // Filter by RM contact
        if (filters.rmContact && Array.isArray(filters.rmContact) && filters.rmContact.length > 0) {
          queryBuilder.andWhere(
            new Brackets(qb => {
              qb.where('registration.rm_contact IN (:...rmContactIds)', {
                rmContactIds: filters.rmContact,
              });
  
              if (getPairAlong) {
                qb.orWhere(
                  subQuery => {
                    return `pair.id IS NOT NULL AND pair.id IN (${subQuery
                      .subQuery()
                      .select('paired.id')
                      .from('registration_pair', 'paired')
                      .innerJoin(
                        'registration_pair_map',
                        'pairMapSearch',
                        'pairMapSearch.registration_pair_id = paired.id AND pairMapSearch.deleted_at IS NULL'
                      )
                      .innerJoin(
                        'hdb_program_registration',
                        'registrationSearch',
                        'registrationSearch.id = pairMapSearch.registration_id AND registrationSearch.deleted_at IS NULL'
                      )
                      .where('registrationSearch.rm_contact IN (:...rmContactIds)')
                      .getQuery()})`;
                  },
                  { rmContactIds: filters.rmContact }
                );
              }
            })
          );
        }
      }
  
      const countQuery = queryBuilder.clone();
      const total = await countQuery.getCount();
  
      // Add default ordering by pair code
      queryBuilder.addOrderBy(
        `CASE WHEN pair.id IS NOT NULL THEN CONCAT(pairProgram.code, pair.pairCode, LPAD(pair.seq_number::text,3, '0')) ELSE NULL END`,
        'DESC',
        'NULLS LAST'
      );
  
      // Add custom sorting
      if (sort && Array.isArray(sort) && sort?.length > 0) {
        sort.forEach((sortOption) => {
          const [field, order] = Object.entries(sortOption)[0];
          let column = '';
          switch (field) {
            case 'age':
              column = "DATE_PART('year', AGE(registration.dob))";
              break;
            case 'noOfHdbs':
              column = 'registration.no_of_hdbs';
              break;
            case 'pairCode':
              column = `CASE WHEN pair.id IS NOT NULL THEN CONCAT(pairProgram.code, pair.pairCode, LPAD(pair.seq_number::text, '0')) ELSE NULL END`;
              break;
            case 'departureDatetime':
              column = 'travelPlan.departure_datetime';
              break;
            default:
              column = `registration.${field}`;
          }
  
          queryBuilder.addOrderBy(`${column}`, order.toUpperCase() as 'ASC' | 'DESC', 'NULLS LAST');
        });
      }
  
      const data = await queryBuilder.limit(limit).offset(offset).getRawMany();
  
      const paginationInfo = {
        totalPages: Math.ceil(total / limit),
        pageNumber: Math.floor(offset / limit) + 1,
        pageSize: limit,
        totalRecords: total,
        numberOfRecords: data.length,
      };
  
      this.logger.log('Registrations by program fetched successfully', {
        programId,
        searchText,
        total,
        count: data.length,
        appliedFilters: filters,
      });
  
      return { data, total, pagination: paginationInfo };
    } catch (error) {
      this.logger.error('Error fetching registrations by program', error.stack, {
        programId,
        limit,
        offset,
        searchText,
        filters,
        sort,
      });
      handleKnownErrors(ERROR_CODES.ROOM_ALLOCATION_GET_FAILED, error);
    }
  }

  /**
   * Get count of registrations based on room allocation status for a specific program
   * Only counts approved registrations - Pure TypeORM implementation without QueryBuilder
   * @param programId - Program ID to filter by
   * @param allocatedProgramId - Optional allocated program ID to filter by
   * @returns Promise<{ allocated: number; notAllocated: number; total: number }> - Count of registrations by allocation status
   */
  async getRegistrationCountByAllocationStatus(
    programId?: number,
    allocatedProgramId?: number
  ): Promise<{ allocated: number; notAllocated: number; total: number }> {
    this.logger.log('Fetching registration count by room allocation status', {
      programId,
      allocatedProgramId,
    });

    try {
      // Build base conditions for filtering registrations
      const baseWhere: any = {
        deletedAt: IsNull(),
        approvals: {
          deletedAt: IsNull(),
          approvalStatus: ApprovalStatusEnum.APPROVED,
        }
      };

      // Add program filter if provided
      if (programId) {
        baseWhere.programId = programId;
      }

      // Add allocated program filter if provided
      if (allocatedProgramId) {
        baseWhere.allocatedProgramId = allocatedProgramId;
      }

      // Get total count
      const total = await this.repo.count({
        where: baseWhere,
        relations: ['approvals']
      });

      // Get allocated count (where allocatedProgramId is NOT null)
      const allocated = await this.repo.count({
        where: {
          ...baseWhere,
          approvals:{
            deletedAt: IsNull(),
            approvalStatus: ApprovalStatusEnum.APPROVED,
          },
          roomAllocations: {
            id: Not(IsNull()),
          }
        },
        relations: ['approvals','roomAllocations']
      });

      // Get not allocated count (where allocatedProgramId is null)
      const notAllocated = await this.repo.count({
        where: {
          ...baseWhere,
          allocatedProgramId: IsNull(),
          roomAllocations: {
            id: IsNull(),
          }
        },
        relations: ['approvals','roomAllocations']
      });

      this.logger.log('Registration count by allocation status fetched successfully', {
        programId,
        allocatedProgramId,
        total,
        allocated,
        notAllocated,
      });

      return { allocated, notAllocated, total };
    } catch (error) {
      this.logger.error('Error fetching registration count by allocation status', error.stack, {
        programId,
        allocatedProgramId,
      });
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }
}
