import { ApiProperty } from '@nestjs/swagger';
import { ArrayMinSize, IsArray, IsIn, IsInt, IsNotEmpty, IsOptional, IsString, ValidateNested } from 'class-validator';
import { SwapRequestStatus } from 'src/common/enum/swap-request-status-enum'; 
import { Type } from 'class-transformer';
import { SwapType } from 'src/common/enum/swap-type-enum';


export class ProgramDto{
  @ApiProperty({ description: 'ID of the program' })
  @IsInt()
  @IsNotEmpty()
  id: number;
}

export class CreateProgramRegistrationSwapDto {
  @ApiProperty({ description: 'ID of the current program registration' })
  @IsInt()
  @IsNotEmpty()
  programRegistrationId: number;

  @ApiProperty({ description: 'Swap requirement type', enum: ['SWAP_REQUEST', 'SWAP_DEMAND'], default: 'SWAP_REQUEST', required: false })
  @IsOptional()
  @IsString()
  @IsIn(['SWAP_REQUEST', 'SWAP_DEMAND'])
  swapRequirement?: 'SWAP_REQUEST' | 'SWAP_DEMAND';

  @ApiProperty({ description: 'List of target programs for the swap' })
  @IsNotEmpty()
  @IsArray()
  @ArrayMinSize(1)
  @ValidateNested({ each: true })
  @Type(() => ProgramDto)
  targetPrograms: ProgramDto[];

  @ApiProperty({ description: 'Swap type (a or b)' })
  @IsString()
  @IsNotEmpty()
  @IsIn(Object.values(SwapType))
  type: SwapType;

  @ApiProperty({ description: 'Comment for the swap' })
  @IsString()
  @IsNotEmpty()
  comment: string;

}

export class MakeSwapRequestDto {

  @ApiProperty({ description: "Type of swapping action ('move' or 'swap')" })
  @IsString()
  @IsNotEmpty()
  @IsIn(['move', 'swap'])
  swappingType: 'move' | 'swap';

  @ApiProperty({ description: 'New status of the swap request' })
  @IsString()
  @IsNotEmpty()
  @IsIn(Object.values(SwapRequestStatus))
  status: SwapRequestStatus;

  @ApiProperty({ description: 'Comment for the swap request' })
  @IsString()
  @IsOptional()
  comment: string;

  @ApiProperty({ description: 'Program Registration ID of the moving seeker registration' })
  @IsInt()
  @IsNotEmpty()
  @IsOptional()
  movingSeekerRegistrationId: number;

  @ApiProperty({ description: 'ID of the Sub Program to swap with' })
  @IsInt()
  @IsOptional()
  movingToSubProgramId: number;

  @ApiProperty({ description: 'Program Registration ID of the outgoing seeker registration' })
  @IsInt()
  @IsNotEmpty()
  @IsOptional()
  outgoingSeekerRegistrationId: number;
}

export class UpdateSwapRequestDto {
  @ApiProperty({ 
    description: 'Action to perform on the swap request',
    enum: ['edit', 'cancel'],
    example: 'edit'
  })
  @IsString()
  @IsNotEmpty()
  @IsIn(['edit', 'cancel'])
  action: 'edit' | 'cancel';

  @ApiProperty({ description: 'Swap requirement type', enum: ['SWAP_REQUEST', 'SWAP_DEMAND'], required: false })
  @IsOptional()
  @IsString()
  @IsIn(['SWAP_REQUEST', 'SWAP_DEMAND'])
  swapRequirement?: 'SWAP_REQUEST' | 'SWAP_DEMAND';

  @ApiProperty({ 
    description: 'List of target programs for the swap (required for edit action)',
    required: false,
    type: [ProgramDto],
    example: [{ id: 123 }]
  })
  @IsOptional()
  @IsArray()
  @ArrayMinSize(1)
  @ValidateNested({ each: true })
  @Type(() => ProgramDto)
  targetPrograms?: ProgramDto[];

  @ApiProperty({ 
    description: 'Swap type to update (optional)',
    enum: Object.values(SwapType),
    required: false,
    example: 'WantsSwap'
  })
  @IsOptional()
  @IsString()
  @IsIn(Object.values(SwapType))
  type?: SwapType;

  @ApiProperty({ 
    description: 'Updated comment for the swap request',
    required: false,
    example: 'Updated swap request comment'
  })
  @IsOptional()
  @IsString()
  comment?: string;
}

export class GetSwapRequestsDto {
  @ApiProperty({ description: 'ID of the swap preference program' })
  @IsOptional()
  @IsInt()
  @IsNotEmpty()
  requestedProgramId: number;

  @ApiProperty({ description: 'Gender of the user making the request' })
  @IsOptional()
  @IsString()
  @IsNotEmpty()
  @IsIn(['Male', 'Female'])
  gender: string;
}

