import {
  Injectable,
  BadRequestException,
  NotFoundException,
  ConflictException,
} from '@nestjs/common';
import { PaymentEditRequestRepository } from '../repositories/payment-edit-request.repository';
import { PaymentEditRequest } from 'src/common/entities/payment-edit-request.entity';
import { PaymentEditRequestStatus } from 'src/common/enum/payment-edit-request-status.enum';
import { RegistrationPaymentDetailService } from 'src/registration-payment-detail/registration-payment-detail.service';
import { CreatePaymentEditRequestDto, UpdatePaymentEditRequestDto } from '../dto/payment-edit-request.dto';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { awsConfig } from 'src/common/config/config';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
// import { RegistrationPaymentDetail } from 'src/common/entities';
import { RegistrationPaymentDetailRepository } from 'src/registration-payment-detail/registration-payment-detail.repository';

@Injectable()
export class PaymentEditRequestService {
  constructor(
    private readonly paymentEditRequestRepository: PaymentEditRequestRepository,
    private readonly registrationPaymentDetailService: RegistrationPaymentDetailService,
  ) {}

  async processRequest(
    status: PaymentEditRequestStatus,
    paymentId: number,
    userId: number,
  ): Promise<PaymentEditRequest | null> {
    try {
      const request = await this.getLatestRequestForPayment(paymentId);
      if (status === PaymentEditRequestStatus.REQUESTED) {
        if (request && request.requestStatus === PaymentEditRequestStatus.REQUESTED) {
          throw new InifniBadRequestException(
            ERROR_CODES.PAYMENT_EDIT_REQUEST_ALREADY_EXISTS,
            null,
            null,
            'Payment edit request already exists',
          );
        } else {
          await this.createRequest({ paymentId }, userId);
        }
      } else if (status === PaymentEditRequestStatus.CLOSED) {
        if (request && request.requestStatus === PaymentEditRequestStatus.REQUESTED) {
          await this.paymentEditRequestRepository.update(request.id, {
            requestStatus: PaymentEditRequestStatus.CLOSED,
            updatedBy: userId,
          });
        } else {
          throw new NotFoundException(`No pending edit request found for payment ${paymentId}`);
        }
      } else if (status === PaymentEditRequestStatus.COMPLETED) {
        if (request && request.requestStatus === PaymentEditRequestStatus.REQUESTED) {
          await this.paymentEditRequestRepository.completeRequest(request.id, userId, status);
        } else {
          throw new NotFoundException(`No pending edit request found for payment ${paymentId}`);
        }
      }
      return await this.getLatestRequestForPayment(paymentId);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PAYMENT_EDIT_REQUEST_PROCESSING_FAILED, error);
    }
  }

  async createRequest(
    createDto: CreatePaymentEditRequestDto,
    requestedBy: number,
  ): Promise<PaymentEditRequest> {
    try {
      // Validate payment exists
      const payment = await this.registrationPaymentDetailService.findOne(createDto.paymentId);
      if (!payment) {
        throw new NotFoundException(`Payment with ID ${createDto.paymentId} not found`);
      }

      // Check if there's already an active request for this payment
      const activeRequest = await this.paymentEditRequestRepository.findActiveRequestForPayment(
        createDto.paymentId,
      );
      if (activeRequest) {
        throw new ConflictException('There is already an active edit request for this payment');
      }

      const requestData = {
        ...createDto,
        requestedBy,
        requestStatus: PaymentEditRequestStatus.REQUESTED,
        createdAt: new Date(),
      };

      return await this.paymentEditRequestRepository.create(requestData);
    } catch (error) {
      handleKnownErrors(
        ERROR_CODES.PAYMENT_EDIT_REQUEST_PROCESSING_FAILED,
        new Error(`Failed to create payment edit request: ${error.message}`),
      );
    }
  }

  async findById(id: number): Promise<PaymentEditRequest> {
    try {
      const request = await this.paymentEditRequestRepository.findById(id);
      if (!request) {
        throw new NotFoundException(`Payment edit request with ID ${id} not found`);
      }
      return request;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PAYMENT_EDIT_REQUEST_NOT_FOUND, error);
    }
  }

  async findByPaymentId(paymentId: number): Promise<PaymentEditRequest[]> {
    try {
      return await this.paymentEditRequestRepository.findByPaymentId(paymentId);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PAYMENT_EDIT_REQUEST_NOT_FOUND, error);
    }
  }

  async findPendingRequests(paymentId: number): Promise<PaymentEditRequest | null> {
    try {
      return await this.paymentEditRequestRepository.findActiveRequestForPayment(paymentId);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PAYMENT_EDIT_REQUEST_NOT_FOUND, error);
    }
  }

  async updateRequest(
    id: number,
    updateDto: UpdatePaymentEditRequestDto,
    updatedBy: number,
  ): Promise<PaymentEditRequest> {
    const existingRequest = await this.findById(id);

    // Only allow updates if request is in requested status
    if (existingRequest.requestStatus !== PaymentEditRequestStatus.REQUESTED) {
      throw new BadRequestException('Only requested requests can be updated');
    }

    // Don't allow changing the payment ID
    if (updateDto.paymentId && updateDto.paymentId !== existingRequest.paymentId) {
      throw new BadRequestException('Payment ID cannot be changed');
    }

    const updateData = {
      ...updateDto,
      updatedBy,
    };

    const updatedRequest = await this.paymentEditRequestRepository.update(id, updateData);
    if (!updatedRequest) {
      throw new NotFoundException(`Payment edit request with ID ${id} not found`);
    }

    return updatedRequest;
  }

  async completeRequest(id: number, completedBy: number): Promise<PaymentEditRequest | null> {
    try {
      const request = await this.findById(id);

      if (request.requestStatus !== PaymentEditRequestStatus.REQUESTED) {
        throw new BadRequestException('Only requested requests can be completed');
      }

      return await this.paymentEditRequestRepository.completeRequest(
        id,
        completedBy,
        PaymentEditRequestStatus.COMPLETED,
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PAYMENT_EDIT_REQUEST_PROCESSING_FAILED, error);
    }
  }

  async cancelRequest(id: number, cancelledBy: number): Promise<PaymentEditRequest | null> {
    try {
      const request = await this.findById(id);

      // Only allow cancellation of requested requests by the requester
      if (request.requestStatus !== PaymentEditRequestStatus.REQUESTED) {
        throw new BadRequestException('Only requested requests can be cancelled');
      }

      if (request.requestedBy !== cancelledBy) {
        throw new BadRequestException('Only the requester can cancel their own request');
      }

      return await this.paymentEditRequestRepository.completeRequest(
        id,
        cancelledBy,
        PaymentEditRequestStatus.CLOSED,
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PAYMENT_EDIT_REQUEST_PROCESSING_FAILED, error);
    }
  }

  async getLatestRequestForPayment(paymentId: number): Promise<PaymentEditRequest | null> {
    try {
      return await this.paymentEditRequestRepository.findLatestByPaymentId(paymentId);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PAYMENT_EDIT_REQUEST_NOT_FOUND, error);
    }
  }
}
