import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, FindOptionsWhere, FindManyOptions } from 'typeorm';
import { PaymentEditRequest } from 'src/common/entities/payment-edit-request.entity';
import { PaymentEditRequestStatus } from 'src/common/enum/payment-edit-request-status.enum';

@Injectable()
export class PaymentEditRequestRepository {
  constructor(
    @InjectRepository(PaymentEditRequest)
    private readonly repository: Repository<PaymentEditRequest>,
  ) {}

  async create(data: Partial<PaymentEditRequest>): Promise<PaymentEditRequest> {
    const entity = this.repository.create(data);
    return await this.repository.save(entity);
  }

  async findById(id: number): Promise<PaymentEditRequest | null> {
    return await this.repository.findOne({
      where: { id },
      relations: ['payment', 'requestedByUser', 'updatedByUser', 'completedByUser'],
    });
  }

  async findByPaymentId(paymentId: number): Promise<PaymentEditRequest[]> {
    return await this.repository.find({
      where: { paymentId },
      relations: ['payment', 'requestedByUser', 'updatedByUser', 'completedByUser'],
      order: { createdAt: 'DESC' },
    });
  }

  async findByRequestedBy(
    requestedBy: number,
    options?: FindManyOptions<PaymentEditRequest>,
  ): Promise<PaymentEditRequest[]> {
    return await this.repository.find({
      where: { requestedBy },
      relations: ['payment', 'requestedByUser', 'updatedByUser', 'completedByUser'],
      order: { createdAt: 'DESC' },
      ...options,
    });
  }

  async findByStatus(
    status: PaymentEditRequestStatus,
    options?: FindManyOptions<PaymentEditRequest>,
  ): Promise<PaymentEditRequest[]> {
    return await this.repository.find({
      where: { requestStatus: status },
      relations: ['payment', 'requestedByUser', 'updatedByUser', 'completedByUser'],
      order: { createdAt: 'DESC' },
      ...options,
    });
  }

  async findAll(options?: FindManyOptions<PaymentEditRequest>): Promise<PaymentEditRequest[]> {
    return await this.repository.find({
      relations: ['payment', 'requestedByUser', 'updatedByUser', 'completedByUser'],
      order: { createdAt: 'DESC' },
      ...options,
    });
  }

  async findWithPagination(
    page: number = 1,
    limit: number = 10,
    where?: FindOptionsWhere<PaymentEditRequest>,
  ): Promise<{ data: PaymentEditRequest[]; total: number; page: number; totalPages: number }> {
    const skip = (page - 1) * limit;

    const [data, total] = await this.repository.findAndCount({
      where,
      relations: ['payment', 'requestedByUser', 'updatedByUser', 'completedByUser'],
      order: { createdAt: 'DESC' },
      skip,
      take: limit,
    });

    return {
      data,
      total,
      page,
      totalPages: Math.ceil(total / limit),
    };
  }

  async update(id: number, data: Partial<PaymentEditRequest>): Promise<PaymentEditRequest | null> {
    await this.repository.update(id, data);
    return await this.findById(id);
  }

  async updateStatus(
    id: number,
    status: PaymentEditRequestStatus,
    updatedBy: number,
  ): Promise<PaymentEditRequest | null> {
    return await this.update(id, {
      requestStatus: status,
      updatedBy,
      updatedAt: new Date(),
    });
  }

  async completeRequest(
    id: number,
    completedBy: number,
    status: PaymentEditRequestStatus = PaymentEditRequestStatus.COMPLETED,
  ): Promise<PaymentEditRequest | null> {
    return await this.update(id, {
      requestStatus: status,
      completedBy,
      updatedBy: completedBy,
      updatedAt: new Date(),
    });
  }

  async delete(id: number): Promise<void> {
    await this.repository.delete(id);
  }

  async count(where?: FindOptionsWhere<PaymentEditRequest>): Promise<number> {
    return await this.repository.count({ where });
  }

  async exists(where: FindOptionsWhere<PaymentEditRequest>): Promise<boolean> {
    const count = await this.repository.count({ where });
    return count > 0;
  }

  async findLatestByPaymentId(paymentId: number): Promise<PaymentEditRequest | null> {
    return await this.repository.findOne({
      where: { paymentId },
      relations: ['payment', 'requestedByUser', 'updatedByUser', 'completedByUser'],
      order: { id: 'DESC' },
    });
  }

  async findActiveRequestForPayment(paymentId: number): Promise<PaymentEditRequest | null> {
    return await this.repository.findOne({
      where: {
        paymentId,
        requestStatus: PaymentEditRequestStatus.REQUESTED,
      },
      relations: ['payment', 'requestedByUser', 'updatedByUser', 'completedByUser'],
    });
  }
}
