import { NestFactory } from '@nestjs/core';
import { AppModule } from './app.module';
import { ConfigService } from '@nestjs/config';
import { appConfig } from './common/config/config';
import { DocumentBuilder, SwaggerModule } from '@nestjs/swagger';
import { SWAGGER } from './common/constants/strings-constants';
import { SecuritySchemeType } from '@nestjs/swagger/dist/interfaces/open-api-spec.interface';
import { HttpExceptionFilter } from './common/filters/http-exception.filter';
import { ValidationPipe } from '@nestjs/common';

async function bootstrap() {
  const app = await NestFactory.create(AppModule);

  // Apply the global exception filter
   app.useGlobalFilters(new HttpExceptionFilter());
  
   // Apply the global validation pipe
   app.useGlobalPipes(new ValidationPipe({ transform: true, whitelist: true }));

   // Security: Disable x-powered-by header to prevent framework detection
   app.getHttpAdapter().getInstance().disable('x-powered-by');

  //Get environment variables
  const configService = app.get(ConfigService);
  const { port, corsOrigin, backendUrl } = appConfig(configService);

  //Set Swagger Documentation
  const swaggerConfig = new DocumentBuilder()
    .setTitle(SWAGGER.TITLE)
    .setDescription(
      `${SWAGGER.DESCRIPTION}<br><br>Access the raw Swagger JSON <a href="/swagger-json" target="_blank">here</a>.`
    )
    .setVersion(SWAGGER.VERSION)
    .addServer(backendUrl, SWAGGER.ENVIRONMENT)
    .addBearerAuth(
      {
        type: SWAGGER.BEARERAUTH.TYPE as SecuritySchemeType,
        name: SWAGGER.BEARERAUTH.NAME,
        description: SWAGGER.BEARERAUTH.DESCRIPTION,
        in: SWAGGER.BEARERAUTH.IN,
      },
      'Authorization',
    )
    .addApiKey(
    {
      type: SWAGGER.USER_ID_AUTH.TYPE as SecuritySchemeType,
      name: SWAGGER.USER_ID_AUTH.NAME,
      in: SWAGGER.USER_ID_AUTH.IN,
      description: SWAGGER.USER_ID_AUTH.DESCRIPTION,
    },
    'userIdAuth',
  )
    .addApiKey(
    {
      type: SWAGGER.ACTIVE_ROLE_AUTH.TYPE as SecuritySchemeType,
      name: SWAGGER.ACTIVE_ROLE_AUTH.NAME,
      in: SWAGGER.ACTIVE_ROLE_AUTH.IN,
      description: SWAGGER.ACTIVE_ROLE_AUTH.DESCRIPTION,
    },
    'activeRoleAuth',
  )
    .build();
  const document = SwaggerModule.createDocument(app, swaggerConfig);
  SwaggerModule.setup(SWAGGER.PATH, app, document, {
    swaggerOptions: {
      persistAuthorization: true,
    },
    customCss: `
      .topbar-wrapper::after {
        content: " | ";
        color: #999;
      }
      .topbar-wrapper::after,
      .topbar-wrapper a.swagger-json-link {
        font-size: 14px;
        color: #61affe;
        text-decoration: none;
        margin-left: 10px;
      }
      .topbar-wrapper a.swagger-json-link:hover {
        text-decoration: underline;
      }
    `,
    customSiteTitle: 'API Documentation',
  });

  app.use('/swagger-json', (req, res) => {
    res.setHeader('Content-Type', 'application/json');
    res.setHeader('Access-Control-Allow-Origin', '*');
    res.setHeader('Access-Control-Allow-Headers', 'Origin, X-Requested-With, Content-Type, Accept');
    res.send(document);
  });

  //Enable CORS
  app.enableCors({
    origin: '*',
    credentials: true,
  });
  await app.listen(port);
}
bootstrap();
