import { Injectable } from '@nestjs/common';
import { LookupDataRepository } from './lookup-data.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { lookupDataConstMessages, ROLE_KEYS } from 'src/common/constants/strings-constants';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { UserRepository } from 'src/user/user.repository';

@Injectable()
export class LookupDataService {
  constructor(
    private readonly repository: LookupDataRepository,
    private readonly userRepository: UserRepository,
    private readonly logger: AppLoggerService,
  ) {}

  async findActiveByCategory(category: string) {
    this.logger.log(lookupDataConstMessages.FINDING_BY_CATEGORY(category));
    try {
      return await this.repository.findActiveByCategory(category);
    } catch (error) {
      this.logger.error('Error finding lookup data by category', error?.stack, { error, category });
      handleKnownErrors(ERROR_CODES.LOOKUP_DATA_GET_FAILED, error);
    }
  }

  async findCategories() {
    this.logger.log(lookupDataConstMessages.FINDING_CATEGORIES);
    try {
      return await this.repository.findActiveCategories();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.LOOKUP_DATA_GET_FAILED, error);
    }
  }

  async findAllWithRM() {
    this.logger.log(lookupDataConstMessages.FINDING_ALL);
    try {
      const lookups = await this.repository.findAllActiveGrouped();
      // if RATRIA_PILLAR_LOCATION category is present, need to append CITY_NAME category values to this
      // if (lookups.RATRIA_PILLAR_LOCATION) {
      //   const cityNames = lookups.CITY_NAME ? [...lookups.CITY_NAME] : [];
      //   lookups.RATRIA_PILLAR_LOCATION.push(...cityNames);
      // }
      const rms = await this.userRepository.getUsersByRoleKey(ROLE_KEYS.RELATIONAL_MANAGER);

      const sortedRms = rms.sort((a, b) => {
        const nameA = a.orgUsrName?.toLowerCase();
        const nameB = b.orgUsrName?.toLowerCase();

        if (nameA === 'other') return 1;
        if (nameB === 'other') return -1;
        return nameA?.localeCompare(nameB);
      });

      lookups.RM = sortedRms.map((user) => ({
        id: Number(user.id),
        key: String(user.id),
        value: user.orgUsrName,
      }));

      return lookups;
    } catch (error) {
      this.logger.error('Error finding all lookup data', error?.stack, { error });
      handleKnownErrors(ERROR_CODES.LOOKUP_DATA_GET_FAILED, error);
    }
  }
}
