import { Controller, Post, Body, Res, HttpStatus } from '@nestjs/common';
import { ApiTags, ApiOperation, ApiResponse, ApiBody } from '@nestjs/swagger';
import { FormSchemaService } from './form-schema.service';
import { ProgramRepository } from 'src/program/program.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { formSchemaConstMessages } from 'src/common/constants/strings-constants';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { ERROR_MESSAGES } from 'src/common/i18n/error-messages';

@ApiTags('form-schema')
@Controller('form-schema')
export class FormSchemaController {
  constructor(
    private readonly formSchemaService: FormSchemaService,
    private readonly programRepo: ProgramRepository,
    private readonly logger: AppLoggerService,
    private readonly responseService: ResponseService,
  ) {}

  @Post('generate')
  @ApiOperation({ summary: 'Generate form schema from program metadata' })
  @ApiBody({
    schema: {
      type: 'object',
      properties: {
        programId: {
          type: 'number',
          description: 'ID of the program',
        },
      },
      required: ['programId'],
    },
  })
  @ApiResponse({ status: HttpStatus.OK, description: 'Form schema generated successfully' })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: 'Invalid program metadata or ID' })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: 'Internal server error' })
  async generateFormSchema(@Body() programMetadata: any, @Res() res: any) {
    try {
      if (!programMetadata || Object.keys(programMetadata).length === 0) {
        throw new InifniBadRequestException(ERROR_CODES.PROGRAM_METADATA_REQUIRED,null,null,ERROR_MESSAGES.FORM_SCHEMA_004);
      }
      if (!programMetadata.programId) {
        throw new InifniBadRequestException(ERROR_CODES.PROGRAM_METADATA_INVALID,null,null,ERROR_MESSAGES.FORM_SCHEMA_005);
      }

      const program = await this.programRepo.findOneById(programMetadata.programId);
      if (!program) {
        throw new InifniBadRequestException(ERROR_CODES.PROGRAM_METADATA_NOT_FOUND, null, null, ERROR_MESSAGES.FORM_SCHEMA_006);
      }

      const respData = await this.formSchemaService.generateSchemaFromMetadata(programMetadata);
      if (respData) {
        return this.responseService.success(
                res,
                formSchemaConstMessages.FORM_SCHEMA_GENERATION_SUCCESS
              );
      }
    } catch (error) {
      this.logger.error('Error in generateFormSchema', error.message, { programMetadata });
      handleControllerError(res, error);
    }
  }
}