import { Injectable } from '@nestjs/common';
import { FeatureFlagRepository } from './feature-flag.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { featureFlagMessages } from 'src/common/constants/strings-constants';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';

@Injectable()
export class FeatureFlagService {
  constructor(
    private readonly repository: FeatureFlagRepository,
    private readonly logger: AppLoggerService,
  ) {}

  async getActiveFlags() {
    this.logger.log(featureFlagMessages.FETCHING_ACTIVE_FLAGS);
    try {
      return this.repository.findActiveFlags();
    } catch (error) {
      this.logger.error('Error getting active feature flags', error?.stack, { error });
      handleKnownErrors(ERROR_CODES.FEATURE_FLAG_GET_FAILED, error);
    }
  }
  async isActiveFeatureEnabled(featureKey: string): Promise<boolean> {
    this.logger.log(`Checking if feature flag "${featureKey}" is enabled`);
    try {
      const flag = await this.repository.findFlagByKey(featureKey);
      const isEnabled = flag ? flag.flag : false;
      this.logger.log(
        `Feature flag "${featureKey}" is ${isEnabled ? 'enabled' : 'disabled'}, feature details: ${JSON.stringify(flag)}`,
      );
      return isEnabled;
    } catch (error) {
      this.logger.error(`Error checking feature flag "${featureKey}"`, error?.stack, {
        error,
        message: error?.message,
        stack: error?.stack,
      });
      handleKnownErrors(ERROR_CODES.FEATURE_FLAG_CHECK_FAILED, error);
    }
  }
}
