import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { CommunicationTrack } from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { CreateCommunicationTrackDto, UpdateCommunicationTrackDto } from './dto/communication-track.dto';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { CommunicationStatusEnum } from 'src/common/enum/communication-status.enum';

@Injectable()
export class CommunicationRepository {
  constructor(
    @InjectRepository(CommunicationTrack)
    private readonly repo: Repository<CommunicationTrack>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  async createEntity(dto: CreateCommunicationTrackDto) {
    try {
      const entity = this.repo.create({
        ...dto,
        registration: { id: dto.registrationId } as any,
        stats: dto.stats || CommunicationStatusEnum.INITIATED,
        createdBy: { id: dto.createdBy } as any,
        updatedBy: { id: dto.updatedBy } as any,
      });
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TRACK_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    try {
      const data = await this.commonDataService.get(
        this.repo,
        undefined,
        {},
        limit,
        offset,
        { id: 'ASC' },
        undefined,
        ['registration'],
      );
      const total = await this.repo.count();
      return {
        data,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TRACK_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    try {
      const data = await this.commonDataService.findOneById(this.repo, id, false, ['registration']);
      if (!data) {
        throw new InifniNotFoundException(ERROR_CODES.COMMUNICATION_TRACK_NOTFOUND, null, null, id.toString());
      }
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TRACK_FIND_BY_ID_FAILED, error);
    }
  }

  async findByRegistrationId(registrationId: number) {
    try {
      const data = await this.commonDataService.get(
        this.repo,
        undefined,
        { registrationId },
        undefined,
        undefined,
        { createdAt: 'DESC' },
        undefined,
        ['registration'],
      );
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TRACK_GET_FAILED, error);
    }
  }

  async updateEntity(id: number, dto: UpdateCommunicationTrackDto) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.COMMUNICATION_TRACK_NOTFOUND, null, null, id.toString());
      }
      Object.assign(entity, dto);
      if (dto.registrationId !== undefined) entity.registration = { id: dto.registrationId } as any;
      if (dto.stats !== undefined) entity.stats = dto.stats;
      entity.updatedBy = { id: dto.updatedBy } as any;
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TRACK_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.COMMUNICATION_TRACK_NOTFOUND, null, null, id.toString());
      }
      await this.repo.remove(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TRACK_DELETE_FAILED, error);
    }
  }

  async updateStatus(id: number, status: CommunicationStatusEnum, failedReason?: string, updatedBy?: number) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.COMMUNICATION_TRACK_NOTFOUND, null, null, id.toString());
      }
      entity.stats = status;
      if (failedReason !== undefined) entity.failedReason = failedReason;
      if (updatedBy !== undefined) entity.updatedBy = { id: updatedBy } as any;
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TRACK_SAVE_FAILED, error);
    }
  }
}