import { Response } from 'express';
import { ErrorHandler } from '../response-handling/error-handling';
import { MainInifniException } from '../exceptions/infini-abstract-exception';

const errorHandler = new ErrorHandler();

/**
 * Handles exceptions from the service layer and constructs proper error responses.
 * @param res - Express Response object.
 * @param error - The exception thrown by the service layer.
 */
export async function handleControllerError(res: Response, error: any): Promise<void> {
  if (error instanceof MainInifniException) {
    switch (error.name) {
      case 'InifniNotFoundException':
        await errorHandler.notFound(res, error.message);
        break;
      case 'InifniBadRequestException':
        await errorHandler.badRequest(res, error.message);
        break;
      case 'InifniDatabaseException':
        await errorHandler.serverError(res, error.error || new Error(error.message));
        break;
      case 'InifniInternalServerErrorException':
        await errorHandler.serverError(res, error.error || new Error(error.message));
        break;
      default:
        await errorHandler.serverError(res, error.error || new Error('Unhandled exception'));
        break;
    }
  } else {
    // Handle unexpected errors
    await errorHandler.serverError(res, error);
  }
}