import { Injectable } from "@nestjs/common";
import {
  Repository,
  FindManyOptions,
  FindOptionsWhere,
  FindOptionsOrder,
  ILike,
  ObjectLiteral,
  IsNull,
  DataSource,
  QueryFailedError,
  FindOptionsSelect,
} from "typeorm";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { ERROR_CODES } from "../constants/error-string-constants";
import { AppLoggerService } from "./logger.service";
import { commonDataServiceMessages } from "../constants/strings-constants";
import InifniDatabaseException from "../exceptions/infini-database-exception";
import { ERROR_MESSAGES } from "../i18n/error-messages";

@Injectable()
/**
 * CommonDataService provides generic methods for data access.
 * @template T - The entity type.
 */
export class CommonDataService {
  constructor(
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService
  ) {}

  /**
   * Fetches data from the database with optional filtering, sorting, and pagination.
   * @param repository - The repository to query.
   * @param select - Fields to select.
   * @param where - Conditions for filtering.
   * @param limit - Number of records to fetch or 'all' for unlimited.
   * @param offset - Offset for pagination or 'all' for no offset.
   * @param sort - Sorting options.
   * @param search - Search parameters.
   * @param relations - Relations to include in the result.
   * @param relationSelect - Specific fields to select from relations.
   * @returns The fetched data.
   */
  async get<T extends ObjectLiteral>(
    repository: Repository<T>,
    select?: (keyof T)[],
    where?: FindOptionsWhere<T>,
    limit: number | 'all' = 500,
    offset: number | 'all' = 0,
    sort?: { [P in keyof T]?: "ASC" | "DESC" }, // Sort parameter
    search?: { [P in keyof T]?: string }, // Search parameter
    relations?: string[],
    relationSelect?: { [relation: string]: (keyof T)[] }
  ): Promise<T[]> {
    try {
      this.logger.log(commonDataServiceMessages.FETCHING_DATA, {
        where,
        limit,
        offset,
        sort,
        search,
      });
      
      const options: FindManyOptions<T> = {};

      // Handle pagination - only apply if not 'all'
      if (limit !== 'all' && typeof limit === 'number') {
        options.take = limit;
      }
      
      if (offset !== 'all' && typeof offset === 'number') {
        options.skip = offset;
      }

      // Handle select fields using TypeORM's preferred format
      if (select && select.length > 0) {
        // Create a select object in the format TypeORM expects
        const selectObj = select.reduce(
          (obj, field) => {
            obj[field] = true;
            return obj;
          },
          {} as Record<keyof T, boolean>
        );

        options.select = selectObj as any;
      }

      // Handle relations
      if (relations) {
        options.relations = relations; // Apply relations dynamically
      }

      if (relationSelect) {
        options.select = options.select || {};
        for (const relation in relationSelect) {
          if (relationSelect[relation]) {
            options.select[relation] = relationSelect[relation].reduce(
              (acc, field) => {
                acc[field] = true;
                return acc;
              },
              {} as Record<keyof T, boolean>
            );
          }
        }
      }

      if (where) {
        options.where = where;
      }

      if (sort) {
        options.order = sort as FindOptionsOrder<T>; // Add sorting
      }
      // To make nulls last, we need to set the 'nulls' property in the order object.
      if (options.order) {
        for (const key in options.order) {
          const v = options.order[key] as any;
          if (typeof v === 'string') {
            // e.g., "ASC" | "DESC"
            options.order[key] = { direction: v, nulls: 'last' } as any;
          } else if (v && typeof v === 'object') {
            // preserve existing object, just force nulls last
            options.order[key] = { ...v, nulls: 'last' } as any;
          }
        }
      }
      if (search) {
        // Create a new where object that combines existing conditions
        const searchConditions = Object.entries(search).reduce(
          (acc, [key, value]) => {
            if (value) {
              // Only add non-empty search values
              (acc as any)[key] = ILike(`%${value}%`);
            }
            return acc;
          },
          {} as FindOptionsWhere<T>
        );

        // Merge with existing where conditions if they exist
        options.where = options.where
          ? { ...options.where, ...searchConditions }
          : searchConditions;
      }

      this.logger.log(commonDataServiceMessages.FINAL_QUERY_OPTIONS, options);
      const results = await repository.find(options);
      this.logger.log(commonDataServiceMessages.DATA_FETCHED_SUCCESS(results.length));
      return results;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMON_DATA_SERVICE_GET_FAILED, error);
    }
  }

    /**
   * Fetches data from the database with optional filtering, sorting, and pagination.
   * @param repository - The repository to query.
   * @param select - Fields to select.
   * @param where - Conditions for filtering.
   * @param limit - Number of records to fetch or 'all' for unlimited.
   * @param offset - Offset for pagination or 'all' for no offset.
   * @param sort - Sorting options.
   * @param search - Search parameters.
   * @param relations - Relations to include in the result.
   * @param relationSelect - Specific fields to select from relations.
   * @returns The fetched data.
   */
  async getWithRelFields<T extends ObjectLiteral>(
    repository: Repository<T>,
    select?: (keyof T | string)[] | FindOptionsSelect<T>,
    where?: FindOptionsWhere<T>,
    limit: number | 'all' = 500,
    offset: number | 'all' = 0,
    sort?: Record<string, any>, // Sort parameter now supports nested sorting
    search?: { [P in keyof T]?: string }, // Search parameter
    relations?: string[],
    relationSelect?: Record<string, (keyof any | string)[]>
  ): Promise<T[]> {
    try {
      this.logger.log(commonDataServiceMessages.FETCHING_DATA, {
        where,
        limit,
        offset,
        sort,
        search,
      });
      
      const options: FindManyOptions<T> = {};

      // Handle pagination - only apply if not 'all'
      if (limit !== 'all' && typeof limit === 'number') {
        options.take = limit;
      }
      
      if (offset !== 'all' && typeof offset === 'number') {
        options.skip = offset;
      }

      // Handle select fields using TypeORM's preferred format
      let selectContainer: Record<string, any> | undefined;
      if (Array.isArray(select) && select.length > 0) {
        // Create a select object in the format TypeORM expects
        selectContainer = select.reduce(
          (obj, field) => {
            obj[field as string] = true;
            return obj;
          },
          {} as Record<string, boolean>
        );

        options.select = selectContainer as any;
      } else if (select) {
        selectContainer = { ...(select as Record<string, any>) };
      }

      // Handle relations
      if (relations) {
        options.relations = relations; // Apply relations dynamically
      }

      if (relationSelect) {
        selectContainer = selectContainer || {};
        for (const relation in relationSelect) {
          const fields = relationSelect[relation];
          if (fields && fields.length > 0) {
            // Handle nested relations properly (e.g., 'user.profileExtension')
            const relationParts = relation.split('.');
            let currentContainer = selectContainer;
            
            // Navigate through nested structure
            for (let i = 0; i < relationParts.length - 1; i++) {
              const part = relationParts[i];
              if (!currentContainer[part]) {
                currentContainer[part] = {};
              }
              currentContainer = currentContainer[part];
            }
            
            // Set the fields for the final relation
            const finalRelation = relationParts[relationParts.length - 1];
            currentContainer[finalRelation] = fields.reduce(
              (acc, field) => {
                acc[field as string] = true;
                return acc;
              },
              {} as Record<string, boolean>
            );
          }
        }
      }

      if (selectContainer) {
        options.select = selectContainer as FindOptionsSelect<T>;
      }

      if (where) {
        options.where = where;
      }

      if (sort) {
        // Simplified nested sorting without complex nulls handling to avoid TypeORM issues
        const setNestedOrder = (
          target: Record<string, any>,
          path: string,
          direction: string
        ) => {
          const parts = path.split('.');
          let current = target;
          for (let index = 0; index < parts.length - 1; index++) {
            const part = parts[index];
            if (!current[part]) {
              current[part] = {};
            }
            current = current[part];
          }
          const finalKey = parts[parts.length - 1];
          current[finalKey] = direction; // Simple string direction
        };

        const order: Record<string, any> = {};
        for (const [key, value] of Object.entries(sort)) {
          if (key.includes('.') && typeof value === 'string') {
            setNestedOrder(order, key, value);
          } else {
            order[key] = value; // Simple assignment for non-nested fields
          }
        }

        if (Object.keys(order).length > 0) {
          options.order = order as FindOptionsOrder<T>;
        }
      }
      if (search) {
        // Create a new where object that combines existing conditions
        const searchConditions = Object.entries(search).reduce(
          (acc, [key, value]) => {
            if (value) {
              // Only add non-empty search values
              (acc as any)[key] = ILike(`%${value}%`);
            }
            return acc;
          },
          {} as FindOptionsWhere<T>
        );

        // Merge with existing where conditions if they exist
        options.where = options.where
          ? { ...options.where, ...searchConditions }
          : searchConditions;
      }

      this.logger.log(commonDataServiceMessages.FINAL_QUERY_OPTIONS, options);
      const results = await repository.find(options);
      this.logger.log(commonDataServiceMessages.DATA_FETCHED_SUCCESS(results.length));
      return results;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMON_DATA_SERVICE_GET_FAILED, error);
    }
  }

  /**
   * Finds an entity by its ID.
   * @param repository - The repository to query.
   * @param id - The ID of the entity.
   * @param hasSoftDelete - Whether to include soft-deleted entities.
   * @param relations - Optional relations to include.
   * @returns The entity if found, or null.
   */
  async findOneById<T extends ObjectLiteral>(
    repository: Repository<T>,
    id: number,
    hasSoftDelete: boolean = true,
    relations?: string[],
    order?: { [key: string]: { [key: string]: "ASC" | "DESC" } }
  ): Promise<T | null> {
    try {
      this.logger.log(commonDataServiceMessages.FINDING_ENTITY_BY_ID(id));
      const whereCondition: any = { id };

      if (hasSoftDelete) {
        whereCondition.deletedAt = IsNull();
      }

      const result = await repository.findOne({
        where: whereCondition,
        relations,
        order: order as FindOptionsOrder<T>
      });
      this.logger.log(commonDataServiceMessages.ENTITY_FOUND_SUCCESS(id));
      return result;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.COMMON_DATA_SERVICE_FINDBYID_FAILED, error);
    }
  }

  /**
   * Saves an entity to the database.
   * @param repository - The repository to save to.
   * @param entity - The entity to save.
   * @returns The saved entity.
   */
  async save<T extends ObjectLiteral>(
    repository: Repository<T>,
    entity: T
  ): Promise<T> {
    try {
      this.logger.log(commonDataServiceMessages.SAVING_ENTITY, { entity });
      const savedEntity = await repository.save(entity);
      this.logger.log(commonDataServiceMessages.ENTITY_SAVED_SUCCESS, { savedEntity });
      return savedEntity;
    } catch (error) {
      if (error instanceof QueryFailedError) {
        throw new InifniDatabaseException(ERROR_CODES.COMMON_DATA_SERVICE_SAVE_FAILED,null,null,ERROR_MESSAGES.CDS_SAVE_FAILED);
      }
      handleKnownErrors(ERROR_CODES.COMMON_DATA_SERVICE_SAVE_FAILED, error);
    }
  }

  /**
   * Finds entities based on specific conditions.
   * @param repository - The repository to query.
   * @param where - Conditions for filtering.
   * @param relations - Optional relations to include.
   * @param sort - Sorting options.
   * @returns The list of entities matching the conditions.
   */
  async findByData<T extends ObjectLiteral>(
    repository: Repository<T>,
    where: FindOptionsWhere<T>,
    relations?: string[],
    sort?: { [P in keyof T]?: "ASC" | "DESC" }
  ): Promise<T[]> {
    try {
      this.logger.log(commonDataServiceMessages.FINDING_ENTITIES_BY_CRITERIA, {
        where,
        relations,
        sort,
      });
      const options: FindManyOptions<T> = {
        where,
      };

      if (relations) {
        options.relations = relations;
      }

      if (sort) {
        options.order = sort as FindOptionsOrder<T>;
      }

      const results = await repository.find({
        where,
        relations,
        order: sort as FindOptionsOrder<T>,
      });
      this.logger.log(commonDataServiceMessages.ENTITIES_FOUND_SUCCESS(results.length));
      return results;
    } catch (error) {
      handleKnownErrors(
        ERROR_CODES.COMMON_DATA_SERVICE_FINDBYDATA_FAILED,
        error
      );
    }
  }
}
