import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  CreateDateColumn,
  UpdateDateColumn,
  ManyToOne,
  JoinColumn,
  DeleteDateColumn,
} from 'typeorm';
import { TravelTypeEnum } from '../enum/travel-type.enum';
import { User, ProgramRegistration } from '../entities/index';
import { TravelStatusEnum } from '../enum/travel-status.enum';
import { Auditable } from 'src/audit-history/decorators/auditable.decorator';
import { SkipAudit } from 'src/audit-history/decorators/skip-audit.decorator';


@Entity('hdb_registration_travel_plan')
@Auditable()
export class RegistrationTravelPlan {
  @PrimaryGeneratedColumn('increment', { type: 'bigint' })
  id: number;

  @Column({ name: 'registration_id', type: 'bigint' })
  registrationId: number;

  @Column({ name: 'travel_type', type: 'enum', enum: TravelTypeEnum })
  travelType: TravelTypeEnum | null;

  @Column({ name: 'return_travel_type', type: 'enum', enum: TravelTypeEnum, nullable: true })
  returnTravelType: TravelTypeEnum | null;

  @Column({ name: 'airline_name', type: 'varchar', length: 100, nullable: true })
  airlineName: string;

  @Column({ name: 'flight_number', type: 'varchar', length: 20, nullable: true })
  flightNumber: string;

  @Column({ name: 'arrival_datetime', type: 'timestamp', nullable: true })
  arrivalDatetime: Date;

  @Column({ name: 'departure_location', type: 'varchar', length: 255, nullable: true })
  departureLocation: string;

  @Column({ name: 'departure_datetime', type: 'timestamptz', nullable: true })
  departureDatetime: Date;

  @Column({ name: 'departure_flight_number', type: 'varchar', length: 20, nullable: true })
  departureFlightNumber: string;

  @Column({ name: 'departure_airline_name', type: 'varchar', length: 100, nullable: true })
  departureAirlineName: string;

  @Column({ name: 'arrival_from', type: 'varchar', length: 255, nullable: true })
  arrivalFrom: string;

  @Column({ name: 'departure_to', type: 'varchar', length: 255, nullable: true })
  departureTo: string;

  @Column({ name: 'other_departure', type: 'varchar', length: 100, nullable: true })
  otherDeparture: string;

  @Column({ name: 'other_arrival', type: 'varchar', length: 100, nullable: true })
  otherArrival: string;

  @Column({ name: 'pickup_time', type: 'time', nullable: true })
  pickupTime: string;

  @Column({ name: 'pickup_location', type: 'varchar', length: 255, nullable: true })
  pickupLocation: string;

  @Column({ name: 'checkin_at', type: 'timestamptz', nullable: true })
  checkinAt: Date;

  @Column({ name: 'checkin_time', type: 'time', nullable: true })
  checkinTime: string;

  @Column({ name: 'checkin_location', type: 'varchar', length: 255, nullable: true })
  checkinLocation: string;

  @Column({
    name: 'travel_plan_status',
    type: 'enum',
    enum: TravelStatusEnum,
    default: TravelStatusEnum.PENDING,
  })
  travelPlanStatus: TravelStatusEnum;

  @Column({ name: 'onward_additional_info', type: 'text', nullable: true })
  onwardAdditionalInfo: string;

  @Column({ name: 'return_additional_info', type: 'text', nullable: true })
  returnAdditionalInfo: string;

  @Column({ name: 'arrival_other_airline_name', type: 'varchar', length: 100, nullable: true })
  arrivalOtherAirlineName: string;

  @Column({ name: 'departure_other_airline_name', type: 'varchar', length: 100, nullable: true })
  departureOtherAirlineName: string;

  @Column({ name: 'onward_terminal', type: 'varchar', length: 100, nullable: true })
  onwardTerminal: string;

  @Column({ name: 'return_terminal', type: 'varchar', length: 100, nullable: true })
  returnTerminal: string;

  @Column({ name: 'onward_journey_ticket_url', type: 'text', nullable: true })
  onwardJourneyTicketUrl: string;

  @Column({ name: 'onward_journey_ticket_signed_url', type: 'text', nullable: true })
  onwardJourneyTicketSignedUrl: string;

  @Column({ name: 'return_journey_ticket_url', type: 'text', nullable: true })
  returnJourneyTicketUrl: string;

  @Column({ name: 'return_journey_ticket_signed_url', type: 'text', nullable: true })
  returnJourneyTicketSignedUrl: string;

  @Column({ name: 'audit_ref_id', type: 'bigint', nullable: true })
  auditRefId: number;

  @Column({ name: 'parent_ref_id', type: 'bigint', nullable: true })
  parentRefId: number;

  @SkipAudit()
  @CreateDateColumn({ name: 'created_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP' })
  createdAt: Date;

  @SkipAudit()
  @UpdateDateColumn({ name: 'updated_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP', onUpdate: 'CURRENT_TIMESTAMP' })
  updatedAt: Date;

  @DeleteDateColumn({ name: 'deleted_at', type: 'timestamptz', nullable: true, default: null })
  deletedAt: Date;
    
  @SkipAudit()
  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'created_by' })
  createdBy: User;

  @SkipAudit()
  @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
  @JoinColumn({ name: 'updated_by' })
  updatedBy: User;

  @SkipAudit()
  @ManyToOne(() => ProgramRegistration, { nullable: false, onDelete: 'CASCADE' })
  @JoinColumn({ name: 'registration_id' })
  registration: ProgramRegistration;

  constructor(partial: Partial<RegistrationTravelPlan>) {
    Object.assign(this, partial);
  }
}