import {
    Entity,
    PrimaryGeneratedColumn,
    Column,
    CreateDateColumn,
    UpdateDateColumn,
    ManyToOne,
    JoinColumn,
    DeleteDateColumn,
    OneToMany,
  } from 'typeorm';
import { ApprovalStatusEnum } from '../enum/approval-status.enum';
import { ProgramRegistration, RegistrationApprovalTrack, User } from '../entities/index'
import { Auditable } from 'src/audit-history/decorators/auditable.decorator';
import { SkipAudit } from 'src/audit-history/decorators/skip-audit.decorator';
  
  @Entity('hdb_registration_approval')
  @Auditable()
  export class RegistrationApproval {
    @PrimaryGeneratedColumn('increment', { type: 'bigint' })
    id: number;
  
    @Column({ name: 'registration_id', type: 'bigint' })
    registrationId: number;
  
    @Column({ name: 'approval_status', type: 'enum', enum: ApprovalStatusEnum, default: ApprovalStatusEnum.PENDING })
    approvalStatus: ApprovalStatusEnum;
  
    @Column({ name: 'approval_date', type: 'timestamptz', nullable: true })
    approvalDate: Date;
  
    @Column({ name: 'approved_by', type: 'bigint', nullable: true })
    approvedBy: number;
  
    @Column({ name: 'rejection_reason', type: 'text', nullable: true })
    rejectionReason: string;
  
    @Column({ name: 'reviewer_comments', type: 'text', nullable: true })
    reviewerComments: string;
  
    @Column({ name: 'auto_approved', type: 'boolean', default: false })
    autoApproved: boolean;

    @Column({ name: 'audit_ref_id', type: 'bigint', nullable: true })
    auditRefId: number;

    @Column({ name: 'parent_ref_id', type: 'bigint', nullable: true })
    parentRefId: number;
  
    @SkipAudit()
    @CreateDateColumn({ name: 'created_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP' })
      createdAt: Date;
    
    @SkipAudit()
    @UpdateDateColumn({ name: 'updated_at', type: 'timestamptz', default: () => 'CURRENT_TIMESTAMP', onUpdate: 'CURRENT_TIMESTAMP' })
    updatedAt: Date;
  
    @DeleteDateColumn({ name: 'deleted_at', type: 'timestamptz', nullable: true, default: null })
    deletedAt: Date;
    
    @SkipAudit()
    @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
    @JoinColumn({ name: 'created_by' })
    createdBy: User;
  
    @SkipAudit()
    @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
    @JoinColumn({ name: 'updated_by' })
    updatedBy: User;

    @SkipAudit()
    @ManyToOne(() => ProgramRegistration, { nullable: false, onDelete: 'CASCADE' })
    @JoinColumn({ name: 'registration_id' })
    registration: ProgramRegistration;
  
    @SkipAudit()
    @ManyToOne(() => User, { nullable: true, onDelete: 'SET NULL' })
    @JoinColumn({ name: 'approved_by' })
    approvedByUser: User;

    @SkipAudit()
    @OneToMany(() => RegistrationApprovalTrack, (track) => track.approval)
    tracks: RegistrationApprovalTrack[];
  
    constructor(partial: Partial<RegistrationApproval>) {
      Object.assign(this, partial);
    }
  }