//Environment
export const NODE_ENVIRONMENT = process.env.NODE_ENV || 'development';

//Swagger string constants
export const SWAGGER = {
  API_DOCS: 'api-docs',
  TITLE: 'Infinitheism Backend',
  DESCRIPTION: 'Infinitheism Backend Documentation',
  VERSION: '1.0',
  PATH: 'api-docs',
  ENVIRONMENT: `${NODE_ENVIRONMENT} environment`,
  BEARERAUTH: {
    TYPE: 'http',
    NAME: 'Authorization',
    DESCRIPTION: 'Enter access token',
    IN: 'header',
  },
  USER_ID_AUTH: {
    TYPE: 'apiKey',
    NAME: 'userid',
    DESCRIPTION: 'Enter user ID',
    IN: 'header',
  },
  ACTIVE_ROLE_AUTH: {
    TYPE: 'apiKey',
    NAME: 'activerole',
    DESCRIPTION: 'Enter active role key (optional) - switches user to specific role context. Valid values: ROLE_VIEWER, ROLE_ADMIN, ROLE_MAHATRIA, ROLE_RM, ROLE_FINANCE_MANAGER, ROLE_RELATIONAL_MANAGER, ROLE_SHOBA, ROLE_OPERATIONAL_MANAGER, ROLE_RM_SUPPORT',
    IN: 'header',
  },
};

export const ROLE_GUARD_STRINGS = {
  MISSING_ROLES: 'Roles are missing in the request.',
  UNAUTHORIZED: 'User does not have the required roles to access this resource.',
};

export const ROLE_KEYS = {
  VIEWER: 'ROLE_VIEWER',
  ADMIN: 'ROLE_ADMIN',
  MAHATRIA: 'ROLE_MAHATRIA',
  RM: 'ROLE_RM',
  FINANCE_MANAGER: 'ROLE_FINANCE_MANAGER',
  RELATIONAL_MANAGER: 'ROLE_RELATIONAL_MANAGER',
  SHOBA: 'ROLE_SHOBA',
  OPERATIONAL_MANAGER: 'ROLE_OPERATIONAL_MANAGER',
  RM_SUPPORT: 'ROLE_RM_SUPPORT',
};

export const ROLE_VALUES = {
  VIEWER: 'viewer',
  ADMIN: 'admin',
  MAHATRIA: 'mahatria',
  RM: 'relational_manager',
  FINANCE_MANAGER: 'finance_manager',
  RELATIONAL_MANAGER: 'relational_manager',
  COORDINATOR: 'shoba',
  OPERATIONAL_MANAGER: 'operational_manger',
  RM_SUPPORT: 'rm_support',
};

export const ENVIRONMENTS = {
  DEVELOPMENT: 'development',
  QA: 'qa',
  STAGING: 'staging',
  PRODUCTION: 'production',
};

export const optionCategoryMessages = {
  CREATE_NEW_CATEGORY_OPTION: 'Create a new option category',
  GET_ALL_OPTION_CATEGORIES_SEARCH: 'Get all option categories with pagination and search',
  GET_OPTION_CATEGORY_BY_ID: 'Get a category by ID',
  GET_CATEGORY_OPTIONS: 'Get options by category ID with pagination and search',
  NO_OPTIONS_FOUND_CATEGORY: 'No options found for this category',
  CATEGORY_CREATED: 'Category created successfully',
  CATEGORY_UPDATED: 'Category updated successfully',
  CATEGORY_DELETED: 'Category deleted successfully',
  CATEGORY_RETRIEVED: 'Category retrieved successfully',
  CATEGORIES_RETRIEVED: 'Categories retrieved successfully',
  OPTIONS_RETRIEVED: 'Options retrieved successfully',
  DELETE_CATEGORY_ID: 'Delete category by ID',
  UPDATE_CATEGORY_ID: 'Update category by ID',
  CATEGORY_NOT_FOUND: 'Category not found',
  USER_NOT_FOUND: 'User not found',
  DUPLICATE_CATEGORY: 'Category with this name already exists',
  FAILED_TO_CREATE_CATEGORY: 'Failed to create category',
  FAILED_TO_UPDATE_CATEGORY: 'Failed to update category',
  FAILED_TO_DELETE_CATEGORY: 'Failed to delete category',
  FAILED_TO_FETCH_DATA: 'Failed to fetch data',
  ERROR_FIND_ALL: 'Error in findAll:',
  ERROR_FIND_CATEGORY_BY_ID: 'Error in findByCategoryWiseOptionsById:',
  CREATE_CATEGORY_REQUEST_RECEIVED: 'Create category request received',
  FIND_ALL_CATEGORIES_REQUEST_RECEIVED: 'Find all categories request received',
  FIND_ONE_CATEGORY_REQUEST_RECEIVED: 'Find one category request received',
  UPDATE_CATEGORY_REQUEST_RECEIVED: 'Update category request received',
  DELETE_CATEGORY_REQUEST_RECEIVED: 'Delete category request received',
  CATEGORY_NOT_FOUND_ID: (id: number) => `Option category with ID ${id} not found.`,
  FAILED_TO_REMOVE_CATEGORY: 'Failed to remove option category.',
  CREATING_CATEGORY: (data: any) => `Creating option category with data: ${JSON.stringify(data)}`,
  FINDING_ALL_CATEGORIES: (limit: number, offset: number, searchText: string) =>
    `Finding all option categories with limit: ${limit}, offset: ${offset}, searchText: ${searchText}`,
  FINDING_OPTIONS_BY_CATEGORY: (categoryId: number, searchText: string) =>
    `Finding options for category ID: ${categoryId} with search text: ${searchText}`,
  FINDING_CATEGORY_BY_ID: (id: number) => `Finding option category by ID: ${id}`,
  FAILED_FINDING_CATEGORY_BY_ID: (id: number) => `Failed to find option category by ID: ${id}`,
  UPDATING_CATEGORY: (id: number) => `Updating option category with ID: ${id}`,
  FAILED_UPDATING_CATEGORY: (id: number) => `Failed to update option category with ID: ${id}`,
  REMOVING_CATEGORY: (id: number) => `Removing option category with ID: ${id}`,
  ERROR_CREATING_CATEGORY: 'Error creating option category',
  SOFT_DELETING_CATEGORY: (id: number) => `Soft deleting option category with ID: ${id}`,
  SOFT_DELETED_CATEGORY: (id: number) => `Soft deleted option category with ID: ${id}`,
};

export const optionConstMessages = {
  CREATE_NEW_OPTION: 'Create a new option',
  GET_ALL_OPTIONS_SEARCH: 'Get all options with pagination and search',
  GET_OPTION_BY_ID: 'Get option by ID',
  NO_OPTIONS_FOUND: 'No options found for this category',
  OPTION_CREATED: 'Option created successfully',
  OPTION_UPDATED: 'Option updated successfully',
  OPTION_DELETED: 'Option deleted successfully',
  OPTION_RETRIEVED: 'Option retrieved successfully',
  OPTIONS_RETRIEVED: 'Options retrieved successfully',
  DELETE_OPTION_ID: 'Delete option by ID',
  UPDATE_OPTION_ID: 'Update option by ID',
  OPTION_NOT_FOUND: 'Option not found',
  USER_NOT_FOUND: 'User not found',
  DUPLICATE_OPTION: 'Option with this name already exists',
  FAILED_TO_CREATE_OPTION: 'Failed to create option',
  FAILED_TO_UPDATE_OPTION: 'Failed to update option',
  FAILED_TO_DELETE_OPTION: 'Failed to delete option',
  FAILED_TO_FETCH_DATA: 'Failed to fetch data',
  ERROR_FIND_ALL: 'Error in findAll:',
  ERROR_FIND_OPTION_BY_ID: 'Error in findById:',
  CATERGORY_NOT_FOUND: 'Category not found',
  BAD_REQUEST: 'Bad request',
  NOT_FOUND: 'Not found',
  OPTION_ID: 'Option ID',
  INTERNAL_SERVER_ERROR: 'Internal server error',
  CREATING_OPTION: 'Creating Option',
  FINDING_ALL_OPTIONS: 'Finding all options',
  FINDING_OPTION_BY_ID: 'Finding option by ID',
  FINDING_OPTIONS_BY_CATEGORY: 'Finding options by category',
  UPDATING_OPTION: 'Updating option:',
  DELETING_OPTION_BY_ID: 'Deleting option with ID:',
  CATEGORY_MUST_BE_ARRAY: 'Category must be an array',
  VALIDATION_ERROR_BAD_REQ: 'Validation error or option already exists',
  CREATE_OPTION_REQUEST_RECEIVED: 'Create option request received',
  FIND_ALL_OPTIONS_REQUEST_RECEIVED: 'Find all options request received',
  FIND_ONE_OPTION_REQUEST_RECEIVED: 'Find one option request received',
  UPDATE_OPTION_REQUEST_RECEIVED: 'Update option request received',
  DELETE_OPTION_REQUEST_RECEIVED: 'Delete option request received',
  OPTION_NOT_FOUND_ID: (id: number) => `Option with ID ${id} not found.`,
};

export const paginationConstants = {
  LIMIT: 'Number of records per page (default: 10)',
  OFFSET: 'Offset for pagination (default: 0)',
  SEARCH_TEXT: 'Search text to filter options (default: empty string)',
  CATEGORY_ID: 'Category ID to filter options (default: [])',
  FILTERS: 'Additional filters (default: empty string)',
  SORT: 'Sort options (default: empty string)',
};

export const roomInventoryMessages = {
  GET_ALL_ROOM_INVENTORY: 'Get paginated room inventory with KPIs, filters, and available reports',
  UPDATE_ROOM_INVENTORY: 'Update room inventory details',
  GET_ROOM_INVENTORY_FILTER_CONFIG: 'Get room inventory filter configuration',
  DOWNLOAD_ROOM_INVENTORY_REPORT: 'Download room inventory reports',
  ROOM_INVENTORY_RETRIEVED: 'Room inventory data retrieved successfully',
  ROOM_INVENTORY_UPDATED: 'Room inventory updated successfully',
  ROOM_INVENTORY_FILTER_CONFIG_RETRIEVED: 'Room inventory filter configuration retrieved successfully',
  ROOM_INVENTORY_NOT_FOUND: 'Room inventory not found',
  ROOM_INVENTORY_UPDATE_REQUEST_RECEIVED: 'Room inventory update request received',
  ROOM_INVENTORY_LIST_REQUEST_RECEIVED: 'Room inventory list request received',
  ROOM_INVENTORY_FILTER_CONFIG_REQUEST_RECEIVED: 'Room inventory filter configuration request received',
  ROOM_INVENTORY_DOWNLOAD_REQUEST_RECEIVED: 'Room inventory download/report request received',
  ROOM_INVENTORY_WITH_KPIS_REQUEST_RECEIVED: 'Room inventory with KPIs and filters request received',
  ROOM_INVENTORY_DATA_PROCESSED: 'Room inventory data processed successfully',
  ROOM_INVENTORY_KPIS_CALCULATED: 'Room inventory KPIs calculated successfully',
  ROOM_INVENTORY_FILTER_CONFIG_GENERATED: 'Room inventory filter configuration generated successfully',
  CALCULATING_ROOM_INVENTORY_KPIS: 'Calculating room inventory KPIs',
  FAILED_TO_UPDATE_ROOM_INVENTORY: 'Failed to update room inventory',
  FAILED_TO_RETRIEVE_ROOM_INVENTORY: 'Failed to retrieve room inventory data',
  FAILED_TO_RETRIEVE_FILTER_CONFIG: 'Failed to retrieve room inventory filter configuration',
  FAILED_TO_GENERATE_REPORT: 'Failed to generate room inventory download/report',
  INVALID_LIMIT_PARAMETER: 'Limit must be between 1 and 100',
  INVALID_OFFSET_PARAMETER: 'Offset must be greater than or equal to 0',
  INVALID_FILTER_FORMAT: 'Invalid filter format - must be valid URL-encoded JSON',
  DOWNLOAD_TYPE_REQUIRED: 'downloadType parameter is required',
  REPORT_CODE_REQUIRED: 'reportCode parameter is required',
  INVALID_DOWNLOAD_TYPE: (types: string[]) => `Invalid downloadType. Must be one of: ${types.join(', ')}`,
  INVALID_REPORT_CODE: (codes: string[]) => `Invalid reportCode. Must be one of: ${codes.join(', ')}`,
  ACCESS_DENIED_TO_REPORT: 'You do not have permission to access this report',
  REPORT_GENERATED_SUCCESSFULLY: 'Report generated successfully',
  FILTER_DECODED_SUCCESSFULLY: 'Filter decoded and parsed successfully',
  FAILED_TO_DECODE_FILTER: 'Failed to decode or parse filter',
  FILTER_IDS_CLEANED: 'Filter IDs cleaned and validated',
  ERROR_IN_ROOM_INVENTORY_RETRIEVAL: 'Error in room inventory retrieval with KPIs service',
  ERROR_CALCULATING_KPIS: 'Error calculating room inventory KPIs',
  ERROR_GENERATING_FILTER_CONFIG: 'Error generating room inventory filter configuration',
  ROOM_STATUS_INVALID_FOR_OCCUPANCY: (requestedStatus: string, remainingOccupancy: number, totalCapacity: number) =>
    `Status ${requestedStatus} is invalid: remainingOccupancy=${remainingOccupancy}, totalCapacity=${totalCapacity}`,
  ROOM_INVENTORY_RECORD_NOT_FOUND: (id: number) => `Room inventory with ID ${id} not found`,
  ROOM_CANNOT_SET_RESERVED_WITH_ALLOCATIONS: 'Cannot set room to reserved status when allocations exist',
  ROOM_HAS_ACTIVE_ALLOCATIONS: (count: number) => `Room has ${count} active allocation(s). Remove allocations before setting to reserved.`,
  REMAINING_OCCUPANCY_CANNOT_BE_NEGATIVE: 'Remaining occupancy cannot be negative',
  REMAINING_OCCUPANCY_EXCEEDS_CAPACITY: 'Remaining occupancy cannot exceed total capacity',
  ROOM_INVENTORY_UPDATED_SUCCESSFULLY: 'Room inventory updated successfully',
  ERROR_UPDATING_ROOM_INVENTORY: 'Error updating room inventory',
  UNKNOWN_FILTER_KEY_IN_CONTROLLER: 'Unknown filter key found in controller filters',
  UNKNOWN_FILTER_KEY_IN_CONFIG: 'Unknown filter key found in configuration',
  ERROR_GENERATING_REPORT: (reportCode: string) => `Error generating report ${reportCode}:`,
  ROOM_INVENTORY_MAPPING_VALIDATION_RECEIVED: 'Room inventory mapping validation request received',
  ROOM_INVENTORY_NOT_FOUND_FOR_SUBPROGRAM: 'Room inventory not found for sub-program',
  ROOM_INVENTORY_NOT_FOUND_FOR_PROGRAM: 'Room inventory not found for program',
  ROOM_INVENTORY_MAPPING_VALIDATED: 'Room inventory mapping validated successfully',
  ERROR_IN_MAPPING_VALIDATION: 'Error in room inventory mapping validation',
  ROOM_CAPACITY_CHECK_RECEIVED: 'Room capacity check request received',
  ROOM_NOT_FOUND_FOR_CAPACITY_CHECK: 'Room inventory not found for capacity check',
  ROOM_CAPACITY_CHECK_COMPLETED: 'Room capacity check completed',
  ERROR_CHECKING_ROOM_CAPACITY: 'Error checking room capacity',
  FINDING_ROOM_BY_ID_AND_PROGRAM: 'Finding room inventory by ID and program ID',
  FINDING_ROOM_BY_ID_AND_SUBPROGRAM: 'Finding room inventory by ID and sub-program ID',
  FINDING_ROOM_BY_ID_WITH_RELATIONS: 'Finding room inventory by ID with relations',
  FINDING_ROOM_BY_ID: 'Finding room inventory by ID',
  UPDATING_REMAINING_OCCUPANCY: 'Updating remaining occupancy',
  SAVING_ROOM_INVENTORY_MAPPING: 'Saving room inventory mapping',
  FINDING_ROOM_WITH_ALLOCATIONS: 'Finding room inventory with allocations',
  INVALID_LIMIT_PARAMETER_WARN: 'Invalid limit parameter provided',
  INVALID_OFFSET_PARAMETER_WARN: 'Invalid offset parameter provided',
  ROOM_INVENTORY_DATA_RETRIEVED: 'Room inventory data retrieved successfully',
  REPORT_GENERATED: 'Report generated successfully',
  ROOM_INVENTORY_FILTER_CONFIG_RETRIEVED_LOG: 'Room inventory filter configuration retrieved successfully',
  ROOM_OCCUPANCY_UPDATE_RECEIVED: 'Room occupancy update request received',
  ROOM_OCCUPANCY_UPDATED: 'Room occupancy updated successfully',
  ERROR_UPDATING_ROOM_OCCUPANCY: 'Error updating room occupancy',
  ROOM_OCCUPANCY_DEALLOCATION_RECEIVED: 'Room occupancy deallocation update request received',
  ROOM_OCCUPANCY_UPDATED_AFTER_DEALLOCATION: 'Room occupancy updated after deallocation successfully',
  ERROR_UPDATING_OCCUPANCY_AFTER_DEALLOCATION: 'Error updating room occupancy after deallocation',
  ROOM_OCCUPANCY_DETAILS_RECEIVED: 'Room occupancy details request received',
  ROOM_OCCUPANCY_DETAILS_RETRIEVED: 'Room occupancy details retrieved successfully',
  ERROR_RETRIEVING_OCCUPANCY_DETAILS: 'Error retrieving room occupancy details',
  ROOM_STATUS_RECALCULATION_RECEIVED: 'Room status recalculation request received',
  ROOM_STATUS_RECALCULATED: 'Room status recalculated successfully',
  ERROR_RECALCULATING_ROOM_STATUS: 'Error recalculating room status',
  INVALID_REPORT_CODE_MESSAGE: (reportCode: string) => `Invalid report code: ${reportCode}`,
  UNSUPPORTED_REPORT_CODE: (reportCode: string) => `Unsupported report code: ${reportCode}`,
  NO_DATA_FOR_REPORT: (reportLabel: string) => `No data found for ${reportLabel} with the current filters.`,
  LABEL_ALL: 'All',
  LABEL_OTHER: 'Other',
  LABEL_UNNAMED_RM: 'Unnamed RM',
  LABEL_YES: 'Yes',
  LABEL_NO: 'No',
  COLUMN_VENUE: 'Venue',
  COLUMN_BLOCK: 'Block',
  COLUMN_FLOOR: 'Floor',
  COLUMN_ROOM_NUMBER: 'Room Number',
  COLUMN_IS_RESERVED: 'Is Reserved',
  COLUMN_RESERVED_FOR: 'Reserved For',
  COLUMN_HDB_NAME: 'HDB Infinitheist Name',
  COLUMN_GENDER: 'Gender',
  COLUMN_MOBILE: 'Mobile',
  COLUMN_RM_CONTACT: 'RM Contact',
  COLUMN_CITY: 'City',
  COLUMN_EMAIL: 'Email',
  FILTER_KEY_VENUES: 'venues',
  FILTER_KEY_BLOCK: 'block',
  FILTER_KEY_FLOOR: 'floor',
  FILTER_KEY_AGE: 'age',
  FILTER_KEY_GENDER: 'gender',
  FILTER_KEY_DEPARTURE_DATETIME: 'departureDatetime',
  FILTER_KEY_NO_OF_HDBS: 'noOfHdbs',
  FILTER_KEY_PREFERRED_ROOMMATE: 'preferredRoommate',
  FILTER_KEY_RM_CONTACT: 'rmContact',
  FILTER_KEY_CITY: 'city',
  FILTER_KEY_PAIRED_STATUS: 'pairedStatus',
  FILTER_KEY_OCCUPANCY: 'occupancy',
  FILTER_KEY_RETURN_TRAVEL_TERMINAL: 'returnTravelTerminal',
  FILTER_PLACEHOLDER_OCCUPANCY: 'Enter room occupancy',
  FILTER_LABEL_OCCUPANCY: 'Occupancy',
  LOOKUP_CATEGORY_CITY_NAME: 'CITY_NAME',
  REPORT_CODE_ROOM_INVENTORY_ALLOCATION: 'room_inventory_allocation_report',
  MIME_TYPE_EXCEL: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
  PAGINATION_ALL: 'all',
  STATUS_KEY_ALLOCATED_SEEKERS: 'ALLOCATED_SEEKERS',
  STATUS_KEY_ALL: 'ALL',
};

export const questionConstMessages = {
  GET_QUESTION_BY_ID: 'Get a question by ID',
  GET_ALL_QUESTIONS: 'Get all questions with pagination and search',
  UPDATE_QUESTION_BY_ID: 'Update question by ID',
  DELETE_QUESTION_BY_ID: 'Delete question by ID',
  CREATE_NEW_QUESTION: 'Create a new question',
  QUESTION_CREATED: 'Question created successfully',
  QUESTION_UPDATED: 'Question updated successfully',
  QUESTION_DELETED: 'Question deleted successfully',
  QUESTION_RETRIEVED: 'Question retrieved successfully',
  QUESTIONS_RETRIEVED: 'Questions retrieved successfully',
  QUESTION_NOT_FOUND: 'Question not found',
  DUPLICATE_QUESTION: 'A question with the same label and type already exists',
  VALIDATION_ERROR_BAD_REQ: 'Validation failed. Please check the input data.',
  FAILED_TO_CREATE_QUESTION: 'Failed to create question. Ensure all data is valid.',
  FAILED_TO_UPDATE_QUESTION: 'Failed to update question. Check data integrity.',
  FAILED_TO_DELETE_QUESTION: 'Failed to delete question.',
  BAD_REQUEST: 'Bad request',
  NOT_FOUND: 'Not found',
  QUESTION_ID: 'Question ID',
  USER_NOT_FOUND: 'User not found',
  INVALID_FILTERS: 'Invalid filters provided',
  FILTER_QUESTIONS_FORM_SECTION_ID: 'Filter questions by Form Section ID',
  CREATE_QUESTION_REQUEST_RECEIVED: 'Create question request received',
  FIND_ALL_QUESTIONS_REQUEST_RECEIVED: 'Find all questions request received',
  FIND_ONE_QUESTION_REQUEST_RECEIVED: 'Find one question request received',
  UPDATE_QUESTION_REQUEST_RECEIVED: 'Update question request received',
  DELETE_QUESTION_REQUEST_RECEIVED: 'Delete question request received',
  QUESTION_NOT_FOUND_ID: (id: number) => `Question with ID ${id} not found.`,
  SOFT_DELETING_QUESTION: (id: number) => `Soft deleting question with ID: ${id}`,
  SOFT_DELETED_QUESTION: (id: number) => `Soft deleted question with ID: ${id}`,
  SOFT_DELETING_ASSOCIATED_RECORDS: (id: number) =>
    `Soft deleting associated question-option map records for question ID: ${id}`,
  SOFT_DELETED_ASSOCIATED_RECORDS: (id: number) =>
    `Soft deleted associated question-option map records for question ID: ${id}`,
  CREATING_QUESTION: (data: any) => `Creating question with data: ${JSON.stringify(data)}`,
  DUPLICATE_QUESTION_FOUND: (label: string) => `Duplicate question found: ${label}`,
  FINDING_ALL_QUESTIONS: (limit: number, offset: number, searchText: string) =>
    `Finding all questions with limit: ${limit}, offset: ${offset}, searchText: ${searchText}`,
  RETRIEVING_QUESTIONS_WHERE_CLAUSE: (whereClause: Record<string, any>) =>
    `Retrieving questions with where clause: ${JSON.stringify(whereClause)}`,
  FINDING_QUESTION_BY_ID: (id: number) => `Finding question by ID: ${id}`,
  UPDATING_QUESTION: (id: number) => `Updating question with ID: ${id}`,
  REMOVING_QUESTION: (id: number) => `Removing question with ID: ${id}`,
  FINDING_QUESTIONS_BY_PROGRAM_ID: (
    programId: number,
    limit: number,
    offset: number,
    searchText: string,
  ) =>
    `Finding questions by program ID: ${programId} with limit: ${limit}, offset: ${offset}, searchText: ${searchText}`,
};

export const userConstMessages = {
  USER_ID_NOT_FOUND: 'User ID not found in the request.',
  USER_NOT_FOUND_BY_FIELD: (field: string, value: any) => `User not found with ${field}: ${value}`,
  USER_FOUND_BY_FIELD: (field: string, value: any) => `User found with ${field}: ${value}`,
  USER_NOT_FOUND_ID: (id: number) => `User with ID ${id} not found.`,
  ROLE_NOT_FOUND: (role: string) => `Role ${role} not found.`,
  FAILED_TO_FETCH_USER: (field: string, value: any) =>
    `Failed to fetch user with ${field}: ${value}`,
  FETCHING_USER_BY_FIELD: (field: string, value: any) => `Fetching user by ${field}: ${value}`,
  FETCHING_USER_BY_CONCATED_FIELDS: (fields: string[], value: string) =>
    `Fetching user by concated fileds ${JSON.stringify(fields)} with ${value} `,
  FETCHING_ALL_USERS: (params: {
    limit: number;
    offset: number;
    searchText: string;
    filters: Record<string, any>;
  }) =>
    `Fetching all users with limit: ${params.limit}, offset: ${params.offset}, searchText: ${params.searchText}, filters: ${JSON.stringify(params.filters)}`,
  FETCHING_RM_LIST: (params: {
    limit: number;
    offset: number;
    searchText: string;
  }) =>
    `Fetching relational manager users with limit: ${params.limit}, offset: ${params.offset}, searchText: ${params.searchText}`,
  RM_LIST_RETRIEVED: 'Relational managers retrieved successfully',
  USER_CREATED: 'User created successfully',
  USER_CREATION_FAILED: 'User creation failed',
};

export const userProgramExperienceConstMessages = {
  CATEGORY: 'SEEKER_PROGRAM_EXPERIENCE',
  CREATE_OPERATION: 'Create seeker program experiences',
  UPDATE_OPERATION: 'Update seeker program experiences',
  EXPERIENCES_CREATED: 'Seeker program experiences recorded successfully',
  EXPERIENCES_UPDATED: 'Seeker program experiences updated successfully',
  CREATE_REQUEST_RECEIVED: (userId: number) =>
    `Create user program experiences request received for user ${userId}`,
  UPDATE_REQUEST_RECEIVED: (userId: number) =>
    `Update user program experiences request received for user ${userId}`,
  USER_VALIDATION: (userId: number) => `Validating user ${userId} for program experiences`,
  REGISTRATION_VALIDATION: (registrationId: number) =>
    `Validating registration ${registrationId} for program experiences`,
  VALIDATING_LOOKUP_DATA: (ids: number[]) =>
    `Validating seeker program experience lookup IDs: ${ids.join(', ')}`,
  INVALID_LOOKUP_DATA: (ids: number[]) =>
    `Invalid seeker program experience lookup IDs: ${ids.join(', ')}`,
  FETCHING_EXISTING: (userId: number, registrationId?: number | null) =>
    `Fetching existing program experiences for user ${userId}${
      registrationId ? ` and registration ${registrationId}` : ''
    }`,
  REMOVING_EXISTING: (userId: number, registrationId?: number | null) =>
    `Removing existing program experiences for user ${userId}${
      registrationId ? ` and registration ${registrationId}` : ''
    }`,
};

export const userRegistrationMapConstMessages = {
  CREATE_OPERATION: 'Create user registration mappings',
  UPDATE_OPERATION: 'Update user registration mappings',
  DELETE_OPERATION: 'Delete user registration mappings',
  GET_OPERATION: 'Get user registration mappings',
  CREATE_REQUEST_RECEIVED: (userId: number) =>
    `Create user registration mappings request received for user ${userId}`,
  UPDATE_REQUEST_RECEIVED: (userId: number) =>
    `Update user registration mappings request received for user ${userId}`,
  DELETE_REQUEST_RECEIVED: (userId: number) =>
    `Delete user registration mappings request received for user ${userId}`,
  GET_REQUEST_RECEIVED: (userId: number) =>
    `Get user registration mappings request received for user ${userId}`,
  VALIDATING_USER: (userId: number) => `Validating user ${userId} for registration mappings`,
  VALIDATING_CREATOR: (userId: number) =>
    `Validating creator ${userId} for registration mappings`,
  VALIDATING_UPDATER: (userId: number) =>
    `Validating updater ${userId} for registration mappings`,
  VALIDATING_REGISTRATIONS: (ids: number[]) =>
    `Validating registration IDs: ${ids.join(', ')}`,
  REGISTRATION_STATUS_INVALID: (registrationId: number) =>
    `Registration ${registrationId} is not completed`,
  REGISTRATION_USER_MISMATCH: (
    registrationId: number,
    expectedUserId: number,
    actualUserId?: number | null,
  ) =>
    `Registration ${registrationId} belongs to user ${
      actualUserId ?? 'unknown'
    }, expected ${expectedUserId}`,
  FETCHING_EXISTING_MAPPINGS: (userId: number) =>
    `Fetching existing registration mappings for user ${userId}`,
  REMOVING_MAPPINGS: (userId: number, registrationIds?: number[]) =>
    `Removing registration mappings for user ${userId}${
      registrationIds?.length ? ` and registrations ${registrationIds.join(', ')}` : ''
    }`,
  CREATING_SUMMARY: (registrationId: number) =>
    `Creating user participation summary for registration ${registrationId}`,
  SUMMARY_EXISTS: (
    userId: number,
    programId: number,
    sessionId?: number | null,
  ) =>
    `Existing participation summary found for user ${userId}, program ${programId}${
      sessionId ? `, session ${sessionId}` : ''
    }`,
  SUMMARY_CREATED: (summaryId: number) =>
    `User participation summary created with ID ${summaryId}`,
  MAPPING_CREATED: (mapId: number) =>
    `User registration mapping created with ID ${mapId}`,
  MAPPING_UPDATED: (mapId: number) =>
    `User registration mapping updated with ID ${mapId}`,
  MAPPINGS_RECORDED: 'User registration mappings recorded successfully',
  MAPPINGS_UPDATED: 'User registration mappings updated successfully',
  MAPPINGS_DELETED: 'User registration mappings deleted successfully',
  MAPPINGS_RETRIEVED: 'User registration mappings retrieved successfully',
};

export const ERROR_OCCURRED = 'Error occurred: ';

export const APP_LOGGER = {
  NOT_INITIALIZED: 'AppLoggerService has not been initialized yet.',
};

export const commonDataServiceMessages = {
  FETCHING_DATA: 'Fetching data with filters',
  DATA_FETCHED_SUCCESS: (count: number) => `Data fetched successfully. Count: ${count}`,
  FINDING_ENTITY_BY_ID: (id: number) => `Finding entity by ID: ${id}`,
  ENTITY_FOUND_SUCCESS: (id: number) => `Entity with ID ${id} found successfully`,
  SAVING_ENTITY: 'Saving entity',
  ENTITY_SAVED_SUCCESS: 'Entity saved successfully',
  FINDING_ENTITIES_BY_CRITERIA: 'Finding entities by criteria',
  ENTITIES_FOUND_SUCCESS: (count: number) => `Entities found successfully. Count: ${count}`,
  FINAL_QUERY_OPTIONS: 'Final query options',
  ERROR_FETCHING_DATA: 'Error occurred while fetching data',
  ERROR_FINDING_ENTITY_BY_ID: (id: number) => `Error occurred while finding entity with ID: ${id}`,
  ERROR_SAVING_ENTITY: 'Error occurred while saving entity',
  ERROR_FINDING_ENTITIES_BY_CRITERIA: 'Error occurred while finding entities by criteria',
};

export const questionOptionConstMessages = {
  CREATE_NEW_MAPPING: 'Create a new question-option mapping',
  GET_ALL_MAPPINGS: 'Get all question-option mappings with pagination and search',
  GET_MAPPING_BY_ID: 'Get a question-option mapping by ID',
  UPDATE_QUESTION_OPTION_BY_ID: 'Update question-option mapping by ID',
  DELETE_QUESTION_OPTION_BY_ID: 'Delete question-option mapping by ID',
  MAPPING_CREATED: 'Mapping created successfully',
  MAPPING_UPDATED: 'Mapping updated successfully',
  MAPPING_DELETED: 'Mapping deleted successfully',
  MAPPING_RETRIEVED: 'Mapping retrieved successfully',
  MAPPINGS_RETRIEVED: 'Mappings retrieved successfully',
  MAPPING_NOT_FOUND: 'Mapping not found',
  DUPLICATE_MAPPING: 'A mapping for this question and option already exists',
  VALIDATION_ERROR_BAD_REQ: 'Validation failed. Please check the input data.',
  FAILED_TO_CREATE_MAPPING: 'Failed to create mapping. Ensure all data is valid.',
  FAILED_TO_UPDATE_MAPPING: 'Failed to update mapping. Check data integrity.',
  FAILED_TO_DELETE_MAPPING: 'Failed to delete mapping.',
  BAD_REQUEST: 'Bad request',
  NOT_FOUND: 'Not found',
  MAPPING_ID: 'Mapping ID',
  QUESTION_NOT_FOUND: 'Question not found',
  OPTION_NOT_FOUND: 'Option not found',
  USER_NOT_FOUND: 'User not found',
  CREATE_MAPPING_REQUEST_RECEIVED: 'Create mapping request received',
  CREATING_QUES_OPT_MAPPING: (data: any) =>
    `Creating question-option mapping with data: ${JSON.stringify(data)}`,
  FIND_ALL_MAPPINGS_REQUEST_RECEIVED: 'Find all mappings request received',
  FINDING_ALL_MAPPINGS: (limit: number, offset: number, searchText: string) =>
    `Finding all mappings with limit: ${limit}, offset: ${offset}, searchText: ${searchText}`,
  FINDING_MAPPING_BY_ID: (id: number) => `Finding mapping by ID: ${id}`,
  FINDING_MAPPING_BY_QUESTION_ID: (questionId: number) =>
    `Finding mapping by question ID: ${questionId}`,
  FIND_ONE_MAPPING_REQUEST_RECEIVED: 'Find one mapping request received',
  UPDATE_MAPPING_REQUEST_RECEIVED: 'Update mapping request received',
  UPDATING_MAPPING: (id: number) => `Updating mapping with ID: ${id}`,
  DELETE_MAPPING_REQUEST_RECEIVED: 'Delete mapping request received',
  REMOVING_MAPPING: (id: number) => `Removing mapping with ID: ${id}`,
  MAPPING_QUES_OPT_ALREADY_EXISTS: (questionId: number, optionId: number) =>
    `Mapping for question ${questionId} and option ${optionId} already exists.`,
  MAPPING_NOT_FOUND_ID: (id: number) => `Mapping with ID ${id} not found.`,
};

export const SWAGGER_API_RESPONSE = {
  SUCCESS: 'Success',
  CREATED: 'Created',
  BAD_REQUEST: 'Bad Request',
  UNAUTHORIZED: 'Unauthorized',
  FORBIDDEN: 'Forbidden',
  NOT_FOUND: 'Not Found',
  INTERNAL_SERVER_ERROR: 'Internal Server Error',
  OK: 'OK',
  NO_CONTENT: 'No Content',
  CONFLICT: 'Conflict',
  UPDATED: 'Updated',
};

export const programConstMessages = {
  CREATING_PROGRAM: (data: any) => `Creating program with data: ${JSON.stringify(data)}`,
  PROGRAM_TYPE_NOT_FOUND_ID: (id: number) => `Program type with ID ${id} not found.`,
  CREATE_NEW_PROGRAM: 'Create a new program',
  PROGRAM_CREATED: 'Program created successfully',
  GET_ALL_PROGRAMS: 'Get all programs with pagination and search',
  PROGRAMS_RETRIEVED: 'Programs retrieved successfully',
  GET_PROGRAM_BY_ID: 'Get a program by ID',
  PROGRAM_ID: 'Program ID',
  PROGRAM_RETRIEVED: 'Program retrieved successfully',
  UPDATE_PROGRAM_BY_ID: 'Update program by ID',
  PROGRAM_UPDATED: 'Program updated successfully',
  DELETE_PROGRAM_BY_ID: 'Delete program by ID',
  PROGRAM_DELETED: 'Program deleted successfully',
  PROGRAM_NOT_FOUND_ID: (id: number) => `Program with ID ${id} not found.`,
  SOFT_DELETING_PROGRAM: (id: number) => `Soft deleting program with ID: ${id}`,
  SOFT_DELETED_PROGRAM: (id: number) => `Soft deleted program with ID: ${id}`,
  FINDING_ALL_PROGRAMS: (limit: number, offset: number, searchText: string) =>
    `Finding all programs with limit: ${limit}, offset: ${offset}, searchText: ${searchText}`,
  FINDING_PROGRAM_BY_ID: (id: number) => `Finding program by ID: ${id}`,
  UPDATING_PROGRAM: (id: number) => `Updating program with ID: ${id}`,
  REMOVING_PROGRAM: (id: number) => `Removing program with ID: ${id}`,
  UPDATE_PROGRAM_REQUEST_RECEIVED: 'Update program request received',
  DELETING_PROGRAM_BY_ID: 'Deleting program by ID',
  CREATE_PROGRAM_REQUEST_RECEIVED: 'Create program request received',
  FIND_ALL_PROGRAMS_REQUEST_RECEIVED: 'Find all programs request received',
  FIND_ONE_PROGRAM_REQUEST_RECEIVED: 'Find one program request received',
  DELETE_PROGRAM_REQUEST_RECEIVED: 'Delete program request received',
  PROGRAM_DUPLICATE_BADREQUEST_CODE: (code: string) => `Duplicate program found with name: ${code}`,
  ERROR_OCCURRED: 'Error occurred: ',
  PROGRAM_BAD_REQUEST: 'Program creation failed due to bad request',
};

export const FORM_SECTION_MESSAGES = {
  FORM_SEC_NOT_FOUND: (id: number) => `Form section with ID ${id} not found.`,
  FIND_ALL_REQUEST_RECEIVED: 'Find all form sections request received',
  FOUND: 'Form sections found successfully',
  FETCHING_ALL_FORM_SECTIONS: (limit: number, offset: number, searchText: string) =>
    `Fetching all form sections with limit: ${limit}, offset: ${offset}, searchText: ${searchText}`,
};

export const programQuestionConstMessages = {
  // API Operation Messages
  PROGRAM_ID: 'Program ID',
  CREATE_NEW_MAPPING: 'Create new program-question mapping',
  CREATE_MAPPING_REQUEST_RECEIVED: 'Create program-question mapping request received',
  MAPPING_CREATED: 'Program-question mapping created successfully',
  FIND_ALL_MAPPINGS: 'Find all program-question mappings',
  FINDING_ALL_MAPPINGS: 'Finding all program-question mappings',
  MAPPINGS_RETRIEVED: 'Program-question mappings retrieved successfully',
  GET_MAPPING_BY_ID: 'Get program-question mapping by ID',
  FINDING_MAPPING_BY_ID: (id: number) => `Finding program-question mapping by ID: ${id}`,
  MAPPING_RETRIEVED: 'Program-question mapping retrieved successfully',
  UPDATE_MAPPING_BY_ID: 'Update program-question mapping by ID',
  UPDATING_MAPPING: (id: number) => `Updating program-question mapping with ID: ${id}`,
  MAPPING_UPDATED: 'Program-question mapping updated successfully',
  DELETE_MAPPING_BY_ID: 'Delete program-question mapping by ID',
  REMOVING_MAPPING: (id: number) => `Removing program-question mapping with ID: ${id}`,
  MAPPING_DELETED: 'Program-question mapping deleted successfully',
  MAPPING_ID: 'Program-question mapping ID',

  // Transaction and Processing Messages
  TRANSACTION_FAILED: 'Transaction failed, rolling back all changes',
  TRANSACTION_STARTED: 'Transaction started for program-question mapping operation',
  TRANSACTION_COMMITTED: 'Transaction committed successfully',

  // Request Handling Messages
  CREATE_PRO_MAP_REQUEST_RECEIVED: 'Create program-question mapping request received',
  FIND_ALL_MAPPINGS_REQUEST_RECEIVED: 'Find all program-question mappings request received',
  FIND_ONE_MAPPING_REQUEST_RECEIVED: 'Find one program-question mapping request received',
  UPDATE_MAPPING_REQUEST_RECEIVED: 'Update program-question mapping request received',
  DELETE_MAPPING_REQUEST_RECEIVED: 'Delete program-question mapping request received',
  BULK_DELETE_REQUEST_RECEIVED: 'Bulk delete program-question mapping request received',

  // Validation Messages
  VALIDATING_PROGRAM: (programId: number) => `Validating program with ID: ${programId}`,
  VALIDATING_PROGRAM_SESSION: (sessionId: number) =>
    `Validating program session with ID: ${sessionId}`,
  VALIDATING_QUESTIONS: (questionIds: number[]) =>
    `Validating questions with IDs: ${questionIds.join(', ')}`,
  VALIDATING_USER: (userId: number) => `Validating user with ID: ${userId}`,
  VALIDATING_FORM_SECTION: (sectionId: number) => `Validating form section with ID: ${sectionId}`,
  VALIDATION_COMPLETED: 'Validation completed successfully',

  // Scope and Duplicate Check Messages
  CHECKING_EXISTING_MAPPINGS: 'Checking for existing mappings in scope',
  EXISTING_MAPPINGS_FOUND: (count: number) => `Found ${count} existing mappings in scope`,
  NO_EXISTING_MAPPINGS: 'No existing mappings found in scope',
  DUPLICATE_MAPPING_DETECTED: (programId: number, questionId: number) =>
    `Duplicate mapping detected for program ${programId} and question ${questionId}`,

  // Custom Question Creation Messages
  PROCESSING_CUSTOM_QUESTIONS: 'Processing custom questions',
  CREATING_CUSTOM_QUESTION: (label: string) => `Creating custom question: ${label}`,
  CUSTOM_QUESTION_CREATED: (id: number) => `Custom question created with ID: ${id}`,
  PROCESSING_QUESTION_OPTIONS: 'Processing question options',
  CREATING_OPTION_MAPPING: (questionId: number, optionId: number) =>
    `Creating option mapping for question ${questionId} and option ${optionId}`,
  OPTION_MAPPING_CREATED: 'Option mapping created successfully',

  // Form Section Processing Messages
  PROCESSING_SECTIONS: 'Processing form sections',
  PROCESSING_EXISTING_SECTION: (sectionId: number) => `Processing existing section: ${sectionId}`,
  PROCESSING_NEW_SECTION: (name: string) => `Processing new section: ${name}`,
  SECTION_CREATED: (id: number) => `Form section created with ID: ${id}`,
  SECTION_UPDATED: (id: number) => `Form section updated with ID: ${id}`,

  // Mapping Creation Messages
  CREATING_MAPPINGS: 'Creating program-question mappings',
  MAPPING_CREATED_WITH_ORDER: (questionId: number, order: number) =>
    `Mapping created for question ${questionId} with display order ${order}`,
  MAPPINGS_SAVED: (count: number) => `${count} mappings saved successfully`,

  // Update Operation Messages
  UPDATE_OPERATION_STARTED: (id: number) => `Update operation started for mapping ID: ${id}`,
  UPDATE_PROGRAM_REFERENCE: (oldId: number, newId: number) =>
    `Updating program reference from ${oldId} to ${newId}`,
  UPDATE_QUESTION_REFERENCE: (oldId: number, newId: number) =>
    `Updating question reference from ${oldId} to ${newId}`,
  UPDATE_SESSION_REFERENCE: (sessionId: number | null) =>
    `Updating session reference to: ${sessionId}`,
  UPDATE_DISPLAY_ORDER: (oldOrder: number, newOrder: number) =>
    `Updating display order from ${oldOrder} to ${newOrder}`,
  UPDATE_REGISTRATION_LEVEL: (oldLevel: string, newLevel: string) =>
    `Updating registration level from ${oldLevel} to ${newLevel}`,

  // Delete Operation Messages
  DELETE_OPERATION_STARTED: (id: number) => `Delete operation started for mapping ID: ${id}`,
  SOFT_DELETING_MAPPING: (id: number) => `Soft deleting mapping with ID: ${id}`,
  MAPPING_SOFT_DELETED: (id: number) => `Mapping with ID ${id} soft deleted successfully`,

  // Bulk Delete Messages
  BULK_DELETE_STARTED: 'Bulk delete operation started',
  PROCESSING_SECTION_DELETIONS: 'Processing section deletions',
  PROCESSING_QUESTION_DELETIONS: 'Processing question deletions',
  DELETING_SECTION: (sectionId: number) => `Deleting section with ID: ${sectionId}`,
  SECTION_DELETED: (sectionId: number, questionCount: number) =>
    `Section ${sectionId} deleted with ${questionCount} associated questions`,
  DELETING_QUESTIONS: (questionIds: number[]) => `Deleting questions: ${questionIds.join(', ')}`,
  QUESTIONS_DELETED: (count: number) => `${count} questions deleted successfully`,
  BULK_DELETE_COMPLETED: (operationCount: number) =>
    `Bulk delete completed with ${operationCount} operations`,

  // Query and Database Messages
  BUILDING_QUERY_CONDITIONS: 'Building query conditions',
  EXECUTING_FIND_QUERY: 'Executing find query',
  QUERY_EXECUTED: (resultCount: number) => `Query executed, found ${resultCount} results`,
  CALCULATING_PAGINATION: 'Calculating pagination information',
  PAGINATION_CALCULATED: (totalPages: number, currentPage: number) =>
    `Pagination: ${currentPage}/${totalPages}`,

  // Error Context Messages
  ERROR_CONTEXT_CREATION: 'Error occurred during creation operation',
  ERROR_CONTEXT_UPDATE: 'Error occurred during update operation',
  ERROR_CONTEXT_DELETE: 'Error occurred during delete operation',
  ERROR_CONTEXT_FETCH: 'Error occurred during fetch operation',
  ERROR_CONTEXT_VALIDATION: 'Error occurred during validation',
  ERROR_CONTEXT_TRANSACTION: 'Error occurred during transaction',

  // Success Messages
  OPERATION_COMPLETED: 'Operation completed successfully',
  DATA_PROCESSED: (count: number) => `${count} items processed successfully`,
  ENTITIES_AFFECTED: (count: number) => `${count} entities affected by operation`,

  // Repository Operation Messages
  REPO_FIND_ALL_CALLED: 'Repository findAll method called',
  REPO_FIND_BY_ID_CALLED: (id: number) => `Repository findById method called with ID: ${id}`,
  REPO_CREATE_CALLED: 'Repository create method called',
  REPO_UPDATE_CALLED: (id: number) => `Repository update method called with ID: ${id}`,
  REPO_DELETE_CALLED: (id: number) => `Repository delete method called with ID: ${id}`,
  REPO_BULK_DELETE_CALLED: 'Repository bulk delete method called',

  // Service Operation Messages
  SERVICE_CREATE_CALLED: 'Service create method called',
  SERVICE_FIND_ALL_CALLED: 'Service findAll method called',
  SERVICE_FIND_ONE_CALLED: (id: number) => `Service findOne method called with ID: ${id}`,
  SERVICE_UPDATE_CALLED: (id: number) => `Service update method called with ID: ${id}`,
  SERVICE_DELETE_CALLED: (id: number) => `Service delete method called with ID: ${id}`,
  SERVICE_BULK_DELETE_CALLED: 'Service bulk delete method called',
};
export const programTypeConstMessages = {
  FINDING_ALL_PROGRAM_TYPES: (limit: number, offset: number, searchText: string) =>
    `Finding all program types with limit: ${limit}, offset: ${offset}, searchText: ${searchText}`,
  FINDING_PROGRAM_TYPE_BY_ID: (id: number) => `Finding program type by ID: ${id}`,
  PROGRAM_TYPE_NOT_FOUND_ID: (id: number) => `Program type with ID ${id} not found.`,
  SOFT_DELETING_PROGRAM_TYPE: (id: number) => `Soft deleting program type with ID: ${id}`,
  SOFT_DELETED_PROGRAM_TYPE: (id: number) => `Soft deleted program type with ID: ${id}`,
  RETRIEVING_PROGRAM_TYPES_WHERE_CLAUSE: (whereClause: Record<string, any>) =>
    `Retrieving program types with where clause: ${JSON.stringify(whereClause)}`,
  CREATING_PROGRAM_TYPE: (data: any) => `Creating program type with data: ${JSON.stringify(data)}`,
  DUPLICATE_PROGRAM_TYPE_FOUND: (name: string) => `Duplicate program type found: ${name}`,
  UPDATING_PROGRAM_TYPE: (id: number) => `Updating program type with ID: ${id}`,
  REMOVING_PROGRAM_TYPE: (id: number) => `Removing program type with ID: ${id}`,
  GET_ALL_PROGRAM_TYPES: 'Get all program types with pagination and search',
  FIND_ALL_PROGRAM_TYPES_REQUEST_RECEIVED: 'Find all program types request received',
  PROGRAM_TYPES_RETRIEVED: 'Program types retrieved successfully',
  PROGRAM_TYPE_ID: 'Program type ID',
  GET_PROGRAM_TYPE_BY_ID: 'Get a program type by ID',
  FIND_ONE_PROGRAM_TYPE_REQUEST_RECEIVED: 'Find one program type request received',
  PROGRAM_TYPE_RETRIEVED: 'Program type retrieved successfully',
  CREATE_NEW_PROGRAM_TYPE: 'Create a new program type',
  CREATE_PROGRAM_TYPE_REQUEST_RECEIVED: 'Create program type request received',
  PROGRAM_TYPE_CREATED: 'Program type created successfully',
  UPDATE_PROGRAM_TYPE_BY_ID: 'Update program type by ID',
  UPDATE_PROGRAM_TYPE_REQUEST_RECEIVED: 'Update program type request received',
  PROGRAM_TYPE_UPDATED: 'Program type updated successfully',
  DELETE_PROGRAM_TYPE_BY_ID: 'Delete program type by ID',
  DELETE_PROGRAM_TYPE_REQUEST_RECEIVED: 'Delete program type request received',
  PROGRAM_TYPE_DELETED: 'Program type deleted successfully',
};

export const formSchemaConstMessages = {
  PROCESSING_FORM_SCHEMA: 'Processing form schema started',
  ANALYZING_QUESTIONS: 'Analyzing questions',
  RETRIEVED_EXISTING_QUESTIONS: 'Retrieved existing questions',
  MISSING_QUESTIONS_DB: 'Missing questions in database',
  PROCESSING_NEW_QUESTION: 'Processing new question',
  FORM_SECTION_LOOKUP: 'Form section lookup result',
  CREATED_QUESTION_ENTITY: 'Created question entity',
  QUESTION_SAVED: 'Question saved successfully',
  PROCESSING_QUESTION_OPTIONS: 'Processing question options',
  PROCESSING_OPTION: 'Processing option',
  UPDATED_EXISTING_OPTION: 'Updated existing option',
  CREATED_NEW_OPTION: 'Created new option',
  MAPPED_OPTIONS_TO_QUESTION: 'Mapped options to question',
  FOUND_PROGRAM_MAPPINGS: 'Found program question mappings',
  CREATED_NEW_MAPPINGS: 'Created new program question mappings',
  NO_NEW_QUESTIONS: 'No new questions to map',
  FORM_SCHEMA_COMPLETED: 'Form schema processing completed',
  FORM_SCHEMA_GENERATION_FAILED: 'Form schema generation failed',
  FORM_SCHEMA_GENERATION_SUCCESS: 'Form schema generated successfully',
};

export const programSessionConstMessages = {
  CREATE_NEW_SESSION: 'Create a new program session',
  GET_ALL_SESSIONS: 'Get all program sessions with pagination and search',
  GET_SESSION_BY_ID: 'Get program session by ID',
  UPDATE_SESSION_BY_ID: 'Update program session by ID',
  DELETE_SESSION_BY_ID: 'Delete program session by ID',
  SESSION_CREATED: 'Program session created successfully',
  SESSION_UPDATED: 'Program session updated successfully',
  SESSION_DELETED: 'Program session deleted successfully',
  SESSION_RETRIEVED: 'Program session retrieved successfully',
  SESSIONS_RETRIEVED: 'Program sessions retrieved successfully',
  SESSION_NOT_FOUND: 'Program session not found',
  DUPLICATE_SESSION: 'A session with the same name/number/code already exists',
  VALIDATION_ERROR_BAD_REQ: 'Validation failed. Please check the input data.',
  FAILED_TO_CREATE_SESSION: 'Failed to create session. Ensure all data is valid.',
  FAILED_TO_UPDATE_SESSION: 'Failed to update session. Check data integrity.',
  FAILED_TO_DELETE_SESSION: 'Failed to delete session.',
  BAD_REQUEST: 'Bad request',
  NOT_FOUND: 'Not found',
  SESSION_ID: 'Program Session ID',
  PROGRAM_NOT_FOUND: 'Program not found',
  USER_NOT_FOUND: 'User not found',
  CREATE_SESSION_REQUEST_RECEIVED: 'Create program session request received',
  FIND_ALL_SESSIONS_REQUEST_RECEIVED: 'Find all program sessions request received',
  FIND_ONE_SESSION_REQUEST_RECEIVED: 'Find one program session request received',
  UPDATE_SESSION_REQUEST_RECEIVED: 'Update program session request received',
  DELETE_SESSION_REQUEST_RECEIVED: 'Delete program session request received',
  SESSION_NOT_FOUND_ID: (id: number) => `Program session with ID ${id} not found.`,
  CREATING_SESSION: (data: any) => `Creating program session with data: ${JSON.stringify(data)}`,
  FINDING_ALL_SESSIONS: (limit: number, offset: number, searchText: string) =>
    `Finding all program sessions with limit: ${limit}, offset: ${offset}, searchText: ${searchText}`,
  FINDING_SESSION_BY_ID: (id: number) => `Finding program session by ID: ${id}`,
  UPDATING_SESSION: (id: number) => `Updating program session with ID: ${id}`,
  REMOVING_SESSION: (id: number) => `Removing program session with ID: ${id}`,
};

export const programRegistrationMessages = {
  CREATE_NEW: 'Create a new program registration',
  REGISTRATION_CREATED: 'Program registration created successfully',
  GET_ALL: 'Get all program registrations',
  REGISTRATIONS_RETRIEVED: 'Program registrations retrieved successfully',
  GET_BY_ID: 'Get program registration by ID',
  REGISTRATION_RETRIEVED: 'Program registration retrieved successfully',
  UPDATE_BY_ID: 'Update program registration by ID',
  REGISTRATION_UPDATED: 'Program registration updated successfully',
  DELETE_BY_ID: 'Delete program registration by ID',
  REGISTRATION_DELETED: 'Program registration deleted successfully',
  REGISTRATION_ID: 'Program registration ID',
  CREATE_REQUEST_RECEIVED: 'Create program registration request received',
  FIND_ALL_REQUEST_RECEIVED: 'Find all program registrations request received',
  FIND_ONE_REQUEST_RECEIVED: 'Find one program registration request received',
  UPDATE_REQUEST_RECEIVED: 'Update program registration request received',
  DELETE_REQUEST_RECEIVED: 'Delete program registration request received',
  POST_RM_RATINGS: 'Post registration module ratings and review for program registered user',
  RATINGS_SUBMITTED: 'Ratings and review submitted successfully.',
  UPDATE_RM_RATINGS: 'Update registration module ratings and review for program registered user',
  RATINGS_UPDATED: 'Ratings and review updated successfully.',
  REQUEST_SWAP: 'Request swap for program registration',
  SWAP_REQUEST_CREATED: 'Swap request created successfully.',
  SWAP_REQUEST_RECEIVED: 'Swap request received successfully.',
  SWAP_REQUEST_NOT_FOUND: 'Swap request not found for the given registration ID',
  SWAP_REQUEST_FILTERS : 'Filters for swap requests',
  UPDATE_SWAP_REQUEST: 'Update swap request for program registration',
  SWAP_REQUEST_UPDATED: 'Swap request updated successfully.',
  SWAP_REQUEST_ALREADY_UPDATED: 'Swap request already updated for the given registration ID',
  SWAP_REQUEST_ALREADY_EXISTS: 'Swap request already exists for the given registration ID',
  SWAP_REQUEST_ACCEPTED: 'Swap request accepted successfully.',
  INVALID_SWAP_REQUEST_STATUS: 'Invalid swap request status provided.',
  EDIT_SWAP_REQUEST: 'Edit swap request for program registration',
  SWAP_REQUEST_EDITED: 'Swap request edited successfully.',
  CANCEL_SWAP_REQUEST: 'Cancel swap request for program registration',
  SWAP_REQUEST_CANCELLED: 'Swap request cancelled successfully.',
  MANAGE_SWAP_REQUEST: 'Manage swap request (edit/cancel) for program registration',
  SWAP_REQUEST_ACTION_COMPLETED: 'Swap request action completed successfully.',
  INVALID_SWAP_REQUEST_ACTION: 'Invalid action provided for swap request.',
  GET_SWAP_REQUESTS : 'Get all swap requests for program registrations',
  SWAP_REQUESTS_RETRIEVED: 'Swap requests retrieved successfully.',
  FIND_BY_REQUEST_RECEIVED : 'Find program registration by request received',
  GET_SUB_PROGRAM_REGISTRATIONS: 'Get all sub program registrations',
  PROGRAM_ID: 'Program ID',
  SUB_PROGRAM_ID : 'Sub Program ID',
  SUB_PROGRAM_REGISTRATIONS_RETRIEVED: 'Sub program registrations retrieved successfully',
  FIND_SUB_PROGRAM_REQUEST_RECEIVED: 'Find sub program registrations request received',
  INVALID_REQUEST: 'Invalid request received for program registration',
  DIRECT_SWAP : 'Direct swap for program registration',
  DIRECT_SWAP_SUCCESS: 'Direct swap for program registration successful',
  INVALID_SWAP_REQUEST: 'Invalid swap request received.',
  FREE_SEAT_KPI_RETRIEVED: 'Free seat KPI retrieved successfully.',
};

export const registrationApprovalMessages = {
  CREATE_NEW: 'Create a new registration approval',
  APPROVAL_CREATED: 'Registration approval created successfully',
  GET_ALL: 'Get all registration approvals',
  APPROVALS_RETRIEVED: 'Registration approvals retrieved successfully',
  GET_BY_ID: 'Get registration approval by ID',
  APPROVAL_RETRIEVED: 'Registration approval retrieved successfully',
  UPDATE_BY_ID: 'Update registration approval by ID',
  APPROVAL_UPDATED: 'Registration approval updated successfully',
  DELETE_BY_ID: 'Delete registration approval by ID',
  APPROVAL_DELETED: 'Registration approval deleted successfully',
  APPROVAL_ID: 'Registration approval ID',
  CREATE_REQUEST_RECEIVED: 'Create registration approval request received',
  FIND_ALL_REQUEST_RECEIVED: 'Find all registration approvals request received',
  FIND_ONE_REQUEST_RECEIVED: 'Find one registration approval request received',
  UPDATE_REQUEST_RECEIVED: 'Update registration approval request received',
  DELETE_REQUEST_RECEIVED: 'Delete registration approval request received',
};

export const registrationCustomResponseMessages = {
  CREATE_NEW: 'Create a new registration custom response',
  RESPONSE_CREATED: 'Registration custom response created successfully',
  GET_ALL: 'Get all registration custom responses',
  RESPONSES_RETRIEVED: 'Registration custom responses retrieved successfully',
  GET_BY_ID: 'Get registration custom response by ID',
  RESPONSE_RETRIEVED: 'Registration custom response retrieved successfully',
  UPDATE_BY_ID: 'Update registration custom response by ID',
  RESPONSE_UPDATED: 'Registration custom response updated successfully',
  DELETE_BY_ID: 'Delete registration custom response by ID',
  RESPONSE_DELETED: 'Registration custom response deleted successfully',
  RESPONSE_ID: 'Registration custom response ID',
  CREATE_REQUEST_RECEIVED: 'Create registration custom response request received',
  FIND_ALL_REQUEST_RECEIVED: 'Find all registration custom responses request received',
  FIND_ONE_REQUEST_RECEIVED: 'Find one registration custom response request received',
  UPDATE_REQUEST_RECEIVED: 'Update registration custom response request received',
  DELETE_REQUEST_RECEIVED: 'Delete registration custom response request received',
};

export const registrationInvoiceDetailMessages = {
  CREATE_NEW: 'Create a new registration invoice detail',
  INVOICE_CREATED: 'Registration invoice detail created successfully',
  GET_ALL: 'Get all registration invoice details',
  INVOICES_RETRIEVED: 'Registration invoice details retrieved successfully',
  GET_BY_ID: 'Get registration invoice detail by ID',
  INVOICE_RETRIEVED: 'Registration invoice detail retrieved successfully',
  UPDATE_BY_ID: 'Update registration invoice detail by ID',
  INVOICE_UPDATED: 'Registration invoice detail updated successfully',
  DELETE_BY_ID: 'Delete registration invoice detail by ID',
  INVOICE_DELETED: 'Registration invoice detail deleted successfully',
  INVOICE_ID: 'Registration invoice detail ID',
  CREATE_REQUEST_RECEIVED: 'Create registration invoice detail request received',
  FIND_ALL_REQUEST_RECEIVED: 'Find all registration invoice details request received',
  FIND_ONE_REQUEST_RECEIVED: 'Find one registration invoice detail request received',
  UPDATE_REQUEST_RECEIVED: 'Update registration invoice detail request received',
  DELETE_REQUEST_RECEIVED: 'Delete registration invoice detail request received',
};

export const registrationPaymentDetailMessages = {
  CREATE_NEW: 'Create a new registration payment detail',
  PAYMENT_CREATED: 'Registration payment detail created successfully',
  GET_ALL: 'Get all registration payment details',
  PAYMENTS_RETRIEVED: 'Registration payment details retrieved successfully',
  GET_BY_ID: 'Get registration payment detail by ID',
  PAYMENT_RETRIEVED: 'Registration payment detail retrieved successfully',
  UPDATE_BY_ID: 'Update registration payment detail by ID',
  PAYMENT_UPDATED: 'Registration payment detail updated successfully',
  DELETE_BY_ID: 'Delete registration payment detail by ID',
  PAYMENT_DELETED: 'Registration payment detail deleted successfully',
  PAYMENT_ID: 'Registration payment detail ID',
  CREATE_REQUEST_RECEIVED: 'Create registration payment detail request received',
  FIND_ALL_REQUEST_RECEIVED: 'Find all registration payment details request received',
  FIND_ONE_REQUEST_RECEIVED: 'Find one registration payment detail request received',
  UPDATE_REQUEST_RECEIVED: 'Update registration payment detail request received',
  DELETE_REQUEST_RECEIVED: 'Delete registration payment detail request received',
};

export const registrationTravelInfoMessages = {
  CREATE_NEW: 'Create a new registration travel info',
  INFO_CREATED: 'Registration travel info created successfully',
  GET_ALL: 'Get all registration travel info records',
  INFOS_RETRIEVED: 'Registration travel info records retrieved successfully',
  GET_BY_ID: 'Get registration travel info by ID',
  INFO_RETRIEVED: 'Registration travel info retrieved successfully',
  UPDATE_BY_ID: 'Update registration travel info by ID',
  INFO_UPDATED: 'Registration travel info updated successfully',
  DELETE_BY_ID: 'Delete registration travel info by ID',
  INFO_DELETED: 'Registration travel info deleted successfully',
  INFO_ID: 'Registration travel info ID',
  CREATE_REQUEST_RECEIVED: 'Create registration travel info request received',
  FIND_ALL_REQUEST_RECEIVED: 'Find all registration travel info request received',
  FIND_ONE_REQUEST_RECEIVED: 'Find one registration travel info request received',
  UPDATE_REQUEST_RECEIVED: 'Update registration travel info request received',
  DELETE_REQUEST_RECEIVED: 'Delete registration travel info request received',
};

export const registrationTravelPlanMessages = {
  CREATE_NEW: 'Create a new registration travel plan',
  PLAN_CREATED: 'Registration travel plan created successfully',
  GET_ALL: 'Get all registration travel plans',
  PLANS_RETRIEVED: 'Registration travel plans retrieved successfully',
  GET_BY_ID: 'Get registration travel plan by ID',
  PLAN_RETRIEVED: 'Registration travel plan retrieved successfully',
  UPDATE_BY_ID: 'Update registration travel plan by ID',
  PLAN_UPDATED: 'Registration travel plan updated successfully',
  DELETE_BY_ID: 'Delete registration travel plan by ID',
  PLAN_DELETED: 'Registration travel plan deleted successfully',
  PLAN_ID: 'Registration travel plan ID',
  RESET_TRAVEL_DETAILS: 'Reset travel details for registration',
  TRAVEL_DETAILS_RESET: 'Travel details reset successfully',
  CREATE_REQUEST_RECEIVED: 'Create registration travel plan request received',
  FIND_ALL_REQUEST_RECEIVED: 'Find all registration travel plans request received',
  FIND_ONE_REQUEST_RECEIVED: 'Find one registration travel plan request received',
  UPDATE_REQUEST_RECEIVED: 'Update registration travel plan request received',
  DELETE_REQUEST_RECEIVED: 'Delete registration travel plan request received',
};

export const registrationMessages = {
  CREATE_REQUEST_RECEIVED: 'Create registration request received',
  UPDATE_REQUEST_RECEIVED: 'Update registration request received',
  DELETE_REQUEST_RECEIVED: 'Delete registration request received',
  DELETE_BY_ID: 'Delete registration by ID',
  REGISTRATION_ID: 'Registration ID',
  GET_ALL: 'Get all registrations',
  REGISTRATIONS_RETRIEVED: 'Registrations retrieved successfully',
  GET_BY_ID: 'Get registration by ID',
  REGISTRATION_RETRIEVED: 'Registration retrieved successfully',
  GET_BY_USER: 'Get registrations by user',
  REGISTRATION_SUCCESS: 'Registration created successfully',
  REGISTRATION_UPDATED: 'Registration updated successfully',
  REGISTRATION_PENDING_APPROVAL: 'Registration pending approval',
  REGISTRATION_WAITLISTED: 'Registration added to waitlist',
  REGISTRATION_NO_SPACE: 'Registration failed due to no space',
  REGISTRATION_ALREADY_EXISTS: 'User already registered for this program',
  GET_QA_REQUEST_RECEIVED: 'Get registration question responses request received',
  QA_RETRIEVED: 'Registration question responses retrieved successfully',
  GET_PROGRAM_DASHBOARD: 'Get program dashboard data',
  GET_SUB_PROGRAM_DASHBOARD: 'Get sub-program dashboard data',
  DASHBOARD_RETRIEVED: 'Program dashboard data retrieved successfully',
  REGISTRATION_DELETED: 'Registration deleted successfully',
  GET_STATUSES: 'Get registration statuses',
  STATUSES_RETRIEVED: 'Registration statuses retrieved successfully',
};

export const preferenceMessages = {
  CREATE_NEW: 'Create preferences for a registration',
  PREFERENCES_CREATED: 'Preferences created successfully',
  GET_ALL: 'Get preferences by registration ID',
  PREFERENCES_RETRIEVED: 'Preferences retrieved successfully',
  UPDATE_BY_REG_ID: 'Replace preferences by registration ID',
  PREFERENCES_UPDATED: 'Preferences updated successfully',
  DELETE_BY_REG_ID: 'Delete preferences by registration ID',
  PREFERENCES_DELETED: 'Preferences deleted successfully',
  REGISTRATION_ID: 'Registration ID',
  CREATE_REQUEST_RECEIVED: 'Create preference request received',
  FIND_ALL_REQUEST_RECEIVED: 'Find preferences request received',
  UPDATE_REQUEST_RECEIVED: 'Update preferences request received',
  DELETE_REQUEST_RECEIVED: 'Delete preferences request received',
};

export const lookupDataConstMessages = {
  FINDING_BY_CATEGORY: (category: string) =>
    `Finding lookup data for category: ${category}`,
  LOOKUP_RETRIEVED: 'Lookup data retrieved successfully',
  CATEGORY_QUERY: 'Lookup category',
  GET_LOOKUP_BY_CATEGORY: 'Get lookup data by category',
  GET_BANKS: 'Get bank names',
  GET_CITIES: 'Get city names',
  GET_CATEGORIES: 'Get lookup categories',
  GET_ALL: 'Get all lookup data',
  FINDING_CATEGORIES: 'Finding lookup categories',
  FINDING_ALL: 'Finding all active lookup data',
  CATEGORIES_RETRIEVED: 'Lookup categories retrieved successfully',
};

export const cancellationReasonConstMessages = {
  FINDING_ACTIVE: 'Finding active cancellation reasons',
  GET_ALL: 'Get cancellation reasons',
  RETRIEVED: 'Cancellation reasons retrieved successfully',
};

export const zeptoEmailCreadentials = {
  ZEPTO_EMAIL: 'hdb@infinitheism.com',
  ZEPTO_EMAIL_NAME: 'Infinitheism',
  ZEPTO_NOREPLY_EMAIL: 'noreply@infinitheism.com',
};

export const messageConstMessages = {
  CREATE_MESSAGE: 'Create a new message',
  MESSAGE_CREATED: 'Message created successfully',
  FIND_ALL_MESSAGES: 'Get messages with filters',
  MESSAGES_RETRIEVED: 'Messages retrieved successfully',
  FIND_ALL_SENDERS: 'Get all senders',
  SENDERS_RETRIEVED: 'Senders retrieved successfully',
  INVALID_SENDER_TYPE: 'Only team members can send messages',
  INVALID_RECEIVER_TYPE: 'Receiver must be a team member',
  MESSAGE_NOTIFICATION_FAILED: 'Message notification failed',
};

export const authConstMessages = {
  LOGIN_ATTEMPT: 'Initiate login attempt',
  OTP_SENT: 'OTP generated successfully',
  RESEND_OTP: 'Resend OTP',
  VALIDATE_OTP: 'Validate OTP',
  OTP_VALIDATED: 'OTP validated successfully',
  LOGOUT: 'Initiate logout',
  LOGOUT_SUCCESS: 'Logout successful',
};

export const featureFlagMessages = {
  FINDING_ACTIVE_FLAGS: 'Finding active feature flags',
  FETCHING_ACTIVE_FLAGS: 'Fetching active feature flags',
  ERROR_FETCHING_FLAGS: 'Error fetching feature flags',
};

export const eInvoiceConstants = {
  COMPANY_NAME: 'Infinitheism Spiritual Foundation Private Limited',
  CURRENCY: { INDIA: 'INR' },
  UNITS: 'oth',
  SERVICE: 'service',
  SHIPPING_INTERSTATE_CHARGE_TAX: 'IGST18',
  SHIPPING_CHANRGE_TAX: 'CSGST', // Note: keeping original typo for compatibility
  SHIPPING_CHARGE_SEC_CODE: '999293',
  GST_TREATMENT: 'business_gst',
  HSN_CODE: 999293,
};

// State GST codes mapping (move this to a constants file)
export const stateGstCodes = [
  { code: '01', state: 'JK', stateName: 'Jammu and Kashmir' },
  { code: '02', state: 'HP', stateName: 'Himachal Pradesh' },
  { code: '03', state: 'PB', stateName: 'Punjab' },
  { code: '04', state: 'CH', stateName: 'Chandigarh' },
  { code: '05', state: 'UT', stateName: 'Uttarakhand' },
  { code: '06', state: 'HR', stateName: 'Haryana' },
  { code: '07', state: 'DL', stateName: 'Delhi' },
  { code: '08', state: 'RJ', stateName: 'Rajasthan' },
  { code: '09', state: 'UP', stateName: 'Uttar Pradesh' },
  { code: '10', state: 'BR', stateName: 'Bihar' },
  { code: '11', state: 'SK', stateName: 'Sikkim' },
  { code: '12', state: 'AR', stateName: 'Arunachal Pradesh' },
  { code: '13', state: 'NL', stateName: 'Nagaland' },
  { code: '14', state: 'MN', stateName: 'Manipur' },
  { code: '15', state: 'MZ', stateName: 'Mizoram' },
  { code: '16', state: 'TR', stateName: 'Tripura' },
  { code: '17', state: 'ML', stateName: 'Meghalaya' },
  { code: '18', state: 'AS', stateName: 'Assam' },
  { code: '19', state: 'WB', stateName: 'West Bengal' },
  { code: '20', state: 'JH', stateName: 'Jharkhand' },
  { code: '21', state: 'OR', stateName: 'Odisha' },
  { code: '22', state: 'CG', stateName: 'Chhattisgarh' },
  { code: '23', state: 'MP', stateName: 'Madhya Pradesh' },
  { code: '24', state: 'GJ', stateName: 'Gujarat' },
  { code: '25', state: 'DD', stateName: 'Daman and Diu' },
  { code: '26', state: 'DN', stateName: 'Dadra and Nagar Haveli' },
  { code: '27', state: 'MH', stateName: 'Maharashtra' },
  { code: '28', state: 'AP', stateName: 'Andhra Pradesh' },
  { code: '29', state: 'KA', stateName: 'Karnataka' },
  { code: '30', state: 'GA', stateName: 'Goa' },
  { code: '31', state: 'LD', stateName: 'Lakshadweep' },
  { code: '32', state: 'KL', stateName: 'Kerala' },
  { code: '33', state: 'TN', stateName: 'Tamil Nadu' },
  { code: '34', state: 'PY', stateName: 'Puducherry' },
  { code: '35', state: 'AN', stateName: 'Andaman and Nicobar Islands' },
  { code: '36', state: 'TS', stateName: 'Telangana' },
  { code: '37', state: 'AD', stateName: 'Andhra Pradesh' },
  { code: '38', state: 'LA', stateName: 'Ladakh' },
];

export const userParticipationConstMessages = {
  GET_USER_PARTICIPATION: 'Get comprehensive user participation data',
  GET_USER_PARTICIPATION_BY_YEAR: 'Get user participation for a specific year',
  GET_USER_PARTICIPATION_STATS: 'Get user participation statistics',
  USER_PARTICIPATION_RETRIEVED: 'User participation data retrieved successfully',
  USER_PARTICIPATION_BY_YEAR_RETRIEVED: 'User participation data for the year retrieved successfully',
  USER_PARTICIPATION_STATS_RETRIEVED: 'User participation statistics retrieved successfully',
  FILTERING_BY_YEAR: 'Filtering participation by year',
  FILTERING_BY_PROGRAM_NAME: 'Filtering participation by program name',
  BUILDING_YEAR_WISE_BREAKDOWN: 'Building year-wise participation breakdown',
  TRANSFORMING_DATA: 'Transforming participation data',
  FETCHING_PARTICIPATION_RECORDS: 'Fetching participation records',
  FETCHING_PARTICIPATION_STATS: 'Fetching participation statistics',
  CHECKING_USER_EXISTS: 'Checking if user has participation records',
  METHOD_GET_USER_PARTICIPATION_SUMMARY: 'getUserParticipationSummary',
  METHOD_GET_PROGRAM_PARTICIPATION_DETAILS: 'getProgramParticipationDetails',
  METHOD_GET_PARTICIPATION_STATISTICS: 'getParticipationStatistics',
  API_OPERATION_SUMMARY: 'Get user participation summary with total and year-wise data',
  API_PARAM_USER_ID: 'User ID to get participation data for',
  API_QUERY_YEAR: 'Filter by specific year (optional)',
  API_RESPONSE_SUCCESS: 'User participation summary retrieved successfully',
  ERROR_USER_NOT_FOUND: 'User not found',
  DETAILED_PROGRAMS_RETRIEVED: 'Detailed program participation retrieved successfully',
};

/**
 * Clear Registration Operation Messages Constants
 */
export const CLEAR_REGISTRATION_MESSAGES = {
  CLEAR_ALL_REQUESTED: 'Clear all registrations requested for program {0} by user {1}',
  CLEARING_REGISTRATIONS: 'Clearing all registrations for programId: {0} by userId: {1}',
  NO_ACTIVE_REGISTRATIONS: 'No active registrations found for programId: {0}',
  SOFT_DELETED_REGISTRATIONS: 'Soft deleted {0} registrations for programId: {1}',
  SUCCESSFULLY_CLEARED: 'Successfully cleared {0} registrations for program {1}',
  OPERATION_NOT_ALLOWED: 'Clear registrations operation is not allowed in {0} environment',
  OPERATION_NOT_ENABLED: 'Clear registrations operation is not enabled. Set ENABLE_CLEAR_REGISTRATIONS=true to enable.',
  ERROR_CLEARING: 'Error clearing registrations for program {0}:',
  CLEAR_COMPLETE: 'Completed clearing all registrations for programId: {0}',
  GET_COUNT_BEFORE_DELETE: 'Getting count of active registrations before deletion',
  USING_QUERY_BUILDER: 'Soft deleting registrations using QueryBuilder',
};
export const seekerExperienceRecordMessages = {
  CREATE_OPERATION: 'Create seeker experience record',
  UPDATE_OPERATION: 'Update seeker experience record',
  GET_OPERATION: 'Get seeker experience record by id',
  LIST_OPERATION: 'List seeker experience records with filters',
  DELETE_OPERATION: 'Soft delete seeker experience record',
  CREATED: 'Seeker experience record created successfully',
  UPDATED: 'Seeker experience record updated successfully',
  FETCHED: 'Seeker experience record retrieved successfully',
  LISTED: 'Seeker experience records retrieved successfully',
  DELETED: 'Seeker experience record deleted successfully',
  CREATE_REQUEST: (userId: number) => `Create seeker experience record for user ${userId}`,
  UPDATE_REQUEST: (id: number) => `Update seeker experience record ${id}`,
  FETCH_REQUEST: (id: number) => `Fetch seeker experience record ${id}`,
  LISTING: (limit?: number, offset?: number, search?: string) =>
    `List seeker experience records with limit ${limit}, offset ${offset}, search ${search ?? ''}`,
  DELETE_REQUEST: (id: number) => `Delete seeker experience record ${id}`,
  EXISTING_RECORD_FOUND: (id: number) => `Existing seeker experience record found with ID: ${id}`,
  ALREADY_DELETED: (id: number) => `Seeker experience record with ID: ${id} is already deleted`,
};

export const CLEARANCE_REASONS = {
  APPROVAL_STATUS_REJECTED: (registrationId: number) => `approval status changed to rejected - registration ${registrationId}`,
  APPROVAL_STATUS_ON_HOLD: (registrationId: number) => `approval status changed to on-hold - registration ${registrationId}`,
  PROGRAM_ALLOCATION_CHANGED: (registrationId: number) => `program allocation changed on approval - registration ${registrationId}`,
  REGISTRATION_CANCELLED: (registrationId: number) => `registration cancellation - registration ${registrationId}`,
  SWAP_REQUEST_ACCEPTED_OUTGOING: (registrationId: number) => `swap request accepted - outgoing seeker ${registrationId}`,
  SWAP_REQUEST_ACCEPTED_INCOMING: (registrationId: number) => `swap request accepted - incoming seeker ${registrationId}`,
  SWAP_REQUEST_ACCEPTED_MOVE: (registrationId: number) => `swap request accepted - move seeker ${registrationId}`,
  SWAP_REQUEST_ON_HOLD: (registrationId: number) => `swap request placed on hold - registration ${registrationId}`,
  SWAP_REQUEST_REJECTED: (registrationId: number) => `swap request rejected - registration ${registrationId}`,
};
