import { ConfigService } from '@nestjs/config';
import { TypeOrmModuleOptions } from '@nestjs/typeorm';
import { entities } from '../entities';
import * as fs from 'fs';
import { AuditHistoryLog } from '../entities/audit-history-log.entity';
import { AuditHistoryLogDetail } from '../entities/audit-history-log-detail.entity';

export const isProd = (configService: ConfigService): boolean => {
  return configService.get<string>('IS_PROD') === 'true' || false;
};

export const dbConfig = (configService: ConfigService): TypeOrmModuleOptions => {
  const isSsl = configService.get<string>('DB_SSL')?.toLowerCase() === 'true';
  let caString: string | undefined;
  if (isSsl) {
    const caPath = configService.get<string>('DB_SSL_CA');
    if (caPath) {
      try {
        const fileContent = fs.readFileSync(caPath, 'utf8');
        // If the file is base64 encoded, decode it; otherwise, use as is
        caString = fileContent.includes('-----BEGIN CERTIFICATE-----')
          ? fileContent
          : Buffer.from(fileContent, 'base64').toString('utf8');
      } catch (err) {
        throw new Error('Unable to read or decode DB SSL CA file');
      }
    }
  }

  return {
    type: (configService.get<string>('DB_TYPE') as 'postgres') || 'postgres',
    host: configService.get<string>('DB_HOST'),
    port: configService.get<number>('DB_PORT'),
    username: configService.get<string>('DB_USERNAME'),
    password: configService.get<string>('DB_PASSWORD'),
    database: configService.get<string>('DB_NAME'),
    entities: [...entities,
      AuditHistoryLog,
      AuditHistoryLogDetail
    ],
    autoLoadEntities: false,
    synchronize: false,
    ssl: isSsl
      ? {
          rejectUnauthorized: true,
          ca: caString,
        }
      : false,
  };
};

export const appConfig = (configService: ConfigService) => ({
  port: configService.get<number>('PORT') ?? 3000,
  corsOrigin: configService.get<string>('CORS_ORIGIN_URL') ?? '*',
  backendUrl: configService.get<string>('BACKEND_URL') ?? 'http://localhost:3000',
});

export const awsConfig = (configService: ConfigService) => {
  const prod = isProd(configService);
  const config: any = {
    region: configService.get<string>('AWS_REGION'),
  };
  if (!prod) {
    config.credentials = {
      accessKeyId: configService.get<string>('AWS_ACCESS_KEY_ID'),
      secretAccessKey: configService.get<string>('AWS_SECRET_ACCESS_KEY'),
    };
  }
  return config;
};

export const s3Config = (configService: ConfigService) => ({
  ...awsConfig(configService),
  bucket: configService.get<string>('AWS_S3_BUCKET_NAME'),
});

export const sqsConfig = (configService: ConfigService) => ({
  ...awsConfig(configService),
  queueUrl: configService.get<string>('AWS_SQS_QUEUE_URL'),
});

export const winstonCloudWatchTransportConfig = (configService: ConfigService) => {
  const prod = isProd(configService);
  const config: any = {
    logGroupName: configService.get<string>('CLOUDWATCH_LOG_GROUP'),
    logStreamName: () =>
      `${configService.get<string>('CLOUDWATCH_LOG_STREAM')}-${configService.get<string>('ENV') || 'dev'}-${new Date()
        .toISOString()
        .split('T')[0]}`, // stream per date
    awsRegion: configService.get<string>('AWS_REGION') || 'us-east-1',
    jsonMessage: true,
  };
  if (!prod) {
    config.awsAccessKeyId = configService.get<string>('AWS_ACCESS_KEY_ID');
    config.awsSecretKey = configService.get<string>('AWS_SECRET_ACCESS_KEY');
  }
  return config;
}

export const azureOpenAIConfig = (configService: ConfigService) => ({
  endpoint: configService.get<string>('AZURE_OPENAI_MODEL_ENDPOINT') || '',
  key: configService.get<string>('AZURE_OPENAI_MODEL_KEY') || '',
  apiVersion: configService.get<string>('AZURE_OPENAI_MODEL_API_VERSION') || '2024-04-01-preview',
});

export const logConfig = (configService: ConfigService) => ({
  log: configService.get<string>('LOG_CLOUDWATCH') === 'true' || false,
});
