import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { UserSession, User } from 'src/common/entities';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { UserSessionStatusEnum } from 'src/common/enum/user-session-status.enum';

@Injectable()
export class AuthRepository {
  constructor(
    @InjectRepository(UserSession)
    private readonly sessionRepo: Repository<UserSession>,
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
    private readonly logger: AppLoggerService,
  ) {}

  async createSession(data: Partial<UserSession>): Promise<UserSession> {
    try {
      const entity = this.sessionRepo.create(data);
      return await this.sessionRepo.save(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_SESSION_SAVE_FAILED, error);
    }
  }

  async findLatestSession(
    phoneNumber?: string,
    countryCode?: string,
    email?: string,
  ): Promise<UserSession | null> {
    try {
      const qb = this.sessionRepo
        .createQueryBuilder('session')
        .leftJoinAndSelect('session.user', 'user')
        .orderBy('session.loginAt', 'DESC');
      if (phoneNumber && countryCode) {
        qb.where('session.phoneNumber = :phoneNumber', { phoneNumber })
          .andWhere('session.countryCode = :countryCode', { countryCode });
      } else if (email) {
        qb.where('session.email = :email', { email });
      } else {
        throw new Error('Either phoneNumber and countryCode or email must be provided');
      }
      qb.andWhere('session.sessionStatus = :status', {
        status: UserSessionStatusEnum.INITIATED,
      });
      return await qb.getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_SESSION_NOTFOUND, error);
    }
  }

  async findByJwtToken(userId: string, token: string): Promise<UserSession | null> {
    try {
      const id = parseInt(userId, 10);
      if (isNaN(id)) {
        return null;
      }
      const result = await this.sessionRepo.findOne({
        where: {
          userId: id,
          jwtToken: token,
          sessionStatus: UserSessionStatusEnum.VALIDATED,
        },
        relations: ['user', 'user.userRoleMaps', 'user.userRoleMaps.role'],
      });
      return result;
    } catch (error){
      return null;
    }
  }

  async updateSession(id: number, updates: Partial<UserSession>) {
    try {
      await this.sessionRepo.update({ id }, updates);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_SESSION_UPDATE_FAILED, error);
    }
  }
}
