import { Test, TestingModule } from '@nestjs/testing';
import { getRepositoryToken } from '@nestjs/typeorm';
import { AuditHistoryService } from '../services/audit-history.service';
import { UserService } from '../../user/user.service';
import { AuditHistoryLog } from 'src/common/entities/audit-history-log.entity';

describe('AuditHistoryService', () => {
  let service: AuditHistoryService;

  const mockRepository = {
    create: jest.fn(),
    save: jest.fn(),
    find: jest.fn(),
  };

  const mockUserService = {
    getUserById: jest.fn(),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        AuditHistoryService,
        {
          provide: getRepositoryToken(AuditHistoryLog),
          useValue: mockRepository,
        },
        {
          provide: UserService,
          useValue: mockUserService,
        },
      ],
    }).compile();

    service = module.get<AuditHistoryService>(AuditHistoryService);
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  describe('createAuditLog', () => {
    it('should create and save an audit log', async () => {
      const mockLog = {
        entityType: 'User',
        entityName: 'User',
        entityId: '123',
        action: 'INSERT',
      };

      const mockEntity = { id: '1', ...mockLog };
      const mockSavedEntity = { id: '1', ...mockLog };

      mockRepository.create.mockReturnValue(mockEntity);
      mockRepository.save.mockResolvedValue(mockSavedEntity);

      const result = await service.createAuditLog(mockLog);

      expect(mockRepository.create).toHaveBeenCalledWith(mockLog);
      expect(mockRepository.save).toHaveBeenCalledWith(mockEntity);
      expect(result).toEqual(mockSavedEntity);
    });
  });

  describe('getIntelligentActivityLog', () => {
    it('should return intelligent activity log with empty activities when no logs found', async () => {
      mockRepository.find.mockResolvedValue([]);

      const result = await service.getIntelligentActivityLog('123', {});

      expect(result).toEqual({
        data: [],
        totalActivities: 0,
        dateRange: {
          startDate: undefined,
          endDate: undefined,
        },
      });
    });

    it('should process logs and return intelligent activity items', async () => {
      const mockLogs = [
        {
          id: '1',
          entityType: 'ProgramRegistration',
          action: 'INSERT',
          userId: '123',
          requestId: 'req-1',
          createdAt: new Date('2024-01-01T10:00:00Z'),
          details: [
            {
              fieldName: 'registrationStatus',
              oldValue: null,
              newValue: 'save_as_draft',
            },
          ],
        },
      ];

      mockRepository.find.mockResolvedValue(mockLogs);
      mockUserService.getUserById.mockResolvedValue({
        fullName: 'Test User',
      });

      const result = await service.getIntelligentActivityLog('123', {});

      expect(result.data).toHaveLength(1);
      expect(result.data[0].message).toContain('Registration');
      expect(result.data[0].performedByName).toBe('Test User');
      expect(result.totalActivities).toBe(1);
    });
  });
});
