import { modifyWebinarData } from "../controllers"; // Import modifyWebinarData from socket controllers
import { AppDataSource, initiateClientConnection } from "../config/dbConfig"; // Import the AppDataSource
import { Users, Webinar, WebinarRegistration } from "../../src/entities"; // Import the Webinar entity
import { IsNull, Raw } from "typeorm"; // import IsNull and Raw date format from typeorm
import { createLogger } from "../utils/winstonLogger"; // Import the logger function
const logger = createLogger; // create a logger instance

/**
 * Retrieves a list of upcoming webinars with relevant details.
 *
 * @returns {Promise<any>} A promise resolving to the modified webinar data.
 * @throws {Error} If an error occurs while fetching webinar data.
 */
export const webinarInfo = async () => {
  logger.info(`webinarInfo service function`);
  await initiateClientConnection();
  let currentDateTime: any = new Date();
  // UTC to IST
  currentDateTime = new Date(currentDateTime.getTime() + 5.5 * 60 * 60 * 1000); // Assuming currentDateTime is the current date and time
  try {
    // Get the repository for the Webinar entity
    const webinarRepository = AppDataSource.getRepository(Webinar);
    // Use the query builder to fetch webinar data
    const webinarData = await webinarRepository
      .createQueryBuilder("webinar")
      .select([
        "webinar.id",
        "webinar.registrationStartsAt",
        "webinar.registrationEndsAt",
        "webinar.startDate",
        "webinar.startAt",
        "webinar.actualMeetingEndsAt",
        "webinar.duration",
        "webinar.title",
        "webinar.joinEnableTime",
        "webinar.maxVideoLimit",
        "webinar.maxNonVideoLimit",
        "webinar.endDate",
      ])
      .where("webinar.actualMeetingEndsAt IS NULL")
      .andWhere("webinar.endDate > :currentDateTime", { currentDateTime })
      .orderBy("webinar.endDate", "ASC")
      .getMany();
    // logger.info(`webinarInfo webinarData object data: ${webinarData}`);
    return modifyWebinarData(webinarData); // Modify and return the webinar data
  } catch (error) {
    logger.error(`webinarInfo Error fetching webinar data: ${error}`);
    throw error;
  }
};

/**
 * Fetches webinar data by its ID.
 *
 * @param {number} webinarId - The ID of the webinar.
 * @returns {Promise<Webinar | null>} The webinar data or null if not found.
 */
export const getWebinarDataById = async (webinarId: number) => {
  logger.info(`getWebinarDataById service funtion data: ${webinarId}`);
  await initiateClientConnection();
  try {
    const webinar = await AppDataSource.getRepository(Webinar).findOneBy({
      id: webinarId,
      deletedAt: IsNull(),
    });
    logger.info(`getWebinarDataById webinar data: ${JSON.stringify(webinar, null, 2)}`);
    return webinar;
  } catch (error) {
    logger.error(`getWebinarDataById Error while retriving webinar data: ${error}`);
    return null;
  }
};

/**
 * Retrieves the number of panelists and attendees registered for a webinar.
 *
 * @param {number} webinarId - The ID of the webinar.
 * @returns {Promise<{ panelistCount: number; attendeeCount: number }>} 
 */
export const getWebinarRegistrationCounts = async (webinarId: number) => {
  try {
    logger.info(`getWebinarRegistrationCounts service funtion data: ${webinarId}`);
    await initiateClientConnection();
    const webinarRegistrationRepository =
      AppDataSource.getRepository(WebinarRegistration);

    const panelistCount = await webinarRegistrationRepository
      .createQueryBuilder("webinar_registration")
      .where("webinar_registration.webinar_id = :webinarId", { webinarId })
      .andWhere("webinar_registration.is_panelist = true")
      .andWhere("webinar_registration.deleted_at IS NULL")
      .getCount();

    const nonPanelistCount = await webinarRegistrationRepository
      .createQueryBuilder("webinar_registration")
      .where("webinar_registration.webinar_id = :webinarId", { webinarId })
      .andWhere("webinar_registration.is_panelist = false")
      .andWhere("webinar_registration.deleted_at IS NULL")
      .getCount();

    logger.info(`getWebinarRegistrationCounts panelistCount data: ${panelistCount} and nonPanelistCount data: ${nonPanelistCount}`);
    return {
      panelistCount: panelistCount,
      attendeeCount: nonPanelistCount,
    };
  } catch (error) {
    logger.error(`getWebinarRegistrationCounts Error getting webinar registration counts:: ${error}`);
    return {
      panelistCount: 0,
      attendeeCount: 0,
    };
  }
};

/**
 * Retrieves user data by user ID.
 *
 * @param {number} userId - The ID of the user.
 * @returns {Promise<any>} The user data or null if not found.
 */
export const getUserData = async (userId: number) => {
  try {
    logger.info(`getUserData service funtion data: ${userId}`);
    await initiateClientConnection();
    return await AppDataSource.getRepository(Users)
      .createQueryBuilder()
      .select([
        "id as id",
        "first_name as first_name",
        "last_name as last_name",
        "email as email",
        "role as role",
        "phone_number as phone_number",
        "full_name as full_name",
      ])
      .where("id = :id", { id: userId })
      .andWhere("is_infipath = :user", { user: true })
      .andWhere("is_infipath_user_active = :status", { status: true })
      .getRawOne();
  } catch (err) {
    logger.error(`getUserData Error fetching user data: ${err}`);
    return null;
  }
};

/**
 * Retrieves webinar data by webinar ID.
 *
 * @param {number} id - The ID of the webinar.
 * @returns {Promise<any>} The webinar data or null if not found.
 */
export const getUserWebinarData = async (id: number) => {
  try {
    logger.info(`getUserWebinarData service funtion data: ${id}`);
    const currentDateTime = new Date(
      new Date().getTime() + 5.5 * 60 * 60 * 1000
    );
    const webinarData = await AppDataSource.getRepository(Webinar)
      .createQueryBuilder()
      .select([
        "id as id",
        "registration_starts_at as registration_starts_at",
        "registration_ends_at as registration_ends_at",
        "start_date as start_date",
        "start_at as start_at",
        "actual_meeting_ends_at as actual_meeting_ends_at",
        "duration as duration",
        "title as title",
        'join_enable_time as "joinEnableTime"',
        'max_video_limit as "max_video_limit"',
        'max_non_video_limit as "max_non_video_limit"',
        'end_date as "end_date"',
      ])
      .where("id = :id", { id }) // Add condition to filter by id
      .andWhere("actual_meeting_ends_at IS NULL")
      .andWhere("end_date > :currentDateTime", { currentDateTime })
      .orderBy("end_date", "ASC")
      .getRawOne();
      logger.info(`getUserWebinarData webinarData data:  ${JSON.stringify(webinarData, null, 2)}`);
    return webinarData;
  } catch (error) {
    logger.error(`getUserWebinarData Error fetching meeting data:${error}`);
    return null;
  }
};
/**
 * Retrieves expired webinar data by webinar ID.
 *
 * @param {number} id - The ID of the webinar.
 * @returns {Promise<any>} The webinar data or null if not found.
 */
export const getExpiredUserWebinarData = async (id: number) => {
  try {
    logger.info(`getExpiredUserWebinarData service funtion data: ${id}`);
    const webinarData = await AppDataSource.getRepository(Webinar)
      .createQueryBuilder()
      .select([
        "id as id",
        "registration_starts_at as registration_starts_at",
        "registration_ends_at as registration_ends_at",
        "start_date as start_date",
        "start_at as start_at",
        "actual_meeting_ends_at as actual_meeting_ends_at",
        "duration as duration",
        "title as title",
        'join_enable_time as "joinEnableTime"',
        'max_video_limit as "max_video_limit"',
        'max_non_video_limit as "max_non_video_limit"',
        'end_date as "end_date"',
      ])
      .where("id = :id", { id }) // Add condition to filter by id
      .andWhere("actual_meeting_ends_at IS NULL")
      .orderBy("end_date", "ASC")
      .getRawOne();

    logger.info(`getExpiredUserWebinarData webinarData: ${JSON.stringify(webinarData, null, 2)}`);
    return webinarData;
  } catch (error) {
    logger.error(`getExpiredUserWebinarData Error fetching meeting data: ${error}`);
    return null;
  }
};
/**
 * Fetches a user's webinar details.
 *
 * @param {string} userId - The user's ID.
 * @param {number} meetingId - The meeting ID.
 * @returns {Promise<any>} The webinar details or null if not found.
 */
export const getUserWebinarDetails = async (
  userId: string,
  meetingId: number
) => {
  try {
    logger.info(`getUserWebinarDetails service funtion userId data: ${userId} and meetingId data: ${meetingId}`);
    await initiateClientConnection();
    const data = AppDataSource.getRepository(WebinarRegistration)
      .createQueryBuilder("webinar_registration")
      .select([
        "webinar_registration.is_panelist as is_panelist",
        "webinar_registration.join_url as join_url",
        "deleted_at as deleted_at",
      ])
      .where("webinar_registration.user_id = :id", { id: userId })
      .andWhere("webinar_registration.webinar_id = :meetingId", {
        meetingId,
      })
      // .andWhere('webinar_registration.deleted_at IS null')
      .orderBy("webinar_registration.created_at", "DESC");
    logger.info(`getUserWebinarDetails sql data: ${data.getSql()}`);
    return await data.getRawOne();
  } catch (err) {
    logger.error(`getUserWebinarDetails Error fetching user meeting details: ${err}`);
    return null;
  }
};

/**
 * Gets the number of registration attempts a user has made for a webinar.
 *
 * @param {number} userId - The user's ID.
 * @param {number} webinarId - The webinar's ID.
 * @returns {Promise<{ panelist_attempts: number; attendee_attempts: number }>} 
 */
export const getUserWebinarRegistrationAttempts = async (
  userId: number,
  webinarId: number
) => {
  try {
    logger.info(`getUserWebinarRegistrationAttempts service funtion userId data: ${userId} and webinarId data: ${webinarId}`);
    await initiateClientConnection();
    const results = await AppDataSource.getRepository(WebinarRegistration).find(
      {
        where: {
          userId: userId,
          webinarId: webinarId,
          createdAt: Raw(() => "DATE(created_at) = CURRENT_DATE"),
        },
        select: {
          isPanelist: true,
        },
      }
    );
    const panelistAttempts = results.filter(
      (registration) => registration.isPanelist
    ).length;
    const attendeeAttempts = results.filter(
      (registration) => !registration.isPanelist
    ).length;
    logger.info(`getUserWebinarRegistrationAttempts panelist_attempts data:${panelistAttempts} and attendee_attempts data: ${attendeeAttempts}`);
    return {
      panelist_attempts: panelistAttempts,
      attendee_attempts: attendeeAttempts,
    };
  } catch (err) {
    logger.error(`getUserWebinarRegistrationAttempts Error fetching webinar users KPI data error: ${err}`);
    return {
      attendee_attempts: 0,
      panelist_attempts: 0,
    };
  }
};
