import {
  Column,
  CreateDateColumn,
  Entity,
  PrimaryGeneratedColumn,
  UpdateDateColumn,
  ManyToOne,
  JoinColumn,
} from "typeorm";
import { WebinarRegistration } from "./webinar_registration";
/**
 * Entity definition for webinar registration transaction table.
 * Represents a webinar registration transaction details in the system
 */
@Entity({ name: "webinar_registration_transaction" })
export class WebinarRegistrationTransaction {
  @PrimaryGeneratedColumn()
  id: number;

  @Column({ name: "webinar_registration_id" })
  webinarRegistrationId: number;

  @Column({
    name: "registration_id_ext",
    type: "varchar",
    length: 255,
    nullable: true,
  })
  registrationIdExt: string;

  @Column({ name: "is_downgraded", type: "boolean", default: false })
  isDowngraded: boolean;

  @Column({ name: "is_upgraded", type: "boolean", default: false })
  isUpgraded: boolean;

  @CreateDateColumn({
    name: "created_at",
    type: "timestamptz",
    default: () => "CURRENT_TIMESTAMP",
  })
  createdAt: Date;

  @UpdateDateColumn({
    name: "updated_at",
    type: "timestamptz",
    default: () => "CURRENT_TIMESTAMP",
    onUpdate: "CURRENT_TIMESTAMP",
  })
  updatedAt: Date;

  @Column({ name: "is_cancelled", type: "boolean", default: false })
  isCancelled: boolean;
  // Relationships
  @ManyToOne(() => WebinarRegistration)
  @JoinColumn({ name: "webinar_registration_id" })
  webinarRegistration: WebinarRegistration;

  /**
   * Represents a constructor for a webinar registration object.
   *
   * @constructor
   * @param {number} webinarRegistrationId - The unique identifier for the webinar registration.
   * @param {string} registrationIdExt - An external reference or extension ID associated with the registration.
   * @param {boolean} isDowngraded - Indicates whether the registration has been downgraded (true if downgraded, false otherwise).
   * @param {boolean} isUpgraded - Indicates whether the registration has been upgraded (true if upgraded, false otherwise).
   */
  constructor(
    webinarRegistrationId: number,
    registrationIdExt: string,
    isDowngraded: boolean,
    isUpgraded: boolean
  ) {
    this.webinarRegistrationId = webinarRegistrationId;
    this.registrationIdExt = registrationIdExt;
    this.isDowngraded = isDowngraded;
    this.isUpgraded = isUpgraded;
  }
}
