import {
  Column,
  CreateDateColumn,
  Entity,
  PrimaryGeneratedColumn,
  UpdateDateColumn,
  ManyToOne,
  JoinColumn,
} from "typeorm";
import { Users } from "./users";
import { Webinar } from "./webinar";
/**
 * Entity definition for webinar registration table.
 * Represents a webinar registration details in the system
 */
@Entity({ name: "webinar_registration" })
export class WebinarRegistration {
  @PrimaryGeneratedColumn()
  id: number;

  @Column({ name: "user_id" })
  userId: number;

  @Column({ name: "webinar_id" })
  webinarId: number;

  @Column({ name: "join_url", type: "varchar", length: 255, nullable: true })
  joinUrl: string;

  @Column({ name: "is_panelist", type: "boolean", default: false })
  isPanelist: boolean;

  @Column({
    name: "registration_id_ext",
    type: "varchar",
    length: 255,
    nullable: true,
  })
  registrationIdExt: string;

  @CreateDateColumn({
    name: "created_at",
    type: "timestamptz",
    default: () => "CURRENT_TIMESTAMP",
  })
  createdAt: Date;

  @UpdateDateColumn({
    name: "updated_at",
    type: "timestamptz",
    default: () => "CURRENT_TIMESTAMP",
    onUpdate: "CURRENT_TIMESTAMP",
  })
  updatedAt: Date;

  @Column({ name: "deleted_at", type: "timestamptz", nullable: true })
  deletedAt: Date;

  // Relationships
  @ManyToOne(() => Users)
  @JoinColumn({ name: "user_id" })
  user: Users;

  @ManyToOne(() => Webinar, (webinar) => webinar.webinarRegistrations)
  @JoinColumn({ name: "webinar_id" })
  webinar: Webinar;

  /**
   * Represents a constructor for a webinar attendee or panelist object.
   *
   * @constructor
   * @param {number} userId - The unique identifier of the user attending the webinar.
   * @param {number} webinarId - The unique identifier of the webinar being attended.
   * @param {string} joinUrl - The URL that allows the user to join the webinar.
   * @param {boolean} isPanelist - Indicates whether the user is a panelist (true if panelist, false otherwise).
   * @param {string} registrationIdExt - An external reference or extension ID associated with the user's registration.
   */
  constructor(
    userId: number,
    webinarId: number,
    joinUrl: string,
    isPanelist: boolean,
    registrationIdExt: string
  ) {
    this.userId = userId;
    this.webinarId = webinarId;
    this.joinUrl = joinUrl;
    this.isPanelist = isPanelist;
    this.registrationIdExt = registrationIdExt;
  }
}
