import {
  Column,
  CreateDateColumn,
  Entity,
  OneToMany,
  PrimaryGeneratedColumn,
  UpdateDateColumn,
} from "typeorm";
import { WebinarRegistration } from "./webinar_registration";
/**
 * Entity definition for webinar table.
 * Represents a webinar details in the system
 */
@Entity({ name: "webinar" })
export class Webinar {
  @PrimaryGeneratedColumn()
  id: number;

  @Column()
  title: string;

  @Column({ name: "program_session_id" })
  programSessionId: number;

  @Column({ name: "webinar_id_ext" })
  webinarIdExt: string;

  @Column()
  password: string;

  @Column({ name: "start_date" })
  startDate: Date;

  @Column({ name: "start_url" })
  startUrl: string;

  @Column()
  duration: number;

  @Column({ name: "end_date" })
  endDate: Date;

  @Column({ name: "start_at" })
  startAt: Date;

  @Column({ name: "started_by" })
  startedBy: string;

  @Column({ name: "created_by" })
  createdBy: string;

  @Column({ name: "registration_starts_at" })
  registrationStartsAt: Date;

  @Column({ name: "registration_ends_at" })
  registrationEndsAt: Date;

  @Column({ type: "json" })
  meta: object;

  @CreateDateColumn({ type: "timestamptz", default: () => "CURRENT_TIMESTAMP" })
  created_at: Date;

  @UpdateDateColumn({
    type: "timestamptz",
    default: () => "CURRENT_TIMESTAMP",
    onUpdate: "CURRENT_TIMESTAMP",
  })
  updated_at: Date;

  @Column({ name: "deleted_at" })
  deletedAt: Date;

  @Column({ name: "max_video_limit" })
  maxVideoLimit: number;

  @Column({ name: "max_non_video_limit" })
  maxNonVideoLimit: number;

  @Column({ name: "actual_meeting_ends_at" })
  actualMeetingEndsAt: Date;

  @Column({ name: "video_limit_count" })
  videoLimitCount: number;

  @Column({ name: "non_video_limit_count" })
  nonVideoLimitCount: number;
  @Column({ name: "join_enable_time" })
  joinEnableTime: number;

  @Column({ name: "webinar_status" })
  webinarStatus: string;

  @OneToMany(() => WebinarRegistration, (registration) => registration.webinar)
  webinarRegistrations: WebinarRegistration[];

  /**
   * Represents a constructor for a webinar session object.
   *
   * @constructor
   * @param {string} title - The title of the webinar.
   * @param {number} programSessionId - The unique identifier of the program session.
   * @param {string} webinarIdExt - The external reference ID for the webinar.
   * @param {string} password - The password required to join the webinar.
   * @param {Date} startDate - The scheduled start date and time of the webinar.
   * @param {string} startUrl - The URL to access the webinar.
   * @param {number} duration - The duration of the webinar in minutes.
   * @param {Date} endDate - The scheduled end date and time of the webinar.
   * @param {Date} startAt - The actual start time of the webinar.
   * @param {string} startedBy - The identifier or name of the person who started the webinar.
   * @param {string} createdBy - The identifier or name of the person who created the webinar session.
   * @param {Date} registrationStartsAt - The date and time when registration for the webinar opens.
   * @param {Date} registrationEndsAt - The date and time when registration for the webinar closes.
   * @param {object} meta - Additional metadata related to the webinar session.
   * @param {Date} deletedAt - The date and time when the webinar session was deleted, if applicable.
   * @param {number} maxVideoLimit - The maximum number of video participants allowed in the webinar.
   * @param {number} maxNonVideoLimit - The maximum number of non-video participants allowed in the webinar.
   * @param {Date} actualMeetingEndsAt - The actual time the webinar ends.
   * @param {number} videoLimitCount - The current number of video participants in the webinar.
   * @param {number} nonVideoLimitCount - The current number of non-video participants in the webinar.
   * @param {number} joinEnableTime - The time in minutes before the webinar starts when participants can join.
   * @param {string} webinarStatus - The current status of the webinar (e.g., 'scheduled', 'started', 'ended').
   */

  constructor(
    title: string,
    programSessionId: number,
    webinarIdExt: string,
    password: string,
    startDate: Date,
    startUrl: string,
    duration: number,
    endDate: Date,
    startAt: Date,
    startedBy: string,
    createdBy: string,
    registrationStartsAt: Date,
    registrationEndsAt: Date,
    meta: object,
    deletedAt: Date,
    maxVideoLimit: number,
    maxNonVideoLimit: number,
    actualMeetingEndsAt: Date,
    videoLimitCount: number,
    nonVideoLimitCount: number,
    joinEnableTime: number,
    webinarStatus: string
  ) {
    this.title = title;
    this.programSessionId = programSessionId;
    this.webinarIdExt = webinarIdExt;
    this.password = password;
    this.startDate = startDate;
    this.startUrl = startUrl;
    this.duration = duration;
    this.endDate = endDate;
    this.startAt = startAt;
    this.startedBy = startedBy;
    this.createdBy = createdBy;
    this.registrationStartsAt = registrationStartsAt;
    this.registrationEndsAt = registrationEndsAt;
    this.meta = meta;
    this.deletedAt = deletedAt;
    this.maxVideoLimit = maxVideoLimit;
    this.maxNonVideoLimit = maxNonVideoLimit;
    this.actualMeetingEndsAt = actualMeetingEndsAt;
    this.videoLimitCount = videoLimitCount;
    this.nonVideoLimitCount = nonVideoLimitCount;
    this.joinEnableTime = joinEnableTime;
    this.webinarStatus = webinarStatus;
  }
}
