import express from "express"; // Import express framework
import http from "http"; // Import Node.js HTTP module
import dotenv from "dotenv"; // Import dotenv for environment variables
import path from "path"; // Import path module
import cors from "cors"; // Import CORS middleware
import { Server } from "socket.io"; // Import Socket.IO server
import { setRoutes } from "./routes/index"; // Import route setting function
import { handleRegisterWebinar, modifySingleWebinarData } from "./controllers"; // Import the controller functions
import { initiateClientConnection } from "./config/dbConfig"; // Import the database initialization function
import { authenticateSocket } from "../src/middleware/auth"; // Import the authenticate socket function
import { createLogger } from "./utils/winstonLogger"; // Import the logger function
// import { emitWebinarData } from "./utils/helpers";

dotenv.config(); // Load environment variables from .env file
const app = express(); // Create an Express application
const server = http.createServer(app); // Create an HTTP server
server.timeout = 0; // Disable the server timeout
const logger = createLogger; // create a logger instance

export const io = new Server(server, {
  cors: {
    origin: "*", // Allow all origins
    methods: ["GET", "POST"], // Allow specific methods
    allowedHeaders: ["Content-Type", "token", "userid"], // Allow specific headers
    credentials: true, // Allow credentials
  },
  pingInterval: 35000, // Send a ping to the client every 35 seconds
  pingTimeout: 30000, // receive a ping from the client every 30 seconds
});

app.use(cors()); // Enable CORS
app.use(express.json()); // Parse JSON request bodies
app.use(express.static(path.join(__dirname, "../public"))); // Serve static files from the public directory

setRoutes(app); // Set up routes

let connectedUsers = 0; // Initialize a counter for connected users
const userSocketMap = new Map(); // Map to store user ID and socket ID
io.use(authenticateSocket); // Socket connection authentication
io.on("connection", (socket) => {
  // socket event connection establish for an event
  // Store the user ID and socket ID mapping
  socket.on("registerUser", (userId) => {
    logger.info(`Register User`);
    if (!userSocketMap.has(userId)) {
      userSocketMap.set(userId, []); // Initialize an array for the user ID if not already present
    }
    const socketIds = userSocketMap.get(userId);
    if (!socketIds.includes(socket.id)) {
      socketIds.push(socket.id); // Add the socket ID only if it doesn't already exist
      connectedUsers++; // Increment the counter when a user connects with a new device
    }
    logger.info(`Connected users: ${connectedUsers}`); // Log the number of connected users
    logger.info(`userSocketMap data : ${JSON.stringify(Object.fromEntries(userSocketMap))}`);
    logger.info(`User ${userId} registered with socket ID ${socket.id}`);
  });

  // Update the Webinar details such as registration start, registration end, webinar join reminder, start and ended.
  socket.on("webinarData", async (data) => {
    try {
      logger.info(`Webinar data updation : ${JSON.stringify(data, null, 2)}`);
      await modifySingleWebinarData(data.id, userSocketMap);
    } catch (error) {
      logger.error(`Error updating webinar data at webinar creation: ${error}`);
      socket.emit("error", { message: "Failed to update webinar data" });
    }
  });

  // Register the user for that webinar
  socket.on("registerWebinar", async (data) => {
    try {
      logger.info(`Register webinar data: ${JSON.stringify(data, null, 2)}`);
      await handleRegisterWebinar(data, userSocketMap, socket);
    } catch (error) {
      logger.error(`Error registering webinar data: ${error}`);
      socket.emit("error", { message: "Failed to register webinar" });
    }
  });

  // Disconnect the socket
  socket.on("disconnect", (reason) => {
    logger.info(`Connection disconnected`);
    // console.log(`Client disconnected Reason: ${reason}`);
    logger.info(`Client disconnected Reason: ${reason}`)
    // Remove the user ID and socket ID mappinß
    for (const [userId, socketIds] of userSocketMap.entries()) {
      const index = socketIds.indexOf(socket.id);
      if (index !== -1) {
        socketIds.splice(index, 1); // Remove the disconnected socket ID from the array
        connectedUsers--; // Decrement the counter when a user disconnects from a device
        if (socketIds.length === 0) {
          userSocketMap.delete(userId); // Remove the user entry if no socket IDs are left
          logger.info(`User ${userId} disconnected and removed from map`);
        } else {
          userSocketMap.set(userId, socketIds); // Update the map with the remaining socket IDs
          logger.info(`Socket ID ${socket.id} for user ${userId} disconnected`);
        }
        break;
      }
    }
    logger.info(`Connected users after disconnect: ${connectedUsers}`); // Log the number of connected users
    logger.info(`existing userSocketMap data : ${JSON.stringify(Object.fromEntries(userSocketMap))}`);
  });
});
const PORT = process.env.PORT || 3200; // Get port from environment or default to 3200
// Global error handler
app.use((err: any, req: express.Request, res: express.Response, next: express.NextFunction) => {
  logger.error(`Global error handler ${err}`);
  res.status(500).send('Something broke!');
});
// setInterval(emitWebinarData, 60000);
// Initialize the database connection
initiateClientConnection()
  .then(() => {
    server.listen(PORT, () => { // running on given port 
      logger.info(`Server is running on port ${PORT}`); // log the server start
    });
  })
  .catch((error) => {
    logger.error(`Failed to initialize database connection: ${error}`);
  });
